library(xpectr)
context("gxs_function()")

test_that("gxs_function() works", {

  # Some arbitrary function
  # Have constraints on the arguments
  # and will output slightly different things depending on
  # the values of the args
  fn <- function(a, a_fn, b, d = NULL, e = 3, f = "hi"){

    # Check arguments ####
    assert_collection <- checkmate::makeAssertCollection()
    checkmate::assert_number(x = a, lower = 4, upper = 6, add = assert_collection)
    checkmate::assert_function(x = a_fn, add = assert_collection)
    checkmate::assert_character(x = b, min.chars = 3,
                                any.missing = FALSE,
                                len = 2,
                                add = assert_collection)
    checkmate::assert_data_frame(
      x = d,
      min.rows = 3,
      min.cols = 1,
      col.names = "named",
      null.ok = TRUE,
      add = assert_collection
    )
    checkmate::assert_count(
      x = e,
      positive = TRUE,
      null.ok = TRUE,
      add = assert_collection
    )
    checkmate::assert_string(x = f, min.chars = 2, add = assert_collection)
    checkmate::reportAssertions(assert_collection)
    # End of argument checks ####

    n <- a_fn(a * e)
    m <- paste0(f, ": ", paste0(b, collapse = ", "))

    if (!is.null(d)){
      d[["column"]] <- n
      output <- list("d" = d, "m" = m)
    } else {
      output <- list("n" = n, "m" = m)
    }

    output
  }

  current_envir <- sys.frame(which = sys.nframe())

  set_test_seed(1)
  # All errors ####
  expected_tests <- c(
    " ",
    "## Testing 'fn'                                                             ####",
    "## Initially generated by xpectr",
    "# Testing different combinations of argument values",
    " ",
    "# Testing fn(a = 1, a_fn = mean, b = \"ah\")",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_12655 <- xpectr::capture_side_effects(fn(a = 1, a_fn = mean, b = \"ah\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12655[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12655[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 2, a_fn = mean, b = \"ah\")",
    "# Changed from baseline: a = 2",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_13721 <- xpectr::capture_side_effects(fn(a = 2, a_fn = mean, b = \"ah\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = NULL, a_fn = mean, b = \"ah\")",
    "# Changed from baseline: a = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_15728 <- xpectr::capture_side_effects(fn(a = NULL, a_fn = mean, b = \"ah\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Must be of type 'number', not 'NULL'.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 1, a_fn = sum, b = \"ah\")",
    "# Changed from baseline: a_fn = sum",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19082 <- xpectr::capture_side_effects(fn(a = 1, a_fn = sum, b = \"ah\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 1, a_fn = NULL, b = \"ah\")",
    "# Changed from baseline: a_fn = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_12016 <- xpectr::capture_side_effects(fn(a = 1, a_fn = NULL, b = \"ah\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['error']]),\n  xpectr::strip(\"3 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'a_fn': Must be a function, not 'NULL'.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 1, a_fn = mean, b = \"eh\")",
    "# Changed from baseline: b = \"eh\"",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_18983 <- xpectr::capture_side_effects(fn(a = 1, a_fn = mean, b = \"eh\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_18983[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'b': Must have length 2, but has length 1.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_18983[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 1, a_fn = mean, b = NULL)",
    "# Changed from baseline: b = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19446 <- xpectr::capture_side_effects(fn(a = 1, a_fn = mean, b = NULL), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['error']]),\n  xpectr::strip(\"2 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\\n * Variable 'b': Must be of type 'character', not 'NULL'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "## Finished testing 'fn'                                                    ####",
    " "
  )

  expect_equal(
    strip(gxs_function(fn, list(
        "a" = list(1, 2),
        a_fn = list(mean, sum),
        "b" = list("ah", "eh")
      ),
      out = "return")),
    strip(expected_tests),
    fixed = TRUE)
  expect_equal(
  length(gxs_function(fn, list(
        "a" = list(1, 2),
        a_fn = list(mean, sum),
        "b" = list("ah", "eh")
      ),
      out = "return") ),
  69L)

  eval_expectations(expected_tests, envir = current_envir)


  # Good baseline values  ####

  # Set seed as assignment names are random
  set_test_seed(1)

  expected_tests2 <- c(
    " ",
    "## Testing 'fn'                                                             ####",
    "## Initially generated by xpectr",
    "# Testing different combinations of argument values",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_12655 <- fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\")",
    "# Testing class",
    "expect_equal(\n  class(output_12655),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_12655,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_12655,\n  list(n = 15, m = \"hii: ahhh, ohhh\"))",
    "# Testing names",
    "expect_equal(\n  names(output_12655),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_12655),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_12655)),\n  2L)",
    " ",
    "# Testing fn(a = 3, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: a = 3",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_13721 <- xpectr::capture_side_effects(fn(a = 3, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'a': Element 1 is not >= 4.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 7, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: a = 7",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_15728 <- xpectr::capture_side_effects(fn(a = 7, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'a': Element 1 is not <= 6.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = NA, a_fn = mean, b = c(\"ahhh\", \"ohhh\"),...",
    "# Changed from baseline: a = NA",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19082 <- xpectr::capture_side_effects(fn(a = NA, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'a': May not be NA.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = NULL, a_fn = mean, b = c(\"ahhh\", \"ohhh\"...",
    "# Changed from baseline: a = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_12016 <- xpectr::capture_side_effects(fn(a = NULL, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'a': Must be of type 'number', not 'NULL'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = sum, b = c(\"ahhh\", \"ohhh\"), d...",
    "# Changed from baseline: a_fn = sum",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_18983 <- fn(a = 5, a_fn = sum, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\")",
    "# Testing class",
    "expect_equal(\n  class(output_18983),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_18983,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_18983,\n  list(n = 15, m = \"hii: ahhh, ohhh\"))",
    "# Testing names",
    "expect_equal(\n  names(output_18983),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_18983),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_18983)),\n  2L)",
    " ",
    "# Testing fn(a = 5, a_fn = NULL, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: a_fn = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19446 <- xpectr::capture_side_effects(fn(a = 5, a_fn = NULL, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'a_fn': Must be a function, not 'NULL'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ehhh\", \"loool\"),...",
    "# Changed from baseline: b = c(\"ehhh\", \"loool\")",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_16607 <- fn(a = 5, a_fn = mean, b = c(\"ehhh\", \"loool\"), d = NULL, e = 3, f = \"hii\")",
    "# Testing class",
    "expect_equal(\n  class(output_16607),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_16607,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_16607,\n  list(n = 15, m = \"hii: ehhh, loool\"))",
    "# Testing names",
    "expect_equal(\n  names(output_16607),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_16607),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_16607)),\n  2L)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"jj\", \"ll\"), d = ...",
    "# Changed from baseline: b = c(\"jj\", \"ll\")",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_16291 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"jj\", \"ll\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16291[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'b': All elements must have at least 3 characters.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16291[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ehhh\", \"loool\", ...",
    "# Changed from baseline: b = c(\"ehhh\", \"loool\", ...",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_10617 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ehhh\", \"loool\", \"heeej\"), d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_10617[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'b': Must have length 2, but has length 3.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_10617[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = NULL, d = NULL, e =...",
    "# Changed from baseline: b = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_12059 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = NULL, d = NULL, e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12059[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'b': Must be of type 'character', not 'NULL'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12059[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: d = data.frame(col1 = c...",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_11765 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = data.frame(col1 = c(1)), e = 3, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_11765[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'd': Must have at least 3 rows, but has 1 rows.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_11765[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: e = NA",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_16870 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = NA, f = \"hii\"), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16870[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'e': May not be NA.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16870[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: e = 6",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_13841 <- fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 6, f = \"hii\")",
    "# Testing class",
    "expect_equal(\n  class(output_13841),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_13841,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_13841,\n  list(n = 30, m = \"hii: ahhh, ohhh\"))",
    "# Testing names",
    "expect_equal(\n  names(output_13841),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_13841),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_13841)),\n  2L)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: e = NULL",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_17698 <- fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = NULL, f = \"hii\")",
    "# Testing class",
    "expect_equal(\n  class(output_17698),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_17698,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_17698,\n  list(n = NaN, m = \"hii: ahhh, ohhh\"))",
    "# Testing names",
    "expect_equal(\n  names(output_17698),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_17698),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_17698)),\n  2L)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: f = \"lol\"",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_14976 <- fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = \"lol\")",
    "# Testing class",
    "expect_equal(\n  class(output_14976),\n  \"list\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_14976,\n  type = \"list\")",
    "# Testing values",
    "expect_equal(\n  output_14976,\n  list(n = 15, m = \"lol: ahhh, ohhh\"))",
    "# Testing names",
    "expect_equal(\n  names(output_14976),\n  c(\"n\", \"m\"),\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_14976),\n  2L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_14976)),\n  2L)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: f = NA",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_17176 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = NA), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_17176[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'f': May not be NA.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_17176[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: f = 3",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19919 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = 3), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19919[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'f': Must be of type 'string', not 'double'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19919[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), ...",
    "# Changed from baseline: f = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_13800 <- xpectr::capture_side_effects(fn(a = 5, a_fn = mean, b = c(\"ahhh\", \"ohhh\"), d = NULL, e = 3, f = NULL), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13800[['error']]),\n  xpectr::strip(\"1 assertions failed:\\n * Variable 'f': Must be of type 'string', not 'NULL'.\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13800[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "## Finished testing 'fn'                                                    ####",
    " "
  )

  set_test_seed(1)
  expect_equal(
    strip(gxs_function(fn, list(
      "a" = list(5, 3, 7, NA),
      a_fn = list(mean, sum),
      "b" = list(c("ahhh","ohhh"), c("ehhh","loool"), c("jj","ll"),  c("ehhh","loool", "heeej")),
      "d" = list(NULL, data.frame("col1" = c(1))),
      "e" = list(3, 6, NA),
      "f" = list("hii", "lol", NA, 3)
    ),
    out = "return")),
    strip(expected_tests2),
    fixed = TRUE)

  eval_expectations(expected_tests2, envir = current_envir)

  ## Second function

  fn2 <- function(x, y, z){
    if (x>3) stop("'x' > 3")
    if (y<0) warning("'y'<0")
    if (z==10) message("'z' was 10!")
    x + y + z
  }

  # assignment names are random
  set_test_seed(1)
  fn2_expected_tests <- c(
    " ",
    "## Testing 'fn2'                                                            ####",
    "## Initially generated by xpectr",
    "# Testing different combinations of argument values",
    " ",
    "# Testing fn2(x = 2, y = 0, z = 5)",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_12655 <- fn2(x = 2, y = 0, z = 5)",
    "# Testing class",
    "expect_equal(\n  class(output_12655),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_12655,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_12655,\n  7,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_12655),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_12655),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_12655)),\n  1L)",
    " ",
    "# Testing fn2(x = 4, y = 0, z = 5)",
    "# Changed from baseline: x = 4",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_13721 <- xpectr::capture_side_effects(fn2(x = 4, y = 0, z = 5), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error']]),\n  xpectr::strip(\"'x' > 3\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_13721[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn2(x = NA, y = 0, z = 5)",
    "# Changed from baseline: x = NA",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_15728 <- xpectr::capture_side_effects(fn2(x = NA, y = 0, z = 5), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error']]),\n  xpectr::strip(\"missing value where TRUE/FALSE needed\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_15728[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn2(x = NULL, y = 0, z = 5)",
    "# Changed from baseline: x = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19082 <- xpectr::capture_side_effects(fn2(x = NULL, y = 0, z = 5), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error']]),\n  xpectr::strip(\"argument is of length zero\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19082[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn2(x = 2, y = -1, z = 5)",
    "# Changed from baseline: y = -1",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_12016 <- xpectr::capture_side_effects(fn2(x = 2, y = -1, z = 5), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['warnings']]),\n  xpectr::strip(\"'y'<0\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_12016[['messages']]),\n  xpectr::strip(character(0)),\n  fixed = TRUE)",
    "# Assigning output",
    "output_12016 <- xpectr::suppress_mw(fn2(x = 2, y = -1, z = 5))",
    "# Testing class",
    "expect_equal(\n  class(output_12016),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_12016,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_12016,\n  6,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_12016),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_12016),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_12016)),\n  1L)",
    " ",
    "# Testing fn2(x = 2, y = NULL, z = 5)",
    "# Changed from baseline: y = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_18983 <- xpectr::capture_side_effects(fn2(x = 2, y = NULL, z = 5), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_18983[['error']]),\n  xpectr::strip(\"argument is of length zero\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_18983[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "# Testing fn2(x = 2, y = 0, z = 10)",
    "# Changed from baseline: z = 10",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_19446 <- xpectr::capture_side_effects(fn2(x = 2, y = 0, z = 10), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['warnings']]),\n  xpectr::strip(character(0)),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_19446[['messages']]),\n  xpectr::strip(\"'z' was 10!\\n\"),\n  fixed = TRUE)",
    "# Assigning output",
    "output_19446 <- xpectr::suppress_mw(fn2(x = 2, y = 0, z = 10))",
    "# Testing class",
    "expect_equal(\n  class(output_19446),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_19446,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_19446,\n  12,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_19446),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_19446),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_19446)),\n  1L)",
    " ",
    "# Testing fn2(x = 2, y = 0, z = 15)",
    "# Changed from baseline: z = 15",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_16607 <- fn2(x = 2, y = 0, z = 15)",
    "# Testing class",
    "expect_equal(\n  class(output_16607),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_16607,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_16607,\n  17,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_16607),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_16607),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_16607)),\n  1L)",
    " ",
    "# Testing fn2(x = 2, y = 0, z = NULL)",
    "# Changed from baseline: z = NULL",
    "xpectr::set_test_seed(42)",
    "# Testing side effects",
    "# Assigning side effects",
    "side_effects_16291 <- xpectr::capture_side_effects(fn2(x = 2, y = 0, z = NULL), reset_seed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16291[['error']]),\n  xpectr::strip(\"argument is of length zero\"),\n  fixed = TRUE)",
    "expect_equal(\n  xpectr::strip(side_effects_16291[['error_class']]),\n  xpectr::strip(c(\"simpleError\", \"error\", \"condition\")),\n  fixed = TRUE)",
    " ",
    "## Finished testing 'fn2'                                                   ####",
    " "
  )

  set_test_seed(1)
  # Create expectations
  # Note: define the list in the call
  expect_equal(strip(gxs_function(fn2,
               list("x" = list(2, 4, NA),
                    "y" = list(0, -1),
                    "z" = list(5, 10, 15)),
               out = "return")),
               strip(fn2_expected_tests),
               fixed = TRUE)

  eval_expectations(fn2_expected_tests, envir = current_envir)


})

test_that("gxs_function() works with multiline function definitions as arg value", {

  fn1 <- function(x, y, z){
    x(1) + y + z
  }

  # gxs_function(fn1, args_values = list(
  #   "x" = list(function(a){
  #     b <- a * a
  #     b
  #   }),
  #   "y" = list(1),
  #   "z" = list(3)
  # ), indentation = 2)

  ## Testing 'fn1'                                                            ####
  ## Initially generated by xpectr
  # Testing different combinations of argument values

  # Testing fn1(x = function(a) { b <- a * a b}, y = 1, ...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- fn1(x = function(a) {
      b <- a * a
      b
  }, y = 1, z = 3)
  # Testing class
  expect_equal(
    class(output_19148),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_19148,
    type = "double")
  # Testing values
  expect_equal(
    output_19148,
    5,
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_19148),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_19148),
    1L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_19148)),
    1L)

  # Testing fn1(x = NULL, y = 1, z = 3)
  # Changed from baseline: x = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19370 <- xpectr::capture_side_effects(fn1(x = NULL, y = 1, z = 3), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error']]),
    xpectr::strip("could not find function \"x\""),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing fn1(x = function(a) { b <- a * a b}, y = NUL...
  # Changed from baseline: y = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_12861 <- fn1(x = function(a) {
      b <- a * a
      b
  }, y = NULL, z = 3)
  # Testing class
  expect_equal(
    class(output_12861),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_12861,
    type = "double")
  # Testing values
  expect_equal(
    output_12861,
    numeric(0),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_12861),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_12861),
    0L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_12861)),
    0L)

  # Testing fn1(x = function(a) { b <- a * a b}, y = 1, ...
  # Changed from baseline: z = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_18304 <- fn1(x = function(a) {
      b <- a * a
      b
  }, y = 1, z = NULL)
  # Testing class
  expect_equal(
    class(output_18304),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_18304,
    type = "double")
  # Testing values
  expect_equal(
    output_18304,
    numeric(0),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_18304),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_18304),
    0L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_18304)),
    0L)

  ## Finished testing 'fn1'                                                   ####
  #

})


test_that("gxs_function() works with extra combinations", {

  # Also checks order when args aren't alphabetically ordered
  fn10 <- function(z, a, x){
    x + a * z + 10
  }

  set_test_seed(1)
  fn10_expected_tests <- c(
    " ",
    "## Testing 'fn10'                                                           ####",
    "## Initially generated by xpectr",
    "# Testing different combinations of argument values",
    " ",
    "# Testing fn10(z = 10, a = 1, x = 3)",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_12655 <- fn10(z = 10, a = 1, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_12655),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_12655,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_12655,\n  23,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_12655),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_12655),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_12655)),\n  1L)",
    " ",
    "# Testing fn10(z = 20, a = 1, x = 3)",
    "# Changed from baseline: z = 20",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_13721 <- fn10(z = 20, a = 1, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_13721),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_13721,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_13721,\n  33,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_13721),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_13721),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_13721)),\n  1L)",
    " ",
    "# Testing fn10(z = NULL, a = 1, x = 3)",
    "# Changed from baseline: z = NULL",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_15728 <- fn10(z = NULL, a = 1, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_15728),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_15728,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_15728,\n  numeric(0),\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_15728),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_15728),\n  0L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_15728)),\n  0L)",
    " ",
    "# Testing fn10(z = 20, a = 1, x = -3)",
    "# Changed from baseline: z, x",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_19082 <- fn10(z = 20, a = 1, x = -3)",
    "# Testing class",
    "expect_equal(\n  class(output_19082),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_19082,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_19082,\n  27,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_19082),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_19082),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_19082)),\n  1L)",
    " ",
    "# Testing fn10(z = 20, a = 2, x = 3)",
    "# Changed from baseline: z, a",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_12016 <- fn10(z = 20, a = 2, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_12016),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_12016,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_12016,\n  53,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_12016),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_12016),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_12016)),\n  1L)",
    " ",
    "# Testing fn10(z = 10, a = 2, x = 3)",
    "# Changed from baseline: a = 2",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_18983 <- fn10(z = 10, a = 2, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_18983),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_18983,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_18983,\n  33,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_18983),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_18983),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_18983)),\n  1L)",
    " ",
    "# Testing fn10(z = 10, a = NULL, x = 3)",
    "# Changed from baseline: a = NULL",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_19446 <- fn10(z = 10, a = NULL, x = 3)",
    "# Testing class",
    "expect_equal(\n  class(output_19446),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_19446,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_19446,\n  numeric(0),\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_19446),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_19446),\n  0L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_19446)),\n  0L)",
    " ",
    "# Testing fn10(z = 10, a = 1, x = -3)",
    "# Changed from baseline: x = -3",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_16607 <- fn10(z = 10, a = 1, x = -3)",
    "# Testing class",
    "expect_equal(\n  class(output_16607),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_16607,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_16607,\n  17,\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_16607),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_16607),\n  1L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_16607)),\n  1L)",
    " ",
    "# Testing fn10(z = 10, a = 1, x = NULL)",
    "# Changed from baseline: x = NULL",
    "xpectr::set_test_seed(42)",
    "# Assigning output",
    "output_16291 <- fn10(z = 10, a = 1, x = NULL)",
    "# Testing class",
    "expect_equal(\n  class(output_16291),\n  \"numeric\",\n  fixed = TRUE)",
    "# Testing type",
    "expect_type(\n  output_16291,\n  type = \"double\")",
    "# Testing values",
    "expect_equal(\n  output_16291,\n  numeric(0),\n  tolerance = 1e-4)",
    "# Testing names",
    "expect_equal(\n  names(output_16291),\n  NULL,\n  fixed = TRUE)",
    "# Testing length",
    "expect_equal(\n  length(output_16291),\n  0L)",
    "# Testing sum of element lengths",
    "expect_equal(\n  sum(xpectr::element_lengths(output_16291)),\n  0L)",
    " ",
    "## Finished testing 'fn10'                                                  ####",
    " "
  )

  set_test_seed(1)
  expect_equal(strip(
    gxs_function(
      fn10,
      args_values = list(
        "z" = list(10, 20),
        "a" = list(1, 2),
        "x" = list(3,-3)
      ),
      extra_combinations = list(list("z" = 20, "x" = -3),
                                list("a" = 2, "z" = 20),
                                list("z" = 10, "a" = 1)),
      # last should be filtered out (not unique)),
      indentation = 2,
      out = "return"
    )
  ),
  strip(fn10_expected_tests),
  fixed = TRUE)


})
