# Xpose 4
# An R-based population pharmacokinetic/
# pharmacodynamic model building aid for NONMEM.
# Copyright (C) 1998-2004 E. Niclas Jonsson and Mats Karlsson.
# Copyright (C) 2005-2008 Andrew C. Hooker, Justin J. Wilkins, 
# Mats O. Karlsson and E. Niclas Jonsson.
# Copyright (C) 2009-2010 Andrew C. Hooker, Mats O. Karlsson and 
# E. Niclas Jonsson.

# This file is a part of Xpose 4.
# Xpose 4 is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation, either version 3
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public License
# along with this program.  A copy can be cound in the R installation
# directory under \share\licenses. If not, see http://www.gnu.org/licenses/.
simprazExample <- function(overwrite=FALSE) {

  writeMod <- function() {
    cat(file="run1.mod","$PROB Simpraz model
;; 1. Based on: First run [nodOFV]
;;    First run with a one compartment model and first order absorption
;; 2. Structural model:
;;    One compartment model with first order absorption
;; 3. Covariate model:
;;    No covariates
;; 4. Inter-individual variability:
;;    ETAs on CL, V and KA
;; 5. Residual variability:
;;    Proportional
;; 6. Estimation:
;;    FOCE INTER
$INPUT ID SEX AGE RACE HT SMOK HCTZ PROP CON AMT WT TIME SECR
       DV RATE EVID OCC
$DATA simpraz.dta IGNORE=@
$SUBROUTINE ADVAN2 TRANS2
$PK
 
    TVCL  = THETA(1)
    TVV   = THETA(2)
    TVKA  = THETA(3)

    CL    = TVCL*EXP(ETA(1))
    V     = TVV *EXP(ETA(2))
    KA    = TVKA*EXP(ETA(3))

    S2=V

$ERROR 

    IPRED = F
    IRES  = DV - F
    W     = F
    IF(W.EQ.0) W = 1
    IWRES = IRES/W

    Y     = IPRED + W*EPS(1)

$THETA 

 (0,18.7)    ; CL (L/h)
 (0,87.3)    ; V   (L)
 (0,2.13)    ; KA (1/h)

$OMEGA 

 0.128       ; omCL
 0.142       ; omV
 1.82        ; omKA

$SIGMA  

 0.0231      ; Prop error

$EST NOABORT METHOD=1 INTER PRINT=1
$COV PRINT=E

$TABLE ID TIME IPRED IWRES CWRES 
       CL V KA ETA1 ETA2 ETA3 
       AGE HT WT SECR 
       SEX RACE SMOK HCTZ PROP CON OCC
       NOPRINT ONEHEADER FILE=xptab1

"
      )
  }

  writeDta <- function() {
    cat(file="simpraz.dta","ID SEX AGE RACE HT SMOK HCTZ PROP CON AMT WT TIME SECR DV RATE EVID OCC
1 2 55 2 154 0 1 1 1 10000 80.97 0 1 0 0 1 0
1 2 55 2 154 0 1 1 1 0 80.97 1 1 71.74 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 2 1 72.61 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 3 1 88.01 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 4 1 53.13 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 5 1 56.83 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 6 1 51.94 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 7 1 52.89 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 9 1 26.95 0 0 0
1 2 55 2 154 0 1 1 1 0 80.97 11 1 26.17 0 0 0
2 1 37 1 179 1 0 0 0 10000 93.21 0 1.2 0 0 1 0
2 1 37 1 179 1 0 0 0 0 93.21 1 1.2 108.75 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 2 1.2 96.6 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 3 1.2 81 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 4 1.2 77.07 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 5 1.2 64.57 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 6 1.2 50.21 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 7 1.2 64.58 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 8 1.2 50.54 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 10 1.2 41.56 0 0 0
2 1 37 1 179 1 0 0 0 0 93.21 12 1.2 42.46 0 0 0
3 1 35 1 188 0 0 0 0 1000 94.35 0 0.9 0 0 1 0
3 1 35 1 188 0 0 0 0 0 94.35 1 0.9 9.35 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 2 0.9 8.66 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 3 0.9 8.18 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 4 0.9 6.19 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 5 0.9 7.08 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 6 0.9 4.68 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 7 0.9 5.3 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 8 0.9 4.2 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 10 0.9 3.92 0 0 0
3 1 35 1 188 0 0 0 0 0 94.35 12 0.9 2.75 0 0 0
4 2 67 2 168 0 0 0 0 5000 74.39 0 0.8 0 0 1 0
4 2 67 2 168 0 0 0 0 0 74.39 1 0.8 63.15 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 2 0.8 62.41 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 3 0.8 41.05 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 4 0.8 24.31 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 5 0.8 18.17 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 6 0.8 13.23 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 7 0.8 8.48 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 9 0.8 3.92 0 0 0
4 2 67 2 168 0 0 0 0 0 74.39 11 0.8 1.76 0 0 0
5 2 69 2 165 0 0 0 1 2000 91.85 0 1 0 0 1 0
5 2 69 2 165 0 0 0 1 0 91.85 1 1 21.49 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 2 1 14.87 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 3 1 15.17 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 4 1 13.68 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 5 1 14.83 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 6 1 9.86 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 7 1 8.75 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 9 1 7.77 0 0 0
5 2 69 2 165 0 0 0 1 0 91.85 11 1 4.36 0 0 0
6 2 52 2 157 0 1 1 1 10000 104.3 0 0.8 0 0 1 0
6 2 52 2 157 0 1 1 1 0 104.3 1 0.8 74.53 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 2 0.8 63.02 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 3 0.8 100.46 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 4 0.8 84.92 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 5 0.8 59.01 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 6 0.8 53.87 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 7 0.8 45.15 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 8 0.8 33.88 0 0 0
6 2 52 2 157 0 1 1 1 0 104.3 10 0.8 26.27 0 0 0
7 2 44 2 140 0 1 1 1 10000 90.04 0 0.9 0 0 1 0
7 2 44 2 140 0 1 1 1 0 90.04 1 0.9 184.78 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 2 0.9 142.73 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 3 0.9 86.18 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 4 0.9 65.65 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 5 0.9 55.07 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 6 0.9 37.95 0 0 0
7 2 44 2 140 0 1 1 1 0 90.04 8 0.9 23.25 0 0 0
8 2 50 2 173 1 1 0 0 5000 98.88 0 0.9 0 0 1 0
8 2 50 2 173 1 1 0 0 0 98.88 1 0.9 37.72 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 2 0.9 19.15 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 3 0.9 14.41 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 4 0.9 11.89 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 5 0.9 5.48 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 6 0.9 4.93 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 7 0.9 4.01 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 9 0.9 1.54 0 0 0
8 2 50 2 173 1 1 0 0 0 98.88 11 0.9 0.6 0 0 0
9 2 61 2 160 0 1 0 1 5000 81.42 0 0.9 0 0 1 0
9 2 61 2 160 0 1 0 1 0 81.42 1 0.9 52.58 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 2 0.9 57.71 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 3 0.9 36.23 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 4 0.9 26.13 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 5 0.9 18.07 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 6 0.9 10.83 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 8 0.9 5.49 0 0 0
9 2 61 2 160 0 1 0 1 0 81.42 10 0.9 3.05 0 0 0
10 2 52 2 168 0 1 1 1 5000 87.32 0 1.8 0 0 1 0
10 2 52 2 168 0 1 1 1 0 87.32 1 1.8 61.76 0 0 0
10 2 52 2 168 0 1 1 1 0 87.32 2 1.8 18.4 0 0 0
10 2 52 2 168 0 1 1 1 0 87.32 3 1.8 4.82 0 0 0
10 2 52 2 168 0 1 1 1 0 87.32 6 1.8 0.08 0 0 0
10 2 52 2 168 0 1 1 1 0 87.32 8 1.8 0.01 0 0 0
12 1 59 2 178 0 0 0 1 2000 98.43 0 1.1 0 0 1 0
12 1 59 2 178 0 0 0 1 0 98.43 1 1.1 16.63 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 2 1.1 16.48 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 3 1.1 11.11 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 4 1.1 11.98 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 5 1.1 7.73 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 6 1.1 8.08 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 7 1.1 5.74 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 9 1.1 3.87 0 0 0
12 1 59 2 178 0 0 0 1 0 98.43 11 1.1 2.73 0 0 0
13 1 54 2 159 0 0 0 0 2000 68.04 0 1.3 0 0 1 0
13 1 54 2 159 0 0 0 0 0 68.04 1 1.3 13.98 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 2 1.3 12.08 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 3 1.3 9.14 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 4 1.3 7.21 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 5 1.3 4.76 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 6 1.3 2.5 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 7 1.3 1.79 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 8 1.3 1.32 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 10 1.3 0.62 0 0 0
13 1 54 2 159 0 0 0 0 0 68.04 12 1.3 0.16 0 0 0
14 1 62 2 180 0 0 0 1 2000 81.65 0 1.1 0 0 1 0
14 1 62 2 180 0 0 0 1 0 81.65 1 1.1 27.05 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 2 1.1 19.34 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 3 1.1 13.79 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 4 1.1 10.04 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 5 1.1 7.52 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 6 1.1 3.78 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 7 1.1 3.47 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 8 1.1 3.1 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 10 1.1 1.01 0 0 0
14 1 62 2 180 0 0 0 1 0 81.65 12 1.1 0.6 0 0 0
15 1 63 1 172 0 1 0 1 2000 83.1 0 1.2 0 0 1 0
15 1 63 1 172 0 1 0 1 0 83.1 1 1.2 8.88 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 2 1.2 11.07 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 3 1.2 14.29 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 4 1.2 8.91 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 5 1.2 9.92 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 6 1.2 8.8 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 7 1.2 6.79 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 8 1.2 5.76 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 10 1.2 4 0 0 0
15 1 63 1 172 0 1 0 1 0 83.1 12 1.2 4.02 0 0 0
16 1 63 1 170 1 0 0 1 2000 83.4 0 1.1 0 0 1 0
16 1 63 1 170 1 0 0 1 0 83.4 1 1.1 9.79 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 2 1.1 16 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 3 1.1 15.77 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 4 1.1 17.11 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 5 1.1 16.56 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 6 1.1 11.53 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 7 1.1 12.31 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 8 1.1 8.91 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 10 1.1 7.08 0 0 0
16 1 63 1 170 1 0 0 1 0 83.4 12 1.1 4.96 0 0 0
17 1 63 1 177 1 1 0 0 10000 104.1 0 1 0 0 1 0
17 1 63 1 177 1 1 0 0 0 104.1 1 1 176.18 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 2 1 138.65 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 3 1 112.75 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 4 1 93.39 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 5 1 86.78 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 6 1 66.81 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 7 1 46.65 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 8 1 45.43 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 10 1 31.86 0 0 0
17 1 63 1 177 1 1 0 0 0 104.1 12 1 16.42 0 0 0
18 1 58 2 187 1 1 0 1 5000 136.8 0 1.5 0 0 1 0
18 1 58 2 187 1 1 0 1 0 136.8 1 1.5 56.35 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 2 1.5 47.18 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 3 1.5 44.91 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 4 1.5 41.94 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 5 1.5 29.04 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 6 1.5 17.36 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 7 1.5 22.15 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 8 1.5 18.05 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 10 1.5 13.64 0 0 0
18 1 58 2 187 1 1 0 1 0 136.8 12 1.5 10.16 0 0 0
19 1 66 1 177 1 0 0 1 5000 97.3 0 1.2 0 0 1 0
19 1 66 1 177 1 0 0 1 0 97.3 1 1.2 56.37 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 2 1.2 47.02 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 3 1.2 58.38 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 4 1.2 36.77 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 5 1.2 38.95 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 6 1.2 28.99 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 7 1.2 32.83 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 8 1.2 24.78 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 10 1.2 16.52 0 0 0
19 1 66 1 177 1 0 0 1 0 97.3 12 1.2 12.04 0 0 0
20 1 67 1 181 1 0 0 1 2000 96.1 0 1.3 0 0 1 0
20 1 67 1 181 1 0 0 1 0 96.1 1 1.3 10.88 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 2 1.3 11.89 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 3 1.3 9.42 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 4 1.3 13.67 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 5 1.3 10.97 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 6 1.3 9.21 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 7 1.3 6.33 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 9 1.3 4.95 0 0 0
20 1 67 1 181 1 0 0 1 0 96.1 11 1.3 2.85 0 0 0
21 1 57 1 180 1 1 0 1 1000 85.9 0 1.2 0 0 1 0
21 1 57 1 180 1 1 0 1 0 85.9 1 1.2 2.23 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 3 1.2 3.53 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 4 1.2 4.13 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 5 1.2 3.84 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 6 1.2 2.9 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 7 1.2 2.61 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 8 1.2 2.5 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 10 1.2 1.43 0 0 0
21 1 57 1 180 1 1 0 1 0 85.9 12 1.2 0.66 0 0 0
22 1 56 2 170 0 1 0 1 3500 88.13 0 0.8 0 0 1 0
22 1 56 2 170 0 1 0 1 0 88.13 1 0.8 19.76 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 2 0.8 23.99 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 3 0.8 15.86 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 4 0.8 12.01 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 5 0.8 7.7 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 6 0.8 6.29 0 0 0
22 1 56 2 170 0 1 0 1 0 88.13 7 0.8 4.11 0 0 0
23 2 57 3 168 0 1 0 1 3500 69.08 0 1.1 0 0 1 0
23 2 57 3 168 0 1 0 1 0 69.08 1 1.1 15.43 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 2 1.1 17.58 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 3 1.1 17.85 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 4 1.1 17.46 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 5 1.1 14.53 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 6 1.1 17.52 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 7 1.1 15.93 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 8 1.1 15.72 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 10 1.1 12.62 0 0 0
23 2 57 3 168 0 1 0 1 0 69.08 12 1.1 10.71 0 0 0
24 2 56 1 175 0 0 0 0 3500 74.6 0 0.8 0 0 1 0
24 2 56 1 175 0 0 0 0 0 74.6 1 0.8 15.35 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 2 0.8 16.73 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 3 0.8 29.66 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 4 0.8 29.59 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 5 0.8 24.34 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 6 0.8 19.81 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 7 0.8 25.61 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 8 0.8 12.9 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 10 0.8 14.45 0 0 0
24 2 56 1 175 0 0 0 0 0 74.6 12 0.8 12.65 0 0 0
25 1 61 1 171 0 1 0 1 3500 96.62 0 1 0 0 1 0
25 1 61 1 171 0 1 0 1 0 96.62 1 1 28.33 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 2 1 23.91 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 3 1 22.45 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 4 1 15.55 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 5 1 12.41 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 6 1 8.77 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 7 1 5.83 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 8 1 4.05 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 9 1 3.38 0 0 0
25 1 61 1 171 0 1 0 1 0 96.62 10 1 2.26 0 0 0
26 2 67 1 157 0 0 0 0 3500 66.4 0 0.9 0 0 1 0
26 2 67 1 157 0 0 0 0 0 66.4 1 0.9 8.83 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 2 0.9 16.84 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 3 0.9 25.07 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 4 0.9 19.91 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 5 0.9 20.68 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 6 0.9 22.01 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 7 0.9 19.94 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 8 0.9 15.65 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 10 0.9 14.46 0 0 0
26 2 67 1 157 0 0 0 0 0 66.4 12 0.9 10.65 0 0 0
27 1 56 1 177 0 0 0 0 5000 97.4 0 1 0 0 1 0
27 1 56 1 177 0 0 0 0 0 97.4 1 1 40.74 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 2 1 54.32 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 3 1 37.44 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 4 1 29.8 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 5 1 26.36 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 6 1 17.04 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 7 1 13.14 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 8 1 7.92 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 10 1 4.21 0 0 0
27 1 56 1 177 0 0 0 0 0 97.4 12 1 2.38 0 0 0
28 1 58 1 173 0 1 0 1 10000 78.7 0 1.4 0 0 1 0
28 1 58 1 173 0 1 0 1 0 78.7 1 1.4 84.19 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 2 1.4 86.2 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 3 1.4 74.03 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 4 1.4 58.2 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 5 1.4 47 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 6 1.4 40.57 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 7 1.4 31.51 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 8 1.4 30.41 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 10 1.4 20.17 0 0 0
28 1 58 1 173 0 1 0 1 0 78.7 12 1.4 10.19 0 0 0
29 1 53 1 180 0 1 1 1 10000 87.6 0 1.2 0 0 1 0
29 1 53 1 180 0 1 1 1 0 87.6 1 1.2 22.06 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 3 1.2 40.41 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 4 1.2 34.71 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 5 1.2 26.09 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 6 1.2 28.7 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 7 1.2 30.07 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 8 1.2 19.96 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 10 1.2 10.01 0 0 0
29 1 53 1 180 0 1 1 1 0 87.6 12 1.2 10.61 0 0 0
30 1 46 1 175 0 1 1 1 10000 84.8 0 1.2 0 0 1 0
30 1 46 1 175 0 1 1 1 0 84.8 1 1.2 57.81 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 2 1.2 103.55 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 3 1.2 76 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 4 1.2 72.99 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 5 1.2 56.53 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 6 1.2 60.08 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 7 1.2 57.04 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 8 1.2 37.89 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 10 1.2 38.8 0 0 0
30 1 46 1 175 0 1 1 1 0 84.8 12 1.2 23.21 0 0 0
31 2 30 1 157 0 0 0 0 2000 61.7 0 1.1 0 0 1 0
31 2 30 1 157 0 0 0 0 0 61.7 1 1.1 13.34 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 2 1.1 17.91 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 3 1.1 17.23 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 4 1.1 11.74 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 5 1.1 10.35 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 6 1.1 10.74 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 7 1.1 9.4 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 8 1.1 6.42 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 10 1.1 4.26 0 0 0
31 2 30 1 157 0 0 0 0 0 61.7 12 1.1 3.56 0 0 0
32 2 56 1 174 0 1 0 1 3500 68.72 0 1.1 0 0 1 0
32 2 56 1 174 0 1 0 1 0 68.72 1 1.1 17.32 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 2 1.1 17.54 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 3 1.1 12.74 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 4 1.1 14.73 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 5 1.1 13.02 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 6 1.1 9.25 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 7 1.1 7.85 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 8 1.1 8.4 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 10 1.1 5.16 0 0 0
32 2 56 1 174 0 1 0 1 0 68.72 12 1.1 5.04 0 0 0
33 1 54 1 180 0 0 0 0 10000 76.43 0 1 0 0 1 0
33 1 54 1 180 0 0 0 0 0 76.43 1 1 105.94 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 2 1 120.44 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 3 1 104.12 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 4 1 91.11 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 5 1 84.46 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 6 1 58.59 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 7 1 53.96 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 8 1 43.25 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 10 1 46.03 0 0 0
33 1 54 1 180 0 0 0 0 0 76.43 12 1 29.78 0 0 0
34 1 34 1 170 0 0 0 1 1000 77.34 0 1 0 0 1 0
34 1 34 1 170 0 0 0 1 0 77.34 2 1 8.48 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 3 1 5.39 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 4 1 3.09 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 5 1 3.46 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 6 1 1.79 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 7 1 1.08 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 9 1 0.59 0 0 0
34 1 34 1 170 0 0 0 1 0 77.34 10 1 0.33 0 0 0
35 1 52 1 183 0 1 0 1 5000 89.36 0 1.1 0 0 1 0
35 1 52 1 183 0 1 0 1 0 89.36 1 1.1 27.99 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 2 1.1 42.82 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 3 1.1 25.66 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 4 1.1 24.67 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 5 1.1 18.75 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 6 1.1 14.15 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 7 1.1 11.46 0 0 0
35 1 52 1 183 0 1 0 1 0 89.36 10 1.1 2.88 0 0 0
36 1 47 1 175 0 1 1 1 10000 93.21 0 1.1 0 0 1 0
36 1 47 1 175 0 1 1 1 0 93.21 1 1.1 134.89 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 2 1.1 107.21 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 3 1.1 63.25 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 4 1.1 38.67 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 5 1.1 26.35 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 6 1.1 14.12 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 7 1.1 11.74 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 8 1.1 6.08 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 10 1.1 2.1 0 0 0
36 1 47 1 175 0 1 1 1 0 93.21 12 1.1 0.89 0 0 0
37 1 66 1 155 0 1 1 1 5000 93.44 0 1.4 0 0 1 0
37 1 66 1 155 0 1 1 1 0 93.44 1 1.4 23.32 0 0 0
37 1 66 1 155 0 1 1 1 0 93.44 2 1.4 25.76 0 0 0
37 1 66 1 155 0 1 1 1 0 93.44 3 1.4 36.97 0 0 0
37 1 66 1 155 0 1 1 1 0 93.44 8 1.4 22.21 0 0 0
37 1 66 1 155 0 1 1 1 0 93.44 12 1.4 8.49 0 0 0
38 1 66 2 160 1 0 0 1 5000 62.14 0 1.1 0 0 1 0
38 1 66 2 160 1 0 0 1 0 62.14 1 1.1 70.92 0 0 0
38 1 66 2 160 1 0 0 1 0 62.14 2 1.1 47.75 0 0 0
38 1 66 2 160 1 0 0 1 0 62.14 3 1.1 40.87 0 0 0
38 1 66 2 160 1 0 0 1 0 62.14 4 1.1 21.24 0 0 0
38 1 66 2 160 1 0 0 1 0 62.14 6 1.1 12.55 0 0 0
39 1 51 1 178 0 1 0 1 1000 97.07 0 1 0 0 1 0
39 1 51 1 178 0 1 0 1 0 97.07 1 1 6.32 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 2 1 7.29 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 3 1 5.58 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 4 1 5.72 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 5 1 3.81 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 7 1 3.48 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 8 1 3.17 0 0 0
39 1 51 1 178 0 1 0 1 0 97.07 10 1 1.93 0 0 0
40 1 24 1 181 0 0 0 1 3500 80.29 0 1.4 0 0 1 0
40 1 24 1 181 0 0 0 1 0 80.29 1 1.4 33.86 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 2 1.4 18.36 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 3 1.4 19.47 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 4 1.4 15.85 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 5 1.4 10.68 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 6 1.4 7.84 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 7 1.4 7.81 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 8 1.4 5.84 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 10 1.4 4.6 0 0 0
40 1 24 1 181 0 0 0 1 0 80.29 12 1.4 1.99 0 0 0
41 1 33 1 176 0 0 0 1 3500 80.74 0 1.3 0 0 1 0
41 1 33 1 176 0 0 0 1 0 80.74 1 1.3 45.4 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 2 1.3 44.07 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 3 1.3 37.28 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 4 1.3 41.31 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 5 1.3 37.36 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 6 1.3 24.23 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 7 1.3 22.39 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 8 1.3 15.46 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 10 1.3 15.16 0 0 0
41 1 33 1 176 0 0 0 1 0 80.74 12 1.3 9.55 0 0 0
42 1 41 1 168 0 1 0 1 10000 84.37 0 1.5 0 0 1 0
42 1 41 1 168 0 1 0 1 0 84.37 1 1.5 110.98 0 0 0
42 1 41 1 168 0 1 0 1 0 84.37 2 1.5 72.34 0 0 0
42 1 41 1 168 0 1 0 1 0 84.37 3 1.5 30.91 0 0 0
42 1 41 1 168 0 1 0 1 0 84.37 4 1.5 14 0 0 0
42 1 41 1 168 0 1 0 1 0 84.37 6 1.5 3.13 0 0 0
42 1 41 1 168 0 1 0 1 0 84.37 8 1.5 0.67 0 0 0
43 1 38 1 173 0 0 0 1 1000 67.13 0 1.2 0 0 1 0
43 1 38 1 173 0 0 0 1 0 67.13 1 1.2 4.69 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 2 1.2 5.8 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 3 1.2 5.9 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 4 1.2 5.23 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 5 1.2 4.68 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 6 1.2 3.56 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 7 1.2 2.48 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 8 1.2 2.5 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 10 1.2 2.14 0 0 0
43 1 38 1 173 0 0 0 1 0 67.13 12 1.2 1.67 0 0 0
44 1 54 1 183 1 0 0 0 3500 103.4 0 1.1 0 0 1 0
44 1 54 1 183 1 0 0 0 0 103.4 1 1.1 28.9 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 2 1.1 21.08 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 3 1.1 15.09 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 4 1.1 8.77 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 5 1.1 3.41 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 6 1.1 1.84 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 7 1.1 1.09 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 8 1.1 0.48 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 10 1.1 0.14 0 0 0
44 1 54 1 183 1 0 0 0 0 103.4 12 1.1 0.03 0 0 0
45 1 51 2 170 1 1 0 0 5000 83.14 0 1.2 0 0 1 0
45 1 51 2 170 1 1 0 0 0 83.14 1 1.2 53 0 0 0
45 1 51 2 170 1 1 0 0 0 83.14 4 1.2 17.98 0 0 0
45 1 51 2 170 1 1 0 0 0 83.14 8 1.2 2.75 0 0 0
45 1 51 2 170 1 1 0 0 0 83.14 11 1.2 0.67 0 0 0
46 2 58 2 159 1 1 0 1 10000 69.31 0 1.1 0 0 1 0
46 2 58 2 159 1 1 0 1 0 69.31 1 1.1 161.86 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 2 1.1 157.57 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 3 1.1 116.89 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 4 1.1 102.18 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 5 1.1 69.69 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 6 1.1 48.29 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 7 1.1 47.6 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 8 1.1 38.98 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 9 1.1 21.22 0 0 0
46 2 58 2 159 1 1 0 1 0 69.31 11 1.1 14.56 0 0 0
47 1 56 1 187 1 1 1 1 10000 108.2 0 1 0 0 1 0
47 1 56 1 187 1 1 1 1 0 108.2 1 1 73.72 0 0 0
47 1 56 1 187 1 1 1 1 0 108.2 4 1 62.77 0 0 0
47 1 56 1 187 1 1 1 1 0 108.2 5 1 52.28 0 0 0
47 1 56 1 187 1 1 1 1 0 108.2 8 1 41.83 0 0 0
47 1 56 1 187 1 1 1 1 0 108.2 12 1 30.84 0 0 0
48 1 63 1 178 1 1 0 1 5000 93.8 0 1 0 0 1 0
48 1 63 1 178 1 1 0 1 0 93.8 1 1 119.97 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 2 1 55.31 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 3 1 52.02 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 4 1 49.41 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 5 1 40.56 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 6 1 33.93 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 7 1 23.22 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 8 1 15.22 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 9 1 12.25 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 10 1 13.64 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 11 1 7.72 0 0 0
48 1 63 1 178 1 1 0 1 0 93.8 12 1 9.18 0 0 0
49 2 50 1 157 1 1 0 1 5000 125.4 0 0.7 0 0 1 0
49 2 50 1 157 1 1 0 1 0 125.4 1 0.7 19.09 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 2 0.7 26.46 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 3 0.7 24.47 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 4 0.7 20.43 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 5 0.7 24.08 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 6 0.7 13.59 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 7 0.7 16.84 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 8 0.7 12.95 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 10 0.7 10.71 0 0 0
49 2 50 1 157 1 1 0 1 0 125.4 12 0.7 9.75 0 0 0
50 2 62 2 147 0 1 0 1 5000 51.03 0 0.8 0 0 1 0
50 2 62 2 147 0 1 0 1 0 51.03 1 0.8 65.72 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 2 0.8 42.13 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 3 0.8 20.42 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 4 0.8 11.54 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 5 0.8 4.39 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 6 0.8 1.93 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 7 0.8 1 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 8 0.8 0.36 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 10 0.8 0.08 0 0 0
50 2 62 2 147 0 1 0 1 0 51.03 12 0.8 0.02 0 0 0
51 1 48 2 185 0 1 1 1 5000 96.3 0 1.1 0 0 1 0
51 1 48 2 185 0 1 1 1 0 96.3 1 1.1 42.61 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 2 1.1 47.68 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 3 1.1 47.65 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 4 1.1 44.96 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 5 1.1 36.56 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 6 1.1 32.37 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 8 1.1 19.77 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 10 1.1 17.05 0 0 0
51 1 48 2 185 0 1 1 1 0 96.3 12 1.1 16.78 0 0 0
52 2 57 1 165 0 0 0 1 5000 70.53 0 1 0 0 1 0
52 2 57 1 165 0 0 0 1 0 70.53 1 1 27.81 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 2 1 44.56 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 3 1 27.71 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 4 1 35.69 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 5 1 32.64 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 6 1 20.07 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 7 1 26.51 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 8 1 24.71 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 10 1 18.05 0 0 0
52 2 57 1 165 0 0 0 1 0 70.53 12 1 13.82 0 0 0
53 2 67 1 160 0 0 0 1 1000 83.24 0 1 0 0 1 0
53 2 67 1 160 0 0 0 1 0 83.24 1 1 14.36 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 2 1 14.72 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 3 1 9.02 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 4 1 7.6 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 5 1 6.45 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 6 1 4.86 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 7 1 3.23 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 8 1 2.11 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 10 1 1.16 0 0 0
53 2 67 1 160 0 0 0 1 0 83.24 12 1 0.7 0 0 0
54 2 39 2 169 1 1 0 1 5000 78.25 0 1 0 0 1 0
54 2 39 2 169 1 1 0 1 0 78.25 1 1 15 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 2 1 20.61 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 3 1 18.4 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 4 1 16 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 5 1 15.3 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 6 1 9.47 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 7 1 7.66 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 8 1 4.37 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 10 1 2.9 0 0 0
54 2 39 2 169 1 1 0 1 0 78.25 12 1 0.91 0 0 0
55 2 47 1 168 0 1 0 1 10000 72.12 0 0.7 0 0 1 0
55 2 47 1 168 0 1 0 1 0 72.12 1 0.7 98.31 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 2 0.7 80.18 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 3 0.7 54.06 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 4 0.7 27.03 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 5 0.7 14.27 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 6 0.7 12.56 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 7 0.7 4.39 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 9 0.7 2.05 0 0 0
55 2 47 1 168 0 1 0 1 0 72.12 11 0.7 0.65 0 0 0
56 2 36 2 157 0 1 0 1 1000 88.45 0 0.9 0 0 1 0
56 2 36 2 157 0 1 0 1 0 88.45 1 0.9 14.96 0 0 0
56 2 36 2 157 0 1 0 1 0 88.45 4 0.9 8.72 0 0 0
56 2 36 2 157 0 1 0 1 0 88.45 5 0.9 8.66 0 0 0
56 2 36 2 157 0 1 0 1 0 88.45 8 0.9 4.38 0 0 0
56 2 36 2 157 0 1 0 1 0 88.45 10 0.9 3.03 0 0 0
56 2 36 2 157 0 1 0 1 0 88.45 12 0.9 2.43 0 0 0
57 1 63 1 173 0 1 0 1 5000 73.94 0 0.9 0 0 1 0
57 1 63 1 173 0 1 0 1 0 73.94 1 0.9 5.99 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 2 0.9 13.26 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 3 0.9 13.19 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 4 0.9 14.15 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 5 0.9 16.3 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 6 0.9 14.39 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 7 0.9 12.63 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 8 0.9 14.82 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 9 0.9 13.44 0 0 0
57 1 63 1 173 0 1 0 1 0 73.94 10 0.9 10.27 0 0 0
58 1 61 1 178 0 0 0 1 3500 67.27 0 1.1 0 0 1 0
58 1 61 1 178 0 0 0 1 0 67.27 1 1.1 25.91 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 4 1.1 13 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 5 1.1 8.31 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 6 1.1 4.54 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 7 1.1 4.87 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 8 1.1 2.93 0 0 0
58 1 61 1 178 0 0 0 1 0 67.27 10 1.1 1.9 0 0 0
59 2 53 1 157 0 1 0 1 5000 55.02 0 0.8 0 0 1 0
59 2 53 1 157 0 1 0 1 0 55.02 1 0.8 36.19 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 2 0.8 40.04 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 3 0.8 35.84 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 4 0.8 48.96 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 6 0.8 30.71 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 7 0.8 30.07 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 8 0.8 29.57 0 0 0
59 2 53 1 157 0 1 0 1 0 55.02 12 0.8 15.6 0 0 0
60 1 55 1 173 0 0 0 1 5000 78.7 0 1.1 0 0 1 0
60 1 55 1 173 0 0 0 1 0 78.7 1 1.1 73.05 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 2 1.1 51.52 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 3 1.1 39.96 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 4 1.1 23.43 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 5 1.1 22.59 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 6 1.1 14.64 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 7 1.1 12.32 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 8 1.1 9.58 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 10 1.1 4.23 0 0 0
60 1 55 1 173 0 0 0 1 0 78.7 12 1.1 2.48 0 0 0
61 1 58 1 179 0 0 0 1 1000 94.57 0 1 0 0 1 0
61 1 58 1 179 0 0 0 1 0 94.57 1 1 10.61 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 2 1 8.08 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 4 1 3.32 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 5 1 2.72 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 6 1 1.49 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 7 1 1.26 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 8 1 0.89 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 9 1 0.49 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 10 1 0.38 0 0 0
61 1 58 1 179 0 0 0 1 0 94.57 11 1 0.25 0 0 0
62 1 56 1 179 0 1 0 1 5000 102.3 0 1.2 0 0 1 0
62 1 56 1 179 0 1 0 1 0 102.3 1 1.2 50.88 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 2 1.2 50.86 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 3 1.2 41.68 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 4 1.2 38.77 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 5 1.2 35.61 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 6 1.2 24.9 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 7 1.2 24.82 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 8 1.2 20.89 0 0 0
62 1 56 1 179 0 1 0 1 0 102.3 10 1.2 15.97 0 0 0
63 1 66 1 182 0 0 0 1 3500 94.8 0 1.1 0 0 1 0
63 1 66 1 182 0 0 0 1 0 94.8 1 1.1 39.62 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 2 1.1 34.31 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 3 1.1 25.9 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 4 1.1 29.15 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 5 1.1 18.82 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 7 1.1 10.61 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 8 1.1 7.07 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 10 1.1 4.5 0 0 0
63 1 66 1 182 0 0 0 1 0 94.8 11 1.1 2.73 0 0 0
64 1 48 1 183 0 0 0 1 5000 111.8 0 1.2 0 0 1 0
64 1 48 1 183 0 0 0 1 0 111.8 1 1.2 26.15 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 2 1.2 27.49 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 3 1.2 27.17 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 4 1.2 30.42 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 5 1.2 24.82 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 6 1.2 20.71 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 8 1.2 10.35 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 9 1.2 14.18 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 11 1.2 7.3 0 0 0
64 1 48 1 183 0 0 0 1 0 111.8 12 1.2 4.51 0 0 0
65 1 64 1 180 0 0 0 1 5000 99.79 0 1.1 0 0 1 0
65 1 64 1 180 0 0 0 1 0 99.79 1 1.1 19.94 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 2 1.1 33.94 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 3 1.1 30.16 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 4 1.1 23.47 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 5 1.1 27.24 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 6 1.1 25.67 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 7 1.1 21.15 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 9 1.1 20.4 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 10 1.1 18.13 0 0 0
65 1 64 1 180 0 0 0 1 0 99.79 12 1.1 20.18 0 0 0
")
    #data("simprazdata")
    #write.table(simprazdata,sep=" ",file="simpraz.dta",quote=F,row.names=F,col.names=T)
    #rm(simprazdata,pos=1)
  }

  writeTab <- function() {
    cat(file="xptab1","TABLE NO.  1
ID TIME IPRED IWRES CWRES CL V KA ETA1 ETA2 ETA3 AGE HT WT SECR SEX RACE SMOK HCTZ PROP CON OCC DV PRED RES WRES
1 0 0 0 0 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 0 0 0 0
1 1 69.193 0.036814 -0.064632 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 71.74 86.417 -14.677 -0.10466
1 2 80.181 -0.094424 -0.94112 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 72.61 89.007 -16.397 -0.75889
1 3 75.33 0.16833 1.1911 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 88.01 75.467 12.543 1.2075
1 4 66.916 -0.20602 -1.5154 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 53.13 61.036 -7.9058 -1.5386
1 5 58.398 -0.026852 -0.05963 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 56.83 48.711 8.1189 -0.1152
1 6 50.666 0.025138 0.40992 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 51.94 38.724 13.216 0.35805
1 7 43.871 0.20557 1.8304 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 52.89 30.748 22.142 2.0246
1 9 32.841 -0.17939 -1.0218 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 26.95 19.372 7.5782 -1.3045
1 11 24.575 0.064921 0.83731 13.579 93.64 1.2249 -0.26774 0.19829 -0.16359 55 154 80.97 1 2 2 0 1 1 1 0 26.17 12.202 13.968 1.3377
2 0 0 0 0 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 0 0 0 0
2 1 96.073 0.13195 0.81197 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 108.75 86.417 22.333 0.78856
2 2 92.138 0.048423 -0.086669 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 96.6 89.007 7.5927 0.20562
2 3 84.073 -0.036555 -0.54055 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 81 75.467 5.5326 -0.16109
2 4 76.514 0.0072629 0.0058848 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 77.07 61.036 16.034 0.33678
2 5 69.625 -0.072605 -0.40244 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 64.57 48.711 15.859 -0.42935
2 6 63.356 -0.20749 -1.2486 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 50.21 38.724 11.486 -1.9657
2 7 57.651 0.12019 1.3843 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 64.58 30.748 33.832 1.713
2 8 52.46 -0.036595 0.33237 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 50.54 24.407 26.133 -0.024531
2 10 43.438 -0.043226 0.4972 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 41.56 15.375 26.185 0.46007
2 12 35.967 0.18052 2.3421 8.7284 92.501 3.0141 -0.7097 0.18605 0.73689 37 179 93.21 1.2 1 1 1 0 0 0 0 42.46 9.6845 32.776 5.4087
3 0 0 0 0 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 0 0 0 0
3 1 8.7842 0.064412 0.39278 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 9.35 8.6417 0.70834 0.43295
3 2 8.7557 -0.010932 -0.40336 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 8.66 8.9007 -0.24073 -0.20004
3 3 7.8877 0.037055 0.061309 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 8.18 7.5467 0.63326 0.23779
3 4 7.0253 -0.11889 -0.96332 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 6.19 6.1036 0.086418 -0.95166
3 5 6.2486 0.13306 1.0866 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 7.08 4.8711 2.2089 1.2177
3 6 5.5568 -0.15779 -0.99186 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 4.68 3.8724 0.80764 -1.4247
3 7 4.9416 0.072537 0.85148 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 5.3 3.0748 2.2252 0.85019
3 8 4.3944 -0.044238 0.049046 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 4.2 2.4407 1.7593 -0.23439
3 10 3.4751 0.12801 1.4633 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 3.92 1.5375 2.3825 2.1772
3 12 2.7482 0.00066506 0.55527 11.024 93.942 2.2304 -0.4762 0.20151 0.43577 35 188 94.35 0.9 1 1 0 0 0 0 0 2.75 0.96845 1.7816 1.4402
4 0 0 0 0 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 0 0 0 0
4 1 63.799 -0.010173 0.44757 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 63.15 43.208 19.942 0.44742
4 2 55.019 0.13433 1.2202 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 62.41 44.504 17.906 1.5121
4 3 39.396 0.041987 0.20166 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 41.05 37.734 3.3163 0.27899
4 4 27.031 -0.10065 -1.0776 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 24.31 30.518 -6.2079 -0.95201
4 5 18.339 -0.0092034 -0.47843 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 18.17 24.356 -6.1855 -0.52772
4 6 12.404 0.066622 0.071143 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 13.23 19.362 -6.1318 -0.23497
4 7 8.3822 0.011663 -0.3334 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 8.48 15.374 -6.8942 -0.40566
4 9 3.826 0.024557 -0.17993 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 3.92 9.6859 -5.7659 -0.043349
4 11 1.7462 0.0079027 -0.25069 19.607 49.991 1.6776 0.099584 -0.42932 0.15097 67 168 74.39 0.8 2 2 0 0 0 0 0 1.76 6.1012 -4.3412 0.38238
5 0 0 0 0 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 0 0 0 0
5 1 19.492 0.10248 0.96785 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 21.49 17.283 4.2067 1.1867
5 2 18.528 -0.19743 -1.5134 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 14.87 17.801 -2.9315 -1.4188
5 3 16.141 -0.060132 -0.42243 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 15.17 15.093 0.076512 -0.38579
5 4 13.944 -0.018938 -0.031704 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 13.68 12.207 1.4728 -0.047322
5 5 12.036 0.2321 1.9294 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 14.83 9.7422 5.0878 2.0694
5 6 10.389 -0.050901 -0.17008 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 9.86 7.7447 2.1153 -0.37002
5 7 8.9667 -0.024163 0.056323 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 8.75 6.1497 2.6003 -0.14125
5 9 6.6798 0.16321 1.4828 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 7.77 3.8744 3.8956 1.7438
5 11 4.9761 -0.12382 -0.71336 12.47 84.706 2.4371 -0.35295 0.09802 0.52438 69 165 91.85 1 2 2 0 0 0 1 0 4.36 2.4405 1.9195 -0.76729
6 0 0 0 0 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 0 0 0 0
6 1 71.066 0.048749 0.25708 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 74.53 86.417 -11.887 0.18787
6 2 88.184 -0.28535 -2.1306 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 63.02 89.007 -25.987 -2.0607
6 3 84.934 0.18281 1.5438 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 100.46 75.467 24.993 1.6366
6 4 75.001 0.13224 1.2209 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 84.92 61.036 23.884 1.3909
6 5 63.797 -0.075034 -0.34688 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 59.01 48.711 10.299 -0.43263
6 6 53.316 0.010383 0.26499 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 53.87 38.724 15.146 0.28554
6 7 44.173 0.022122 0.30314 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 45.15 30.748 14.402 0.31129
6 8 36.439 -0.070216 -0.45133 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 33.88 24.407 9.4728 -0.5882
6 10 24.659 0.065321 0.4657 14.395 73.084 0.86833 -0.20941 -0.049554 -0.50761 52 157 104.3 0.8 2 2 0 1 1 1 0 26.27 15.375 10.895 0.43734
7 0 0 0 0 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 0 0 0 0
7 1 163.49 0.1302 1.8578 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 184.78 86.417 98.363 2.2385
7 2 129.36 0.10339 0.96819 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 142.73 89.007 53.723 1.201
7 3 96.217 -0.10431 -0.72471 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 86.18 75.467 10.713 -0.65983
7 4 71.277 -0.078944 -0.53927 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 65.65 61.036 4.6142 -0.27781
7 5 52.787 0.043248 0.4085 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 55.07 48.711 6.3589 0.6249
7 6 39.093 -0.029236 -0.11275 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 37.95 38.724 -0.7736 -0.032028
7 8 21.441 0.08439 0.7694 14.093 46.925 2.9834 -0.23062 -0.49263 0.72663 44 140 90.04 0.9 2 2 0 1 1 1 0 23.25 24.407 -1.1572 0.31745
8 0 0 0 0 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 0 0 0 0
8 1 34.26 0.101 -0.14372 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 37.72 43.208 -5.4883 0.35818
8 2 23.971 -0.20112 -2.5728 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 19.15 44.504 -25.354 -1.6326
8 3 16.11 -0.1055 -1.5138 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 14.41 37.734 -23.324 -1.123
8 4 10.808 0.10012 0.34578 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 11.89 30.518 -18.628 -0.41616
8 5 7.2505 -0.24419 -2.0502 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 5.48 24.356 -18.876 -1.0534
8 6 4.864 0.013561 0.012393 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 4.93 19.362 -14.432 -0.42656
8 7 3.263 0.22891 1.6775 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 4.01 15.374 -11.364 -0.041559
8 9 1.4685 0.048684 0.26851 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 1.54 9.6859 -8.1459 -0.093338
8 11 0.66089 -0.092132 -0.85236 42.15 105.58 3.5462 0.86495 0.31834 0.89946 50 173 98.88 0.9 2 2 1 1 0 0 0 0.6 6.1012 -5.5012 0.018256
9 0 0 0 0 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 0 0 0 0
9 1 55.324 -0.049604 -0.0014771 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 52.58 43.208 9.3717 -0.044004
9 2 49.276 0.17117 1.26 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 57.71 44.504 13.206 1.4433
9 3 36.235 -0.00013393 -0.32564 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 36.23 37.734 -1.5037 -0.24112
9 4 25.457 0.02645 -0.23908 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 26.13 30.518 -4.3879 -0.28108
9 5 17.661 0.023164 -0.24129 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 18.07 24.356 -6.2855 -0.39395
9 6 12.208 -0.11291 -1.17 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 10.83 19.362 -8.5318 -0.93881
9 8 5.82 -0.056694 -0.50093 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 5.49 12.204 -6.7136 -0.32287
9 10 2.7731 0.099863 0.82323 21.33 57.54 1.6074 0.18382 -0.28869 0.10819 61 160 81.42 0.9 2 2 0 1 0 1 0 3.05 7.6874 -4.6374 0.47335
10 0 0 0 0 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 0 0 0 0
10 1 56.258 0.097797 -0.40766 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 61.76 43.208 18.552 1.9491
10 2 17.428 0.055796 -2.5277 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 18.4 44.504 -26.104 -2.2177
10 3 4.9485 -0.025964 -1.8592 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 4.82 37.734 -32.914 -2.3911
10 6 0.1103 -0.27472 -0.40417 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 0.08 19.362 -19.282 -0.59368
10 8 0.0087302 0.14545 2.4362 44.279 34.914 3.5597 0.91423 -0.78827 0.90326 52 168 87.32 1.8 2 2 0 1 1 1 0 0.01 12.204 -12.194 0.15934
12 0 0 0 0 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 0 0 0 0
12 1 16.198 0.026669 0.041568 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 16.63 17.283 -0.65332 0.05277
12 2 15.599 0.056483 0.10904 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 16.48 17.801 -1.3215 0.062575
12 3 13.092 -0.15138 -1.4208 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 11.11 15.093 -3.9835 -1.2067
12 4 10.749 0.11453 0.67923 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 11.98 12.207 -0.22716 0.55165
12 5 8.7913 -0.12072 -1.0203 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 7.73 9.7422 -2.0122 -0.83406
12 6 7.1853 0.12451 0.89666 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 8.08 7.7447 0.33528 0.73228
12 7 5.872 -0.022481 -0.17281 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 5.74 6.1497 -0.40966 -0.12474
12 9 3.9214 -0.013119 -0.063403 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 3.87 3.8744 -0.0043587 -0.025436
12 11 2.6188 0.04246 0.35116 18.695 92.607 1.9254 0.051956 0.18719 0.28872 59 178 98.43 1.1 1 2 0 0 0 1 0 2.73 2.4405 0.28953 0.28459
13 0 0 0 0 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 0 0 0 0
13 1 13.355 0.046786 -0.33884 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 13.98 17.283 -3.3033 -0.22478
13 2 12.928 -0.065582 -1.0392 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 12.08 17.801 -5.7215 -0.63982
13 3 9.7203 -0.059697 -0.90908 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 9.14 15.093 -5.9535 -0.61478
13 4 6.7047 0.07536 0.14479 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 7.21 12.207 -4.9972 -0.16638
13 5 4.4567 0.068053 0.092608 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 4.76 9.7422 -4.9822 -0.34387
13 6 2.9115 -0.14133 -1.5011 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 2.5 7.7447 -5.2447 -0.93137
13 7 1.886 -0.050882 -0.83497 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 1.79 6.1497 -4.3597 -0.6245
13 8 1.2165 0.08505 0.16766 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 1.32 4.8814 -3.5614 -0.2851
13 10 0.50351 0.23136 1.2465 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 0.62 3.075 -2.455 0.12675
13 12 0.20789 -0.23037 -2.1245 34.673 78.32 1.1217 0.66968 0.019628 -0.25153 54 159 68.04 1.3 1 2 0 0 0 0 0 0.16 1.9369 -1.7769 0.19744
14 0 0 0 0 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 0 0 0 0
14 1 25.038 0.08036 1.1038 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 27.05 17.283 9.7667 1.3163
14 2 19.718 -0.019169 -0.1943 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 19.34 17.801 1.5385 -0.20323
14 3 14.053 -0.01869 -0.35064 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 13.79 15.093 -1.3035 -0.27732
14 4 9.9046 0.013668 -0.1362 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 10.04 12.207 -2.1672 -0.11491
14 5 6.9719 0.078621 0.36998 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 7.52 9.7422 -2.2222 0.14978
14 6 4.9067 -0.22963 -1.9321 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 3.78 7.7447 -3.9647 -1.2185
14 7 3.4532 0.0048506 -0.12578 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 3.47 6.1497 -2.6797 -0.2328
14 8 2.4303 0.27555 1.9526 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 3.1 4.8814 -1.7814 0.60454
14 10 1.2037 -0.16095 -1.2921 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 1.01 3.075 -2.065 -0.36266
14 12 0.59621 0.0063547 0.037434 20.27 57.701 2.48 0.13286 -0.2859 0.54182 62 180 81.65 1.1 1 2 0 0 0 1 0 0.6 1.9369 -1.3369 0.19247
15 0 0 0 0 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 0 0 0 0
15 1 9.2441 -0.039389 -0.9602 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 8.88 17.283 -8.4033 -0.76309
15 2 11.725 -0.055844 -1.1138 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 11.07 17.801 -6.7315 -0.93745
15 3 11.639 0.22775 1.1714 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 14.29 15.093 -0.80349 0.77297
15 4 10.663 -0.16442 -1.6265 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 8.91 12.207 -3.2972 -1.3204
15 5 9.4555 0.049129 0.15252 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 9.92 9.7422 0.17779 0.19822
15 6 8.2647 0.064765 0.41888 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 8.8 7.7447 1.0553 0.46686
15 7 7.1767 -0.053888 -0.34849 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 6.79 6.1497 0.64034 -0.25555
15 8 6.2131 -0.072925 -0.38045 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 5.76 4.8814 0.87856 -0.28711
15 10 4.6404 -0.138 -0.70211 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 4 3.075 0.92505 -0.55263
15 12 3.4608 0.16158 1.67 17.392 118.44 0.90406 -0.020267 0.43321 -0.46728 63 172 83.1 1.2 1 1 0 1 0 1 0 4.02 1.9369 2.0831 2.1753
16 0 0 0 0 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 0 0 0 0
16 1 10.618 -0.07796 -1.3419 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 9.79 17.283 -7.4933 -0.7232
16 2 15.114 0.058591 -0.12685 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 16 17.801 -1.8015 -0.20988
16 3 16.3 -0.032503 -0.50437 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 15.77 15.093 0.67651 -0.56435
16 4 15.778 0.084399 0.65615 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 17.11 12.207 4.9028 0.78632
16 5 14.454 0.14572 1.3152 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 16.56 9.7422 6.8178 1.7317
16 6 12.824 -0.10093 -0.43961 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 11.53 7.7447 3.7853 -0.74092
16 7 11.156 0.10349 1.1475 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 12.31 6.1497 6.1603 1.5476
16 8 9.5804 -0.069975 -0.17708 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 8.91 4.8814 4.0286 -0.48247
16 10 6.9067 0.025086 0.42631 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 7.08 3.075 4.005 0.45306
16 12 4.8984 0.012575 0.17456 12.654 70.52 0.53142 -0.33827 -0.085275 -0.99863 63 170 83.4 1.1 1 1 1 0 0 1 0 4.96 1.9369 3.0231 0.16488
17 0 0 0 0 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 0 0 0 0
17 1 159.08 0.10746 1.7129 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 176.18 86.417 89.763 2.3032
17 2 142.4 -0.026341 0.32532 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 138.65 89.007 49.643 0.27595
17 3 117.07 -0.036901 0.13158 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 112.75 75.467 37.283 0.20541
17 4 95.486 -0.02195 0.18219 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 93.39 61.036 32.354 0.3705
17 5 77.821 0.11513 1.1642 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 86.78 48.711 38.069 1.7808
17 6 63.419 0.05347 0.64493 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 66.81 38.724 28.086 0.94805
17 7 51.682 -0.097366 -0.54632 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 46.65 30.748 15.902 -0.80267
17 8 42.117 0.078656 0.7392 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 45.43 24.407 21.023 0.88122
17 10 27.97 0.13906 1.1194 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 31.86 15.375 16.485 1.1844
17 12 18.575 -0.11604 -0.86228 10.286 50.262 2.5232 -0.54547 -0.42393 0.55911 63 177 104.1 1 1 1 1 1 0 0 0 16.42 9.6845 6.7355 -1.2682
18 0 0 0 0 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 0 0 0 0
18 1 52.403 0.075311 0.72633 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 56.35 43.208 13.142 0.81678
18 2 49.435 -0.045611 -0.47678 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 47.18 44.504 2.6764 -0.49041
18 3 42.292 0.06191 0.40132 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 44.91 37.734 7.1763 0.46323
18 4 35.799 0.17153 1.3563 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 41.94 30.518 11.422 1.5409
18 5 30.267 -0.040529 -0.13071 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 29.04 24.356 4.6845 -0.10866
18 6 25.585 -0.32149 -2.1582 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 17.36 19.362 -2.0018 -2.4648
18 7 21.628 0.02414 0.53511 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 22.15 15.374 6.7758 0.61812
18 8 18.282 -0.012716 0.31563 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 18.05 12.204 5.8464 0.35718
18 10 13.064 0.044092 0.81309 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 13.64 7.6874 5.9526 0.97913
18 12 9.335 0.088372 1.1591 12.916 76.863 2.3225 -0.3178 0.00085212 0.47621 58 187 136.8 1.5 1 2 1 1 0 1 0 10.16 4.8422 5.3178 1.5056
19 0 0 0 0 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 0 0 0 0
19 1 52.559 0.072516 0.76543 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 56.37 43.208 13.162 0.95268
19 2 56.065 -0.16134 -0.90235 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 47.02 44.504 2.5164 -0.95875
19 3 50.272 0.16128 1.5843 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 58.38 37.734 20.646 2.0735
19 4 43.456 -0.15385 -0.78352 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 36.77 30.518 6.2521 -1.1225
19 5 37.236 0.046037 0.72311 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 38.95 24.356 14.594 0.79995
19 6 31.837 -0.08943 -0.3239 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 28.99 19.362 9.6282 -0.76221
19 7 27.207 0.20668 1.8825 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 32.83 15.374 17.456 2.4129
19 8 23.247 0.065957 0.77573 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 24.78 12.204 12.576 0.83367
19 10 16.971 -0.026555 -0.021789 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 16.52 7.6874 8.8326 -0.13776
19 12 12.389 -0.028155 -0.13583 10.697 67.987 1.5497 -0.50627 -0.12186 0.071634 66 177 97.3 1.2 1 1 1 0 0 1 0 12.04 4.8422 7.1978 0.089378
20 0 0 0 0 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 0 0 0 0
20 1 9.9678 0.091519 0.20135 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 10.88 17.283 -6.4033 -0.30566
20 2 13.099 -0.092299 -1.0564 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 11.89 17.801 -5.9115 -0.91293
20 3 13.113 -0.28162 -2.2839 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 9.42 15.093 -5.6735 -1.8347
20 4 11.842 0.15432 1.1607 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 13.67 12.207 1.4628 1.2351
20 5 10.167 0.078928 0.65618 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 10.97 9.7422 1.2278 0.811
20 6 8.489 0.084934 0.69922 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 9.21 7.7447 1.4653 0.80048
20 7 6.9724 -0.09213 -0.68983 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 6.33 6.1497 0.18034 -0.60324
20 9 4.5823 0.080241 0.45165 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 4.95 3.8744 1.0756 0.21005
20 11 2.9624 -0.03795 -0.60782 19.285 86.198 0.66436 0.08303 0.11548 -0.77536 67 181 96.1 1.3 1 1 1 0 0 1 0 2.85 2.4405 0.40953 -0.8834
21 0 0 0 0 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 0 0 0 0
21 1 2.5587 -0.12846 -1.8527 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 2.23 8.6417 -6.4117 -1.3363
21 3 3.9986 -0.11719 -1.9728 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 3.53 7.5467 -4.0167 -1.5215
21 4 3.8485 0.073141 -0.36714 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 4.13 6.1036 -1.9736 -0.35491
21 5 3.4729 0.1057 0.063667 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 3.84 4.8711 -1.0311 0.17075
21 6 3.0089 -0.03619 -0.84869 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 2.9 3.8724 -0.97236 -0.31258
21 7 2.5347 0.029706 -0.22688 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 2.61 3.0748 -0.46483 0.1279
21 8 2.0919 0.1951 1.11 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 2.5 2.4407 0.05928 0.96638
21 10 1.364 0.048414 0.090333 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 1.43 1.5375 -0.10748 0.03258
21 12 0.8541 -0.22726 -1.9483 29.942 96.832 0.34333 0.52299 0.23181 -1.4355 57 180 85.9 1.2 1 1 1 1 0 1 0 0.66 0.96845 -0.30845 -1.387
22 0 0 0 0 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 0 0 0 0
22 1 21.425 -0.077719 -1.1746 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 19.76 30.246 -10.486 -0.7882
22 2 21.267 0.12806 0.29417 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 23.99 31.153 -7.1625 0.14107
22 3 16.61 -0.04515 -1.019 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 15.86 26.414 -10.554 -0.75533
22 4 12.021 -0.00092238 -0.64037 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 12.01 21.363 -9.3525 -0.57866
22 5 8.4441 -0.088118 -1.2237 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 7.7 17.049 -9.3489 -0.84649
22 6 5.8572 0.073894 0.090724 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 6.29 13.553 -7.2633 -0.24963
22 7 4.0406 0.017175 -0.25591 34.239 91.155 1.185 0.65709 0.17139 -0.19664 56 170 88.13 0.8 1 2 0 1 0 1 0 4.11 10.762 -6.6519 -0.28444
23 0 0 0 0 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 0 0 0 0
23 1 14.522 0.062535 -0.24086 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 15.43 30.246 -14.816 -0.54914
23 2 18.514 -0.050452 -1.2076 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 17.58 31.153 -13.573 -0.68755
23 3 18.962 -0.058653 -1.1808 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 17.85 26.414 -8.5636 -0.61799
23 4 18.257 -0.04368 -0.91078 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 17.46 21.363 -3.9025 -0.56835
23 5 17.216 -0.15601 -1.5863 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 14.53 17.049 -2.5189 -1.4462
23 6 16.113 0.087317 0.42626 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 17.52 13.553 3.9667 0.27997
23 7 15.04 0.059171 0.37385 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 15.93 10.762 5.1681 0.15942
23 8 14.025 0.12089 0.98726 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 15.72 8.5425 7.1775 0.9871
23 10 12.184 0.035777 0.59187 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 12.62 5.3812 7.2388 1.0645
23 12 10.582 0.01205 0.60485 10.711 151.96 1.0701 -0.50501 0.68246 -0.29865 57 168 69.08 1.1 2 3 0 1 0 1 0 10.71 3.3896 7.3204 2.3721
24 0 0 0 0 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 0 0 0 0
24 1 15.286 0.0041836 -0.62099 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 15.35 30.246 -14.896 -0.38078
24 2 22.651 -0.26139 -2.583 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 16.73 31.153 -14.423 -1.9738
24 3 25.459 0.165 0.89281 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 29.66 26.414 3.2464 0.76662
24 4 25.717 0.15061 1.0648 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 29.59 21.363 8.2275 1.2045
24 5 24.609 -0.010938 0.091851 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 24.34 17.049 7.2911 0.076003
24 6 22.832 -0.13234 -0.62816 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 19.81 13.553 6.2567 -1.0711
24 7 20.785 0.23215 2.276 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 25.61 10.762 14.848 3.3746
24 8 18.694 -0.30994 -1.7252 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 12.9 8.5425 4.3575 -3.2514
24 10 14.808 -0.024145 0.53011 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 14.45 5.3812 9.0688 0.82837
24 12 11.558 0.094509 1.4235 11.016 84.57 0.50433 -0.47696 0.096414 -1.0509 56 175 74.6 0.8 2 1 0 0 0 0 0 12.65 3.3896 9.2604 3.2902
25 0 0 0 0 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 0 0 0 0
25 1 27.108 0.045074 0.062087 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 28.33 30.246 -1.9158 0.059699
25 2 26.713 -0.10494 -1.0418 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 23.91 31.153 -7.2425 -0.82566
25 3 21.158 0.061049 0.18656 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 22.45 26.414 -3.9636 0.13293
25 4 15.759 -0.013279 -0.42687 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 15.55 21.363 -5.8125 -0.36554
25 5 11.498 0.079363 0.22884 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 12.41 17.049 -4.6389 0.015907
25 6 8.3268 0.053231 0.0056508 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 8.77 13.553 -4.7833 -0.1556
25 7 6.0144 -0.030653 -0.63268 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 5.83 10.762 -4.9319 -0.45801
25 8 4.3399 -0.066798 -0.89231 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 4.05 8.5425 -4.4925 -0.49874
25 9 3.1305 0.079697 0.24096 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 3.38 6.7801 -3.4001 -0.019112
25 10 2.2578 0.00095507 -0.32484 25.561 78.202 1.3308 0.3648 0.018126 -0.080614 61 171 96.62 1 1 1 0 1 0 1 0 2.26 5.3812 -3.1212 -0.1289
26 0 0 0 0 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 0 0 0 0
26 1 10.933 -0.19237 -2.3438 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 8.83 30.246 -21.416 -1.0485
26 2 17.159 -0.018601 -1.1484 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 16.84 31.153 -14.313 -1.0352
26 3 20.26 0.23743 0.95609 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 25.07 26.414 -1.3436 0.47871
26 4 21.327 -0.066453 -1.0793 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 19.91 21.363 -1.4525 -1.0219
26 5 21.111 -0.020411 -0.45536 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 20.68 17.049 3.6311 -0.41992
26 6 20.12 0.093937 0.65715 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 22.01 13.553 8.4567 0.94118
26 7 18.697 0.066492 0.65103 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 19.94 10.762 9.1781 0.9748
26 8 17.068 -0.083069 -0.33027 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 15.65 8.5425 7.1075 -0.52165
26 10 13.724 0.053655 0.86037 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 14.46 5.3812 9.0788 1.5548
26 12 10.703 -0.0049923 0.42307 12.968 85.947 0.34318 -0.31378 0.11256 -1.4359 67 157 66.4 0.9 2 1 0 0 0 0 0 10.65 3.3896 7.2604 1.2194
27 0 0 0 0 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 0 0 0 0
27 1 43.322 -0.059603 -0.67606 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 40.74 43.208 -2.4683 -0.5567
27 2 47.499 0.1436 1.1242 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 54.32 44.504 9.8164 1.2218
27 3 40.227 -0.069291 -0.41888 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 37.44 37.734 -0.29372 -0.37531
27 4 31.112 -0.042162 -0.29649 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 29.8 30.518 -0.71791 -0.25469
27 5 23.106 0.14081 0.93692 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 26.36 24.356 2.0045 0.80361
27 6 16.823 0.012891 -0.18216 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 17.04 19.362 -2.3218 -0.1913
27 7 12.124 0.083804 0.2425 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 13.14 15.374 -2.2342 0.048293
27 8 8.6906 -0.088675 -1.1212 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 7.92 12.204 -4.2836 -0.77634
27 10 4.4333 -0.050362 -0.81843 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 4.21 7.6874 -3.4774 -0.48029
27 12 2.2534 0.056204 0.075791 19.905 58.701 0.9571 0.11469 -0.26872 -0.41027 56 177 97.4 1 1 1 0 0 0 0 0 2.38 4.8422 -2.4622 -0.0035763
28 0 0 0 0 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 0 0 0 0
28 1 82.497 0.020518 -0.0042123 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 84.19 86.417 -2.2266 -0.0087439
28 2 84.503 0.020086 -0.016928 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 86.2 89.007 -2.8073 -0.015087
28 3 72.533 0.020632 0.021114 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 74.03 75.467 -1.4374 0.025091
28 4 59.932 -0.028907 -0.31221 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 58.2 61.036 -2.8358 -0.28023
28 5 49.06 -0.041983 -0.37872 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 47 48.711 -1.7111 -0.34016
28 6 40.065 0.012614 0.054478 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 40.57 38.724 1.8464 0.053999
28 7 32.699 -0.036365 -0.30664 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 31.51 30.748 0.76168 -0.26431
28 8 26.684 0.13965 1.0269 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 30.41 24.407 6.0028 0.88148
28 10 17.767 0.13523 0.98635 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 20.17 15.375 4.7952 0.79859
28 12 11.83 -0.13865 -1.0902 17.21 84.631 1.5687 -0.030799 0.097126 0.083819 58 173 78.7 1.4 1 1 0 1 0 1 0 10.19 9.6845 0.50551 -0.77558
29 0 0 0 0 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 0 0 0 0
29 1 23.035 -0.042346 -1.6466 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 22.06 86.417 -64.357 -1.4038
29 3 36.649 0.10262 -0.54209 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 40.41 75.467 -35.057 -0.77187
29 4 35.703 -0.027824 -1.2294 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 34.71 61.036 -26.326 -0.78813
29 5 32.679 -0.20164 -2.2282 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 26.09 48.711 -22.621 -1.1968
29 6 28.777 -0.0026812 -0.43391 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 28.7 38.724 -10.024 0.013747
29 7 24.689 0.21794 1.4808 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 30.07 30.748 -0.67832 1.2974
29 8 20.793 -0.04007 -0.27172 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 19.96 24.407 -4.4472 0.081953
29 10 14.201 -0.29513 -1.9231 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 10.01 15.375 -5.3648 -1.1756
29 12 9.3842 0.13063 1.4557 30.478 127.08 0.40332 0.54072 0.50365 -1.2744 53 180 87.6 1.2 1 1 0 1 1 1 0 10.61 9.6845 0.92551 0.62989
30 0 0 0 0 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 0 0 0 0
30 1 67.041 -0.13769 -1.3959 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 57.81 86.417 -28.607 -1.0171
30 2 83.337 0.24255 1.62 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 103.55 89.007 14.543 1.5038
30 3 81.768 -0.070546 -0.5672 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 76 75.467 0.53256 -0.52349
30 4 74.513 -0.020446 -0.045599 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 72.99 61.036 11.954 -0.042819
30 5 66.006 -0.14356 -0.87616 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 56.53 48.711 7.8189 -1.158
30 6 57.792 0.039596 0.57833 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 60.08 38.724 21.356 0.58925
30 7 50.351 0.13285 1.3292 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 57.04 30.748 26.292 1.5879
30 8 43.775 -0.13445 -0.66524 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 37.89 24.407 13.483 -1.1351
30 10 33.016 0.1752 1.6968 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 38.8 15.375 23.425 2.4769
30 12 24.88 -0.067135 -0.14878 12.163 85.935 0.98068 -0.37785 0.11242 -0.38594 46 175 84.8 1.2 1 1 0 1 1 1 0 23.21 9.6845 13.526 0.15975
31 0 0 0 0 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 0 0 0 0
31 1 13.951 -0.043782 -0.75289 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 13.34 17.283 -3.9433 -0.59817
31 2 16.571 0.080816 0.37808 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 17.91 17.801 0.10855 0.3276
31 3 15.57 0.10664 0.74387 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 17.23 15.093 2.1365 0.74397
31 4 13.608 -0.13727 -0.99211 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 11.74 12.207 -0.46716 -1.0323
31 5 11.573 -0.10568 -0.69112 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 10.35 9.7422 0.60779 -0.75571
31 6 9.7344 0.1033 0.92165 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 10.74 7.7447 2.9953 0.9828
31 7 8.1504 0.15332 1.3131 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 9.4 6.1497 3.2503 1.4048
31 8 6.811 -0.057401 -0.27956 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 6.42 4.8814 1.5386 -0.33037
31 10 4.747 -0.10258 -0.63756 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 4.26 3.075 1.185 -0.64779
31 12 3.3061 0.076789 0.68937 15.109 83.503 1.0404 -0.16099 0.083716 -0.32683 30 157 61.7 1.1 2 1 0 0 0 0 0 3.56 1.9369 1.6231 0.84381
32 0 0 0 0 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 0 0 0 0
32 1 16.89 0.025433 -0.60485 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 17.32 30.246 -12.926 -0.50287
32 2 17.248 0.016928 -0.89339 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 17.54 31.153 -13.613 -0.72191
32 3 15.402 -0.17284 -2.192 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 12.74 26.414 -13.674 -1.4279
32 4 13.47 0.093527 0.038483 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 14.73 21.363 -6.6325 -0.10769
32 5 11.739 0.10912 0.36482 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 13.02 17.049 -4.0289 0.16554
32 6 10.224 -0.095282 -0.99892 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 9.25 13.553 -4.3033 -0.66201
32 7 8.9039 -0.11836 -1.0185 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 7.85 10.762 -2.9119 -0.64996
32 8 7.7539 0.083324 0.63662 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 8.4 8.5425 -0.14252 0.55234
32 10 5.8803 -0.1225 -0.73011 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 5.16 5.3812 -0.22117 -0.2945
32 12 4.4595 0.13018 1.2982 22.273 161.06 1.8949 0.2271 0.74059 0.27276 56 174 68.72 1.1 2 1 0 1 0 1 0 5.04 3.3896 1.6504 1.509
33 0 0 0 0 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 0 0 0 0
33 1 104.97 0.009229 0.17992 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 105.94 86.417 19.523 0.2723
33 2 110.96 0.085435 0.65142 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 120.44 89.007 31.433 0.92096
33 3 100.28 0.038245 0.34694 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 104.12 75.467 28.653 0.52264
33 4 88.03 0.034989 0.43044 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 91.11 61.036 30.074 0.54588
33 5 76.805 0.099661 1.0321 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 84.46 48.711 35.749 1.316
33 6 66.926 -0.12456 -0.56434 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 58.59 38.724 19.866 -1.1113
33 7 58.301 -0.074464 -0.10166 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 53.96 30.748 23.212 -0.47337
33 8 50.785 -0.14837 -0.59329 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 43.25 24.407 18.843 -1.2681
33 10 38.534 0.19454 2.0928 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 46.03 15.375 30.655 3.498
33 12 29.238 0.018546 0.80486 9.8142 71.098 1.6821 -0.59245 -0.077108 0.15361 54 180 76.43 1 1 1 0 0 0 0 0 29.78 9.6845 20.096 1.8245
34 0 0 0 0 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 0 0 0 0
34 2 8.056 0.052634 0.17397 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 8.48 8.9007 -0.42073 0.31395
34 3 5.8506 -0.078729 -0.73935 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 5.39 7.5467 -2.1567 -0.61276
34 4 4.0098 -0.22938 -1.9093 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 3.09 6.1036 -3.0136 -1.4096
34 5 2.6972 0.28283 1.8805 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 3.46 4.8711 -1.4111 0.62606
34 6 1.8027 -0.0070462 -0.40939 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 1.79 3.8724 -2.0824 -0.49897
34 7 1.2022 -0.10166 -1.2036 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 1.08 3.0748 -1.9948 -0.67305
34 9 0.53374 0.10542 0.29326 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 0.59 1.9372 -1.3472 0.0034805
34 10 0.35555 -0.071856 -1.0502 27.237 67.039 1.4591 0.4283 -0.1359 0.011414 34 170 77.34 1 1 1 0 0 0 1 0 0.33 1.5375 -1.2075 -0.066464
35 0 0 0 0 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 0 0 0 0
35 1 30.587 -0.084912 -1.2655 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 27.99 43.208 -15.218 -0.97925
35 2 35.888 0.19317 1.0618 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 42.82 44.504 -1.6836 0.86691
35 3 31.81 -0.19333 -1.6674 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 25.66 37.734 -12.074 -1.2151
35 4 25.24 -0.0226 -0.30112 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 24.67 30.518 -5.8479 -0.13315
35 5 18.906 -0.0082363 -0.23895 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 18.75 24.356 -5.6055 -0.15238
35 6 13.685 0.033996 -0.055101 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 14.15 19.362 -5.2118 -0.11957
35 7 9.6916 0.18246 0.88941 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 11.46 15.374 -3.9142 0.27518
35 10 3.2064 -0.1018 -1.6228 26.102 65.982 0.69312 0.38575 -0.15179 -0.73298 52 183 89.36 1.1 1 1 0 1 0 1 0 2.88 7.6874 -4.8074 -0.87327
36 0 0 0 0 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 0 0 0 0
36 1 130.23 0.035805 0.77244 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 134.89 86.417 48.473 1.006
36 2 100.26 0.069354 0.56696 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 107.21 89.007 18.203 0.74098
36 3 65.01 -0.027069 -0.51794 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 63.25 75.467 -12.217 -0.41929
36 4 40.676 -0.04932 -0.87035 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 38.67 61.036 -22.366 -0.7827
36 5 25.238 0.044073 -0.23503 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 26.35 48.711 -22.361 -0.538
36 6 15.627 -0.096435 -1.3136 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 14.12 38.724 -24.604 -0.94514
36 7 9.6713 0.2139 1.0438 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 11.74 30.748 -19.008 -0.13923
36 8 5.9847 0.015922 -0.42729 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 6.08 24.407 -18.327 -0.32327
36 10 2.2916 -0.083591 -1.0714 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 2.1 15.375 -13.275 0.12922
36 12 0.87743 0.014327 -0.18771 23.108 48.141 1.89 0.26389 -0.46703 0.27015 47 175 93.21 1.1 1 1 0 1 1 1 0 0.89 9.6845 -8.7945 0.74818
37 0 0 0 0 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 0 0 0 0
37 1 22.057 0.057282 -0.20782 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 23.32 43.208 -19.888 -0.46085
37 2 31.667 -0.18653 -1.9498 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 25.76 44.504 -18.744 -1.5352
37 3 34.285 0.078307 0.31516 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 36.97 37.734 -0.76372 0.36346
37 8 19.366 0.14685 1.3638 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 22.21 12.204 10.006 1.7239
37 12 9.1165 -0.068725 -0.65874 15.989 75.886 0.46892 -0.1044 -0.011938 -1.1237 66 155 93.44 1.4 1 1 0 1 1 1 0 8.49 4.8422 3.6478 -1.1485
38 0 0 0 0 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 0 0 0 0
38 1 65.378 0.084771 1.2398 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 70.92 43.208 27.712 1.5131
38 2 50.928 -0.06241 -0.51061 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 47.75 44.504 3.2464 -0.5589
38 3 35.807 0.14138 0.80403 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 40.87 37.734 3.1363 0.7627
38 4 24.883 -0.14639 -1.4329 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 21.24 30.518 -9.2779 -1.1218
38 6 11.979 0.047651 0.082904 19.983 54.656 2.4626 0.1186 -0.34011 0.53479 66 160 62.14 1.1 1 2 1 0 0 1 0 12.55 19.362 -6.8118 -0.093915
39 0 0 0 0 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 0 0 0 0
39 1 6.3145 0.00087858 -0.46855 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 6.32 8.6417 -2.3217 -0.42192
39 2 6.8335 0.066807 0.016756 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 7.29 8.9007 -1.6107 -0.048601
39 3 6.1532 -0.093156 -1.0872 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 5.58 7.5467 -1.9667 -0.87159
39 4 5.3149 0.076212 0.32471 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 5.72 6.1036 -0.38358 0.26075
39 5 4.5416 -0.16108 -1.3518 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 3.81 4.8711 -1.0611 -1.1438
39 7 3.2943 0.05637 0.46374 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 3.48 3.0748 0.40517 0.43342
39 8 2.804 0.13052 1.0776 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 3.17 2.4407 0.72928 1.019
39 10 2.0312 -0.049827 -0.2315 17.788 110.33 1.4649 0.002242 0.36232 0.015385 51 178 97.07 1 1 1 0 1 0 1 0 1.93 1.5375 0.39252 -0.034015
40 0 0 0 0 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 0 0 0 0
40 1 29.012 0.1671 0.91597 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 33.86 30.246 3.6142 0.9144
40 2 23.845 -0.23003 -2.336 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 18.36 31.153 -12.793 -1.8667
40 3 18.905 0.02989 -0.19343 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 19.47 26.414 -6.9436 -0.2196
40 4 14.968 0.058919 0.2017 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 15.85 21.363 -5.5125 0.12838
40 5 11.85 -0.098769 -0.84781 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 10.68 17.049 -6.3689 -0.51569
40 6 9.3822 -0.16437 -1.2338 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 7.84 13.553 -5.7133 -0.73973
40 7 7.428 0.051428 0.47817 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 7.81 10.762 -2.9519 0.24226
40 8 5.8808 -0.0069455 0.088658 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 5.84 8.5425 -2.7025 -0.020674
40 10 3.6862 0.24791 2.0591 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 4.6 5.3812 -0.78117 0.82918
40 12 2.3105 -0.13873 -0.87306 22.991 98.438 3.5003 0.25881 0.24826 0.88642 24 181 80.29 1.4 1 1 0 0 0 1 0 1.99 3.3896 -1.3996 -0.55768
41 0 0 0 0 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 0 0 0 0
41 1 42.425 0.07013 0.82671 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 45.4 30.246 15.154 1.1902
41 2 46.534 -0.052953 0.08916 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 44.07 31.153 12.917 0.17488
41 3 42.051 -0.11346 -0.31044 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 37.28 26.414 10.866 -0.57616
41 4 36.263 0.13918 1.5993 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 41.31 21.363 19.947 2.2912
41 5 30.864 0.21047 2.1101 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 37.36 17.049 20.311 3.0955
41 6 26.169 -0.074097 -0.083042 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 24.23 13.553 10.677 -0.57839
41 7 22.163 0.010232 0.50993 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 22.39 10.762 11.628 0.42658
41 8 18.764 -0.1761 -0.9463 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 15.46 8.5425 6.9175 -2.0041
41 10 13.447 0.12735 1.2584 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 15.16 5.3812 9.7788 2.0242
41 12 9.6366 -0.0089838 0.14241 9.3156 55.911 1.385 -0.64459 -0.3174 -0.040757 33 176 80.74 1.3 1 1 0 0 0 1 0 9.55 3.3896 6.1604 0.50509
42 0 0 0 0 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 0 0 0 0
42 1 111.85 -0.0077447 0.040727 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 110.98 86.417 24.563 0.83778
42 2 64.951 0.11376 -0.32425 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 72.34 89.007 -16.667 -0.1692
42 3 31.46 -0.017498 -1.6388 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 30.91 75.467 -44.557 -1.4801
42 4 14.636 -0.043427 -1.5594 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 14 61.036 -47.036 -1.5545
42 6 3.0945 0.011466 -0.37315 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 3.13 38.724 -35.594 -0.75353
42 8 0.65147 0.028437 0.030147 37.266 47.824 2.104 0.74181 -0.47365 0.37743 41 168 84.37 1.5 1 1 0 1 0 1 0 0.67 24.407 -23.737 0.26485
43 0 0 0 0 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 0 0 0 0
43 1 4.8318 -0.029349 -0.92316 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 4.69 8.6417 -3.9517 -0.73637
43 2 5.7721 0.0048286 -0.7521 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 5.8 8.9007 -3.1007 -0.68968
43 3 5.5047 0.071813 -0.13728 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 5.9 7.5467 -1.6467 -0.20736
43 4 4.9146 0.064187 -0.026981 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 5.23 6.1036 -0.87358 -0.040734
43 5 4.2865 0.091789 0.36311 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 4.68 4.8711 -0.19111 0.32682
43 6 3.7062 -0.039441 -0.45566 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 3.56 3.8724 -0.31236 -0.31008
43 7 3.1936 -0.22345 -1.6905 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 2.48 3.0748 -0.59483 -1.3559
43 8 2.7484 -0.090374 -0.54712 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 2.5 2.4407 0.05928 -0.35711
43 10 2.033 0.052632 0.74239 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 2.14 1.5375 0.60252 0.89509
43 12 1.5033 0.11092 1.3101 19.034 126.07 1.0944 0.069941 0.4957 -0.27622 38 173 67.13 1.2 1 1 0 0 0 1 0 1.67 0.96845 0.70155 1.6255
44 0 0 0 0 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 0 0 0 0
44 1 29.118 -0.0074969 -0.39423 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 28.9 30.246 -1.3458 0.07159
44 2 22.868 -0.078172 -1.292 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 21.08 31.153 -10.073 -0.61229
44 3 13.778 0.095262 -0.23298 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 15.09 26.414 -11.324 -0.33481
44 4 7.5363 0.1637 0.18808 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 8.77 21.363 -12.593 -0.61694
44 5 3.9402 -0.13456 -1.9837 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 3.41 17.049 -13.639 -1.3182
44 6 2.0122 -0.085575 -1.3889 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 1.84 13.553 -11.713 -1.0482
44 7 1.0145 0.074461 0.067997 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 1.09 10.762 -9.6719 -0.64268
44 8 0.50775 -0.054661 -0.7355 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 0.48 8.5425 -8.0625 -0.36252
44 10 0.12585 0.11245 0.59701 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 0.14 5.3812 -5.2412 0.36631
44 12 0.031006 -0.032431 -0.53686 40.086 57.104 1.2388 0.81476 -0.2963 -0.1523 54 183 103.4 1.1 1 1 1 0 0 0 0 0.03 3.3896 -3.3596 0.92899
45 0 0 0 0 27.758 59.022 1.7103 0.44725 -0.26326 0.17026 51 170 83.14 1.2 1 2 1 1 0 0 0 0 0 0 0
45 1 51.88 0.021598 0.29266 27.758 59.022 1.7103 0.44725 -0.26326 0.17026 51 170 83.14 1.2 1 2 1 1 0 0 0 53 43.208 9.7917 0.48391
45 4 17.683 0.016803 -0.90724 27.758 59.022 1.7103 0.44725 -0.26326 0.17026 51 170 83.14 1.2 1 2 1 1 0 0 0 17.98 30.518 -12.538 -0.97096
45 8 2.7139 0.013314 -0.69663 27.758 59.022 1.7103 0.44725 -0.26326 0.17026 51 170 83.14 1.2 1 2 1 1 0 0 0 2.75 12.204 -9.4536 -0.69102
45 11 0.662 0.012079 -0.3429 27.758 59.022 1.7103 0.44725 -0.26326 0.17026 51 170 83.14 1.2 1 2 1 1 0 0 0 0.67 6.1012 -5.4312 0.18093
46 0 0 0 0 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 0 0 0 0
46 1 155.17 0.043118 1.0758 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 161.86 86.417 75.443 1.3233
46 2 148.42 0.061672 1.2032 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 157.57 89.007 68.563 1.6001
46 3 119.59 -0.022587 0.38414 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 116.89 75.467 41.423 0.44874
46 4 92.993 0.09879 1.1092 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 102.18 61.036 41.144 1.5771
46 5 71.708 -0.02814 -0.016067 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 69.69 48.711 20.979 0.029125
46 6 55.183 -0.12491 -0.87664 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 48.29 38.724 9.5664 -1.0206
46 7 42.445 0.12145 0.89345 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 47.6 30.748 16.852 1.0753
46 8 32.644 0.19411 1.3792 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 38.98 24.407 14.573 1.4718
46 9 25.105 -0.15474 -1.2994 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 21.22 19.372 1.8482 -1.2219
46 11 14.848 -0.019389 -0.29839 11.674 44.455 1.6743 -0.41893 -0.5467 0.14899 58 159 69.31 1.1 2 2 1 1 0 1 0 14.56 12.202 2.3577 -0.26327
47 0 0 0 0 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 0 0 0 0
47 1 70.883 0.04003 -0.21059 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 73.72 86.417 -12.697 -0.12092
47 4 62.328 0.0070931 -0.3373 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 62.77 61.036 1.7342 -0.14199
47 5 56.48 -0.074364 -0.72454 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 52.28 48.711 3.5689 -0.70447
47 8 41.969 -0.0033161 0.36407 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 41.83 24.407 17.423 0.19481
47 12 28.244 0.091917 1.5126 11.253 113.65 1.9601 -0.45564 0.39195 0.30658 56 187 108.2 1 1 1 1 1 1 1 0 30.84 9.6845 21.156 2.9455
48 0 0 0 0 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 0 0 0 0
48 1 91.965 0.30452 3.3693 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 119.97 43.208 76.762 4.9656
48 2 74.907 -0.26162 -1.4145 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 55.31 44.504 10.806 -2.7221
48 3 59.571 -0.12675 -0.46176 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 52.02 37.734 14.286 -0.36738
48 4 47.346 0.04359 0.79343 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 49.41 30.518 18.892 1.7875
48 5 37.63 0.077869 1.0297 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 40.56 24.356 16.204 2.0446
48 6 29.907 0.1345 1.4411 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 33.93 19.362 14.568 2.3457
48 7 23.77 -0.023129 0.23451 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 23.22 15.374 7.8458 0.46965
48 8 18.892 -0.19436 -1.0742 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 15.22 12.204 3.0164 -1.3597
48 9 15.015 -0.18414 -1.0108 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 12.25 9.6859 2.5641 -1.4216
48 10 11.933 0.14301 1.4484 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 13.64 7.6874 5.9526 1.0879
48 11 9.4844 -0.18603 -1.0597 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 7.72 6.1012 1.6188 -1.6641
48 12 7.538 0.21783 1.9742 10.28 44.757 3.9249 -0.54605 -0.53992 1.0009 63 178 93.8 1 1 1 1 1 0 1 0 9.18 4.8422 4.3378 1.1273
49 0 0 0 0 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 0 0 0 0
49 1 19.858 -0.038661 -1.1673 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 19.09 43.208 -24.118 -0.91699
49 2 24.671 0.072531 -0.3769 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 26.46 44.504 -18.044 -0.49753
49 3 24.37 0.0041092 -0.75043 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 24.47 37.734 -13.264 -0.59118
49 4 22.474 -0.090953 -1.2758 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 20.43 30.518 -10.088 -0.89483
49 5 20.214 0.19125 1.0555 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 24.08 24.356 -0.27553 0.84064
49 6 18.006 -0.24524 -2.0661 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 13.59 19.362 -5.7718 -1.6002
49 7 15.977 0.054027 0.35755 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 16.84 15.374 1.4658 0.40266
49 8 14.155 -0.085109 -0.55881 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 12.95 12.204 0.7464 -0.36749
49 10 11.094 -0.034575 0.029225 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 10.71 7.6874 3.0226 0.34774
49 12 8.6901 0.12196 1.3404 18.413 150.76 1.0291 0.036796 0.67452 -0.33777 50 157 125.4 0.7 2 1 1 1 0 1 0 9.75 4.8422 4.9078 2.1237
50 0 0 0 0 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 0 0 0 0
50 1 64.334 0.021547 0.47761 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 65.72 43.208 22.512 1.2342
50 2 40.817 0.032168 -0.29339 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 42.13 44.504 -2.3736 0.047346
50 3 20.617 -0.0095633 -1.2297 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 20.42 37.734 -17.314 -1.0224
50 4 9.7311 0.18589 -0.050894 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 11.54 30.518 -18.978 -0.88576
50 5 4.4821 -0.020538 -1.5527 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 4.39 24.356 -19.966 -1.2817
50 6 2.0451 -0.05628 -1.4799 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 1.93 19.362 -17.432 -1.0146
50 7 0.92971 0.0756 -0.028513 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 1 15.374 -14.374 -0.55654
50 8 0.42203 -0.14699 -1.2748 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 0.36 12.204 -11.844 -0.19676
50 10 0.086844 -0.078808 -0.19622 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 0.08 7.6874 -7.6074 0.58305
50 12 0.017862 0.11968 1.2797 31.166 39.413 1.7095 0.56306 -0.66706 0.16978 62 147 51.03 0.8 2 2 0 1 0 1 0 0.02 4.8422 -4.8222 1.153
51 0 0 0 0 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 0 0 0 0
51 1 41.926 0.016316 -0.052739 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 42.61 43.208 -0.59831 0.057734
51 2 48.138 -0.0095153 -0.23855 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 47.68 44.504 3.1764 -0.11163
51 3 45.308 0.051693 0.32564 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 47.65 37.734 9.9163 0.44199
51 4 40.582 0.10787 0.88095 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 44.96 30.518 14.442 1.0462
51 5 35.829 0.020405 0.35171 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 36.56 24.356 12.204 0.28686
51 6 31.494 0.027807 0.5304 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 32.37 19.362 13.008 0.46519
51 8 24.259 -0.18506 -0.87912 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 19.77 12.204 7.5664 -1.6853
51 10 18.673 -0.086927 0.004544 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 17.05 7.6874 9.3626 -0.089808
51 12 14.372 0.16752 2.0196 10.521 80.379 1.3062 -0.52289 0.045589 -0.09933 48 185 96.3 1.1 1 2 0 1 1 1 0 16.78 4.8422 11.938 4.0157
52 0 0 0 0 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 0 0 0 0
52 1 29.898 -0.069827 -0.98192 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 27.81 43.208 -15.398 -0.77681
52 2 37.118 0.20049 1.1901 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 44.56 44.504 0.05637 1.0472
52 3 36.741 -0.2458 -2.0019 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 27.71 37.734 -10.024 -1.6992
52 4 34.017 0.049183 0.39343 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 35.69 30.518 5.1721 0.50046
52 5 30.753 0.061365 0.61588 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 32.64 24.356 8.2845 0.64794
52 6 27.552 -0.27155 -1.8036 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 20.07 19.362 0.7082 -2.4367
52 7 24.597 0.077774 0.91206 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 26.51 15.374 11.136 0.91045
52 8 21.929 0.12681 1.3331 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 24.71 12.204 12.506 1.5632
52 10 17.407 0.036912 0.69738 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 18.05 7.6874 10.363 1.0647
52 12 13.812 0.00054611 0.40869 11.746 101.52 1.0476 -0.4128 0.27905 -0.31988 57 165 70.53 1 2 1 0 0 0 1 0 13.82 4.8422 8.9778 1.3942
53 0 0 0 0 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 0 0 0 0
53 1 13.954 0.029075 0.76368 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 14.36 8.6417 5.7183 0.88273
53 2 13.293 0.10737 1.4841 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 14.72 8.9007 5.8193 1.9432
53 3 10.429 -0.13514 -0.52822 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 9.02 7.5467 1.4733 -0.73631
53 4 7.8078 -0.026608 0.034805 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 7.6 6.1036 1.4964 0.094219
53 5 5.7685 0.11813 0.88532 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 6.45 4.8711 1.5789 1.028
53 6 4.2456 0.14472 0.89283 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 4.86 3.8724 0.98764 0.90276
53 7 3.1211 0.034878 -0.072182 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 3.23 3.0748 0.15517 -0.05445
53 8 2.2937 -0.080107 -1.021 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 2.11 2.4407 -0.33072 -0.78514
53 10 1.2385 -0.06341 -0.93106 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 1.16 1.5375 -0.37748 -0.58548
53 12 0.66873 0.046761 -0.051008 14.31 46.439 1.5242 -0.2153 -0.50303 0.055049 67 160 83.24 1 2 1 0 0 0 1 0 0.7 0.96845 -0.26845 -0.026616
54 0 0 0 0 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 0 0 0 0
54 1 16.696 -0.10158 -1.538 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 15 43.208 -28.208 -1.0821
54 2 20.886 -0.013198 -1.0871 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 20.61 44.504 -23.894 -0.95479
54 3 19.596 -0.061036 -1.4116 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 18.4 37.734 -19.334 -0.93684
54 4 16.344 -0.021058 -1.0055 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 16 30.518 -14.518 -0.58044
54 5 12.781 0.19712 0.7455 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 15.3 24.356 -9.0555 0.33649
54 6 9.5949 -0.013021 -0.76683 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 9.47 19.362 -9.8918 -0.37099
54 7 7.0036 0.093723 0.097576 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 7.66 15.374 -7.7142 -0.064523
54 8 5.0081 -0.12741 -1.5299 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 4.37 12.204 -7.8336 -0.6603
54 10 2.4512 0.1831 0.91602 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 2.9 7.6874 -4.7874 -0.070088
54 12 1.152 -0.21007 -1.9179 41.226 90.404 0.48268 0.84279 0.16312 -1.0948 39 169 78.25 1 2 2 1 1 0 1 0 0.91 4.8422 -3.9322 -0.55164
55 0 0 0 0 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 0 0 0 0
55 1 98.934 -0.0063036 -0.051249 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 98.31 86.417 11.893 0.23794
55 2 77.626 0.0329 -0.16284 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 80.18 89.007 -8.8273 0.085764
55 3 49.121 0.10056 0.080936 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 54.06 75.467 -21.407 -0.094394
55 4 29.293 -0.077255 -1.3093 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 27.03 61.036 -34.006 -1.1052
55 5 17.132 -0.16705 -1.8756 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 14.27 48.711 -34.441 -1.3095
55 6 9.9522 0.26203 1.5476 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 12.56 38.724 -26.164 -0.34688
55 7 5.7677 -0.23886 -2.0685 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 4.39 30.748 -26.358 -0.83137
55 9 1.9333 0.060356 0.41203 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 2.05 19.372 -17.322 0.12404
55 11 0.64769 0.0035692 0.014354 31.516 57.635 1.5806 0.57423 -0.28705 0.091398 47 168 72.12 0.7 2 1 0 1 0 1 0 0.65 12.202 -11.552 0.73948
56 0 0 0 0 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 0 0 0 0
56 1 13.86 0.07937 1.2736 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 14.96 8.6417 6.3183 1.6119
56 4 9.2232 -0.054558 -0.030028 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 8.72 6.1036 2.6164 -0.030142
56 5 7.7254 0.12097 1.3176 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 8.66 4.8711 3.7889 1.8568
56 8 4.5377 -0.034762 0.14923 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 4.38 2.4407 1.9393 0.02635
56 10 3.1826 -0.047938 0.022892 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 3.03 1.5375 1.4925 -0.12526
56 12 2.2321 0.088658 1.0122 10.241 57.74 2.3185 -0.54985 -0.28522 0.47449 36 157 88.45 0.9 2 2 0 1 0 1 0 2.43 0.96845 1.4616 1.3828
57 0 0 0 0 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 0 0 0 0
57 1 6.9227 -0.13473 -1.9686 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 5.99 43.208 -37.218 -1.2801
57 2 11.293 0.17418 0.02493 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 13.26 44.504 -31.244 -1.0617
57 3 13.817 -0.045355 -1.688 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 13.19 37.734 -24.544 -1.3239
57 4 15.026 -0.058309 -1.681 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 14.15 30.518 -16.368 -1.0914
57 5 15.32 0.063948 -0.57015 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 16.3 24.356 -8.0555 -0.27575
57 6 14.995 -0.040379 -1.1387 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 14.39 19.362 -4.9718 -0.44399
57 7 14.27 -0.11492 -1.4757 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 12.63 15.374 -2.7442 -0.61941
57 8 13.302 0.11408 0.47274 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 14.82 12.204 2.6164 1.0388
57 9 12.207 0.10101 0.56827 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 13.44 9.6859 3.7541 1.2499
57 10 11.063 -0.071712 -0.57104 24.453 122.69 0.20826 0.32048 0.46847 -1.9354 63 173 73.94 0.9 1 1 0 1 0 1 0 10.27 7.6874 2.5826 0.23467
58 0 0 0 0 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 0 0 0 0
58 1 25.711 0.0077392 -0.52934 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 25.91 30.246 -4.3358 -0.31093
58 4 11.812 0.10057 -0.14975 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 13 21.363 -8.3625 -0.40013
58 5 8.5492 -0.027977 -0.79564 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 8.31 17.049 -8.7389 -0.71563
58 6 6.1835 -0.26579 -2.2875 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 4.54 13.553 -9.0133 -1.2258
58 7 4.4719 0.089028 0.65405 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 4.87 10.762 -5.8919 -0.0065646
58 8 3.234 -0.093995 -0.53679 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 2.93 8.5425 -5.6125 -0.26766
58 10 1.6913 0.12338 1.3084 31.161 96.146 2.0525 0.56289 0.2247 0.35264 61 178 67.27 1.1 1 1 0 0 0 1 0 1.9 5.3812 -3.4812 0.45902
59 0 0 0 0 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 0 0 0 0
59 1 33.55 0.078676 0.24571 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 36.19 43.208 -7.0183 0.087441
59 2 43.409 -0.077621 -0.72811 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 40.04 44.504 -4.4636 -0.48629
59 3 43.977 -0.18503 -1.3119 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 35.84 37.734 -1.8937 -1.2469
59 4 41.13 0.19036 1.7016 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 48.96 30.518 18.442 2.1969
59 6 33.242 -0.076159 -0.13002 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 30.71 19.362 11.348 -0.64824
59 7 29.481 0.019978 0.63302 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 30.07 15.374 14.696 0.38181
59 8 26.068 0.13434 1.5092 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 29.57 12.204 17.366 1.8318
59 12 15.805 -0.012985 0.29449 10.252 81.606 0.88693 -0.54877 0.060736 -0.48642 53 157 55.02 0.8 2 1 0 1 0 1 0 15.6 4.8422 10.758 0.87419
60 0 0 0 0 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 0 0 0 0
60 1 65.817 0.1099 1.3867 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 73.05 43.208 29.842 1.603
60 2 52.331 -0.015504 -0.11814 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 51.52 44.504 7.0164 -0.19234
60 3 38.758 0.031012 0.13663 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 39.96 37.734 2.2263 0.23536
60 4 28.55 -0.17933 -1.464 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 23.43 30.518 -7.0879 -1.1412
60 5 21.021 0.074619 0.4641 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 22.59 24.356 -1.7655 0.46111
60 6 15.478 -0.054116 -0.50451 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 14.64 19.362 -4.7218 -0.3316
60 7 11.396 0.081105 0.51746 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 12.32 15.374 -3.0542 0.26559
60 8 8.3904 0.14178 0.96932 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 9.58 12.204 -2.6236 0.4183
60 10 4.5484 -0.070011 -0.65437 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 4.23 7.6874 -3.4574 -0.40335
60 12 2.4657 0.0057951 -0.095081 17.665 57.7 2.833 -0.0047027 -0.28592 0.67493 55 173 78.7 1.1 1 1 0 0 0 1 0 2.48 4.8422 -2.3622 -0.14452
61 0 0 0 0 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 0 0 0 0
61 1 10.173 0.042983 0.39558 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 10.61 8.6417 1.9683 0.56282
61 2 7.8553 0.0286 -0.31271 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 8.08 8.9007 -0.82073 -0.2251
61 4 3.6987 -0.1024 -1.3165 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 3.32 6.1036 -2.7836 -1.0066
61 5 2.5157 0.081193 0.18395 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 2.72 4.8711 -2.1511 -0.25076
61 6 1.7107 -0.129 -1.3038 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 1.49 3.8724 -2.3824 -0.85046
61 7 1.1632 0.083215 0.38056 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 1.26 3.0748 -1.8148 -0.17939
61 8 0.79094 0.12525 0.75252 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 0.89 2.4407 -1.5507 -0.0099744
61 9 0.53781 -0.088893 -0.83357 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 0.49 1.9372 -1.4472 -0.24132
61 10 0.36569 0.039134 0.15552 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 0.38 1.5375 -1.1575 0.096285
61 11 0.24865 0.0054093 -0.087711 26.588 68.93 2.3835 0.40416 -0.10808 0.50213 58 179 94.57 1 1 1 0 0 0 1 0 0.25 1.2202 -0.97023 0.20726
62 0 0 0 0 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 0 0 0 0
62 1 48.825 0.042082 0.38354 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 50.88 43.208 7.6717 0.46762
62 2 49.427 0.028999 0.13628 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 50.86 44.504 6.3564 0.23936
62 3 43.616 -0.044391 -0.37604 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 41.68 37.734 3.9463 -0.35678
62 4 37.655 0.029611 0.27649 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 38.77 30.518 8.2521 0.31782
62 5 32.384 0.099605 0.89461 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 35.61 24.356 11.254 1.012
62 6 27.833 -0.10536 -0.5843 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 24.9 19.362 5.5382 -0.84881
62 7 23.918 0.03773 0.55242 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 24.82 15.374 9.4458 0.57698
62 8 20.553 0.016402 0.42908 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 20.89 12.204 8.6864 0.45776
62 10 15.177 0.052263 0.73677 11.93 78.686 1.8773 -0.39721 0.024295 0.26342 56 179 102.3 1.2 1 1 0 1 0 1 0 15.97 7.6874 8.2826 1.0815
63 0 0 0 0 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 0 0 0 0
63 1 36.355 0.089813 0.7464 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 39.62 30.246 9.3742 0.79999
63 2 37.855 -0.093635 -0.29394 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 34.31 31.153 3.1575 -0.30342
63 3 31.309 -0.17276 -0.86927 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 25.9 26.414 -0.5136 -0.93294
63 4 24.159 0.2066 1.8465 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 29.15 21.363 7.7875 2.0214
63 5 18.174 0.035573 0.3308 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 18.82 17.049 1.7711 0.37394
63 7 10.041 0.056623 0.092125 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 10.61 10.762 -0.15191 0.040888
63 8 7.4372 -0.049368 -0.83804 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 7.07 8.5425 -1.4725 -0.65804
63 10 4.0733 0.10476 0.21042 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 4.5 5.3812 -0.88117 -0.074768
63 11 3.0138 -0.094157 -1.3027 16.987 56.372 1.1992 -0.043846 -0.30921 -0.18473 66 182 94.8 1.1 1 1 0 0 0 1 0 2.73 4.2708 -1.5408 -0.79538
64 0 0 0 0 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 0 0 0 0
64 1 23.523 0.11166 0.12184 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 26.15 43.208 -17.058 -0.37981
64 2 31.116 -0.11653 -1.4177 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 27.49 44.504 -17.014 -1.1551
64 3 31.287 -0.13158 -1.263 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 27.17 37.734 -10.564 -0.95383
64 4 28.326 0.073933 0.50581 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 30.42 30.518 -0.09791 0.6314
64 5 24.338 0.019816 0.2278 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 24.82 24.356 0.46447 0.43468
64 6 20.305 0.019944 0.28571 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 20.71 19.362 1.3482 0.45596
64 8 13.495 -0.23305 -1.6602 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 10.35 12.204 -1.8536 -1.3981
64 9 10.865 0.30509 2.3606 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 14.18 9.6859 4.4941 1.8678
64 11 6.9591 0.048992 0.32082 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 7.3 6.1012 1.1988 -0.047073
64 12 5.5504 -0.18745 -1.5037 20.352 88.502 0.63827 0.13691 0.14186 -0.81541 48 183 111.8 1.2 1 1 0 0 0 1 0 4.51 4.8422 -0.33225 -1.4288
65 0 0 0 0 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 0 0 0 0
65 1 22.273 -0.10476 -1.4684 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 19.94 43.208 -23.268 -1.0476
65 2 28.523 0.18992 0.6221 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 33.94 44.504 -10.564 0.40245
65 3 29.321 0.028608 -0.5076 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 30.16 37.734 -7.5737 -0.18064
65 4 28.322 -0.17131 -1.8497 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 23.47 30.518 -7.0479 -1.4358
65 5 26.784 0.017023 -0.22955 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 27.24 24.356 2.8845 -0.19065
65 6 25.138 0.021166 -0.0012876 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 25.67 19.362 6.3082 -0.12218
65 7 23.527 -0.10105 -0.73737 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 21.15 15.374 5.7758 -1.1615
65 9 20.559 -0.0077409 0.30669 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 20.4 9.6859 10.714 0.35509
65 10 19.212 -0.056329 0.086922 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 18.13 7.6874 10.443 0.29468
65 12 16.775 0.20297 2.3031 9.5706 141.07 1.0608 -0.61758 0.60809 -0.30738 64 180 99.79 1.1 1 1 0 0 0 1 0 20.18 4.8422 15.338 5.9987

"
        )
                                        #data("xptab1")
    #cat("TABLE NO.  1\n",file="xptab1")
    #cat(names(xptab1),file="xptab1",append=T)
    #cat("\n",file="xptab1",append=T)
    #write.table(xptab1,sep=" ",file="xptab1",append=T,quote=F,row.names=F,col.names=F)
    #rm(xptab1,pos=1)
  }

  writeExt <- function() {
    cat(file="run1.ext","  TABLE NO.  1: First Order Conditional Estimation with Interaction: Goal Function=MINIMUM VALUE OF OBJECTIVE FUNCTION
 ITERATION    THETA1       THETA2       THETA3       SIGMA(1,1)   OMEGA(1,1)   OMEGA(2,1)   OMEGA(2,2)   OMEGA(3,1)   OMEGA(3,2)   OMEGA(3,3)   OBJ
0 18.7 87.3 2.13 0.0231046 0.128 0 0.142 0 0 1.82 1934.14785559591
1 16.8148 72.6252 2.083 0.0126826 0.160664 0 0.149731 0 0 1.51568 1923.70325075889
2 16.9495 73.2307 2.05859 0.0184253 0.166834 0 0.149006 0 0 1.4402 1899.85237046683
3 18.4026 80.7013 1.85358 0.0182676 0.222467 0 0.142799 0 0 0.917776 1890.66304696867
4 17.9049 76.5891 1.64892 0.0181099 0.254935 0 0.139153 0 0 0.639403 1885.75479711981
5 17.1666 80.222 1.41973 0.0181011 0.227412 0 0.138093 0 0 0.525486 1884.41372762353
6 17.9265 78.9975 1.35571 0.0181095 0.211068 0 0.14052 0 0 0.500335 1883.89273761533
7 17.8964 79.06 1.37145 0.0181037 0.207343 0 0.142404 0 0 0.490887 1883.84808484012
8 17.7293 78.2516 1.38609 0.0179329 0.210006 0 0.15096 0 0 0.528111 1883.3535752913
9 17.6299 77.0015 1.42104 0.0177301 0.206074 0 0.139223 0 0 0.568591 1882.84023590007
10 17.5286 76.6146 1.42608 0.0175907 0.206282 0 0.139413 0 0 0.57689 1882.82787864752
11 17.5286 76.6146 1.42608 0.0175836 0.206282 0 0.139413 0 0 0.57689 1882.82787864752
12 17.7222 76.732 1.43993 0.0174915 0.206303 0 0.139799 0 0 0.586448 1882.75522657785
13 17.7481 76.7973 1.44256 0.0174662 0.206217 0 0.139895 0 0 0.585529 1882.75345663931
14 17.7481 76.7973 1.44256 0.0174662 0.206217 0 0.139895 0 0 0.585529 1882.75345663931
-1000000000 17.7481 76.7973 1.44256 0.0174697 0.206217 0 0.139895 0 0 0.585529 1882.75345663931
-1000000001 1.04869 3.96763 0.15611 0.0011367 0.0261889 1e+10 0.0267082 1e+10 1e+10 0.169681 0
-1000000002 0.485426 0.646933 0.753416 0.982995 1.21987 1.33922 1.57214 0 0 0 0
-1000000003 3.23868 0.485426 1.57214 0 0 0 0 0 0 0 0

"
        )
    #data("run1ext")
    #cat("  TABLE NO.  1: First Order Conditional Estimation with Interaction: Goal Function=MINIMUM VALUE OF OBJECTIVE FUNCTION
 #ITERATION    THETA1       THETA2       THETA3       SIGMA(1,1)   OMEGA(1,1)   OMEGA(2,1)   OMEGA(2,2)   OMEGA(3,1)   OMEGA(3,2)   OMEGA(3,3)   OBJ\n",file="run1.ext")
    #write.table(run1ext,sep=" ",file="run1.ext",append=T,quote=F,row.names=F,col.names=F)
    #rm(run1ext,pos=1)
  }


#  TABLE NO.  1: First Order Conditional Estimation with Interaction: Goal Function=MINIMUM VALUE OF OBJECTIVE FUNCTION
# ITERATION    THETA1       THETA2       THETA3       SIGMA(1,1)   OMEGA(1,1)   OMEGA(2,1)   OMEGA(2,2)   OMEGA(3,1)   OMEGA(3,2)   OMEGA(3,3)   OBJ

  
  writeLst <- function() {
    
    cat(file="run1.lst","Sun Oct  9 19:17:21 CEST 2011
ifort
$PROB Simpraz model
;; 1. Based on: First run [nodOFV]
;;    First run with a one compartment model and first order absorption
;; 2. Structural model:
;;    One compartment model with first order absorption
;; 3. Covariate model:
;;    No covariates
;; 4. Inter-individual variability:
;;    ETAs on CL, V and KA
;; 5. Residual variability:
;;    Proportional
;; 6. Estimation:
;;    FOCE INTER
$INPUT ID SEX AGE RACE HT SMOK HCTZ PROP CON AMT WT TIME SECR
       DV RATE EVID OCC
$DATA simpraz.dta IGNORE=@
$SUBROUTINE ADVAN2 TRANS2
$PK
 
    TVCL  = THETA(1)
    TVV   = THETA(2)
    TVKA  = THETA(3)

    CL    = TVCL*EXP(ETA(1))
    V     = TVV *EXP(ETA(2))
    KA    = TVKA*EXP(ETA(3))

    S2=V

$ERROR 

    IPRED = F
    IRES  = DV - F
    W     = F
    IF(W.EQ.0) W = 1
    IWRES = IRES/W

    Y     = IPRED + W*EPS(1)

$THETA 

 (0,18.7)    ; CL (L/h)
 (0,87.3)    ; V   (L)
 (0,2.13)    ; KA (1/h)

$OMEGA 

 0.128       ; omCL
 0.142       ; omV
 1.82        ; omKA

$SIGMA  

 0.0231      ; Prop error

$EST NOABORT METHOD=1 INTER PRINT=1
$COV PRINT=E

$TABLE ID TIME IPRED IWRES CWRES 
       CL V KA ETA1 ETA2 ETA3 
       AGE HT WT SECR 
       SEX RACE SMOK HCTZ PROP CON OCC
       NOPRINT ONEHEADER FILE=xptab1

 
 
  
 WARNINGS AND ERRORS (IF ANY) FOR PROBLEM    1
             
 (WARNING  2) NM-TRAN INFERS THAT THE DATA ARE POPULATION.
 CREATING MUMODEL ROUTINE...
 
 
License Registered to: Exprimo NV
Expiration Date:    14 SEP 2012
Current Date:        9 OCT 2011
Days until program expires : 340
1NONLINEAR MIXED EFFECTS MODEL PROGRAM (NONMEM) VERSION 7.1.0
 ORIGINALLY DEVELOPED BY STUART BEAL, LEWIS SHEINER, AND ALISON BOECKMANN
 CURRENT DEVELOPERS ARE ROBERT BAUER, ICON DEVELOPMENT SOLUTIONS,
 AND ALISON BOECKMANN. IMPLEMENTATION, EFFICIENCY, AND STANDARDIZATION
 PERFORMED BY NOUS INFOSYSTEMS.
 
 PROBLEM NO.:         1
 Simpraz model                                                           
0DATA CHECKOUT RUN:              NO 
 DATA SET LOCATED ON UNIT NO.:    2
 THIS UNIT TO BE REWOUND:        NO 
 NO. OF DATA RECS IN DATA SET:      640
 NO. OF DATA ITEMS IN DATA SET:  18
 ID DATA ITEM IS DATA ITEM NO.:   1
 DEP VARIABLE IS DATA ITEM NO.:  14
 MDV DATA ITEM IS DATA ITEM NO.: 18
0INDICES PASSED TO SUBROUTINE PRED:
 16 12 10 15  0  0  0  0  0
  0  0
 0LABELS FOR DATA ITEMS:
  ID SEX AGE RACE HT SMOK HCTZ PROP CON AMT WT TIME SECR DV RATE EVID OCC MDV
0(NONBLANK) LABELS FOR PRED-DEFINED ITEMS:
 CL      V       KA      IPRE    IWRE
0FORMAT FOR DATA:
 (E3.0,E2.0,E3.0,E2.0,E4.0,4E2.0,2E6.0,E3.0,E4.0,E7.0,3E2.0,1F2.0)               
 
 TOT. NO. OF OBS RECS:      576
 TOT. NO. OF INDIVIDUALS:     64
0LENGTH OF THETA:  3
0DEFAULT THETA BOUNDARY TEST OMITTED:    NO 
0OMEGA HAS SIMPLE DIAGONAL FORM WITH DIMENSION:  3
0DEFAULT OMEGA BOUNDARY TEST OMITTED:    NO 
0SIGMA HAS SIMPLE DIAGONAL FORM WITH DIMENSION:  1
0DEFAULT SIGMA BOUNDARY TEST OMITTED:    NO 
0INITIAL ESTIMATE OF THETA:
 LOWER BOUND    INITIAL EST    UPPER BOUND
  0.0000E+00     0.1870E+02     0.1000E+07
  0.0000E+00     0.8730E+02     0.1000E+07
  0.0000E+00     0.2130E+01     0.1000E+07
0INITIAL ESTIMATE OF OMEGA:
 0.1280E+00
 0.0000E+00   0.1420E+00
 0.0000E+00   0.0000E+00   0.1820E+01
0INITIAL ESTIMATE OF SIGMA:
 0.2310E-01
0ESTIMATION STEP OMITTED:           NO 
 CONDITIONAL ESTIMATES USED:       YES 
 CENTERED ETA:                      NO 
 EPS-ETA INTERACTION:              YES 
 LAPLACIAN OBJ. FUNC.:              NO 
 NO. OF FUNCT. EVALS. ALLOWED:        360
 NO. OF SIG. FIGURES REQUIRED:       3
 INTERMEDIATE PRINTOUT:            YES 
 ESTIMATE OUTPUT TO MSF:            NO 
 ABORT WITH PRED EXIT CODE 1:       NO 
 IND. OBJ. FUNC. VALUES SORTED:     NO 
0COVARIANCE STEP OMITTED:    NO 
 EIGENVLS. PRINTED:         YES 
 SPECIAL COMPUTATION:        NO 
 COMPRESSED FORMAT:          NO 
0TABLES STEP OMITTED:    NO 
 NO. OF TABLES:           1
0-- TABLE  1 --
 PRINTED:                NO 
 HEADER:                YES 
 FILE TO BE FORWARDED:   NO 
 0USER-CHOSEN ITEMS:
  ID TIME IPRED IWRES CWRES CL V KA ETA1 ETA2 ETA3 AGE HT WT SECR SEX RACE SMOK 
 HCTZ PROP CON OCC
 
 THE FOLLOWING LABELS ARE EQUIVALENT
 PRED=PREDI
 RES=RESI
 WRES=WRESI
1DOUBLE PRECISION PREDPP VERSION 7.1.0
 
 ONE COMPARTMENT MODEL WITH FIRST-ORDER ABSORPTION (ADVAN2)
0MAXIMUM NO. OF BASIC PK PARAMETERS:   3
0BASIC PK PARAMETERS (AFTER TRANSLATION):
   ELIMINATION RATE (K) IS BASIC PK PARAMETER NO.:  1
   ABSORPTION RATE (KA) IS BASIC PK PARAMETER NO.:  3
 
 TRANSLATOR WILL CONVERT PARAMETERS 
 CLEARANCE (CL) AND VOLUME (V) TO K (TRANS2)
0COMPARTMENT ATTRIBUTES 
 COMPT. NO.   FUNCTION   INITIAL    ON/OFF      DOSE      DEFAULT    DEFAULT
                         STATUS     ALLOWED    ALLOWED    FOR DOSE   FOR OBS.
    1         DEPOT        OFF        YES        YES        YES        NO 
    2         CENTRAL      ON         NO         YES        NO         YES
    3         OUTPUT       OFF        YES        NO         NO         NO 
1
 ADDITIONAL PK PARAMETERS - ASSIGNMENT OF ROWS IN GG
 COMPT. NO.                             INDICES
              SCALE      BIOAVAIL.   ZERO-ORDER  ZERO-ORDER  ABSORB
                         FRACTION    RATE        DURATION    LAG  
    1           *           *           *           *           *
    2           4           *           *           *           *
    3           *           -           -           -           -
             - PARAMETER IS NOT ALLOWED FOR THIS MODEL
             * PARAMETER IS NOT SUPPLIED BY PK SUBROUTINE;
               WILL DEFAULT TO ONE IF APPLICABLE
0DATA ITEM INDICES USED BY PRED ARE:
   EVENT ID DATA ITEM IS DATA ITEM NO.:     16
   TIME DATA ITEM IS DATA ITEM NO.:         12
   DOSE AMOUNT DATA ITEM IS DATA ITEM NO.:  10
   DOSE RATE DATA ITEM IS DATA ITEM NO.:    15
 
0PK SUBROUTINE CALLED WITH EVERY EVENT RECORD.
 PK SUBROUTINE NOT CALLED AT NONEVENT (ADDITIONAL OR LAGGED) DOSE TIMES.
0ERROR SUBROUTINE CALLED WITH EVERY EVENT RECORD.
1
 
 
 #METH: First Order Conditional Estimation with Interaction
 MONITORING OF SEARCH:

0ITERATION NO.:    0    OBJECTIVE VALUE:   1934.14785559591        NO. OF FUNC. EVALS.: 6
 CUMULATIVE NO. OF FUNC. EVALS.:        6
 PARAMETER:  1.0000E-01  1.0000E-01  1.0000E-01  1.0000E-01  1.0000E-01  1.0000E-01  1.0000E-01
 GRADIENT:   6.5022E+01  1.1261E+02  1.3652E+01 -6.9534E+01 -1.6219E+01  5.5978E+01  1.8356E+02
0ITERATION NO.:    1    OBJECTIVE VALUE:   1923.70325075889        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       13
 PARAMETER: -6.2664E-03 -8.4038E-02  7.7688E-02  2.1364E-01  1.2651E-01  8.5138E-03 -2.0000E-01
 GRADIENT:  -1.2724E+01 -1.3237E+01  1.8799E+01 -3.0046E+01  3.8710E+00  4.0724E+01 -2.9791E+02
0ITERATION NO.:    2    OBJECTIVE VALUE:   1899.85237046683        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       20
 PARAMETER:  1.7137E-03 -7.5736E-02  6.5897E-02  2.3249E-01  1.2408E-01 -1.7028E-02 -1.3154E-02
 GRADIENT:  -1.3973E+01 -1.6776E+01  1.7423E+01 -2.3244E+01  3.5247E+00  3.7366E+01  4.6643E+01
0ITERATION NO.:    3    OBJECTIVE VALUE:   1890.66304696867        NO. OF FUNC. EVALS.:10
 CUMULATIVE NO. OF FUNC. EVALS.:       30
 PARAMETER:  8.3970E-02  2.1405E-02 -3.9001E-02  3.7637E-01  1.0280E-01 -2.4232E-01 -1.7453E-02
 GRADIENT:   3.2964E+01  4.7376E+01  2.3632E+01  8.1454E+00  1.0004E-01  2.0086E+01  3.3806E+01
0ITERATION NO.:    4    OBJECTIVE VALUE:   1885.75479711981        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       37
 PARAMETER:  5.6551E-02 -3.0896E-02 -1.5600E-01  4.4449E-01  8.9873E-02 -4.2303E-01 -2.1788E-02
 GRADIENT:   1.5210E+01 -6.1085E-01  2.0623E+01  2.3389E+01 -1.0150E+00  6.9499E-01  2.9375E+01
0ITERATION NO.:    5    OBJECTIVE VALUE:   1884.41372762353        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       44
 PARAMETER:  1.4439E-02  1.5448E-02 -3.0566E-01  3.8737E-01  8.6050E-02 -5.2113E-01 -2.2030E-02
 GRADIENT:  -9.3667E+00  3.7344E+01 -2.0295E+00  1.0027E+01 -4.5124E+00 -4.3621E+00  2.5094E+01
0ITERATION NO.:    6    OBJECTIVE VALUE:   1883.89273761533        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       51
 PARAMETER:  5.7757E-02  6.5886E-05 -3.5180E-01  3.5007E-01  9.4762E-02 -5.4566E-01 -2.1798E-02
 GRADIENT:   1.4259E+01  2.4904E+01 -9.4935E+00  2.2601E+00 -1.1557E+00 -7.5028E+00  2.4382E+01
0ITERATION NO.:    7    OBJECTIVE VALUE:   1883.84808484012        NO. OF FUNC. EVALS.: 8
 CUMULATIVE NO. OF FUNC. EVALS.:       59
 PARAMETER:  5.6074E-02  8.5652E-04 -3.4025E-01  3.4117E-01  1.0142E-01 -5.5519E-01 -2.1957E-02
 GRADIENT:   1.3534E+01  2.4501E+01 -7.2871E+00  1.2409E-01 -2.3582E-01 -8.7672E+00  2.3990E+01
0ITERATION NO.:    8    OBJECTIVE VALUE:   1883.35357529130        NO. OF FUNC. EVALS.: 8
 CUMULATIVE NO. OF FUNC. EVALS.:       67
 PARAMETER:  4.6693E-02 -9.4217E-03 -3.2964E-01  3.4755E-01  1.3059E-01 -5.1864E-01 -2.6699E-02
 GRADIENT:   8.6482E+00  1.6435E+01 -5.9177E+00  1.9289E+00  6.2091E+00 -4.1343E+00  1.8676E+01
0ITERATION NO.:    9    OBJECTIVE VALUE:   1882.84023590007        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       74
 PARAMETER:  4.1073E-02 -2.5526E-02 -3.0473E-01  3.3810E-01  9.0126E-02 -4.8171E-01 -3.2384E-02
 GRADIENT:   5.4469E+00  6.6298E+00 -1.2150E+00 -1.8173E-01 -6.4254E-01 -1.3516E+00  1.1441E+01
0ITERATION NO.:   10    OBJECTIVE VALUE:   1882.82787864752        NO. OF FUNC. EVALS.: 7
 CUMULATIVE NO. OF FUNC. EVALS.:       81
 PARAMETER:  3.5313E-02 -3.0562E-02 -3.0119E-01  3.3861E-01  9.0808E-02 -4.7447E-01 -3.6332E-02
 GRADIENT:   2.2140E+00  3.0353E+00 -7.0031E-01 -6.0453E-02 -3.5587E-01 -8.2821E-01  5.7532E+00
0ITERATION NO.:   11    OBJECTIVE VALUE:   1882.82787864752        NO. OF FUNC. EVALS.:12
 CUMULATIVE NO. OF FUNC. EVALS.:       93
 PARAMETER:  3.5313E-02 -3.0562E-02 -3.0119E-01  3.3861E-01  9.0808E-02 -4.7447E-01 -3.6332E-02
 GRADIENT:  -7.7981E+00 -2.1031E+00 -1.8360E+00 -6.0453E-02 -3.5587E-01 -8.2821E-01  5.6385E+00
0ITERATION NO.:   12    OBJECTIVE VALUE:   1882.75522657785        NO. OF FUNC. EVALS.:13
 CUMULATIVE NO. OF FUNC. EVALS.:      106
 PARAMETER:  4.6293E-02 -2.9032E-02 -2.9153E-01  3.3866E-01  9.2190E-02 -4.6625E-01 -3.8958E-02
 GRADIENT:  -9.2314E-01 -6.4563E-01 -2.9440E-01  6.0926E-02 -5.7781E-03  1.0360E-01  1.3233E+00
0ITERATION NO.:   13    OBJECTIVE VALUE:   1882.75345663931        NO. OF FUNC. EVALS.:11
 CUMULATIVE NO. OF FUNC. EVALS.:      117
 PARAMETER:  4.7755E-02 -2.8181E-02 -2.8970E-01  3.3845E-01  9.2533E-02 -4.6704E-01 -3.9682E-02
 GRADIENT:   3.6958E-03 -2.5536E-02  1.9604E-02  6.6066E-04  3.6072E-03  1.2488E-03  7.9552E-02
0ITERATION NO.:   14    OBJECTIVE VALUE:   1882.75345663931        NO. OF FUNC. EVALS.: 8
 CUMULATIVE NO. OF FUNC. EVALS.:      125
 PARAMETER:  4.7755E-02 -2.8181E-02 -2.8970E-01  3.3845E-01  9.2533E-02 -4.6704E-01 -3.9682E-02
 GRADIENT:   3.6958E-03 -2.5536E-02  1.9604E-02  6.6066E-04  3.6072E-03  1.2488E-03  7.9552E-02
 Elapsed estimation time in seconds:     1.45
 
 #TERM:
0MINIMIZATION SUCCESSFUL
 NO. OF FUNCTION EVALUATIONS USED:      125
 NO. OF SIG. DIGITS IN FINAL EST.:  3.4

 ETABAR IS THE ARITHMETIC MEAN OF THE ETA-ESTIMATES,
 AND THE P-VALUE IS GIVEN FOR THE NULL HYPOTHESIS THAT THE TRUE MEAN IS 0.

 ETABAR:         7.5792E-03 -7.8643E-03 -7.3866E-02
 SE:             5.5890E-02  4.3281E-02  7.8359E-02

 P VAL.:         8.9213E-01  8.5582E-01  3.4585E-01
 
 ETAshrink(%):   7.6098E-01  6.6939E+00  1.7430E+01
 EPSshrink(%):   1.4530E+01
 
 #TERE:
 Elapsed covariance time in seconds:     1.45
1
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 #OBJT:**************                       MINIMUM VALUE OF OBJECTIVE FUNCTION                      ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 





 #OBJV:********************************************     1882.753       **************************************************
1
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                             FINAL PARAMETER ESTIMATE                           ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 


 THETA - VECTOR OF FIXED EFFECTS PARAMETERS   *********


         TH 1      TH 2      TH 3     
 
         1.77E+01  7.68E+01  1.44E+00
 


 OMEGA - COV MATRIX FOR RANDOM EFFECTS - ETAS  ********


            ETA1      ETA2      ETA3
 
 ETA1
+        2.06E-01
 
 ETA2
+        0.00E+00  1.40E-01
 
 ETA3
+        0.00E+00  0.00E+00  5.86E-01
 


 SIGMA - COV MATRIX FOR RANDOM EFFECTS - EPSILONS  ****


            EPS1
 
 EPS1
+        1.75E-02
 
1
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                            STANDARD ERROR OF ESTIMATE                          ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 


 THETA - VECTOR OF FIXED EFFECTS PARAMETERS   *********


         TH 1      TH 2      TH 3     
 
         1.05E+00  3.97E+00  1.56E-01
 


 OMEGA - COV MATRIX FOR RANDOM EFFECTS - ETAS  ********


            ETA1      ETA2      ETA3
 
 ETA1
+        2.62E-02
 
 ETA2
+       .........  2.67E-02
 
 ETA3
+       ......... .........  1.70E-01
 


 SIGMA - COV MATRIX FOR RANDOM EFFECTS - EPSILONS  ****


            EPS1
 
 EPS1
+        1.14E-03
 
1
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                          COVARIANCE MATRIX OF ESTIMATE                         ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 

            TH 1      TH 2      TH 3      OM11      OM12      OM13      OM22      OM23      OM33      SG11
 
 TH 1
+        1.10E+00
 
 TH 2
+       -5.35E-01  1.57E+01
 
 TH 3
+       -2.77E-02 -1.01E-01  2.44E-02
 
 OM11
+        6.96E-03 -5.01E-03  5.42E-04  6.86E-04
 
 OM12
+       ......... ......... ......... ......... .........
 
 OM13
+       ......... ......... ......... ......... ......... .........
 
 OM22
+        8.25E-04  2.61E-02  8.89E-04  9.46E-05 ......... .........  7.13E-04
 
 OM23
+       ......... ......... ......... ......... ......... ......... ......... .........
 
 OM33
+        2.14E-02 -1.63E-01 -1.89E-03  3.23E-04 ......... ......... -7.79E-04 .........  2.88E-02
 
 SG11
+        9.34E-05  2.55E-04 -1.41E-05  3.95E-06 ......... ......... -3.01E-06 .........  3.54E-05  1.29E-06
 
1
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                          CORRELATION MATRIX OF ESTIMATE                        ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 

            TH 1      TH 2      TH 3      OM11      OM12      OM13      OM22      OM23      OM33      SG11
 
 TH 1
+        1.00E+00
 
 TH 2
+       -1.29E-01  1.00E+00
 
 TH 3
+       -1.69E-01 -1.62E-01  1.00E+00
 
 OM11
+        2.54E-01 -4.82E-02  1.33E-01  1.00E+00
 
 OM12
+       ......... ......... ......... ......... .........
 
 OM13
+       ......... ......... ......... ......... ......... .........
 
 OM22
+        2.95E-02  2.46E-01  2.13E-01  1.35E-01 ......... .........  1.00E+00
 
 OM23
+       ......... ......... ......... ......... ......... ......... ......... .........
 
 OM33
+        1.21E-01 -2.41E-01 -7.15E-02  7.27E-02 ......... ......... -1.72E-01 .........  1.00E+00
 
 SG11
+        7.84E-02  5.65E-02 -7.92E-02  1.33E-01 ......... ......... -9.93E-02 .........  1.84E-01  1.00E+00
 
1
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                      INVERSE COVARIANCE MATRIX OF ESTIMATE                     ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 

            TH 1      TH 2      TH 3      OM11      OM12      OM13      OM22      OM23      OM33      SG11
 
 TH 1
+        1.06E+00
 
 TH 2
+        4.76E-02  7.84E-02
 
 TH 3
+        1.75E+00  5.06E-01  4.91E+01
 
 OM11
+       -1.08E+01  1.44E-01 -4.56E+01  1.67E+03
 
 OM12
+       ......... ......... ......... ......... .........
 
 OM13
+       ......... ......... ......... ......... ......... .........
 
 OM22
+       -4.24E+00 -3.29E+00 -7.15E+01 -1.91E+02 ......... .........  1.68E+03
 
 OM23
+       ......... ......... ......... ......... ......... ......... ......... .........
 
 OM33
+       -3.56E-01  3.90E-01  3.12E+00 -1.19E+01 ......... .........  2.26E+01 .........  3.94E+01
 
 SG11
+       -3.39E+01 -3.22E+01  1.95E+02 -4.95E+03 ......... .........  4.06E+03 ......... -1.01E+03  8.37E+05
 
1
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 ************************************************************************************************************************
 ********************                                                                                ********************
 ********************               FIRST ORDER CONDITIONAL ESTIMATION WITH INTERACTION              ********************
 ********************                      EIGENVALUES OF COR MATRIX OF ESTIMATE                     ********************
 ********************                                                                                ********************
 ************************************************************************************************************************
 

             1         2         3         4         5         6         7
 
         4.85E-01  6.47E-01  7.53E-01  9.83E-01  1.22E+00  1.34E+00  1.57E+00
 
Sun Oct  9 19:17:24 CEST 2011
"
        )
  }

  
  if(!overwrite && any(file.exists("run1.mod","simpraz.dta","run1.lst","run1.ext","xptab1"))){
    cat("One of:\n",
        "run1.mod, simpraz.dta, run1.lst, run1.ext, xptab1\n",
        "already exist in\n",
        getwd(),"\nNo files will be created.\n",sep="")
    return()
  }
  
  writeMod()
  writeDta()
  writeTab()
  writeLst()
  writeExt()

}


