YHAT_URL <- "http://api.yhathq.com/"

yhat.login <- function() {
  username <- scan(, what="")
  apikey <- scan(, what="")
  yhat.login(username, apikey)
}

#' A function for logging into Yhat's api.
#' 
#' @param username Your Yhat username
#' @param apikey Your Yhat apikey
#' 
#' @export
#' @examples
#' yhat.login("hmardukas", "abcd1234")
yhat.login <- function(username, apikey) {
  print("Please create yhat.config.")
}

#' Private function for performing a GET request
#' 
#' @param endpoint /path for REST request
#' @param query url parameters for request
yhat.get <- function(endpoint, query=c()) {
  AUTH <- get("yhat.config")
  if (length(AUTH)==0) {
    stop("You must login. Execute yhat.login(username, apikey).")
  }

  if ("env" %in% names(AUTH)) {
    url <- AUTH[["env"]]
    AUTH <- AUTH[!names(AUTH)=="env"]
  } else {
    url <- YHAT_URL
  }

  query <- c(query, AUTH)
  query <- paste(names(query), query, collapse="&", sep="=")
  url <- paste(url, endpoint, "?", query, sep="")
  httr::GET(url)
}

#' Private function for performing a POST request
#' 
#' @param endpoint /path for REST request
#' @param query url parameters for request
#' @param data payload to be converted to raw JSON
yhat.post <- function(endpoint, query=c(), data) {
  AUTH <- get("yhat.config")
  if (length(AUTH)==0) {
    stop("You must login. Execute yhat.login(username, apikey).")
  }

  if ("env" %in% names(AUTH)) {
    url <- AUTH[["env"]]
    AUTH <- AUTH[!names(AUTH)=="env"]
  } else {
    url <- YHAT_URL
  }

  query <- c(query, AUTH)
  query <- paste(names(query), query, collapse="&", sep="=")
  url <- paste(url, endpoint, "?", query, sep="")
  httr::POST(url, httr::add_headers("Content-Type"="application/json"),
       body = rjson::toJSON(list(
         data = data)
        )
  )
}

#' Private function for checking the size of the user's image.
#' 
check.image.size <- function() {
  total.img.size <- 0
  for (obj in ls()) {
    obj.size <- object.size(get(obj))*9.53674e-7#convert to MB
    total.img.size <- total.img.size + obj.size
    if (obj.size > 20) {
      msg <- paste("Object: ", obj, " is pretty big. Are you sure you want to send it to Yhat?", sep="")
#       print(msg)
    }
  }
  total.img.mb <- total.img.size[1]
  if (total.img.mb > 50) {
    stop("Sorry, your model is too big for a free account.
         Try removing some large objects from your workspace using the rm() command.")
  }
}

#' Shows which models you have deployed on Yhat.
#' 
#' This function queries the Yhat API and finds the models that have been deployed
#' for your account.
#' 
#' @export
#' @examples
#' yhat.config <- c(
#'  username = "your username",
#'  apikey = "your apikey"
#' )
#' yhat.show_models()
#' # some output here
#' #    username className                  name version
#' # 1      greg                 MySMSClassifier       1
#' # 2      greg                 MySMSClassifier       2
#' # 3      greg                 MySMSClassifier       3
#' # 4      greg                 MySMSClassifier       4
yhat.show_models <- function() {
  rsp <- yhat.get("showmodels")
  js <- httr::content(rsp)
  js <- lapply(js$models, function(model) {
    if (is.null(model$className)) {
      model$className <- ""
    }
    model
  })
  plyr::ldply(js, data.frame)
}

#' Calls Yhat's REST API and returns a JSON document containing both the prediction
#' and associated metadata.
#' 
#' @param model_name the name of the model you want to call
#' @param version the version number of the model you want to call
#' @param data input data for the model
#' 
#' @export
#' @examples
#' yhat.config <- c(
#'  username = "your username",
#'  apikey = "your apikey"
#' )
#' yhat.predict_raw("irisModel", 1, iris) 
yhat.predict_raw <- function(model_name, version, data) {
  AUTH <- get("yhat.config")
  if ("env" %in% names(AUTH)) {
    endpoint <- paste("models", model_name, "", sep="/")
  } else {
    endpoint <- "predict"
  }
  rsp <- yhat.post(endpoint, c(model = model_name, 
                                version = version),
                   data = data)
  httr::content(rsp)
}
#' Make a prediction using Yhat.
#' 
#' This function calls Yhat's REST API and returns a response formatted as a
#' data frame.
#' 
#' @param model_name the name of the model you want to call
#' @param version the version number of the model you want to call
#' @param data input data for the model
#' 
#' @keywords predict
#' @export
#' @examples
#' yhat.config <- c(
#'  username = "your username",
#'  apikey = "your apikey"
#' )
#' yhat.predict("irisModel", 1, iris) 
yhat.predict <- function(model_name, version, data) {
  raw_rsp <- yhat.predict_raw(model_name, version, data)
  if ("prediction" %in% raw_rsp) {
    data.frame(raw_rsp$prediction)
  } else {
    data.frame(raw_rsp)
  }
}

#' Deploy a model to Yhat's servers
#' 
#' This function takes model.transform and model.predict and creates
#' a model on Yhat's servers which can be called from any programming language
#' via Yhat's REST API (see \code{\link{yhat.predict}}).
#' 
#' @param model_name name of your model
#' @keywords deploy
#' @export
#' @examples
#' yhat.config <- c(
#'  username = "your username",
#'  apikey = "your apikey"
#' )
#' iris$Sepal.Width_sq <- iris$Sepal.Width^2
#' fit <- glm(I(Species)=="virginica" ~ ., data=iris)
#' 
#' model.require <- function() {
#'  # require("randomForest")
#' }
#' 
#' model.transform <- function(df) {
#'  df$Sepal.Width_sq <- df$Sepal.Width^2
#'  df
#' }
#' model.predict <- function(df) {
#'  data.frame("prediction"=predict(fit, df, type="response"))
#' } 
#' yhat.deploy("irisModel")
yhat.deploy <- function(model_name) {
  if ("env" %in% get("yhat.config")) {
    # do nothing
  } else {
    check.image.size()
  }
  AUTH <- get("yhat.config")
  if (length(AUTH)==0) {
    stop("You must login. execute yhat.login(username, apikey).")
  }
  if ("env" %in% names(AUTH)) {
    url <- AUTH[["env"]]
    AUTH <- AUTH[!names(AUTH)=="env"]
    query <- AUTH
    query <- paste(names(query), query, collapse="&", sep="=")
    url <- paste(url, "model", "?", query, sep="")
  } else {
    url <- YHAT_URL
    query <- AUTH
    query <- paste(names(query), query, collapse="&", sep="=")
    url <- paste(url, "model/R", "?", query, sep="")
  }
  
  image_file <- ".yhatdeployment.img"
  save.image(image_file)
  rsp <- httr::POST(url,
       body=list(
         "model_image" = httr::upload_file(image_file),
         "modelname" = model_name
         )
  )
  unlink(image_file)
  js <- httr::content(rsp)
  data.frame(js)
}

#' Quick function for setting up a basic scaffolding of functions for deploying on Yhat.
#' 
#' @export
#' @examples
#' yhat.scaffolding()
yhat.scaffolding <- function() {
  txt <- c(
    "model_transform <- function(df) {
  df.transformed <- transform(df, x2=x^2)
  df.transformed
}
model_predict <- function(df) {
  pred <- predict(df)
  data.frame('myPrediction' = pred)
}
model_require <- function() {
  require('library1')
  require('library2')
}"
)
  con <- file("yhatExample.R", open="w")
  writeLines(txt, con)
  close(con)
}

#' Documents a column from a given data.frame
#' 
#' Generates documentation for a column.
#' @param df the data.farme frame you're documenting
#' @param col the column as a string that you're documenting
documentColumn <- function(df, col) {
  doc <- list()
  doc[["name"]] <- col
  doc[["dtype"]] <- ifelse(is.numeric(df[,col]), "number",
                           ifelse(is.factor(df[,col]), "factor",
                                  "something else"))
  if (doc[["dtype"]]=="number") {
    step <- abs(max(df[,col], na.rm=T) - min(df[,col], na.rm=T) )/ 10 + 0.01
    doc[["step"]] <- 10^round(log10(step) - log10(5.5) + 0.5)
    doc[["placeholder"]] <- median(df[,col], na.rm=T)
  } else if(doc[["dtype"]]=="factor") {
    factorLevels <- list()
    levs <- sort(levels(df[,col]))
    doc[["levels"]] <- lapply(levs, I)
  } else {
    doc[["placeholder"]] <- "your text here"
  }
  doc
}

#' Private method for generating documentation for a data frame
#' 
#' Documents a data frame using the \link{documentColumn} function.
#' @param data the data.frame you're documenting
documentData <- function(data) {
  docs <- list()
  for(col in names(data)) {
    docs[[length(docs)+1]] <- documentColumn(data, col)
  }
  docs
}

#' Document a model using Yhat
#' 
#' Takes a specific model along with data that is required to execute a prediction
#' and generates a webapp/documentation.
#' 
#' @param model name of your model
#' @param version version of your model
#' @param df example data for your model
#' @keywords document
#' @export
yhat.document <- function(model, version, df) {
  docs <- documentData(df)
  
  AUTH <- get("yhat.config")
  if (length(AUTH)==0) {
    stop("You must login. execute yhat.login(username, apikey).")
  }

  if ("env" %in% names(AUTH)) {
    url <- AUTH[["env"]]
    AUTH <- AUTH[!names(AUTH)=="env"]
  } else {
    url <- YHAT_URL
  }

  query <- AUTH
  query["model"] <- model
  query["version"] <- version
  query <- paste(names(query), query, collapse="&", sep="=")
  url <- paste(url, "document", "?", query, sep="")
  
  
  rsp <- httr::POST(url, httr::add_headers("Content-Type"="application/json"),
             body = rjson::toJSON(list(
               docs = docs)
             ))
  httr::content(rsp)
}


