# AutoPlots is a package for quickly creating high quality visualizations under a common and easy api.
# Copyright (C) <year>  <name of author>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.


# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ ----
# > Distribution Plot Functions                                               ----
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ ----

#' @title Density
#'
#' @description Density plots, by groups, with transparent continuous plots
#'
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param SampleSize = 100000L
#' @param XVar X-Axis variable name
#' @param GroupVar Character variable
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ShowLabels character
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param ContainLabel  TRUE
#' @param areaStyle.color color or hex
#' @param areaStyle.opacity numeric
#' @param showSymbol logical
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)))
#'
#' # Create plot
#' AutoPlots::Density(
#'   dt = dt,
#'   SampleSize = 30000L,
#'   XVar = "Y",
#'   GroupVar = NULL,
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   Height = NULL,
#'   Width = NULL,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   TimeLine = FALSE,
#'   ShowLabels = FALSE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Density <- function(dt = NULL,
                    SampleSize = 100000L,
                    XVar = NULL,
                    XVarTrans = "Identity",
                    GroupVar = NULL,
                    FacetRows = 1,
                    FacetCols = 1,
                    FacetLevels = NULL,
                    TimeLine = FALSE,
                    showSymbol = FALSE,
                    areaStyle.color = NULL,
                    areaStyle.opacity = NULL,
                    Height = NULL,
                    Width = NULL,
                    MouseScroll = FALSE,
                    ShowLabels = FALSE,
                    Theme = "dark",
                    ContainLabel = TRUE,
                    title.text = "Density Plot",
                    title.subtext = NULL,
                    title.link = NULL,
                    title.sublink = NULL,
                    title.Align = NULL,
                    title.top = NULL,
                    title.left = NULL,
                    title.right = NULL,
                    title.bottom = NULL,
                    title.padding = NULL,
                    title.itemGap = NULL,
                    title.backgroundColor = NULL,
                    title.borderColor = NULL,
                    title.borderWidth = NULL,
                    title.borderRadius = NULL,
                    title.shadowColor = NULL,
                    title.shadowBlur = NULL,
                    title.shadowOffsetX = NULL,
                    title.shadowOffsetY = NULL,
                    title.textStyle.color = NULL,
                    title.textStyle.fontStyle = NULL,
                    title.textStyle.fontWeight = NULL,
                    title.textStyle.fontFamily = "Segoe UI",
                    title.textStyle.fontSize = NULL,
                    title.textStyle.lineHeight = NULL,
                    title.textStyle.width = NULL,
                    title.textStyle.height = NULL,
                    title.textStyle.textBorderColor = NULL,
                    title.textStyle.textBorderWidth = NULL,
                    title.textStyle.textBorderType = NULL,
                    title.textStyle.textBorderDashOffset = NULL,
                    title.textStyle.textShadowColor = NULL,
                    title.textStyle.textShadowBlur = NULL,
                    title.textStyle.textShadowOffsetX = NULL,
                    title.textStyle.textShadowOffsetY = NULL,
                    title.subtextStyle.color = NULL,
                    title.subtextStyle.align = NULL,
                    title.subtextStyle.fontStyle = NULL,
                    title.subtextStyle.fontWeight = NULL,
                    title.subtextStyle.fontFamily = "Segoe UI",
                    title.subtextStyle.fontSize = NULL,
                    title.subtextStyle.lineHeight = NULL,
                    title.subtextStyle.width = NULL,
                    title.subtextStyle.height = NULL,
                    title.subtextStyle.textBorderColor = NULL,
                    title.subtextStyle.textBorderWidth = NULL,
                    title.subtextStyle.textBorderType = NULL,
                    title.subtextStyle.textBorderDashOffset = NULL,
                    title.subtextStyle.textShadowColor = NULL,
                    title.subtextStyle.textShadowBlur = NULL,
                    title.subtextStyle.textShadowOffsetX = NULL,
                    title.subtextStyle.textShadowOffsetY = NULL,
                    xAxis.title = NULL,
                    xAxis.min = NULL,
                    xAxis.max = NULL,
                    xAxis.position = "bottom",
                    xAxis.nameLocation = 'center',
                    xAxis.axisTick.customValues = NULL,
                    xAxis.nameTextStyle.color = NULL,
                    xAxis.nameTextStyle.padding = 25,
                    xAxis.nameTextStyle.align = NULL,
                    xAxis.nameTextStyle.fontStyle = NULL,
                    xAxis.nameTextStyle.fontWeight = "bolder",
                    xAxis.nameTextStyle.fontSize = NULL,
                    xAxis.nameTextStyle.fontFamily = "Segoe UI",
                    xAxis.splitNumber = NULL,
                    xAxis.axisLabel.rotate = NULL,
                    xAxis.axisLabel.margin = NULL,
                    xAxis.axisLabel.color = NULL,
                    xAxis.axisLabel.fontStyle = NULL,
                    xAxis.axisLabel.fontWeight = NULL,
                    xAxis.axisLabel.fontFamily = "Segoe UI",
                    xAxis.axisLabel.fontSize = 14,
                    xAxis.axisLabel.align = NULL,
                    xAxis.axisLabel.verticalAlign = NULL,
                    xAxis.axisLabel.backgroundColor = NULL,
                    xAxis.axisLabel.borderColor = NULL,
                    xAxis.axisLabel.borderWidth = NULL,
                    xAxis.axisLabel.borderType = NULL,
                    xAxis.axisLabel.borderRadius = NULL,
                    xAxis.axisLabel.padding = NULL,
                    xAxis.axisLabel.shadowColor = NULL,
                    xAxis.axisLabel.shadowBlur = NULL,
                    xAxis.axisLabel.shadowOffsetX = NULL,
                    xAxis.axisLabel.shadowOffsetY = NULL,
                    xAxis.axisLabel.textBorderColor = NULL,
                    xAxis.axisLabel.textBorderWidth = NULL,
                    xAxis.axisLabel.textBorderType = NULL,
                    xAxis.axisLabel.textShadowColor = NULL,
                    xAxis.axisLabel.textShadowBlur = NULL,
                    xAxis.axisLabel.textShadowOffsetX = NULL,
                    xAxis.axisLabel.textShadowOffsetY = NULL,
                    xAxis.axisLabel.overflow = NULL,
                    yAxis.title = NULL,
                    yAxis.min = NULL,
                    yAxis.max = NULL,
                    yAxis.position = "center",
                    yAxis.nameLocation = 'center',
                    yAxis.axisTick.customValues = NULL,
                    yAxis.nameTextStyle.color = NULL,
                    yAxis.nameTextStyle.padding = 25,
                    yAxis.nameTextStyle.align = NULL,
                    yAxis.nameTextStyle.fontStyle = NULL,
                    yAxis.nameTextStyle.fontWeight = "bolder",
                    yAxis.nameTextStyle.fontSize = NULL,
                    yAxis.nameTextStyle.fontFamily = "Segoe UI",
                    yAxis.splitNumber = NULL,
                    yAxis.axisLabel.rotate = NULL,
                    yAxis.axisLabel.margin = NULL,
                    yAxis.axisLabel.color = NULL,
                    yAxis.axisLabel.fontStyle = NULL,
                    yAxis.axisLabel.fontWeight = NULL,
                    yAxis.axisLabel.fontFamily = "Segoe UI",
                    yAxis.axisLabel.fontSize = 14,
                    yAxis.axisLabel.align = NULL,
                    yAxis.axisLabel.verticalAlign = NULL,
                    yAxis.axisLabel.backgroundColor = NULL,
                    yAxis.axisLabel.borderColor = NULL,
                    yAxis.axisLabel.borderWidth = NULL,
                    yAxis.axisLabel.borderType = NULL,
                    yAxis.axisLabel.borderRadius = NULL,
                    yAxis.axisLabel.padding = NULL,
                    yAxis.axisLabel.shadowColor = NULL,
                    yAxis.axisLabel.shadowBlur = NULL,
                    yAxis.axisLabel.shadowOffsetX = NULL,
                    yAxis.axisLabel.shadowOffsetY = NULL,
                    yAxis.axisLabel.textBorderColor = NULL,
                    yAxis.axisLabel.textBorderWidth = NULL,
                    yAxis.axisLabel.textBorderType = NULL,
                    yAxis.axisLabel.textShadowColor = NULL,
                    yAxis.axisLabel.textShadowBlur = NULL,
                    yAxis.axisLabel.textShadowOffsetX = NULL,
                    yAxis.axisLabel.textShadowOffsetY = NULL,
                    yAxis.axisLabel.overflow = NULL,
                    legend.show = TRUE,
                    legend.type = "scroll",
                    legend.selector = NULL,
                    legend.icon = NULL,
                    legend.align = NULL,
                    legend.padding = NULL,
                    legend.itemGap = NULL,
                    legend.itemWidth = NULL,
                    legend.orient = "vertical",
                    legend.width = NULL,
                    legend.height = NULL,
                    legend.left = NULL,
                    legend.right = 10,
                    legend.top = 30,
                    legend.bottom = NULL,
                    legend.backgroundColor = NULL,
                    legend.borderColor = NULL,
                    legend.borderWidth = NULL,
                    legend.borderRadius = NULL,
                    legend.shadowBlur = NULL,
                    legend.shadowColor = NULL,
                    legend.shadowOffsetX = NULL,
                    legend.shadowOffsetY = NULL,
                    legend.itemStyle.color = NULL,
                    legend.itemStyle.borderColor = NULL,
                    legend.itemStyle.borderWidth = NULL,
                    legend.itemStyle.borderType = NULL,
                    legend.itemStyle.shadowBlur = NULL,
                    legend.itemStyle.shadowColor = NULL,
                    legend.itemStyle.shadowOffsetX = NULL,
                    legend.itemStyle.shadowOffsetY = NULL,
                    legend.itemStyle.opacity = NULL,
                    legend.lineStyle.color = NULL,
                    legend.lineStyle.width = NULL,
                    legend.lineStyle.type = NULL,
                    legend.lineStyle.shadowBlur = NULL,
                    legend.lineStyle.shadowColor = NULL,
                    legend.lineStyle.shadowOffsetX = NULL,
                    legend.lineStyle.shadowOffsetY = NULL,
                    legend.lineStyle.opacity = NULL,
                    legend.lineStyle.inactiveColor = NULL,
                    legend.lineStyle.inactiveWidth = NULL,
                    legend.textStyle.color = NULL,
                    legend.textStyle.fontStyle = NULL,
                    legend.textStyle.fontWeight = NULL,
                    legend.textStyle.fontFamily = "Segoe UI",
                    legend.textStyle.fontSize = NULL,
                    legend.textStyle.backgroundColor = NULL,
                    legend.textStyle.borderColor = NULL,
                    legend.textStyle.borderWidth = NULL,
                    legend.textStyle.borderType = NULL,
                    legend.textStyle.borderRadius = NULL,
                    legend.textStyle.padding = NULL,
                    legend.textStyle.shadowColor = NULL,
                    legend.textStyle.shadowBlur = NULL,
                    legend.textStyle.shadowOffsetX = NULL,
                    legend.textStyle.shadowOffsetY = NULL,
                    legend.textStyle.width = NULL,
                    legend.textStyle.height = NULL,
                    legend.textStyle.textBorderColor = NULL,
                    legend.textStyle.textBorderWidth = NULL,
                    legend.textStyle.textBorderType = NULL,
                    legend.textStyle.textShadowColor = NULL,
                    legend.textStyle.textShadowBlur = NULL,
                    legend.textStyle.textShadowOffsetX = NULL,
                    legend.textStyle.textShadowOffsetY = NULL,
                    legend.pageTextStyle.color = NULL,
                    legend.pageTextStyle.fontStyle = NULL,
                    legend.pageTextStyle.fontWeight = NULL,
                    legend.pageTextStyle.fontFamily = "Segoe UI",
                    legend.pageTextStyle.fontSize = NULL,
                    legend.pageTextStyle.lineHeight = NULL,
                    legend.pageTextStyle.width = NULL,
                    legend.pageTextStyle.height = NULL,
                    legend.pageTextStyle.textBorderColor = NULL,
                    legend.pageTextStyle.textBorderWidth = NULL,
                    legend.pageTextStyle.textBorderType = NULL,
                    legend.pageTextStyle.textShadowColor = NULL,
                    legend.pageTextStyle.textShadowBlur = NULL,
                    legend.pageTextStyle.textShadowOffsetX = NULL,
                    legend.pageTextStyle.textShadowOffsetY = NULL,
                    legend.emphasis.selectorLabel.show = NULL,
                    legend.emphasis.selectorLabel.distance = NULL,
                    legend.emphasis.selectorLabel.rotate = NULL,
                    legend.emphasis.selectorLabel.color = NULL,
                    legend.emphasis.selectorLabel.fontStyle = NULL,
                    legend.emphasis.selectorLabel.fontWeight = NULL,
                    legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                    legend.emphasis.selectorLabel.fontSize = NULL,
                    legend.emphasis.selectorLabel.align = NULL,
                    legend.emphasis.selectorLabel.verticalAlign = NULL,
                    legend.emphasis.selectorLabel.lineHeight = NULL,
                    legend.emphasis.selectorLabel.backgroundColor = NULL,
                    legend.emphasis.selectorLabel.borderColor = NULL,
                    legend.emphasis.selectorLabel.borderWidth = NULL,
                    legend.emphasis.selectorLabel.borderType = NULL,
                    legend.emphasis.selectorLabel.borderRadius = NULL,
                    legend.emphasis.selectorLabel.padding = NULL,
                    legend.emphasis.selectorLabel.shadowColor = NULL,
                    legend.emphasis.selectorLabel.shadowBlur = NULL,
                    legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                    legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                    legend.emphasis.selectorLabel.width = NULL,
                    legend.emphasis.selectorLabel.height = NULL,
                    legend.emphasis.selectorLabel.textBorderColor = NULL,
                    legend.emphasis.selectorLabel.textBorderWidth = NULL,
                    legend.emphasis.selectorLabel.textBorderType = NULL,
                    legend.emphasis.selectorLabel.textShadowColor = NULL,
                    legend.emphasis.selectorLabel.textShadowBlur = NULL,
                    legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                    legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                    tooltip.show = TRUE,
                    tooltip.trigger = "axis",
                    tooltip.backgroundColor = NULL,
                    tooltip.borderColor = NULL,
                    tooltip.borderWidth = NULL,
                    tooltip.padding = NULL,
                    tooltip.axisPointer.type = "cross",
                    tooltip.axisPointer.lineStyle.color = NULL,
                    tooltip.axisPointer.shadowStyle.color = NULL,
                    tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                    tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                    tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                    tooltip.axisPointer.shadowStyle.opacity = NULL,
                    tooltip.textStyle.color = NULL,
                    tooltip.textStyle.fontStyle = NULL,
                    tooltip.textStyle.fontWeight = NULL,
                    tooltip.textStyle.fontFamily = "Segoe UI",
                    tooltip.textStyle.lineHeight = NULL,
                    tooltip.textStyle.width = NULL,
                    tooltip.textStyle.height = NULL,
                    tooltip.textStyle.textBorderColor = NULL,
                    tooltip.textStyle.textBorderWidth = NULL,
                    tooltip.textStyle.textBorderType = NULL,
                    tooltip.textStyle.textShadowColor = NULL,
                    tooltip.textStyle.textShadowBlur = NULL,
                    tooltip.textStyle.textShadowOffsetX = NULL,
                    tooltip.textStyle.textShadowOffsetY = NULL,
                    toolbox.show = TRUE,
                    toolbox.orient = "horizontal",
                    toolbox.itemSize = 15,
                    toolbox.itemGap = 8,
                    toolbox.top = NULL,
                    toolbox.left = NULL,
                    toolbox.right = NULL,
                    toolbox.bottom = NULL,
                    toolbox.width = NULL,
                    toolbox.heigth = NULL,
                    toolbox.feature.saveAsImage.show = TRUE,
                    toolbox.feature.restore.show = TRUE,
                    toolbox.feature.dataZoom.show = TRUE,
                    toolbox.feature.magicType.show = TRUE,
                    toolbox.feature.magicType.type = c("line", "bar", "stack"),
                    toolbox.feature.dataView.show = TRUE,
                    toolbox.iconStyle.color = NULL,
                    toolbox.iconStyle.borderColor = NULL,
                    toolbox.emphasis.iconStyle.borderColor = NULL,
                    toolbox.iconStyle.shadowBlur = NULL,
                    toolbox.iconStyle.shadowColor = NULL,
                    toolbox.iconStyle.shadowOffsetX = NULL,
                    toolbox.iconStyle.shadowOffsetY = NULL,
                    Debug = FALSE) {

  # Cap number of records
  if(length(SampleSize) == 0L) SampleSize <- 30000
  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  if(dt[, .N] > SampleSize) {
    dt1 <- dt[order(stats::runif(.N))][seq_len(SampleSize)]
  } else {
    dt1 <- data.table::copy(dt)
  }
  GroupVar <- tryCatch({GroupVar[1L]}, error = function(x) NULL)

  # Faceting shrink
  if(length(GroupVar) > 0L && (FacetRows > 1L || FacetCols > 1L) && length(FacetLevels) > 0L) {
    dt1 <- dt1[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(XVar,GroupVar)]
  } else {
    dt1 <- dt1[, .SD, .SDcols = c(XVar,GroupVar)]
  }

  # Multiple YVars
  if(length(XVar) > 1L) {
    sqroots <- sqrt(length(XVar))
    if(FacetCols == 1 && FacetRows == 1L) {
      FacetCols <- max(ceiling(sqroots), 6)
      FacetRows <- ceiling(sqroots)
      if((FacetRows - 1L) * FacetCols == length(XVar)) {
        FacetRows <- FacetRows - 1L
      } else if(FacetRows * FacetCols < length(XVar)) {
        while(FacetRows * FacetCols < length(XVar)) {
          FacetRows <- FacetRows + 1L
        }
      }
    }
    XVar <- NULL
    GroupVar <- NULL
    dt1[, temp__ := "a"]
    dt1 <- data.table::melt.data.table(data = dt1, id.vars = "temp__", measure.vars = XVar, variable.name = "Measures", value.name = "Values")
    dt1[, temp__ := NULL]
    GroupVar <- "Measures"
  }

  # Transformation
  if(XVarTrans != "Identity") {
    for(ggss in XVar) {
      dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = ggss, Methods = XVarTrans)$Data
    }
  }

  # Create base plot object
  if(Debug) print('Create Plot with only data')

  if(length(GroupVar) == 0L) {

    p1 <- echarts4r::e_charts_(
      dt1,
      x = NULL,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)

    p1 <- e_density_full(
      e = p1,
      serie = XVar,
      smooth = TRUE,
      showSymbol = showSymbol,
      areaStyle.color = areaStyle.color,
      areaStyle.opacity = areaStyle.opacity
    )

    if(!MouseScroll) {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }

    p1 <- echarts4r::e_theme(e = p1, name = Theme)
    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
    p1 <- echarts4r::e_brush(e = p1)

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else NULL, index = 0, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    return(p1)

  } else {

    data.table::setorderv(x = dt1, cols = GroupVar[1L], 1)

    if(ShowLabels) {
      p1 <- echarts4r::e_charts_(
        dt1 |> dplyr::group_by(get(GroupVar[1L])),
        timeline = TimeLine,
        dispose = TRUE,
        darkMode = TRUE,
        emphasis = list(focus = "series"),
        width = Width,
        height = Height,
        label = list(show = TRUE))
    } else {
      p1 <- echarts4r::e_charts_(
        dt1 |> dplyr::group_by(get(GroupVar[1L])),
        timeline = TimeLine,
        dispose = TRUE,
        darkMode = TRUE,
        emphasis = list(focus = "series"),
        width = Width,
        height = Height)
    }

    p1 <- e_density_full(
      e = p1,
      serie = XVar,
      smooth = TRUE,
      showSymbol = showSymbol,
      areaStyle.color = areaStyle.color,
      areaStyle.opacity = areaStyle.opacity
    )

    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else NULL, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if(FacetRows > 1L || FacetCols > 1L) {
      p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    }

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    return(p1)
  }
}


#' @title ProbabilityPlot
#'
#' @description Build a normal probability plot
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param SampleSize An integer for the number of rows to use. Sampled data is randomized. If NULL then ignored
#' @param YVar Y-Axis variable name
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme "macaron"
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param ContainLabel Default TRUE
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)))
#'
#' # Create plot
#' AutoPlots::ProbabilityPlot(
#'   dt = dt,
#'   SampleSize = 1000L,
#'   YVar = "Y",
#'   YVarTrans = "Identity",
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "blue",
#'   ContainLabel = TRUE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
ProbabilityPlot <- function(dt = NULL,
                            SampleSize = 1000L,
                            YVar = NULL,
                            YVarTrans = "Identity",
                            Height = NULL,
                            Width = NULL,
                            ShowLabels = FALSE,
                            Theme = "dark",
                            ContainLabel = TRUE,
                            title.text = "Probability Plot",
                            title.subtext = NULL,
                            title.link = NULL,
                            title.sublink = NULL,
                            title.Align = NULL,
                            title.top = NULL,
                            title.left = NULL,
                            title.right = NULL,
                            title.bottom = NULL,
                            title.padding = NULL,
                            title.itemGap = NULL,
                            title.backgroundColor = NULL,
                            title.borderColor = NULL,
                            title.borderWidth = NULL,
                            title.borderRadius = NULL,
                            title.shadowColor = NULL,
                            title.shadowBlur = NULL,
                            title.shadowOffsetX = NULL,
                            title.shadowOffsetY = NULL,
                            title.textStyle.color = NULL,
                            title.textStyle.fontStyle = NULL,
                            title.textStyle.fontWeight = NULL,
                            title.textStyle.fontFamily = "Segoe UI",
                            title.textStyle.fontSize = NULL,
                            title.textStyle.lineHeight = NULL,
                            title.textStyle.width = NULL,
                            title.textStyle.height = NULL,
                            title.textStyle.textBorderColor = NULL,
                            title.textStyle.textBorderWidth = NULL,
                            title.textStyle.textBorderType = NULL,
                            title.textStyle.textBorderDashOffset = NULL,
                            title.textStyle.textShadowColor = NULL,
                            title.textStyle.textShadowBlur = NULL,
                            title.textStyle.textShadowOffsetX = NULL,
                            title.textStyle.textShadowOffsetY = NULL,
                            title.subtextStyle.color = NULL,
                            title.subtextStyle.align = NULL,
                            title.subtextStyle.fontStyle = NULL,
                            title.subtextStyle.fontWeight = NULL,
                            title.subtextStyle.fontFamily = "Segoe UI",
                            title.subtextStyle.fontSize = NULL,
                            title.subtextStyle.lineHeight = NULL,
                            title.subtextStyle.width = NULL,
                            title.subtextStyle.height = NULL,
                            title.subtextStyle.textBorderColor = NULL,
                            title.subtextStyle.textBorderWidth = NULL,
                            title.subtextStyle.textBorderType = NULL,
                            title.subtextStyle.textBorderDashOffset = NULL,
                            title.subtextStyle.textShadowColor = NULL,
                            title.subtextStyle.textShadowBlur = NULL,
                            title.subtextStyle.textShadowOffsetX = NULL,
                            title.subtextStyle.textShadowOffsetY = NULL,
                            Debug = FALSE) {

  # Subset cols, define Target - Predicted, NULL YVar in data, Update YVar def, Ensure GroupVar is length(1)
  if(length(SampleSize) == 0L) SampleSize <- 30000L
  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  if(Debug) print("here 1")
  if(Debug) print(utils::head(dt))

  # Subset columns
  dt1 <- dt[, .SD, .SDcols = c(YVar)]

  # Transformation
  # "PercRank"  "Standardize"
  # "Asinh"  "Log"  "LogPlus1"  "Sqrt"  "Asin"  "Logit"  "BoxCox"  "YeoJohnson"
  if(YVarTrans != "Identity") {
    dt1 <- tryCatch({AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data}, error = function(x) dt1)
  }

  # Theoretical Quantiles
  data.table::setorderv(x = dt1, cols = YVar, 1)
  dt1[, temp_i := seq_len(.N)]
  dt1[, `Theoretical Quantiles` := stats::qnorm((temp_i-0.5)/.N)]
  dt1[, temp_i := NULL]

  # Normal Line
  meanX <- dt1[, mean(get(YVar), na.rm = TRUE)]
  sdX <- dt1[, stats::sd(get(YVar), na.rm = TRUE)]
  dt1[, `Normal Line` := eval(meanX) + sdX * `Theoretical Quantiles`]

  # Actual Quantiles
  p1 <- AutoPlots::Scatter(
    dt = dt1,
    SampleSize = SampleSize,
    XVar = "Theoretical Quantiles",
    YVar = YVar,
    YVarTrans = "Identity",
    Height = Height,
    Width = Width,
    Theme = Theme,

    ContainLabel = ContainLabel,
    title.text = title.text,
    title.subtext = title.subtext,
    title.link = title.link,
    title.sublink = title.sublink,
    title.Align = title.Align,
    title.top = title.top,
    title.left = title.left,
    title.right = title.right,
    title.bottom = title.bottom,
    title.padding = title.padding,
    title.itemGap = title.itemGap,
    title.backgroundColor = title.backgroundColor,
    title.borderColor = title.borderColor,
    title.borderWidth = title.borderWidth,
    title.borderRadius = title.borderRadius,
    title.shadowColor = title.shadowColor,
    title.shadowBlur = title.shadowBlur,
    title.shadowOffsetX = title.shadowOffsetX,
    title.shadowOffsetY = title.shadowOffsetY,
    title.textStyle.color = title.textStyle.color,
    title.textStyle.fontStyle = title.textStyle.fontStyle,
    title.textStyle.fontWeight = title.textStyle.fontWeight,
    title.textStyle.fontFamily = title.textStyle.fontFamily,
    title.textStyle.fontSize = title.textStyle.fontSize,
    title.textStyle.lineHeight = title.textStyle.lineHeight,
    title.textStyle.width = title.textStyle.width,
    title.textStyle.height = title.textStyle.height,
    title.textStyle.textBorderColor = title.textStyle.textBorderColor,
    title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
    title.textStyle.textBorderType = title.textStyle.textBorderType,
    title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
    title.textStyle.textShadowColor = title.textStyle.textShadowColor,
    title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
    title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
    title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
    title.subtextStyle.color = title.subtextStyle.color,
    title.subtextStyle.align = title.subtextStyle.align,
    title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
    title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
    title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
    title.subtextStyle.fontSize = title.subtextStyle.fontSize,
    title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
    title.subtextStyle.width = title.subtextStyle.width,
    title.subtextStyle.height = title.subtextStyle.height,
    title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
    title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
    title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
    title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
    title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
    title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
    title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
    title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY,
    Debug = Debug)

  # Add Normal Line
  p1 <- echarts4r::e_line_(e = p1, "Normal Line")
  return(p1)
}

#' @title Histogram
#'
#' @description Build a histogram plot by simply passing arguments to a single function. It will sample your data using SampleSize number of rows. Sampled data is randomized.
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param SampleSize An integer for the number of rows to use. Sampled data is randomized. If NULL then ignored
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param NumberBins = 30
#' @param Height "400px"
#' @param Width "200px"
#' @param Theme = Theme,
#' @param TimeLine logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ShowLabels FALSE
#' @param backgroundStyle.color vector of colors
#' @param backgroundStyle.opacity vector of opacities
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)))
#'
#' # Create plot
#' AutoPlots::Histogram(
#'   dt = dt,
#'   SampleSize = 30000L,
#'   XVar = NULL,
#'   GroupVar = NULL,
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   NumberBins = 30,
#'   Height = NULL,
#'   Width = NULL,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   TimeLine = FALSE,
#'   ShowLabels = FALSE,
#'   Debug = FALSE)
#' @return plot
#' @export
Histogram <- function(dt = NULL,
                      SampleSize = 30000L,
                      XVar = NULL,
                      GroupVar = NULL,
                      XVarTrans = "Identity",
                      FacetRows = 1,
                      FacetCols = 1,
                      FacetLevels = NULL,
                      NumberBins = 30,
                      Height = NULL,
                      Width = NULL,
                      Theme = "dark",
                      MouseScroll = FALSE,
                      TimeLine = FALSE,
                      ShowLabels = FALSE,
                      backgroundStyle.color = NULL,
                      backgroundStyle.opacity = NULL,
                      title.text = "Histogram",
                      title.subtext = NULL,
                      title.link = NULL,
                      title.sublink = NULL,
                      title.Align = NULL,
                      title.top = NULL,
                      title.left = NULL,
                      title.right = NULL,
                      title.bottom = NULL,
                      title.padding = NULL,
                      title.itemGap = NULL,
                      title.backgroundColor = NULL,
                      title.borderColor = NULL,
                      title.borderWidth = NULL,
                      title.borderRadius = NULL,
                      title.shadowColor = NULL,
                      title.shadowBlur = NULL,
                      title.shadowOffsetX = NULL,
                      title.shadowOffsetY = NULL,
                      title.textStyle.color = NULL,
                      title.textStyle.fontStyle = NULL,
                      title.textStyle.fontWeight = NULL,
                      title.textStyle.fontFamily = "Segoe UI",
                      title.textStyle.fontSize = NULL,
                      title.textStyle.lineHeight = NULL,
                      title.textStyle.width = NULL,
                      title.textStyle.height = NULL,
                      title.textStyle.textBorderColor = NULL,
                      title.textStyle.textBorderWidth = NULL,
                      title.textStyle.textBorderType = NULL,
                      title.textStyle.textBorderDashOffset = NULL,
                      title.textStyle.textShadowColor = NULL,
                      title.textStyle.textShadowBlur = NULL,
                      title.textStyle.textShadowOffsetX = NULL,
                      title.textStyle.textShadowOffsetY = NULL,
                      title.subtextStyle.color = NULL,
                      title.subtextStyle.align = NULL,
                      title.subtextStyle.fontStyle = NULL,
                      title.subtextStyle.fontWeight = NULL,
                      title.subtextStyle.fontFamily = "Segoe UI",
                      title.subtextStyle.fontSize = NULL,
                      title.subtextStyle.lineHeight = NULL,
                      title.subtextStyle.width = NULL,
                      title.subtextStyle.height = NULL,
                      title.subtextStyle.textBorderColor = NULL,
                      title.subtextStyle.textBorderWidth = NULL,
                      title.subtextStyle.textBorderType = NULL,
                      title.subtextStyle.textBorderDashOffset = NULL,
                      title.subtextStyle.textShadowColor = NULL,
                      title.subtextStyle.textShadowBlur = NULL,
                      title.subtextStyle.textShadowOffsetX = NULL,
                      title.subtextStyle.textShadowOffsetY = NULL,
                      xAxis.title = NULL,
                      xAxis.min = NULL,
                      xAxis.max = NULL,
                      xAxis.position = "bottom",
                      xAxis.nameLocation = 'center',
                      xAxis.axisTick.customValues = NULL,
                      xAxis.nameTextStyle.color = NULL,
                      xAxis.nameTextStyle.padding = 25,
                      xAxis.nameTextStyle.align = NULL,
                      xAxis.nameTextStyle.fontStyle = NULL,
                      xAxis.nameTextStyle.fontWeight = "bolder",
                      xAxis.nameTextStyle.fontSize = NULL,
                      xAxis.nameTextStyle.fontFamily = "Segoe UI",
                      xAxis.splitNumber = NULL,
                      xAxis.axisLabel.rotate = NULL,
                      xAxis.axisLabel.margin = NULL,
                      xAxis.axisLabel.color = NULL,
                      xAxis.axisLabel.fontStyle = NULL,
                      xAxis.axisLabel.fontWeight = NULL,
                      xAxis.axisLabel.fontFamily = "Segoe UI",
                      xAxis.axisLabel.fontSize = 14,
                      xAxis.axisLabel.align = NULL,
                      xAxis.axisLabel.verticalAlign = NULL,
                      xAxis.axisLabel.backgroundColor = NULL,
                      xAxis.axisLabel.borderColor = NULL,
                      xAxis.axisLabel.borderWidth = NULL,
                      xAxis.axisLabel.borderType = NULL,
                      xAxis.axisLabel.borderRadius = NULL,
                      xAxis.axisLabel.padding = NULL,
                      xAxis.axisLabel.shadowColor = NULL,
                      xAxis.axisLabel.shadowBlur = NULL,
                      xAxis.axisLabel.shadowOffsetX = NULL,
                      xAxis.axisLabel.shadowOffsetY = NULL,
                      xAxis.axisLabel.textBorderColor = NULL,
                      xAxis.axisLabel.textBorderWidth = NULL,
                      xAxis.axisLabel.textBorderType = NULL,
                      xAxis.axisLabel.textShadowColor = NULL,
                      xAxis.axisLabel.textShadowBlur = NULL,
                      xAxis.axisLabel.textShadowOffsetX = NULL,
                      xAxis.axisLabel.textShadowOffsetY = NULL,
                      xAxis.axisLabel.overflow = NULL,
                      yAxis.title = NULL,
                      yAxis.min = NULL,
                      yAxis.max = NULL,
                      yAxis.position = "center",
                      yAxis.nameLocation = 'center',
                      yAxis.axisTick.customValues = NULL,
                      yAxis.nameTextStyle.color = NULL,
                      yAxis.nameTextStyle.padding = 25,
                      yAxis.nameTextStyle.align = NULL,
                      yAxis.nameTextStyle.fontStyle = NULL,
                      yAxis.nameTextStyle.fontWeight = "bolder",
                      yAxis.nameTextStyle.fontSize = NULL,
                      yAxis.nameTextStyle.fontFamily = "Segoe UI",
                      yAxis.splitNumber = NULL,
                      yAxis.axisLabel.rotate = NULL,
                      yAxis.axisLabel.margin = NULL,
                      yAxis.axisLabel.color = NULL,
                      yAxis.axisLabel.fontStyle = NULL,
                      yAxis.axisLabel.fontWeight = NULL,
                      yAxis.axisLabel.fontFamily = "Segoe UI",
                      yAxis.axisLabel.fontSize = 14,
                      yAxis.axisLabel.align = NULL,
                      yAxis.axisLabel.verticalAlign = NULL,
                      yAxis.axisLabel.backgroundColor = NULL,
                      yAxis.axisLabel.borderColor = NULL,
                      yAxis.axisLabel.borderWidth = NULL,
                      yAxis.axisLabel.borderType = NULL,
                      yAxis.axisLabel.borderRadius = NULL,
                      yAxis.axisLabel.padding = NULL,
                      yAxis.axisLabel.shadowColor = NULL,
                      yAxis.axisLabel.shadowBlur = NULL,
                      yAxis.axisLabel.shadowOffsetX = NULL,
                      yAxis.axisLabel.shadowOffsetY = NULL,
                      yAxis.axisLabel.textBorderColor = NULL,
                      yAxis.axisLabel.textBorderWidth = NULL,
                      yAxis.axisLabel.textBorderType = NULL,
                      yAxis.axisLabel.textShadowColor = NULL,
                      yAxis.axisLabel.textShadowBlur = NULL,
                      yAxis.axisLabel.textShadowOffsetX = NULL,
                      yAxis.axisLabel.textShadowOffsetY = NULL,
                      yAxis.axisLabel.overflow = NULL,
                      legend.show = TRUE,
                      legend.type = "scroll",
                      legend.selector = NULL,
                      legend.icon = NULL,
                      legend.align = NULL,
                      legend.padding = NULL,
                      legend.itemGap = NULL,
                      legend.itemWidth = NULL,
                      legend.orient = "vertical",
                      legend.width = NULL,
                      legend.height = NULL,
                      legend.left = NULL,
                      legend.right = 10,
                      legend.top = 30,
                      legend.bottom = NULL,
                      legend.backgroundColor = NULL,
                      legend.borderColor = NULL,
                      legend.borderWidth = NULL,
                      legend.borderRadius = NULL,
                      legend.shadowBlur = NULL,
                      legend.shadowColor = NULL,
                      legend.shadowOffsetX = NULL,
                      legend.shadowOffsetY = NULL,
                      legend.itemStyle.color = NULL,
                      legend.itemStyle.borderColor = NULL,
                      legend.itemStyle.borderWidth = NULL,
                      legend.itemStyle.borderType = NULL,
                      legend.itemStyle.shadowBlur = NULL,
                      legend.itemStyle.shadowColor = NULL,
                      legend.itemStyle.shadowOffsetX = NULL,
                      legend.itemStyle.shadowOffsetY = NULL,
                      legend.itemStyle.opacity = NULL,
                      legend.lineStyle.color = NULL,
                      legend.lineStyle.width = NULL,
                      legend.lineStyle.type = NULL,
                      legend.lineStyle.shadowBlur = NULL,
                      legend.lineStyle.shadowColor = NULL,
                      legend.lineStyle.shadowOffsetX = NULL,
                      legend.lineStyle.shadowOffsetY = NULL,
                      legend.lineStyle.opacity = NULL,
                      legend.lineStyle.inactiveColor = NULL,
                      legend.lineStyle.inactiveWidth = NULL,
                      legend.textStyle.color = NULL,
                      legend.textStyle.fontStyle = NULL,
                      legend.textStyle.fontWeight = NULL,
                      legend.textStyle.fontFamily = "Segoe UI",
                      legend.textStyle.fontSize = NULL,
                      legend.textStyle.backgroundColor = NULL,
                      legend.textStyle.borderColor = NULL,
                      legend.textStyle.borderWidth = NULL,
                      legend.textStyle.borderType = NULL,
                      legend.textStyle.borderRadius = NULL,
                      legend.textStyle.padding = NULL,
                      legend.textStyle.shadowColor = NULL,
                      legend.textStyle.shadowBlur = NULL,
                      legend.textStyle.shadowOffsetX = NULL,
                      legend.textStyle.shadowOffsetY = NULL,
                      legend.textStyle.width = NULL,
                      legend.textStyle.height = NULL,
                      legend.textStyle.textBorderColor = NULL,
                      legend.textStyle.textBorderWidth = NULL,
                      legend.textStyle.textBorderType = NULL,
                      legend.textStyle.textShadowColor = NULL,
                      legend.textStyle.textShadowBlur = NULL,
                      legend.textStyle.textShadowOffsetX = NULL,
                      legend.textStyle.textShadowOffsetY = NULL,
                      legend.pageTextStyle.color = NULL,
                      legend.pageTextStyle.fontStyle = NULL,
                      legend.pageTextStyle.fontWeight = NULL,
                      legend.pageTextStyle.fontFamily = "Segoe UI",
                      legend.pageTextStyle.fontSize = NULL,
                      legend.pageTextStyle.lineHeight = NULL,
                      legend.pageTextStyle.width = NULL,
                      legend.pageTextStyle.height = NULL,
                      legend.pageTextStyle.textBorderColor = NULL,
                      legend.pageTextStyle.textBorderWidth = NULL,
                      legend.pageTextStyle.textBorderType = NULL,
                      legend.pageTextStyle.textShadowColor = NULL,
                      legend.pageTextStyle.textShadowBlur = NULL,
                      legend.pageTextStyle.textShadowOffsetX = NULL,
                      legend.pageTextStyle.textShadowOffsetY = NULL,
                      legend.emphasis.selectorLabel.show = NULL,
                      legend.emphasis.selectorLabel.distance = NULL,
                      legend.emphasis.selectorLabel.rotate = NULL,
                      legend.emphasis.selectorLabel.color = NULL,
                      legend.emphasis.selectorLabel.fontStyle = NULL,
                      legend.emphasis.selectorLabel.fontWeight = NULL,
                      legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                      legend.emphasis.selectorLabel.fontSize = NULL,
                      legend.emphasis.selectorLabel.align = NULL,
                      legend.emphasis.selectorLabel.verticalAlign = NULL,
                      legend.emphasis.selectorLabel.lineHeight = NULL,
                      legend.emphasis.selectorLabel.backgroundColor = NULL,
                      legend.emphasis.selectorLabel.borderColor = NULL,
                      legend.emphasis.selectorLabel.borderWidth = NULL,
                      legend.emphasis.selectorLabel.borderType = NULL,
                      legend.emphasis.selectorLabel.borderRadius = NULL,
                      legend.emphasis.selectorLabel.padding = NULL,
                      legend.emphasis.selectorLabel.shadowColor = NULL,
                      legend.emphasis.selectorLabel.shadowBlur = NULL,
                      legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                      legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                      legend.emphasis.selectorLabel.width = NULL,
                      legend.emphasis.selectorLabel.height = NULL,
                      legend.emphasis.selectorLabel.textBorderColor = NULL,
                      legend.emphasis.selectorLabel.textBorderWidth = NULL,
                      legend.emphasis.selectorLabel.textBorderType = NULL,
                      legend.emphasis.selectorLabel.textShadowColor = NULL,
                      legend.emphasis.selectorLabel.textShadowBlur = NULL,
                      legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                      legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                      tooltip.show = TRUE,
                      tooltip.trigger = "axis",
                      tooltip.backgroundColor = NULL,
                      tooltip.borderColor = NULL,
                      tooltip.borderWidth = NULL,
                      tooltip.padding = NULL,
                      tooltip.axisPointer.type = "cross",
                      tooltip.axisPointer.lineStyle.color = NULL,
                      tooltip.axisPointer.shadowStyle.color = NULL,
                      tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                      tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                      tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                      tooltip.axisPointer.shadowStyle.opacity = NULL,
                      tooltip.textStyle.color = NULL,
                      tooltip.textStyle.fontStyle = NULL,
                      tooltip.textStyle.fontWeight = NULL,
                      tooltip.textStyle.fontFamily = "Segoe UI",
                      tooltip.textStyle.lineHeight = NULL,
                      tooltip.textStyle.width = NULL,
                      tooltip.textStyle.height = NULL,
                      tooltip.textStyle.textBorderColor = NULL,
                      tooltip.textStyle.textBorderWidth = NULL,
                      tooltip.textStyle.textBorderType = NULL,
                      tooltip.textStyle.textShadowColor = NULL,
                      tooltip.textStyle.textShadowBlur = NULL,
                      tooltip.textStyle.textShadowOffsetX = NULL,
                      tooltip.textStyle.textShadowOffsetY = NULL,
                      toolbox.show = TRUE,
                      toolbox.orient = "horizontal",
                      toolbox.itemSize = 15,
                      toolbox.itemGap = 8,
                      toolbox.top = NULL,
                      toolbox.left = NULL,
                      toolbox.right = NULL,
                      toolbox.bottom = NULL,
                      toolbox.width = NULL,
                      toolbox.heigth = NULL,
                      toolbox.feature.saveAsImage.show = TRUE,
                      toolbox.feature.restore.show = TRUE,
                      toolbox.feature.dataZoom.show = TRUE,
                      toolbox.feature.magicType.show = FALSE,
                      toolbox.feature.magicType.type = NULL,
                      toolbox.feature.dataView.show = TRUE,
                      toolbox.iconStyle.color = NULL,
                      toolbox.iconStyle.borderColor = NULL,
                      toolbox.emphasis.iconStyle.borderColor = NULL,
                      toolbox.iconStyle.shadowBlur = NULL,
                      toolbox.iconStyle.shadowColor = NULL,
                      toolbox.iconStyle.shadowOffsetX = NULL,
                      toolbox.iconStyle.shadowOffsetY = NULL,
                      Debug = FALSE) {

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  TimeLine <- FALSE

  # Cap number of records
  if(length(SampleSize) == 0L) SampleSize <- 30000
  if(dt[, .N] > SampleSize) {
    dt1 <- dt[order(stats::runif(.N))][seq_len(SampleSize)]
  } else {
    dt1 <- data.table::copy(dt)
  }

  # Define Plotting Variable
  if(length(XVar) == 0) return(NULL)
  GroupVar <- tryCatch({GroupVar[1L]}, error = function(x) NULL)

  # Faceting shrink
  if(length(GroupVar) > 0L && (FacetRows > 1L || FacetCols > 1L)) {
    dt1 <- dt1[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(XVar,GroupVar)]
  } else {
    dt1 <- dt1[, .SD, .SDcols = c(XVar,GroupVar)]
  }

  # Multiple YVars
  if(length(XVar) > 1L) {
    sqroots <- sqrt(length(XVar))
    if(FacetCols == 1 && FacetRows == 1L) {
      FacetCols <- max(ceiling(sqroots), 6)
      FacetRows <- ceiling(sqroots)
      if((FacetRows - 1L) * FacetCols == length(XVar)) {
        FacetRows <- FacetRows - 1L
      } else if(FacetRows * FacetCols < length(XVar)) {
        while(FacetRows * FacetCols < length(XVar)) {
          FacetRows <- FacetRows + 1L
        }
      }
    }
    XVar <- NULL
    GroupVar <- NULL
    dt1[, temp__ := "a"]
    dt1 <- data.table::melt.data.table(data = dt1, id.vars = "temp__", measure.vars = XVar, variable.name = "Measures", value.name = "Values")
    dt1[, temp__ := NULL]
    GroupVar <- "Measures"
    XVar <- "Values"
  }

  # Transformation
  if(XVarTrans != "Identity") {
    for(ggss in XVar) {
      dt1 <- tryCatch({AutoTransformationCreate(data = dt1, ColumnNames = ggss, Methods = XVarTrans)$Data}, error = function(x) dt1)
    }
  }

  # Create histogram data
  if(length(GroupVar) == 0L) {
    Min <- dt1[, min(get(XVar), na.rm = TRUE)]
    Max <- dt1[, max(get(XVar), na.rm = TRUE)]
    Range <- Max - Min
    if(Range < NumberBins) {
      acc <- round(Range / NumberBins, 2)
      dt1[, Buckets := round(get(XVar) / acc) * acc]
      dt1 <- dt1[, .N, by = "Buckets"][order(Buckets)]
    } else {
      acc <- ceiling(Range / NumberBins)
      dt1[, Buckets := round(get(XVar) / acc) * acc]
      dt1 <- dt1[, .N, by = "Buckets"][order(Buckets)]
    }

  } else {
    levs <- unique(as.character(dt1[[GroupVar]]))
    gg <- list()
    for(i in levs) {# i <- levs[1]
      temp <- dt1[get(GroupVar) == eval(i)]
      Min <- temp[, min(get(XVar), na.rm = TRUE)]
      Max <- temp[, max(get(XVar), na.rm = TRUE)]
      Range <- Max - Min
      if(Range < NumberBins) {
        acc <- round(Range / NumberBins, 2)
      } else {
        acc <- ceiling(Range / NumberBins)
      }
      temp[, Buckets := round(get(XVar) / acc) * acc]
      gg[[i]] <- temp[, .N, by = c("Buckets",GroupVar)][order(Buckets)]
    }
    dt1 <- data.table::rbindlist(gg)
  }

  # Run Bar Plot for no Group and Stacked Bar for Groups?
  dt1[, Buckets := as.character(Buckets)]
  if(length(GroupVar) == 0L) {
    p1 <- Bar(
      dt = dt1,
      PreAgg = TRUE,
      XVar = "Buckets",
      YVar = "N",
      Height = Height,
      Width = Width,
      Theme = Theme,
      MouseScroll = MouseScroll,
      TimeLine = TimeLine,

      backgroundStyle.color = backgroundStyle.color,
      backgroundStyle.opacity = backgroundStyle.opacity,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY,
      xAxis.title = xAxis.title,
      xAxis.min = xAxis.min,
      xAxis.max = xAxis.max,
      xAxis.position = xAxis.position,
      xAxis.nameLocation = xAxis.nameLocation,
      xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding,
      xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle,
      xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize,
      xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily,
      xAxis.splitNumber = xAxis.splitNumber,
      xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin,
      xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle,
      xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily,
      xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align,
      xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor,
      xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth,
      xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius,
      xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor,
      xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX,
      xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor,
      xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType,
      xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur,
      xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY,
      xAxis.axisLabel.overflow = xAxis.axisLabel.overflow,
      yAxis.title = yAxis.title,
      yAxis.min = yAxis.min,
      yAxis.max = yAxis.max,
      yAxis.position = yAxis.position,
      yAxis.nameLocation = yAxis.nameLocation,
      yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding,
      yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle,
      yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize,
      yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily,
      yAxis.splitNumber = yAxis.splitNumber,
      yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin,
      yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle,
      yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily,
      yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align,
      yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor,
      yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth,
      yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius,
      yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor,
      yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX,
      yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor,
      yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType,
      yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur,
      yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY,
      yAxis.axisLabel.overflow = yAxis.axisLabel.overflow,
      legend.show = legend.show,
      legend.type = legend.type,
      legend.selector = legend.selector,
      legend.icon = legend.icon,
      legend.align = legend.align,
      legend.padding = legend.padding,
      legend.itemGap = legend.itemGap,
      legend.itemWidth = legend.itemWidth,
      legend.orient = legend.orient,
      legend.width = legend.width,
      legend.height = legend.height,
      legend.left = legend.left,
      legend.right = legend.right,
      legend.top = legend.top,
      legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor,
      legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth,
      legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur,
      legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX,
      legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color,
      legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth,
      legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur,
      legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX,
      legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity,
      legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width,
      legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur,
      legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX,
      legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity,
      legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth,
      legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle,
      legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily,
      legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor,
      legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth,
      legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius,
      legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor,
      legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX,
      legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width,
      legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor,
      legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType,
      legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur,
      legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY,
      legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle,
      legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily,
      legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight,
      legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height,
      legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth,
      legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor,
      legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX,
      legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show,
      legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate,
      legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle,
      legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily,
      legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align,
      legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight,
      legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor,
      legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType,
      legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding,
      legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur,
      legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY,
      legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height,
      legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth,
      legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor,
      legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX,
      legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY,
      Debug = Debug)

  } else {

    p1 <- Bar(
      dt = dt1,
      PreAgg = TRUE,
      XVar = "Buckets",
      YVar = "N",
      GroupVar = GroupVar,
      FacetRows = FacetRows,
      FacetCols = FacetCols,
      FacetLevels = FacetLevels,
      Height = Height,
      Width = Width,
      MouseScroll = MouseScroll,
      Theme = Theme,
      TimeLine = TimeLine,

      backgroundStyle.color = backgroundStyle.color,
      backgroundStyle.opacity = backgroundStyle.opacity,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY,
      xAxis.title = xAxis.title,
      xAxis.min = xAxis.min,
      xAxis.max = xAxis.max,
      xAxis.position = xAxis.position,
      xAxis.nameLocation = xAxis.nameLocation,
      xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding,
      xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle,
      xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize,
      xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily,
      xAxis.splitNumber = xAxis.splitNumber,
      xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin,
      xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle,
      xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily,
      xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align,
      xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor,
      xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth,
      xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius,
      xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor,
      xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX,
      xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor,
      xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType,
      xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur,
      xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY,
      xAxis.axisLabel.overflow = xAxis.axisLabel.overflow,
      yAxis.title = yAxis.title,
      yAxis.min = yAxis.min,
      yAxis.max = yAxis.max,
      yAxis.position = yAxis.position,
      yAxis.nameLocation = yAxis.nameLocation,
      yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding,
      yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle,
      yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize,
      yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily,
      yAxis.splitNumber = yAxis.splitNumber,
      yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin,
      yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle,
      yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily,
      yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align,
      yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor,
      yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth,
      yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius,
      yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor,
      yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX,
      yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor,
      yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType,
      yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur,
      yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY,
      yAxis.axisLabel.overflow = yAxis.axisLabel.overflow,
      legend.show = legend.show,
      legend.type = legend.type,
      legend.selector = legend.selector,
      legend.icon = legend.icon,
      legend.align = legend.align,
      legend.padding = legend.padding,
      legend.itemGap = legend.itemGap,
      legend.itemWidth = legend.itemWidth,
      legend.orient = legend.orient,
      legend.width = legend.width,
      legend.height = legend.height,
      legend.left = legend.left,
      legend.right = legend.right,
      legend.top = legend.top,
      legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor,
      legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth,
      legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur,
      legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX,
      legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color,
      legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth,
      legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur,
      legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX,
      legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity,
      legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width,
      legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur,
      legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX,
      legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity,
      legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth,
      legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle,
      legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily,
      legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor,
      legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth,
      legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius,
      legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor,
      legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX,
      legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width,
      legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor,
      legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType,
      legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur,
      legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY,
      legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle,
      legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily,
      legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight,
      legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height,
      legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth,
      legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor,
      legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX,
      legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show,
      legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate,
      legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle,
      legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily,
      legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align,
      legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight,
      legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor,
      legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType,
      legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding,
      legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur,
      legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY,
      legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height,
      legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth,
      legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor,
      legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX,
      legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY,
      Debug = Debug)
  }
  return(p1)
}


#' @title Pie
#'
#' @description Build a pie chart by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo","essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired","jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal","sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)), GV = sample(LETTERS, 1000, TRUE))
#'
#' # Create plot
#' AutoPlots::Pie(
#'   dt = dt,
#'   PreAgg = TRUE,
#'   XVar = "GV",
#'   YVar = "Y",
#'   GroupVar = NULL,
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   AggMethod = 'mean',
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   TimeLine = FALSE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Pie <- function(dt = NULL,
                PreAgg = TRUE,
                XVar = NULL,
                YVar = NULL,
                GroupVar = NULL,
                YVarTrans = "Identity",
                XVarTrans = "Identity",
                FacetRows = 1,
                FacetCols = 1,
                FacetLevels = NULL,
                AggMethod = 'mean',
                Height = NULL,
                Width = NULL,
                ShowLabels = TRUE,
                Theme = "dark",
                TimeLine = FALSE,
                title.text = "Pie Chart",
                title.subtext = NULL,
                title.link = NULL,
                title.sublink = NULL,
                title.Align = NULL,
                title.top = NULL,
                title.left = NULL,
                title.right = NULL,
                title.bottom = NULL,
                title.padding = NULL,
                title.itemGap = NULL,
                title.backgroundColor = NULL,
                title.borderColor = NULL,
                title.borderWidth = NULL,
                title.borderRadius = NULL,
                title.shadowColor = NULL,
                title.shadowBlur = NULL,
                title.shadowOffsetX = NULL,
                title.shadowOffsetY = NULL,
                title.textStyle.color = NULL,
                title.textStyle.fontStyle = NULL,
                title.textStyle.fontWeight = NULL,
                title.textStyle.fontFamily = "Segoe UI",
                title.textStyle.fontSize = NULL,
                title.textStyle.lineHeight = NULL,
                title.textStyle.width = NULL,
                title.textStyle.height = NULL,
                title.textStyle.textBorderColor = NULL,
                title.textStyle.textBorderWidth = NULL,
                title.textStyle.textBorderType = NULL,
                title.textStyle.textBorderDashOffset = NULL,
                title.textStyle.textShadowColor = NULL,
                title.textStyle.textShadowBlur = NULL,
                title.textStyle.textShadowOffsetX = NULL,
                title.textStyle.textShadowOffsetY = NULL,
                title.subtextStyle.color = NULL,
                title.subtextStyle.align = NULL,
                title.subtextStyle.fontStyle = NULL,
                title.subtextStyle.fontWeight = NULL,
                title.subtextStyle.fontFamily = "Segoe UI",
                title.subtextStyle.fontSize = NULL,
                title.subtextStyle.lineHeight = NULL,
                title.subtextStyle.width = NULL,
                title.subtextStyle.height = NULL,
                title.subtextStyle.textBorderColor = NULL,
                title.subtextStyle.textBorderWidth = NULL,
                title.subtextStyle.textBorderType = NULL,
                title.subtextStyle.textBorderDashOffset = NULL,
                title.subtextStyle.textShadowColor = NULL,
                title.subtextStyle.textShadowBlur = NULL,
                title.subtextStyle.textShadowOffsetX = NULL,
                title.subtextStyle.textShadowOffsetY = NULL,
                legend.show = TRUE,
                legend.type = "scroll",
                legend.selector = NULL,
                legend.icon = NULL,
                legend.align = NULL,
                legend.padding = NULL,
                legend.itemGap = NULL,
                legend.itemWidth = NULL,
                legend.orient = "vertical",
                legend.width = NULL,
                legend.height = NULL,
                legend.left = NULL,
                legend.right = 10,
                legend.top = 30,
                legend.bottom = NULL,
                legend.backgroundColor = NULL,
                legend.borderColor = NULL,
                legend.borderWidth = NULL,
                legend.borderRadius = NULL,
                legend.shadowBlur = NULL,
                legend.shadowColor = NULL,
                legend.shadowOffsetX = NULL,
                legend.shadowOffsetY = NULL,
                legend.itemStyle.color = NULL,
                legend.itemStyle.borderColor = NULL,
                legend.itemStyle.borderWidth = NULL,
                legend.itemStyle.borderType = NULL,
                legend.itemStyle.shadowBlur = NULL,
                legend.itemStyle.shadowColor = NULL,
                legend.itemStyle.shadowOffsetX = NULL,
                legend.itemStyle.shadowOffsetY = NULL,
                legend.itemStyle.opacity = NULL,
                legend.lineStyle.color = NULL,
                legend.lineStyle.width = NULL,
                legend.lineStyle.type = NULL,
                legend.lineStyle.shadowBlur = NULL,
                legend.lineStyle.shadowColor = NULL,
                legend.lineStyle.shadowOffsetX = NULL,
                legend.lineStyle.shadowOffsetY = NULL,
                legend.lineStyle.opacity = NULL,
                legend.lineStyle.inactiveColor = NULL,
                legend.lineStyle.inactiveWidth = NULL,
                legend.textStyle.color = NULL,
                legend.textStyle.fontStyle = NULL,
                legend.textStyle.fontWeight = NULL,
                legend.textStyle.fontFamily = "Segoe UI",
                legend.textStyle.fontSize = NULL,
                legend.textStyle.backgroundColor = NULL,
                legend.textStyle.borderColor = NULL,
                legend.textStyle.borderWidth = NULL,
                legend.textStyle.borderType = NULL,
                legend.textStyle.borderRadius = NULL,
                legend.textStyle.padding = NULL,
                legend.textStyle.shadowColor = NULL,
                legend.textStyle.shadowBlur = NULL,
                legend.textStyle.shadowOffsetX = NULL,
                legend.textStyle.shadowOffsetY = NULL,
                legend.textStyle.width = NULL,
                legend.textStyle.height = NULL,
                legend.textStyle.textBorderColor = NULL,
                legend.textStyle.textBorderWidth = NULL,
                legend.textStyle.textBorderType = NULL,
                legend.textStyle.textShadowColor = NULL,
                legend.textStyle.textShadowBlur = NULL,
                legend.textStyle.textShadowOffsetX = NULL,
                legend.textStyle.textShadowOffsetY = NULL,
                legend.pageTextStyle.color = NULL,
                legend.pageTextStyle.fontStyle = NULL,
                legend.pageTextStyle.fontWeight = NULL,
                legend.pageTextStyle.fontFamily = "Segoe UI",
                legend.pageTextStyle.fontSize = NULL,
                legend.pageTextStyle.lineHeight = NULL,
                legend.pageTextStyle.width = NULL,
                legend.pageTextStyle.height = NULL,
                legend.pageTextStyle.textBorderColor = NULL,
                legend.pageTextStyle.textBorderWidth = NULL,
                legend.pageTextStyle.textBorderType = NULL,
                legend.pageTextStyle.textShadowColor = NULL,
                legend.pageTextStyle.textShadowBlur = NULL,
                legend.pageTextStyle.textShadowOffsetX = NULL,
                legend.pageTextStyle.textShadowOffsetY = NULL,
                legend.emphasis.selectorLabel.show = NULL,
                legend.emphasis.selectorLabel.distance = NULL,
                legend.emphasis.selectorLabel.rotate = NULL,
                legend.emphasis.selectorLabel.color = NULL,
                legend.emphasis.selectorLabel.fontStyle = NULL,
                legend.emphasis.selectorLabel.fontWeight = NULL,
                legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                legend.emphasis.selectorLabel.fontSize = NULL,
                legend.emphasis.selectorLabel.align = NULL,
                legend.emphasis.selectorLabel.verticalAlign = NULL,
                legend.emphasis.selectorLabel.lineHeight = NULL,
                legend.emphasis.selectorLabel.backgroundColor = NULL,
                legend.emphasis.selectorLabel.borderColor = NULL,
                legend.emphasis.selectorLabel.borderWidth = NULL,
                legend.emphasis.selectorLabel.borderType = NULL,
                legend.emphasis.selectorLabel.borderRadius = NULL,
                legend.emphasis.selectorLabel.padding = NULL,
                legend.emphasis.selectorLabel.shadowColor = NULL,
                legend.emphasis.selectorLabel.shadowBlur = NULL,
                legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                legend.emphasis.selectorLabel.width = NULL,
                legend.emphasis.selectorLabel.height = NULL,
                legend.emphasis.selectorLabel.textBorderColor = NULL,
                legend.emphasis.selectorLabel.textBorderWidth = NULL,
                legend.emphasis.selectorLabel.textBorderType = NULL,
                legend.emphasis.selectorLabel.textShadowColor = NULL,
                legend.emphasis.selectorLabel.textShadowBlur = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                tooltip.show = TRUE,
                tooltip.trigger = "item",
                tooltip.backgroundColor = NULL,
                tooltip.borderColor = NULL,
                tooltip.borderWidth = NULL,
                tooltip.padding = NULL,
                tooltip.axisPointer.type = "cross",
                tooltip.axisPointer.lineStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                tooltip.axisPointer.shadowStyle.opacity = NULL,
                tooltip.textStyle.color = NULL,
                tooltip.textStyle.fontStyle = NULL,
                tooltip.textStyle.fontWeight = NULL,
                tooltip.textStyle.fontFamily = "Segoe UI",
                tooltip.textStyle.lineHeight = NULL,
                tooltip.textStyle.width = NULL,
                tooltip.textStyle.height = NULL,
                tooltip.textStyle.textBorderColor = NULL,
                tooltip.textStyle.textBorderWidth = NULL,
                tooltip.textStyle.textBorderType = NULL,
                tooltip.textStyle.textShadowColor = NULL,
                tooltip.textStyle.textShadowBlur = NULL,
                tooltip.textStyle.textShadowOffsetX = NULL,
                tooltip.textStyle.textShadowOffsetY = NULL,
                toolbox.show = TRUE,
                toolbox.orient = "horizontal",
                toolbox.itemSize = 15,
                toolbox.itemGap = 8,
                toolbox.top = NULL,
                toolbox.left = NULL,
                toolbox.right = NULL,
                toolbox.bottom = NULL,
                toolbox.width = NULL,
                toolbox.heigth = NULL,
                toolbox.feature.saveAsImage.show = TRUE,
                toolbox.feature.restore.show = TRUE,
                toolbox.feature.dataZoom.show = TRUE,
                toolbox.feature.magicType.show = TRUE,
                toolbox.feature.magicType.type = c("line", "bar", "stack"),
                toolbox.feature.dataView.show = TRUE,
                toolbox.iconStyle.color = NULL,
                toolbox.iconStyle.borderColor = NULL,
                toolbox.emphasis.iconStyle.borderColor = NULL,
                toolbox.iconStyle.shadowBlur = NULL,
                toolbox.iconStyle.shadowColor = NULL,
                toolbox.iconStyle.shadowOffsetX = NULL,
                toolbox.iconStyle.shadowOffsetY = NULL,
                Debug = FALSE) {

  if(length(YVar) > 0L) YVar <- YVar[1L]
  if(length(XVar) > 0L) XVar <- XVar[1L]

  # Used multiple times
  check1 <- length(XVar) != 0 && length(YVar) != 0

  if(!PreAgg) {
    if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
      dt <- data.table::as.data.table(dt)
    })
    aggFunc <- SummaryFunction(AggMethod)
  }

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  if(length(XVar) > 0L && class(dt[[XVar]])[1L] == "factor") {
    dt[, eval(XVar) := as.character(get(XVar))]
  }

  # Create base plot object
  numvars <- c()
  byvars <- c()
  if(check1) {
    if(!PreAgg) {
      if(tryCatch({class(dt[[eval(YVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        numvars <- unique(c(numvars, YVar))
      } else {
        byvars <- unique(c(byvars, YVar))
      }
      if(tryCatch({class(dt[[eval(XVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        if(length(numvars) > 0) {
          x <- length(unique(dt[[XVar]]))
          y <- length(unique(dt[[YVar]]))
          if(x > y) {
            byvars <- unique(c(byvars, YVar))
            numvars[1L] <- XVar
          } else {
            byvars <- unique(c(byvars, XVar))
          }
        } else {
          numvars <- unique(c(numvars, XVar))
        }
      } else {
        byvars <- unique(c(byvars, XVar))
      }
      if(!is.null(byvars)) {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        for(i in byvars) {
          if(class(temp[[i]])[1L] %in% c('numeric','integer')) {
            temp[, eval(i) := as.character(get(i))]
          }
        }
      } else {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
      }
    } else {
      temp <- data.table::copy(dt)
      numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
      byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
    }

    # Transformation
    if(YVarTrans != "Identity") {
      temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
    }

    p1 <- echarts4r::e_charts_(
      temp,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      width = Width, height = Height)

    p1 <- echarts4r::e_pie_(
      e = p1,
      YVar,
      stack = XVar
    )

    if (ShowLabels) {
      p1 <- echarts4r::e_labels(
        e = p1,
        show = TRUE,
        formatter = htmlwidgets::JS(
          "function(params) {
             return params.name + ': ' + params.value.toFixed(1) + ' (' + params.percent.toFixed(1) + '%)';
           }"
        ),
        position = "outside"
      )
    }

    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
    p1 <- echarts4r::e_brush(e = p1)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    return(p1)
  }
}

#' @title Donut
#'
#' @description Build a donut plot by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo","essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired","jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal","sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#'
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)), GV = sample(LETTERS, 1000, TRUE))
#'
#' # Create plot
#' AutoPlots::Donut(
#'   dt = dt,
#'   PreAgg = TRUE,
#'   XVar = "GV",
#'   YVar = "Y",
#'   GroupVar = NULL,
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   AggMethod = 'mean',
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   TimeLine = FALSE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Donut <- function(dt = NULL,
                  PreAgg = TRUE,
                  XVar = NULL,
                  YVar = NULL,
                  GroupVar = NULL,
                  YVarTrans = "Identity",
                  XVarTrans = "Identity",
                  FacetRows = 1,
                  FacetCols = 1,
                  FacetLevels = NULL,
                  AggMethod = 'mean',
                  Height = NULL,
                  Width = NULL,
                  ShowLabels = TRUE,
                  Theme = "dark",
                  TimeLine = FALSE,
                  title.text = "Donut Plot",
                  title.subtext = NULL,
                  title.link = NULL,
                  title.sublink = NULL,
                  title.Align = NULL,
                  title.top = NULL,
                  title.left = NULL,
                  title.right = NULL,
                  title.bottom = NULL,
                  title.padding = NULL,
                  title.itemGap = NULL,
                  title.backgroundColor = NULL,
                  title.borderColor = NULL,
                  title.borderWidth = NULL,
                  title.borderRadius = NULL,
                  title.shadowColor = NULL,
                  title.shadowBlur = NULL,
                  title.shadowOffsetX = NULL,
                  title.shadowOffsetY = NULL,
                  title.textStyle.color = NULL,
                  title.textStyle.fontStyle = NULL,
                  title.textStyle.fontWeight = NULL,
                  title.textStyle.fontFamily = "Segoe UI",
                  title.textStyle.fontSize = NULL,
                  title.textStyle.lineHeight = NULL,
                  title.textStyle.width = NULL,
                  title.textStyle.height = NULL,
                  title.textStyle.textBorderColor = NULL,
                  title.textStyle.textBorderWidth = NULL,
                  title.textStyle.textBorderType = NULL,
                  title.textStyle.textBorderDashOffset = NULL,
                  title.textStyle.textShadowColor = NULL,
                  title.textStyle.textShadowBlur = NULL,
                  title.textStyle.textShadowOffsetX = NULL,
                  title.textStyle.textShadowOffsetY = NULL,
                  title.subtextStyle.color = NULL,
                  title.subtextStyle.align = NULL,
                  title.subtextStyle.fontStyle = NULL,
                  title.subtextStyle.fontWeight = NULL,
                  title.subtextStyle.fontFamily = "Segoe UI",
                  title.subtextStyle.fontSize = NULL,
                  title.subtextStyle.lineHeight = NULL,
                  title.subtextStyle.width = NULL,
                  title.subtextStyle.height = NULL,
                  title.subtextStyle.textBorderColor = NULL,
                  title.subtextStyle.textBorderWidth = NULL,
                  title.subtextStyle.textBorderType = NULL,
                  title.subtextStyle.textBorderDashOffset = NULL,
                  title.subtextStyle.textShadowColor = NULL,
                  title.subtextStyle.textShadowBlur = NULL,
                  title.subtextStyle.textShadowOffsetX = NULL,
                  title.subtextStyle.textShadowOffsetY = NULL,
                  legend.show = TRUE,
                  legend.type = "scroll",
                  legend.selector = NULL,
                  legend.icon = NULL,
                  legend.align = NULL,
                  legend.padding = NULL,
                  legend.itemGap = NULL,
                  legend.itemWidth = NULL,
                  legend.orient = "vertical",
                  legend.width = NULL,
                  legend.height = NULL,
                  legend.left = NULL,
                  legend.right = 10,
                  legend.top = 30,
                  legend.bottom = NULL,
                  legend.backgroundColor = NULL,
                  legend.borderColor = NULL,
                  legend.borderWidth = NULL,
                  legend.borderRadius = NULL,
                  legend.shadowBlur = NULL,
                  legend.shadowColor = NULL,
                  legend.shadowOffsetX = NULL,
                  legend.shadowOffsetY = NULL,
                  legend.itemStyle.color = NULL,
                  legend.itemStyle.borderColor = NULL,
                  legend.itemStyle.borderWidth = NULL,
                  legend.itemStyle.borderType = NULL,
                  legend.itemStyle.shadowBlur = NULL,
                  legend.itemStyle.shadowColor = NULL,
                  legend.itemStyle.shadowOffsetX = NULL,
                  legend.itemStyle.shadowOffsetY = NULL,
                  legend.itemStyle.opacity = NULL,
                  legend.lineStyle.color = NULL,
                  legend.lineStyle.width = NULL,
                  legend.lineStyle.type = NULL,
                  legend.lineStyle.shadowBlur = NULL,
                  legend.lineStyle.shadowColor = NULL,
                  legend.lineStyle.shadowOffsetX = NULL,
                  legend.lineStyle.shadowOffsetY = NULL,
                  legend.lineStyle.opacity = NULL,
                  legend.lineStyle.inactiveColor = NULL,
                  legend.lineStyle.inactiveWidth = NULL,
                  legend.textStyle.color = NULL,
                  legend.textStyle.fontStyle = NULL,
                  legend.textStyle.fontWeight = NULL,
                  legend.textStyle.fontFamily = "Segoe UI",
                  legend.textStyle.fontSize = NULL,
                  legend.textStyle.backgroundColor = NULL,
                  legend.textStyle.borderColor = NULL,
                  legend.textStyle.borderWidth = NULL,
                  legend.textStyle.borderType = NULL,
                  legend.textStyle.borderRadius = NULL,
                  legend.textStyle.padding = NULL,
                  legend.textStyle.shadowColor = NULL,
                  legend.textStyle.shadowBlur = NULL,
                  legend.textStyle.shadowOffsetX = NULL,
                  legend.textStyle.shadowOffsetY = NULL,
                  legend.textStyle.width = NULL,
                  legend.textStyle.height = NULL,
                  legend.textStyle.textBorderColor = NULL,
                  legend.textStyle.textBorderWidth = NULL,
                  legend.textStyle.textBorderType = NULL,
                  legend.textStyle.textShadowColor = NULL,
                  legend.textStyle.textShadowBlur = NULL,
                  legend.textStyle.textShadowOffsetX = NULL,
                  legend.textStyle.textShadowOffsetY = NULL,
                  legend.pageTextStyle.color = NULL,
                  legend.pageTextStyle.fontStyle = NULL,
                  legend.pageTextStyle.fontWeight = NULL,
                  legend.pageTextStyle.fontFamily = "Segoe UI",
                  legend.pageTextStyle.fontSize = NULL,
                  legend.pageTextStyle.lineHeight = NULL,
                  legend.pageTextStyle.width = NULL,
                  legend.pageTextStyle.height = NULL,
                  legend.pageTextStyle.textBorderColor = NULL,
                  legend.pageTextStyle.textBorderWidth = NULL,
                  legend.pageTextStyle.textBorderType = NULL,
                  legend.pageTextStyle.textShadowColor = NULL,
                  legend.pageTextStyle.textShadowBlur = NULL,
                  legend.pageTextStyle.textShadowOffsetX = NULL,
                  legend.pageTextStyle.textShadowOffsetY = NULL,
                  legend.emphasis.selectorLabel.show = NULL,
                  legend.emphasis.selectorLabel.distance = NULL,
                  legend.emphasis.selectorLabel.rotate = NULL,
                  legend.emphasis.selectorLabel.color = NULL,
                  legend.emphasis.selectorLabel.fontStyle = NULL,
                  legend.emphasis.selectorLabel.fontWeight = NULL,
                  legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                  legend.emphasis.selectorLabel.fontSize = NULL,
                  legend.emphasis.selectorLabel.align = NULL,
                  legend.emphasis.selectorLabel.verticalAlign = NULL,
                  legend.emphasis.selectorLabel.lineHeight = NULL,
                  legend.emphasis.selectorLabel.backgroundColor = NULL,
                  legend.emphasis.selectorLabel.borderColor = NULL,
                  legend.emphasis.selectorLabel.borderWidth = NULL,
                  legend.emphasis.selectorLabel.borderType = NULL,
                  legend.emphasis.selectorLabel.borderRadius = NULL,
                  legend.emphasis.selectorLabel.padding = NULL,
                  legend.emphasis.selectorLabel.shadowColor = NULL,
                  legend.emphasis.selectorLabel.shadowBlur = NULL,
                  legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                  legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                  legend.emphasis.selectorLabel.width = NULL,
                  legend.emphasis.selectorLabel.height = NULL,
                  legend.emphasis.selectorLabel.textBorderColor = NULL,
                  legend.emphasis.selectorLabel.textBorderWidth = NULL,
                  legend.emphasis.selectorLabel.textBorderType = NULL,
                  legend.emphasis.selectorLabel.textShadowColor = NULL,
                  legend.emphasis.selectorLabel.textShadowBlur = NULL,
                  legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                  legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                  tooltip.show = TRUE,
                  tooltip.trigger = "item",
                  tooltip.backgroundColor = NULL,
                  tooltip.borderColor = NULL,
                  tooltip.borderWidth = NULL,
                  tooltip.padding = NULL,
                  tooltip.axisPointer.type = "cross",
                  tooltip.axisPointer.lineStyle.color = NULL,
                  tooltip.axisPointer.shadowStyle.color = NULL,
                  tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                  tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                  tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                  tooltip.axisPointer.shadowStyle.opacity = NULL,
                  tooltip.textStyle.color = NULL,
                  tooltip.textStyle.fontStyle = NULL,
                  tooltip.textStyle.fontWeight = NULL,
                  tooltip.textStyle.fontFamily = "Segoe UI",
                  tooltip.textStyle.lineHeight = NULL,
                  tooltip.textStyle.width = NULL,
                  tooltip.textStyle.height = NULL,
                  tooltip.textStyle.textBorderColor = NULL,
                  tooltip.textStyle.textBorderWidth = NULL,
                  tooltip.textStyle.textBorderType = NULL,
                  tooltip.textStyle.textShadowColor = NULL,
                  tooltip.textStyle.textShadowBlur = NULL,
                  tooltip.textStyle.textShadowOffsetX = NULL,
                  tooltip.textStyle.textShadowOffsetY = NULL,
                  toolbox.show = TRUE,
                  toolbox.orient = "horizontal",
                  toolbox.itemSize = 15,
                  toolbox.itemGap = 8,
                  toolbox.top = NULL,
                  toolbox.left = NULL,
                  toolbox.right = NULL,
                  toolbox.bottom = NULL,
                  toolbox.width = NULL,
                  toolbox.heigth = NULL,
                  toolbox.feature.saveAsImage.show = TRUE,
                  toolbox.feature.restore.show = TRUE,
                  toolbox.feature.dataZoom.show = TRUE,
                  toolbox.feature.magicType.show = TRUE,
                  toolbox.feature.magicType.type = c("line", "bar", "stack"),
                  toolbox.feature.dataView.show = TRUE,
                  toolbox.iconStyle.color = NULL,
                  toolbox.iconStyle.borderColor = NULL,
                  toolbox.emphasis.iconStyle.borderColor = NULL,
                  toolbox.iconStyle.shadowBlur = NULL,
                  toolbox.iconStyle.shadowColor = NULL,
                  toolbox.iconStyle.shadowOffsetX = NULL,
                  toolbox.iconStyle.shadowOffsetY = NULL,
                  Debug = FALSE) {

  if(length(YVar) > 0L) YVar <- YVar[1L]
  if(length(XVar) > 0L) XVar <- XVar[1L]

  # Used multiple times
  check1 <- length(XVar) != 0 && length(YVar) != 0

  if(!PreAgg) {
    if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
      dt <- data.table::as.data.table(dt)
    })
    aggFunc <- SummaryFunction(AggMethod)
  }

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  if(length(XVar) > 0L && class(dt[[XVar]])[1L] == "factor") {
    dt[, eval(XVar) := as.character(get(XVar))]
  }

  # Create base plot object
  numvars <- c()
  byvars <- c()
  if(check1) {
    if(Debug) print("BarPlot 2.b")
    if(!PreAgg) {
      if(tryCatch({class(dt[[eval(YVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        numvars <- unique(c(numvars, YVar))
      } else {
        byvars <- unique(c(byvars, YVar))
      }
      if(tryCatch({class(dt[[eval(XVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        if(length(numvars) > 0) {
          x <- length(unique(dt[[XVar]]))
          y <- length(unique(dt[[YVar]]))
          if(x > y) {
            byvars <- unique(c(byvars, YVar))
            numvars[1L] <- XVar
          } else {
            byvars <- unique(c(byvars, XVar))
          }
        } else {
          numvars <- unique(c(numvars, XVar))
        }
      } else {
        byvars <- unique(c(byvars, XVar))
      }
      if(!is.null(byvars)) {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        for(i in byvars) {
          if(class(temp[[i]])[1L] %in% c('numeric','integer')) {
            temp[, eval(i) := as.character(get(i))]
          }
        }
      } else {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
      }
    } else {
      temp <- data.table::copy(dt)
      numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
      byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
    }

    yvar <- temp[[YVar]]
    xvar <- temp[[XVar]]

    # Transformation
    if(YVarTrans != "Identity") {
      temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
    }

    p1 <- echarts4r::e_charts_(
      temp,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      width = Width, height = Height)

    p1 <- echarts4r::e_pie_(
      e = p1,
      YVar,
      stack = XVar,
      radius = c("50%", "70%")
    )

    if (ShowLabels) {
      p1 <- echarts4r::e_labels(
        e = p1,
        show = TRUE,
        formatter = htmlwidgets::JS(
          "function(params) {
             return params.name + ': ' + params.value.toFixed(1) + ' (' + params.percent.toFixed(1) + '%)';
           }"
        ),
        position = "outside"
      )
    }

    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
    p1 <- echarts4r::e_brush(e = p1)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    return(p1)
  }
}


#' @title Rosetype
#'
#' @description Build a donut plot by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo","essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired","jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal","sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)), GV = sample(LETTERS, 1000, TRUE))
#'
#' # Create plot
#' AutoPlots::Rosetype(
#'   dt = dt,
#'   PreAgg = TRUE,
#'   XVar = "GV",
#'   YVar = "Y",
#'   GroupVar = NULL,
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   AggMethod = 'mean',
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   TimeLine = FALSE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Rosetype <- function(dt = NULL,
                     PreAgg = TRUE,
                     XVar = NULL,
                     YVar = NULL,
                     GroupVar = NULL,
                     YVarTrans = "Identity",
                     XVarTrans = "Identity",
                     FacetRows = 1,
                     FacetCols = 1,
                     FacetLevels = NULL,
                     AggMethod = 'mean',
                     Height = NULL,
                     Width = NULL,
                     ShowLabels = TRUE,
                     Theme = "dark",
                     TimeLine = FALSE,
                     title.text = "Rosetype Plot",
                     title.subtext = NULL,
                     title.link = NULL,
                     title.sublink = NULL,
                     title.Align = NULL,
                     title.top = NULL,
                     title.left = NULL,
                     title.right = NULL,
                     title.bottom = NULL,
                     title.padding = NULL,
                     title.itemGap = NULL,
                     title.backgroundColor = NULL,
                     title.borderColor = NULL,
                     title.borderWidth = NULL,
                     title.borderRadius = NULL,
                     title.shadowColor = NULL,
                     title.shadowBlur = NULL,
                     title.shadowOffsetX = NULL,
                     title.shadowOffsetY = NULL,
                     title.textStyle.color = NULL,
                     title.textStyle.fontStyle = NULL,
                     title.textStyle.fontWeight = NULL,
                     title.textStyle.fontFamily = "Segoe UI",
                     title.textStyle.fontSize = NULL,
                     title.textStyle.lineHeight = NULL,
                     title.textStyle.width = NULL,
                     title.textStyle.height = NULL,
                     title.textStyle.textBorderColor = NULL,
                     title.textStyle.textBorderWidth = NULL,
                     title.textStyle.textBorderType = NULL,
                     title.textStyle.textBorderDashOffset = NULL,
                     title.textStyle.textShadowColor = NULL,
                     title.textStyle.textShadowBlur = NULL,
                     title.textStyle.textShadowOffsetX = NULL,
                     title.textStyle.textShadowOffsetY = NULL,
                     title.subtextStyle.color = NULL,
                     title.subtextStyle.align = NULL,
                     title.subtextStyle.fontStyle = NULL,
                     title.subtextStyle.fontWeight = NULL,
                     title.subtextStyle.fontFamily = "Segoe UI",
                     title.subtextStyle.fontSize = NULL,
                     title.subtextStyle.lineHeight = NULL,
                     title.subtextStyle.width = NULL,
                     title.subtextStyle.height = NULL,
                     title.subtextStyle.textBorderColor = NULL,
                     title.subtextStyle.textBorderWidth = NULL,
                     title.subtextStyle.textBorderType = NULL,
                     title.subtextStyle.textBorderDashOffset = NULL,
                     title.subtextStyle.textShadowColor = NULL,
                     title.subtextStyle.textShadowBlur = NULL,
                     title.subtextStyle.textShadowOffsetX = NULL,
                     title.subtextStyle.textShadowOffsetY = NULL,
                     legend.show = TRUE,
                     legend.type = "scroll",
                     legend.selector = NULL,
                     legend.icon = NULL,
                     legend.align = NULL,
                     legend.padding = NULL,
                     legend.itemGap = NULL,
                     legend.itemWidth = NULL,
                     legend.orient = "vertical",
                     legend.width = NULL,
                     legend.height = NULL,
                     legend.left = NULL,
                     legend.right = 10,
                     legend.top = 30,
                     legend.bottom = NULL,
                     legend.backgroundColor = NULL,
                     legend.borderColor = NULL,
                     legend.borderWidth = NULL,
                     legend.borderRadius = NULL,
                     legend.shadowBlur = NULL,
                     legend.shadowColor = NULL,
                     legend.shadowOffsetX = NULL,
                     legend.shadowOffsetY = NULL,
                     legend.itemStyle.color = NULL,
                     legend.itemStyle.borderColor = NULL,
                     legend.itemStyle.borderWidth = NULL,
                     legend.itemStyle.borderType = NULL,
                     legend.itemStyle.shadowBlur = NULL,
                     legend.itemStyle.shadowColor = NULL,
                     legend.itemStyle.shadowOffsetX = NULL,
                     legend.itemStyle.shadowOffsetY = NULL,
                     legend.itemStyle.opacity = NULL,
                     legend.lineStyle.color = NULL,
                     legend.lineStyle.width = NULL,
                     legend.lineStyle.type = NULL,
                     legend.lineStyle.shadowBlur = NULL,
                     legend.lineStyle.shadowColor = NULL,
                     legend.lineStyle.shadowOffsetX = NULL,
                     legend.lineStyle.shadowOffsetY = NULL,
                     legend.lineStyle.opacity = NULL,
                     legend.lineStyle.inactiveColor = NULL,
                     legend.lineStyle.inactiveWidth = NULL,
                     legend.textStyle.color = NULL,
                     legend.textStyle.fontStyle = NULL,
                     legend.textStyle.fontWeight = NULL,
                     legend.textStyle.fontFamily = "Segoe UI",
                     legend.textStyle.fontSize = NULL,
                     legend.textStyle.backgroundColor = NULL,
                     legend.textStyle.borderColor = NULL,
                     legend.textStyle.borderWidth = NULL,
                     legend.textStyle.borderType = NULL,
                     legend.textStyle.borderRadius = NULL,
                     legend.textStyle.padding = NULL,
                     legend.textStyle.shadowColor = NULL,
                     legend.textStyle.shadowBlur = NULL,
                     legend.textStyle.shadowOffsetX = NULL,
                     legend.textStyle.shadowOffsetY = NULL,
                     legend.textStyle.width = NULL,
                     legend.textStyle.height = NULL,
                     legend.textStyle.textBorderColor = NULL,
                     legend.textStyle.textBorderWidth = NULL,
                     legend.textStyle.textBorderType = NULL,
                     legend.textStyle.textShadowColor = NULL,
                     legend.textStyle.textShadowBlur = NULL,
                     legend.textStyle.textShadowOffsetX = NULL,
                     legend.textStyle.textShadowOffsetY = NULL,
                     legend.pageTextStyle.color = NULL,
                     legend.pageTextStyle.fontStyle = NULL,
                     legend.pageTextStyle.fontWeight = NULL,
                     legend.pageTextStyle.fontFamily = "Segoe UI",
                     legend.pageTextStyle.fontSize = NULL,
                     legend.pageTextStyle.lineHeight = NULL,
                     legend.pageTextStyle.width = NULL,
                     legend.pageTextStyle.height = NULL,
                     legend.pageTextStyle.textBorderColor = NULL,
                     legend.pageTextStyle.textBorderWidth = NULL,
                     legend.pageTextStyle.textBorderType = NULL,
                     legend.pageTextStyle.textShadowColor = NULL,
                     legend.pageTextStyle.textShadowBlur = NULL,
                     legend.pageTextStyle.textShadowOffsetX = NULL,
                     legend.pageTextStyle.textShadowOffsetY = NULL,
                     legend.emphasis.selectorLabel.show = NULL,
                     legend.emphasis.selectorLabel.distance = NULL,
                     legend.emphasis.selectorLabel.rotate = NULL,
                     legend.emphasis.selectorLabel.color = NULL,
                     legend.emphasis.selectorLabel.fontStyle = NULL,
                     legend.emphasis.selectorLabel.fontWeight = NULL,
                     legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                     legend.emphasis.selectorLabel.fontSize = NULL,
                     legend.emphasis.selectorLabel.align = NULL,
                     legend.emphasis.selectorLabel.verticalAlign = NULL,
                     legend.emphasis.selectorLabel.lineHeight = NULL,
                     legend.emphasis.selectorLabel.backgroundColor = NULL,
                     legend.emphasis.selectorLabel.borderColor = NULL,
                     legend.emphasis.selectorLabel.borderWidth = NULL,
                     legend.emphasis.selectorLabel.borderType = NULL,
                     legend.emphasis.selectorLabel.borderRadius = NULL,
                     legend.emphasis.selectorLabel.padding = NULL,
                     legend.emphasis.selectorLabel.shadowColor = NULL,
                     legend.emphasis.selectorLabel.shadowBlur = NULL,
                     legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                     legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                     legend.emphasis.selectorLabel.width = NULL,
                     legend.emphasis.selectorLabel.height = NULL,
                     legend.emphasis.selectorLabel.textBorderColor = NULL,
                     legend.emphasis.selectorLabel.textBorderWidth = NULL,
                     legend.emphasis.selectorLabel.textBorderType = NULL,
                     legend.emphasis.selectorLabel.textShadowColor = NULL,
                     legend.emphasis.selectorLabel.textShadowBlur = NULL,
                     legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                     legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                     tooltip.show = TRUE,
                     tooltip.trigger = "item",
                     tooltip.backgroundColor = NULL,
                     tooltip.borderColor = NULL,
                     tooltip.borderWidth = NULL,
                     tooltip.padding = NULL,
                     tooltip.axisPointer.type = "cross",
                     tooltip.axisPointer.lineStyle.color = NULL,
                     tooltip.axisPointer.shadowStyle.color = NULL,
                     tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                     tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                     tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                     tooltip.axisPointer.shadowStyle.opacity = NULL,
                     tooltip.textStyle.color = NULL,
                     tooltip.textStyle.fontStyle = NULL,
                     tooltip.textStyle.fontWeight = NULL,
                     tooltip.textStyle.fontFamily = "Segoe UI",
                     tooltip.textStyle.lineHeight = NULL,
                     tooltip.textStyle.width = NULL,
                     tooltip.textStyle.height = NULL,
                     tooltip.textStyle.textBorderColor = NULL,
                     tooltip.textStyle.textBorderWidth = NULL,
                     tooltip.textStyle.textBorderType = NULL,
                     tooltip.textStyle.textShadowColor = NULL,
                     tooltip.textStyle.textShadowBlur = NULL,
                     tooltip.textStyle.textShadowOffsetX = NULL,
                     tooltip.textStyle.textShadowOffsetY = NULL,
                     toolbox.show = TRUE,
                     toolbox.orient = "horizontal",
                     toolbox.itemSize = 15,
                     toolbox.itemGap = 8,
                     toolbox.top = NULL,
                     toolbox.left = NULL,
                     toolbox.right = NULL,
                     toolbox.bottom = NULL,
                     toolbox.width = NULL,
                     toolbox.heigth = NULL,
                     toolbox.feature.saveAsImage.show = TRUE,
                     toolbox.feature.restore.show = TRUE,
                     toolbox.feature.dataZoom.show = FALSE,
                     toolbox.feature.magicType.show = FALSE,
                     toolbox.feature.magicType.type = NULL,
                     toolbox.feature.dataView.show = TRUE,
                     toolbox.iconStyle.color = NULL,
                     toolbox.iconStyle.borderColor = NULL,
                     toolbox.emphasis.iconStyle.borderColor = NULL,
                     toolbox.iconStyle.shadowBlur = NULL,
                     toolbox.iconStyle.shadowColor = NULL,
                     toolbox.iconStyle.shadowOffsetX = NULL,
                     toolbox.iconStyle.shadowOffsetY = NULL,
                     Debug = FALSE) {

  if(length(YVar) > 0L) YVar <- YVar[1L]
  if(length(XVar) > 0L) XVar <- XVar[1L]

  # Used multiple times
  check1 <- length(XVar) != 0 && length(YVar) != 0

  if(!PreAgg) {
    if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
      dt <- data.table::as.data.table(dt)
    })
    aggFunc <- SummaryFunction(AggMethod)
  }

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  if(length(XVar) > 0L && class(dt[[XVar]])[1L] == "factor") {
    dt[, eval(XVar) := as.character(get(XVar))]
  }

  # Create base plot object
  numvars <- c()
  byvars <- c()
  if(check1) {
    if(Debug) print("BarPlot 2.b")
    if(!PreAgg) {
      if(tryCatch({class(dt[[eval(YVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        numvars <- unique(c(numvars, YVar))
      } else {
        byvars <- unique(c(byvars, YVar))
      }
      if(tryCatch({class(dt[[eval(XVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
        if(length(numvars) > 0) {
          x <- length(unique(dt[[XVar]]))
          y <- length(unique(dt[[YVar]]))
          if(x > y) {
            byvars <- unique(c(byvars, YVar))
            numvars[1L] <- XVar
          } else {
            byvars <- unique(c(byvars, XVar))
          }
        } else {
          numvars <- unique(c(numvars, XVar))
        }
      } else {
        byvars <- unique(c(byvars, XVar))
      }
      if(!is.null(byvars)) {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        for(i in byvars) {
          if(class(temp[[i]])[1L] %in% c('numeric','integer')) {
            temp[, eval(i) := as.character(get(i))]
          }
        }
      } else {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
      }
    } else {
      temp <- data.table::copy(dt)
      numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
      byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
    }

    yvar <- temp[[YVar]]
    xvar <- temp[[XVar]]

    # Transformation
    if(YVarTrans != "Identity") {
      temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
    }

    p1 <- echarts4r::e_charts_(
      temp,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      width = Width, height = Height)

    p1 <- echarts4r::e_pie_(
      e = p1,
      YVar,
      stack = XVar,
      roseType = "radius"
    )

    if (ShowLabels) {
      p1 <- echarts4r::e_labels(
        e = p1,
        show = TRUE,
        formatter = htmlwidgets::JS(
          "function(params) {
             return params.name + ': ' + params.value.toFixed(1) + ' (' + params.percent.toFixed(1) + '%)';
           }"
        ),
        position = "outside"
      )
    }

    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
    p1 <- echarts4r::e_brush(e = p1)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    return(p1)
  }
}


#' @title Box
#'
#' @description Build a box plot by simply passing arguments to a single function. It will sample your data using SampleSize number of rows. Sampled data is randomized.
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param SampleSize numeric
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Character variable
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine Logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ContainLabel TRUE
#' @param itemStyle.color Fill color. Can be a single color or vector of multiple colors for gradient.
#' @param itemStyle.opacity transparency
#' @param itemStyle.borderColor hex or color name
#' @param itemStyle.borderWidth numeric
#' @param itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param itemStyle.borderCap 'butt' 'round' 'square'
#' @param itemStyle.shadowBlur numeric
#' @param itemStyle.shadowColor hex or name
#' @param itemStyle.shadowOffsetX numeric
#' @param itemStyle.shadowOffsetY numeric
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- data.table::data.table(Y = qnorm(p = runif(10000)), GV = sample(LETTERS, 1000, TRUE))
#'
#' AutoPlots::Box(
#'   dt = dt,
#'   SampleSize = 100000L,
#'   XVar = "GV",
#'   YVar = "Y",
#'   GroupVar = NULL,
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   TimeLine = FALSE,
#'   ContainLabel = TRUE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Box <- function(dt = NULL,
                SampleSize = 100000L,
                XVar = NULL,
                YVar = NULL,
                GroupVar = NULL,
                YVarTrans = "Identity",
                XVarTrans = "Identity",
                FacetRows = 1,
                FacetCols = 1,
                FacetLevels = NULL,
                Height = NULL,
                Width = NULL,
                ShowLabels = FALSE,
                Theme = "dark",
                MouseScroll = FALSE,
                TimeLine = FALSE,
                ContainLabel = TRUE,
                itemStyle.color = NULL,
                itemStyle.opacity = NULL,
                itemStyle.borderColor = NULL,
                itemStyle.borderWidth = NULL,
                itemStyle.borderType = NULL,
                itemStyle.borderCap = NULL,
                itemStyle.shadowBlur = NULL,
                itemStyle.shadowColor = NULL,
                itemStyle.shadowOffsetX = NULL,
                itemStyle.shadowOffsetY = NULL,
                title.text = "BoxPlot",
                title.subtext = NULL,
                title.link = NULL,
                title.sublink = NULL,
                title.Align = NULL,
                title.top = NULL,
                title.left = NULL,
                title.right = NULL,
                title.bottom = NULL,
                title.padding = NULL,
                title.itemGap = NULL,
                title.backgroundColor = NULL,
                title.borderColor = NULL,
                title.borderWidth = NULL,
                title.borderRadius = NULL,
                title.shadowColor = NULL,
                title.shadowBlur = NULL,
                title.shadowOffsetX = NULL,
                title.shadowOffsetY = NULL,
                title.textStyle.color = NULL,
                title.textStyle.fontStyle = NULL,
                title.textStyle.fontWeight = NULL,
                title.textStyle.fontFamily = "Segoe UI",
                title.textStyle.fontSize = NULL,
                title.textStyle.lineHeight = NULL,
                title.textStyle.width = NULL,
                title.textStyle.height = NULL,
                title.textStyle.textBorderColor = NULL,
                title.textStyle.textBorderWidth = NULL,
                title.textStyle.textBorderType = NULL,
                title.textStyle.textBorderDashOffset = NULL,
                title.textStyle.textShadowColor = NULL,
                title.textStyle.textShadowBlur = NULL,
                title.textStyle.textShadowOffsetX = NULL,
                title.textStyle.textShadowOffsetY = NULL,
                title.subtextStyle.color = NULL,
                title.subtextStyle.align = NULL,
                title.subtextStyle.fontStyle = NULL,
                title.subtextStyle.fontWeight = NULL,
                title.subtextStyle.fontFamily = "Segoe UI",
                title.subtextStyle.fontSize = NULL,
                title.subtextStyle.lineHeight = NULL,
                title.subtextStyle.width = NULL,
                title.subtextStyle.height = NULL,
                title.subtextStyle.textBorderColor = NULL,
                title.subtextStyle.textBorderWidth = NULL,
                title.subtextStyle.textBorderType = NULL,
                title.subtextStyle.textBorderDashOffset = NULL,
                title.subtextStyle.textShadowColor = NULL,
                title.subtextStyle.textShadowBlur = NULL,
                title.subtextStyle.textShadowOffsetX = NULL,
                title.subtextStyle.textShadowOffsetY = NULL,
                xAxis.title = NULL,
                xAxis.min = NULL,
                xAxis.max = NULL,
                xAxis.position = "bottom",
                xAxis.nameLocation = 'center',
                xAxis.axisTick.customValues = NULL,
                xAxis.nameTextStyle.color = NULL,
                xAxis.nameTextStyle.padding = 25,
                xAxis.nameTextStyle.align = NULL,
                xAxis.nameTextStyle.fontStyle = NULL,
                xAxis.nameTextStyle.fontWeight = "bolder",
                xAxis.nameTextStyle.fontSize = NULL,
                xAxis.nameTextStyle.fontFamily = "Segoe UI",
                xAxis.splitNumber = NULL,
                xAxis.axisLabel.rotate = NULL,
                xAxis.axisLabel.margin = NULL,
                xAxis.axisLabel.color = NULL,
                xAxis.axisLabel.fontStyle = NULL,
                xAxis.axisLabel.fontWeight = NULL,
                xAxis.axisLabel.fontFamily = "Segoe UI",
                xAxis.axisLabel.fontSize = 14,
                xAxis.axisLabel.align = NULL,
                xAxis.axisLabel.verticalAlign = NULL,
                xAxis.axisLabel.backgroundColor = NULL,
                xAxis.axisLabel.borderColor = NULL,
                xAxis.axisLabel.borderWidth = NULL,
                xAxis.axisLabel.borderType = NULL,
                xAxis.axisLabel.borderRadius = NULL,
                xAxis.axisLabel.padding = NULL,
                xAxis.axisLabel.shadowColor = NULL,
                xAxis.axisLabel.shadowBlur = NULL,
                xAxis.axisLabel.shadowOffsetX = NULL,
                xAxis.axisLabel.shadowOffsetY = NULL,
                xAxis.axisLabel.textBorderColor = NULL,
                xAxis.axisLabel.textBorderWidth = NULL,
                xAxis.axisLabel.textBorderType = NULL,
                xAxis.axisLabel.textShadowColor = NULL,
                xAxis.axisLabel.textShadowBlur = NULL,
                xAxis.axisLabel.textShadowOffsetX = NULL,
                xAxis.axisLabel.textShadowOffsetY = NULL,
                xAxis.axisLabel.overflow = NULL,
                yAxis.title = NULL,
                yAxis.min = NULL,
                yAxis.max = NULL,
                yAxis.position = "center",
                yAxis.nameLocation = 'center',
                yAxis.axisTick.customValues = NULL,
                yAxis.nameTextStyle.color = NULL,
                yAxis.nameTextStyle.padding = 25,
                yAxis.nameTextStyle.align = NULL,
                yAxis.nameTextStyle.fontStyle = NULL,
                yAxis.nameTextStyle.fontWeight = "bolder",
                yAxis.nameTextStyle.fontSize = NULL,
                yAxis.nameTextStyle.fontFamily = "Segoe UI",
                yAxis.splitNumber = NULL,
                yAxis.axisLabel.rotate = NULL,
                yAxis.axisLabel.margin = NULL,
                yAxis.axisLabel.color = NULL,
                yAxis.axisLabel.fontStyle = NULL,
                yAxis.axisLabel.fontWeight = NULL,
                yAxis.axisLabel.fontFamily = "Segoe UI",
                yAxis.axisLabel.fontSize = 14,
                yAxis.axisLabel.align = NULL,
                yAxis.axisLabel.verticalAlign = NULL,
                yAxis.axisLabel.backgroundColor = NULL,
                yAxis.axisLabel.borderColor = NULL,
                yAxis.axisLabel.borderWidth = NULL,
                yAxis.axisLabel.borderType = NULL,
                yAxis.axisLabel.borderRadius = NULL,
                yAxis.axisLabel.padding = NULL,
                yAxis.axisLabel.shadowColor = NULL,
                yAxis.axisLabel.shadowBlur = NULL,
                yAxis.axisLabel.shadowOffsetX = NULL,
                yAxis.axisLabel.shadowOffsetY = NULL,
                yAxis.axisLabel.textBorderColor = NULL,
                yAxis.axisLabel.textBorderWidth = NULL,
                yAxis.axisLabel.textBorderType = NULL,
                yAxis.axisLabel.textShadowColor = NULL,
                yAxis.axisLabel.textShadowBlur = NULL,
                yAxis.axisLabel.textShadowOffsetX = NULL,
                yAxis.axisLabel.textShadowOffsetY = NULL,
                yAxis.axisLabel.overflow = NULL,
                legend.show = TRUE,
                legend.type = "scroll",
                legend.selector = NULL,
                legend.icon = NULL,
                legend.align = NULL,
                legend.padding = NULL,
                legend.itemGap = NULL,
                legend.itemWidth = NULL,
                legend.orient = "vertical",
                legend.width = NULL,
                legend.height = NULL,
                legend.left = NULL,
                legend.right = 10,
                legend.top = 30,
                legend.bottom = NULL,
                legend.backgroundColor = NULL,
                legend.borderColor = NULL,
                legend.borderWidth = NULL,
                legend.borderRadius = NULL,
                legend.shadowBlur = NULL,
                legend.shadowColor = NULL,
                legend.shadowOffsetX = NULL,
                legend.shadowOffsetY = NULL,
                legend.itemStyle.color = NULL,
                legend.itemStyle.borderColor = NULL,
                legend.itemStyle.borderWidth = NULL,
                legend.itemStyle.borderType = NULL,
                legend.itemStyle.shadowBlur = NULL,
                legend.itemStyle.shadowColor = NULL,
                legend.itemStyle.shadowOffsetX = NULL,
                legend.itemStyle.shadowOffsetY = NULL,
                legend.itemStyle.opacity = NULL,
                legend.lineStyle.color = NULL,
                legend.lineStyle.width = NULL,
                legend.lineStyle.type = NULL,
                legend.lineStyle.shadowBlur = NULL,
                legend.lineStyle.shadowColor = NULL,
                legend.lineStyle.shadowOffsetX = NULL,
                legend.lineStyle.shadowOffsetY = NULL,
                legend.lineStyle.opacity = NULL,
                legend.lineStyle.inactiveColor = NULL,
                legend.lineStyle.inactiveWidth = NULL,
                legend.textStyle.color = NULL,
                legend.textStyle.fontStyle = NULL,
                legend.textStyle.fontWeight = NULL,
                legend.textStyle.fontFamily = "Segoe UI",
                legend.textStyle.fontSize = NULL,
                legend.textStyle.backgroundColor = NULL,
                legend.textStyle.borderColor = NULL,
                legend.textStyle.borderWidth = NULL,
                legend.textStyle.borderType = NULL,
                legend.textStyle.borderRadius = NULL,
                legend.textStyle.padding = NULL,
                legend.textStyle.shadowColor = NULL,
                legend.textStyle.shadowBlur = NULL,
                legend.textStyle.shadowOffsetX = NULL,
                legend.textStyle.shadowOffsetY = NULL,
                legend.textStyle.width = NULL,
                legend.textStyle.height = NULL,
                legend.textStyle.textBorderColor = NULL,
                legend.textStyle.textBorderWidth = NULL,
                legend.textStyle.textBorderType = NULL,
                legend.textStyle.textShadowColor = NULL,
                legend.textStyle.textShadowBlur = NULL,
                legend.textStyle.textShadowOffsetX = NULL,
                legend.textStyle.textShadowOffsetY = NULL,
                legend.pageTextStyle.color = NULL,
                legend.pageTextStyle.fontStyle = NULL,
                legend.pageTextStyle.fontWeight = NULL,
                legend.pageTextStyle.fontFamily = "Segoe UI",
                legend.pageTextStyle.fontSize = NULL,
                legend.pageTextStyle.lineHeight = NULL,
                legend.pageTextStyle.width = NULL,
                legend.pageTextStyle.height = NULL,
                legend.pageTextStyle.textBorderColor = NULL,
                legend.pageTextStyle.textBorderWidth = NULL,
                legend.pageTextStyle.textBorderType = NULL,
                legend.pageTextStyle.textShadowColor = NULL,
                legend.pageTextStyle.textShadowBlur = NULL,
                legend.pageTextStyle.textShadowOffsetX = NULL,
                legend.pageTextStyle.textShadowOffsetY = NULL,
                legend.emphasis.selectorLabel.show = NULL,
                legend.emphasis.selectorLabel.distance = NULL,
                legend.emphasis.selectorLabel.rotate = NULL,
                legend.emphasis.selectorLabel.color = NULL,
                legend.emphasis.selectorLabel.fontStyle = NULL,
                legend.emphasis.selectorLabel.fontWeight = NULL,
                legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                legend.emphasis.selectorLabel.fontSize = NULL,
                legend.emphasis.selectorLabel.align = NULL,
                legend.emphasis.selectorLabel.verticalAlign = NULL,
                legend.emphasis.selectorLabel.lineHeight = NULL,
                legend.emphasis.selectorLabel.backgroundColor = NULL,
                legend.emphasis.selectorLabel.borderColor = NULL,
                legend.emphasis.selectorLabel.borderWidth = NULL,
                legend.emphasis.selectorLabel.borderType = NULL,
                legend.emphasis.selectorLabel.borderRadius = NULL,
                legend.emphasis.selectorLabel.padding = NULL,
                legend.emphasis.selectorLabel.shadowColor = NULL,
                legend.emphasis.selectorLabel.shadowBlur = NULL,
                legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                legend.emphasis.selectorLabel.width = NULL,
                legend.emphasis.selectorLabel.height = NULL,
                legend.emphasis.selectorLabel.textBorderColor = NULL,
                legend.emphasis.selectorLabel.textBorderWidth = NULL,
                legend.emphasis.selectorLabel.textBorderType = NULL,
                legend.emphasis.selectorLabel.textShadowColor = NULL,
                legend.emphasis.selectorLabel.textShadowBlur = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                tooltip.show = TRUE,
                tooltip.trigger = "axis",
                tooltip.backgroundColor = NULL,
                tooltip.borderColor = NULL,
                tooltip.borderWidth = NULL,
                tooltip.padding = NULL,
                tooltip.axisPointer.type = "cross",
                tooltip.axisPointer.lineStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                tooltip.axisPointer.shadowStyle.opacity = NULL,
                tooltip.textStyle.color = NULL,
                tooltip.textStyle.fontStyle = NULL,
                tooltip.textStyle.fontWeight = NULL,
                tooltip.textStyle.fontFamily = "Segoe UI",
                tooltip.textStyle.lineHeight = NULL,
                tooltip.textStyle.width = NULL,
                tooltip.textStyle.height = NULL,
                tooltip.textStyle.textBorderColor = NULL,
                tooltip.textStyle.textBorderWidth = NULL,
                tooltip.textStyle.textBorderType = NULL,
                tooltip.textStyle.textShadowColor = NULL,
                tooltip.textStyle.textShadowBlur = NULL,
                tooltip.textStyle.textShadowOffsetX = NULL,
                tooltip.textStyle.textShadowOffsetY = NULL,
                toolbox.show = TRUE,
                toolbox.orient = "horizontal",
                toolbox.itemSize = 15,
                toolbox.itemGap = 8,
                toolbox.top = NULL,
                toolbox.left = NULL,
                toolbox.right = NULL,
                toolbox.bottom = NULL,
                toolbox.width = NULL,
                toolbox.heigth = NULL,
                toolbox.feature.saveAsImage.show = TRUE,
                toolbox.feature.restore.show = TRUE,
                toolbox.feature.dataZoom.show = TRUE,
                toolbox.feature.magicType.show = TRUE,
                toolbox.feature.magicType.type = c("line", "bar", "stack"),
                toolbox.feature.dataView.show = TRUE,
                toolbox.iconStyle.color = NULL,
                toolbox.iconStyle.borderColor = NULL,
                toolbox.emphasis.iconStyle.borderColor = NULL,
                toolbox.iconStyle.shadowBlur = NULL,
                toolbox.iconStyle.shadowColor = NULL,
                toolbox.iconStyle.shadowOffsetX = NULL,
                toolbox.iconStyle.shadowOffsetY = NULL,
                Debug = FALSE) {

  if(Debug) print("Box 1")

  # Turn off Faceting until I can figure out how to supply it
  FacetRows <- 1L
  FacetCols <- 1L

  # Ensure data.table
  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  if(Debug) print("Box 2")

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  if(Debug) print("Box 3")

  if(length(XVar) > 0L && class(dt[[XVar]])[1L] == "factor") {
    dt[, eval(XVar) := as.character(get(XVar))]
  }

  if(Debug) print("Box 4")

  if(Debug) print("BoxPlot 1")

  # Cap number of records
  if(length(YVar) > 0L) {
    SampleSize <- SampleSize / length(YVar)
  }
  if(dt[,.N] > SampleSize) {
    dt1 <- dt[order(stats::runif(.N))][seq_len(SampleSize)]
  } else {
    dt1 <- data.table::copy(dt)
  }

  if(Debug) print("Box 5")

  if(length(YVar) > 0L && length(XVar) == 0L && length(GroupVar) > 0L) {
    XVar <- GroupVar; GroupVar <- NULL
    CoordFlip <- FALSE
  } else if(length(XVar) > 0L && length(YVar) == 0L && length(GroupVar) > 0L) {
    YVar <- XVar; XVar <- GroupVar; GroupVar <- NULL
    CoordFlip <- TRUE
  } else {
    CoordFlip <- FALSE
    if(length(XVar) > 0L && class(dt1[[XVar]])[1L] %in% c("numeric","integer")) {
      YVarTrans <- XVarTrans
      YVar <- XVar
      XVar <- NULL
    }
  }

  # Multiple YVars
  if(length(YVar) > 1L) {
    XVar <- NULL
    GroupVar <- NULL
    dt1[, temp__ := "a"]
    dt1 <- data.table::melt.data.table(data = dt1, id.vars = "temp__", measure.vars = YVar, variable.name = "Measures", value.name = "Values")
    dt1[, temp__ := NULL]
    XVar <- "Measures"
    YVar <- "Values"
  }

  if(Debug) print("Box 6")

  if(length(GroupVar) > 0L && FacetRows > 1L && FacetCols > 1L) {
    dt1 <- dt1[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(YVar,XVar,GroupVar)]
  }

  if(Debug) print("Box 7")

  # Transformation
  if(YVarTrans != "Identity") {
    for(ggss in YVar) {
      dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = ggss, Methods = YVarTrans)$Data
    }
  }

  if(Debug) print("Box 8")

  # Build Plot Based on Available Variables
  # Create logic checks to determine each case distinctly
  if(Debug) print("BoxPlot 2")
  X_and_Y_and_GroupVars <- length(XVar) > 0L && length(YVar) > 0L && length(GroupVar) > 0L
  X_and_Y <- length(XVar) > 0L && length(YVar) > 0L

  if(Debug) print("Box 9")

  # X,Y,GroupVar
  if(X_and_Y_and_GroupVars) {

    if(Debug) print("Box 10")

    if(Debug) print("Box Echarts")
    p1 <- echarts4r::e_charts_(
      data = dt1 |> dplyr::group_by(get(GroupVar[1L])),
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      width = Width,
      height = Height)

    if(Debug) print("Box 11")

    p1 <- e_boxplot_full(
      e = p1,
      serie = YVar,
      label = ShowLabels,
      itemStyle.color = itemStyle.color,
      itemStyle.opacity = itemStyle.opacity,
      itemStyle.borderColor = itemStyle.borderColor,
      itemStyle.borderWidth = itemStyle.borderWidth,
      itemStyle.borderType = itemStyle.borderType,
      itemStyle.borderCap = itemStyle.borderCap,
      itemStyle.shadowBlur = itemStyle.shadowBlur,
      itemStyle.shadowColor = itemStyle.shadowColor,
      itemStyle.shadowOffsetX = itemStyle.shadowOffsetX,
      itemStyle.shadowOffsetY = itemStyle.shadowOffsetY)

    if(Debug) print("Box 12")

    p1 <- echarts4r::e_visual_map_(e = p1, YVar, show = FALSE)
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }

    if(Debug) print("Box 13")

    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    if(Debug) print("Box 14")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    if(Debug) print("Box 16")

    if(CoordFlip) p1 <- echarts4r::e_flip_coords(e = p1)
    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if(FacetRows > 1L || FacetCols > 1L) p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    return(p1)
  }

  # X,Y
  if(X_and_Y) {

    if(Debug) print("Box 10.a")

    if(Debug) print("Box X_and_Y")
    if(Debug) print("Box Echarts")
    p1 <- echarts4r::e_charts_(
      dt1 |> dplyr::group_by(get(XVar)),
      x = YVar,
      darkMode = TRUE,
      dispose = TRUE,
      color = GroupVar,
      width = Width,
      height = Height)

    p1 <- e_boxplot_full(
      e = p1,
      serie = YVar,
      label = ShowLabels,
      itemStyle.color = itemStyle.color,
      itemStyle.opacity = itemStyle.opacity,
      itemStyle.borderColor = itemStyle.borderColor,
      itemStyle.borderWidth = itemStyle.borderWidth,
      itemStyle.borderType = itemStyle.borderType,
      itemStyle.borderCap = itemStyle.borderCap,
      itemStyle.shadowBlur = itemStyle.shadowBlur,
      itemStyle.shadowColor = itemStyle.shadowColor,
      itemStyle.shadowOffsetX = itemStyle.shadowOffsetX,
      itemStyle.shadowOffsetY = itemStyle.shadowOffsetY)

    p1 <- echarts4r::e_visual_map_(e = p1, YVar, show = FALSE)
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    if(CoordFlip) p1 <- echarts4r::e_flip_coords(e = p1)
    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if(FacetRows > 1L || FacetCols > 1L) p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")



    # Return
    return(p1)
  }

  # Y Only
  if(length(YVar) > 0L) {

    if(Debug) print("Box 10.b")

    if(Debug) print("Box Y Only")

    if(Debug) print("Box Echarts")
    p1 <- echarts4r::e_charts_(
      dt1,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)

    p1 <- e_boxplot_full(
      e = p1,
      serie = YVar,
      label = ShowLabels,
      itemStyle.color = itemStyle.color,
      itemStyle.opacity = itemStyle.opacity,
      itemStyle.borderColor = itemStyle.borderColor,
      itemStyle.borderWidth = itemStyle.borderWidth,
      itemStyle.borderType = itemStyle.borderType,
      itemStyle.borderCap = itemStyle.borderCap,
      itemStyle.shadowBlur = itemStyle.shadowBlur,
      itemStyle.shadowColor = itemStyle.shadowColor,
      itemStyle.shadowOffsetX = itemStyle.shadowOffsetX,
      itemStyle.shadowOffsetY = itemStyle.shadowOffsetY)

    p1 <- echarts4r::e_visual_map_(e = p1, YVar, show = FALSE)
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)


    return(p1)
  }

  # X Only
  if(length(XVar) > 0L) {

    if(Debug) print("Box 10.c")

    if(Debug) print("Box X Only")

    if(Debug) print("Box Echarts")
    p1 <- echarts4r::e_charts_(
      dt1,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)
    p1 <- e_boxplot_full(
      e = p1,
      serie = XVar,
      label = ShowLabels,
      itemStyle.color = itemStyle.color,
      itemStyle.opacity = itemStyle.opacity,
      itemStyle.borderColor = itemStyle.borderColor,
      itemStyle.borderWidth = itemStyle.borderWidth,
      itemStyle.borderType = itemStyle.borderType,
      itemStyle.borderCap = itemStyle.borderCap,
      itemStyle.shadowBlur = itemStyle.shadowBlur,
      itemStyle.shadowColor = itemStyle.shadowColor,
      itemStyle.shadowOffsetX = itemStyle.shadowOffsetX,
      itemStyle.shadowOffsetY = itemStyle.shadowOffsetY)
    p1 <- echarts4r::e_visual_map_(e = p1, XVar, show = FALSE)
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    # Return
    return(p1)
  }
  return(NULL)
}

#' @title WordCloud
#'
#' @description WordCloud plots
#'
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param YVar Y-Axis variable name
#' @param Height "400px"
#' @param Width "200px"
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' dt <- FakeDataGenerator(AddComment = TRUE)
#'
#' # Create plot
#' AutoPlots::WordCloud(
#'   dt = dt,
#'   YVar = "Comment",
#'   Height = NULL,
#'   Width = NULL,
#'   Theme = "dark",
#'   Debug = FALSE)
#'
#' @return plot
#' @export
WordCloud <- function(dt = NULL,
                      YVar = NULL,
                      Height = NULL,
                      Width = NULL,
                      Theme = "dark",
                      title.text = "Word Cloud",
                      title.subtext = NULL,
                      title.link = NULL,
                      title.sublink = NULL,
                      title.Align = NULL,
                      title.top = NULL,
                      title.left = NULL,
                      title.right = NULL,
                      title.bottom = NULL,
                      title.padding = NULL,
                      title.itemGap = NULL,
                      title.backgroundColor = NULL,
                      title.borderColor = NULL,
                      title.borderWidth = NULL,
                      title.borderRadius = NULL,
                      title.shadowColor = NULL,
                      title.shadowBlur = NULL,
                      title.shadowOffsetX = NULL,
                      title.shadowOffsetY = NULL,
                      title.textStyle.color = NULL,
                      title.textStyle.fontStyle = NULL,
                      title.textStyle.fontWeight = NULL,
                      title.textStyle.fontFamily = "Segoe UI",
                      title.textStyle.fontSize = NULL,
                      title.textStyle.lineHeight = NULL,
                      title.textStyle.width = NULL,
                      title.textStyle.height = NULL,
                      title.textStyle.textBorderColor = NULL,
                      title.textStyle.textBorderWidth = NULL,
                      title.textStyle.textBorderType = NULL,
                      title.textStyle.textBorderDashOffset = NULL,
                      title.textStyle.textShadowColor = NULL,
                      title.textStyle.textShadowBlur = NULL,
                      title.textStyle.textShadowOffsetX = NULL,
                      title.textStyle.textShadowOffsetY = NULL,
                      title.subtextStyle.color = NULL,
                      title.subtextStyle.align = NULL,
                      title.subtextStyle.fontStyle = NULL,
                      title.subtextStyle.fontWeight = NULL,
                      title.subtextStyle.fontFamily = "Segoe UI",
                      title.subtextStyle.fontSize = NULL,
                      title.subtextStyle.lineHeight = NULL,
                      title.subtextStyle.width = NULL,
                      title.subtextStyle.height = NULL,
                      title.subtextStyle.textBorderColor = NULL,
                      title.subtextStyle.textBorderWidth = NULL,
                      title.subtextStyle.textBorderType = NULL,
                      title.subtextStyle.textBorderDashOffset = NULL,
                      title.subtextStyle.textShadowColor = NULL,
                      title.subtextStyle.textShadowBlur = NULL,
                      title.subtextStyle.textShadowOffsetX = NULL,
                      title.subtextStyle.textShadowOffsetY = NULL,
                      Debug = FALSE) {

  if(Theme == 'auritus') {
    ColorVals <- c("#3e4359", "#c5a805", "#4d267e", "#22904f", "red")
  } else if(Theme == 'azul') {
    ColorVals <- c("#bfcca6", "#b07a9a", "#65deff", "#f73372", "#d08e1f")
  } else if(Theme == 'bee-inspired') {
    ColorVals <- c("#24243b", "#c2ba38", "#deeb25", "#ebc625", "#ffe700")
  } else if(Theme == 'blue') {
    ColorVals <- c("#2e69aa", "#99b8d9", "#3a84d4", "#1b67b9", "#046fe1")
  } else if(Theme == 'caravan') {
    ColorVals <- c("#18536d", "#d44545", "#eba565", "#e1c3a7", "#e1dda7")
  } else if(Theme == 'carp') {
    ColorVals <- c("#ff3300", "#fff0bb", "#679898", "#ff8870", "#4d3935")
  } else if(Theme == 'chalk') {
    ColorVals <- c("#e8c69e", "#54afec", "#d9dc89", "#f1a7d6", "#927294")
  } else if(Theme == 'cool') {
    ColorVals <- c("#20146a", "#591b89", "#911ea6", "#8081ba", "#2a74c4")
  } else if(Theme == 'dark-bold') {
    ColorVals <- c("#922e2e", "#d06363", "#d0a463", "#5c845e", "#63d0b9")
  } else if(Theme == 'dark') {
    ColorVals <- c("#e17d7d", "#c1ba54", "#66d5b0", "#b366d5", "#66a9d5")
  } else if(Theme == 'eduardo') {
    ColorVals <- c("#352a61", "#696284", "#c190ba", "#9e8a9b", "#615b60")
  } else if(Theme == 'essos') {
    ColorVals <- c("#753751", "#cfc995", "#c2b53c", "#d89c41")
  } else if(Theme == 'forest') {
    ColorVals <- c("#101010", "#bdb892", "#6c7955", "#3e6e86", "#37412e")
  } else if(Theme == 'fresh-cut') {
    ColorVals <- c("#74b936", "#76e314", "#cfbcb2", "#26609e", "#11b1cf")
  } else if(Theme == 'fruit') {
    ColorVals <- c("#dc965e", "#955828", "#c2b3a6", "#a16464", "#ae8c74")
  } else if(Theme == 'gray') {
    ColorVals <- c("#333333", "#696969", "#989898", "#bababa", "#e3e3e3")
  } else if(Theme == 'green') {
    ColorVals <- c("#2c5e25", "#387830", "#56a14d", "#7cbe74", "#b5e3af")
  } else if(Theme == 'halloween') {
    ColorVals <- c("#d1d134", "#d1953c", "#cc735d", "#7a5dcc", "#564f6a")
  } else if(Theme == 'helianthus') {
    ColorVals <- c("#6235e1", "#e16235", "#e1c135", "#c46aa5", "#5bcf3e")
  } else if(Theme == 'infographic') {
    ColorVals <- c("#d5cb2b", "#b4e771", "#cc4d3d", "#e78971", "#82b053")
  } else if(Theme == 'inspired') {
    ColorVals <- c("#8e1212", "#0f6310", "#d39f03", "#ff0000", "#265d82")
  } else if(Theme == 'jazz') {
    ColorVals <- c("#5e4832", "#000000", "#265057", "#d5dcdd")
  } else if(Theme == 'london') {
    ColorVals <- c("#881010", "#b8d1d4", "#227e89", "#041137", "#1c86c4")
  } else if(Theme == 'macarons') {
    ColorVals <- c("#6382cf", "#8776b9", "#318c9d", "#6d5739", "#7f7f98")
  } else if(Theme == 'macarons2') {
    ColorVals <- c("#6d6ddb", "#d45315", "#6e9fe4", "#b9bc89", "#d37c7c")
  } else if(Theme == 'mint') {
    ColorVals <- c("#c3ebd6", "#859d90", "#6dbaba", "#6dba9b", "#62d17f")
  } else if(Theme == 'purple-passion') {
    ColorVals <- c("#9385ba", "#779fbe", "#b86aac", "#5d9dc8", "#5f3a89")
  } else if(Theme == 'red-velvet') {
    ColorVals <- c("#6f4c41", "#db8469", "#f13d67", "#5e1d2c", "#ff00a9")
  } else if(Theme == 'red') {
    ColorVals <- c("#b4342a", "#8a4d49", "#c08480", "#df745a", "#cca69d")
  } else if(Theme == 'roma') {
    ColorVals <- c("#a580e9", "#d56426", "#cd1450", "#8bbec0", "#91836b")
  } else if(Theme == 'royal') {
    ColorVals <- c("#a06156", "#756054", "#5fac21", "#34708a", "#692525")
  } else if(Theme == 'sakura') {
    ColorVals <- c("#d75869", "#cb979e", "#b12a3d", "#adabc7", "#d0a79b")
  } else if(Theme == 'shine') {
    ColorVals <- c("#3d5995", "#296537", "#3390f7", "#b81717", "#50868c")
  } else if(Theme == 'tech-blue') {
    ColorVals <- c("#356499", "#4e487e", "#524f4b", "#b9addc", "#1c70d8")
  } else if(Theme == 'vintage') {
    ColorVals <- c("#a47e5f", "#638176", "#a46969", "#5d3a3a", "#4f8090")
  } else if(Theme == 'walden') {
    ColorVals <- c("#3b96c4", "#8babba", "#a5d9a2", "#535d84", "#7f79ad")
  } else if(Theme == 'wef') {
    ColorVals <- c("#5981d5", "#3268d9", "#9d938a", "#1f457c", "#524e48")
  } else if(Theme == 'weforum') {
    ColorVals <- c("#8a1b6f", "#4d2876", "#d5bf24", "#2792aa", "#a27322")
  } else if(Theme == 'westeros') {
    ColorVals <- c("#4b4d66", "#a681b0", "#8acccf", "#41a7cf")
  } else if(Theme == 'wonderland') {
    ColorVals <- c("#629291", "#3ec5c2", "#cf95ad", "#cd7097")
  }

  # Cap number of records
  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(YVar) > 0L && class(dt[[YVar]])[1L] == "factor") {
    dt[, eval(YVar) := as.character(get(YVar))]
  }

  # Copy Data
  dt1 <- data.table::copy(dt)

  # Define Plotting Variable
  if(length(YVar) == 0L) return(NULL)

  # Data YVar <- "Comment"
  # dt <- AutoNLP::FakeDataGenerator(N = 1000, AddComment = TRUE)
  dt1 <- quanteda::tokens(dt[[YVar]], remove_punct = TRUE)
  dt2 <- quanteda::dfm(dt1)
  dt3 <- data.table::setDT(quanteda.textstats::textstat_frequency(dt2))
  dt4 <- dt3[, .SD, .SDcols = c("feature", "frequency")]
  data.table::setnames(dt4, c("feature", "frequency"),c("term", "freq"))

  # Create base plot object
  if(Debug) print('Create Plot with only data')
  dt5 <- echarts4r::e_color_range_(
    data = dt4,
    input = "freq",
    output = "Color",
    colors = ColorVals)
  p1 <- echarts4r::e_charts(data = dt5)
  p1 <- echarts4r::e_cloud_(e = p1, "term", "freq", "Color", shape = "circle", sizeRange = c(20, 42))
  p1 <- e_title_full(
    e = p1,
    title.text = title.text,
    title.subtext = title.subtext,
    title.link = title.link,
    title.sublink = title.sublink,
    title.Align = title.Align,
    title.top = title.top,
    title.left = title.left,
    title.right = title.right,
    title.bottom = title.bottom,
    title.padding = title.padding,
    title.itemGap = title.itemGap,
    title.backgroundColor = title.backgroundColor,
    title.borderColor = title.borderColor,
    title.borderWidth = title.borderWidth,
    title.borderRadius = title.borderRadius,
    title.shadowColor = title.shadowColor,
    title.shadowBlur = title.shadowBlur,
    title.shadowOffsetX = title.shadowOffsetX,
    title.shadowOffsetY = title.shadowOffsetY,
    title.textStyle.color = title.textStyle.color,
    title.textStyle.fontStyle = title.textStyle.fontStyle,
    title.textStyle.fontWeight = title.textStyle.fontWeight,
    title.textStyle.fontFamily = title.textStyle.fontFamily,
    title.textStyle.fontSize = title.textStyle.fontSize,
    title.textStyle.lineHeight = title.textStyle.lineHeight,
    title.textStyle.width = title.textStyle.width,
    title.textStyle.height = title.textStyle.height,
    title.textStyle.textBorderColor = title.textStyle.textBorderColor,
    title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
    title.textStyle.textBorderType = title.textStyle.textBorderType,
    title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
    title.textStyle.textShadowColor = title.textStyle.textShadowColor,
    title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
    title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
    title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
    title.subtextStyle.color = title.subtextStyle.color,
    title.subtextStyle.align = title.subtextStyle.align,
    title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
    title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
    title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
    title.subtextStyle.fontSize = title.subtextStyle.fontSize,
    title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
    title.subtextStyle.width = title.subtextStyle.width,
    title.subtextStyle.height = title.subtextStyle.height,
    title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
    title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
    title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
    title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
    title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
    title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
    title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
    title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

  p1 <- echarts4r::e_theme(e = p1, name = Theme)
  return(p1)
}

# ----

# ----

# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ ----
# > Aggreagated Plot Functions                                                ----
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ ----

#' @title Line
#'
#' @description This function automatically builds calibration plots and calibration boxplots for model evaluation using regression, stats::quantile regression, and binary and multinomial classification
#'
#' @author Adrian Antico
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param AggMethod character
#' @param YVar Y-Axis variable name. You can supply multiple YVars
#' @param DualYVar Secondary Y-Axis variables. Leave NULL for no secondary axis. Only one variable is allowed and when this is set only one YVar is allowed. An error will be thrown if those conditions are not met
#' @param XVar X-Axis variable name
#' @param GroupVar One Grouping Variable
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param DualYVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height NULL
#' @param Width NULL
#' @param ShowLabels character
#' @param Theme Provide an "Echarts" theme
#' @param TimeLine Logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param Smooth = TRUE
#' @param ShowSymbol = FALSE
#' @param lineStyle.color hex or name
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param ContainLabel TRUE
#' @param DarkMode FALSE
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 1000)
#'
#' # Build Line plot
#' AutoPlots::Line(
#'   dt = data,
#'   PreAgg = TRUE,
#'   AggMethod = "mean",
#'   XVar = "DateTime",
#'   YVar = "Independent_Variable3",
#'   YVarTrans = "LogPlus1",
#'   DualYVar = "Independent_Variable6",
#'   DualYVarTrans = "LogPlus1",
#'   GroupVar = NULL,
#'   Theme = "macarons")
#'
#' @return plot
#' @export
Line <- function(dt = NULL,
                 AggMethod = "mean",
                 PreAgg = TRUE,
                 XVar = NULL,
                 YVar = NULL,
                 DualYVar = NULL,
                 GroupVar = NULL,
                 YVarTrans = "Identity",
                 DualYVarTrans = "Identity",
                 XVarTrans = "Identity",
                 FacetRows = 1,
                 FacetCols = 1,
                 FacetLevels = NULL,
                 Height = NULL,
                 Width = NULL,
                 ShowLabels = FALSE,
                 Theme = "dark",
                 MouseScroll = FALSE,
                 TimeLine = FALSE,
                 Smooth = TRUE,
                 ShowSymbol = FALSE,
                 ContainLabel = TRUE,
                 DarkMode = FALSE,
                 lineStyle.color = NULL,
                 title.text = "Line Plot",
                 title.subtext = NULL,
                 title.link = NULL,
                 title.sublink = NULL,
                 title.Align = NULL,
                 title.top = NULL,
                 title.left = NULL,
                 title.right = NULL,
                 title.bottom = NULL,
                 title.padding = NULL,
                 title.itemGap = NULL,
                 title.backgroundColor = NULL,
                 title.borderColor = NULL,
                 title.borderWidth = NULL,
                 title.borderRadius = NULL,
                 title.shadowColor = NULL,
                 title.shadowBlur = NULL,
                 title.shadowOffsetX = NULL,
                 title.shadowOffsetY = NULL,
                 title.textStyle.color = NULL,
                 title.textStyle.fontStyle = NULL,
                 title.textStyle.fontWeight = NULL,
                 title.textStyle.fontFamily = "Segoe UI",
                 title.textStyle.fontSize = NULL,
                 title.textStyle.lineHeight = NULL,
                 title.textStyle.width = NULL,
                 title.textStyle.height = NULL,
                 title.textStyle.textBorderColor = NULL,
                 title.textStyle.textBorderWidth = NULL,
                 title.textStyle.textBorderType = NULL,
                 title.textStyle.textBorderDashOffset = NULL,
                 title.textStyle.textShadowColor = NULL,
                 title.textStyle.textShadowBlur = NULL,
                 title.textStyle.textShadowOffsetX = NULL,
                 title.textStyle.textShadowOffsetY = NULL,
                 title.subtextStyle.color = NULL,
                 title.subtextStyle.align = NULL,
                 title.subtextStyle.fontStyle = NULL,
                 title.subtextStyle.fontWeight = NULL,
                 title.subtextStyle.fontFamily = "Segoe UI",
                 title.subtextStyle.fontSize = NULL,
                 title.subtextStyle.lineHeight = NULL,
                 title.subtextStyle.width = NULL,
                 title.subtextStyle.height = NULL,
                 title.subtextStyle.textBorderColor = NULL,
                 title.subtextStyle.textBorderWidth = NULL,
                 title.subtextStyle.textBorderType = NULL,
                 title.subtextStyle.textBorderDashOffset = NULL,
                 title.subtextStyle.textShadowColor = NULL,
                 title.subtextStyle.textShadowBlur = NULL,
                 title.subtextStyle.textShadowOffsetX = NULL,
                 title.subtextStyle.textShadowOffsetY = NULL,
                 xAxis.title = NULL,
                 xAxis.min = NULL,
                 xAxis.max = NULL,
                 xAxis.position = "bottom",
                 xAxis.nameLocation = 'center',
                 xAxis.axisTick.customValues = NULL,
                 xAxis.nameTextStyle.color = NULL,
                 xAxis.nameTextStyle.padding = 25,
                 xAxis.nameTextStyle.align = NULL,
                 xAxis.nameTextStyle.fontStyle = NULL,
                 xAxis.nameTextStyle.fontWeight = "bolder",
                 xAxis.nameTextStyle.fontSize = NULL,
                 xAxis.nameTextStyle.fontFamily = "Segoe UI",
                 xAxis.splitNumber = NULL,
                 xAxis.axisLabel.rotate = NULL,
                 xAxis.axisLabel.margin = NULL,
                 xAxis.axisLabel.color = NULL,
                 xAxis.axisLabel.fontStyle = NULL,
                 xAxis.axisLabel.fontWeight = NULL,
                 xAxis.axisLabel.fontFamily = "Segoe UI",
                 xAxis.axisLabel.fontSize = 14,
                 xAxis.axisLabel.align = NULL,
                 xAxis.axisLabel.verticalAlign = NULL,
                 xAxis.axisLabel.backgroundColor = NULL,
                 xAxis.axisLabel.borderColor = NULL,
                 xAxis.axisLabel.borderWidth = NULL,
                 xAxis.axisLabel.borderType = NULL,
                 xAxis.axisLabel.borderRadius = NULL,
                 xAxis.axisLabel.padding = NULL,
                 xAxis.axisLabel.shadowColor = NULL,
                 xAxis.axisLabel.shadowBlur = NULL,
                 xAxis.axisLabel.shadowOffsetX = NULL,
                 xAxis.axisLabel.shadowOffsetY = NULL,
                 xAxis.axisLabel.textBorderColor = NULL,
                 xAxis.axisLabel.textBorderWidth = NULL,
                 xAxis.axisLabel.textBorderType = NULL,
                 xAxis.axisLabel.textShadowColor = NULL,
                 xAxis.axisLabel.textShadowBlur = NULL,
                 xAxis.axisLabel.textShadowOffsetX = NULL,
                 xAxis.axisLabel.textShadowOffsetY = NULL,
                 xAxis.axisLabel.overflow = NULL,
                 yAxis.title = NULL,
                 yAxis.min = NULL,
                 yAxis.max = NULL,
                 yAxis.position = "center",
                 yAxis.nameLocation = 'center',
                 yAxis.axisTick.customValues = NULL,
                 yAxis.nameTextStyle.color = NULL,
                 yAxis.nameTextStyle.padding = 25,
                 yAxis.nameTextStyle.align = NULL,
                 yAxis.nameTextStyle.fontStyle = NULL,
                 yAxis.nameTextStyle.fontWeight = "bolder",
                 yAxis.nameTextStyle.fontSize = NULL,
                 yAxis.nameTextStyle.fontFamily = "Segoe UI",
                 yAxis.splitNumber = NULL,
                 yAxis.axisLabel.rotate = NULL,
                 yAxis.axisLabel.margin = NULL,
                 yAxis.axisLabel.color = NULL,
                 yAxis.axisLabel.fontStyle = NULL,
                 yAxis.axisLabel.fontWeight = NULL,
                 yAxis.axisLabel.fontFamily = "Segoe UI",
                 yAxis.axisLabel.fontSize = 14,
                 yAxis.axisLabel.align = NULL,
                 yAxis.axisLabel.verticalAlign = NULL,
                 yAxis.axisLabel.backgroundColor = NULL,
                 yAxis.axisLabel.borderColor = NULL,
                 yAxis.axisLabel.borderWidth = NULL,
                 yAxis.axisLabel.borderType = NULL,
                 yAxis.axisLabel.borderRadius = NULL,
                 yAxis.axisLabel.padding = NULL,
                 yAxis.axisLabel.shadowColor = NULL,
                 yAxis.axisLabel.shadowBlur = NULL,
                 yAxis.axisLabel.shadowOffsetX = NULL,
                 yAxis.axisLabel.shadowOffsetY = NULL,
                 yAxis.axisLabel.textBorderColor = NULL,
                 yAxis.axisLabel.textBorderWidth = NULL,
                 yAxis.axisLabel.textBorderType = NULL,
                 yAxis.axisLabel.textShadowColor = NULL,
                 yAxis.axisLabel.textShadowBlur = NULL,
                 yAxis.axisLabel.textShadowOffsetX = NULL,
                 yAxis.axisLabel.textShadowOffsetY = NULL,
                 yAxis.axisLabel.overflow = NULL,
                 legend.show = TRUE,
                 legend.type = "scroll",
                 legend.selector = NULL,
                 legend.icon = NULL,
                 legend.align = NULL,
                 legend.padding = NULL,
                 legend.itemGap = NULL,
                 legend.itemWidth = NULL,
                 legend.orient = "vertical",
                 legend.width = NULL,
                 legend.height = NULL,
                 legend.left = NULL,
                 legend.right = 10,
                 legend.top = 30,
                 legend.bottom = NULL,
                 legend.backgroundColor = NULL,
                 legend.borderColor = NULL,
                 legend.borderWidth = NULL,
                 legend.borderRadius = NULL,
                 legend.shadowBlur = NULL,
                 legend.shadowColor = NULL,
                 legend.shadowOffsetX = NULL,
                 legend.shadowOffsetY = NULL,
                 legend.itemStyle.color = NULL,
                 legend.itemStyle.borderColor = NULL,
                 legend.itemStyle.borderWidth = NULL,
                 legend.itemStyle.borderType = NULL,
                 legend.itemStyle.shadowBlur = NULL,
                 legend.itemStyle.shadowColor = NULL,
                 legend.itemStyle.shadowOffsetX = NULL,
                 legend.itemStyle.shadowOffsetY = NULL,
                 legend.itemStyle.opacity = NULL,
                 legend.lineStyle.color = NULL,
                 legend.lineStyle.width = NULL,
                 legend.lineStyle.type = NULL,
                 legend.lineStyle.shadowBlur = NULL,
                 legend.lineStyle.shadowColor = NULL,
                 legend.lineStyle.shadowOffsetX = NULL,
                 legend.lineStyle.shadowOffsetY = NULL,
                 legend.lineStyle.opacity = NULL,
                 legend.lineStyle.inactiveColor = NULL,
                 legend.lineStyle.inactiveWidth = NULL,
                 legend.textStyle.color = NULL,
                 legend.textStyle.fontStyle = NULL,
                 legend.textStyle.fontWeight = NULL,
                 legend.textStyle.fontFamily = "Segoe UI",
                 legend.textStyle.fontSize = NULL,
                 legend.textStyle.backgroundColor = NULL,
                 legend.textStyle.borderColor = NULL,
                 legend.textStyle.borderWidth = NULL,
                 legend.textStyle.borderType = NULL,
                 legend.textStyle.borderRadius = NULL,
                 legend.textStyle.padding = NULL,
                 legend.textStyle.shadowColor = NULL,
                 legend.textStyle.shadowBlur = NULL,
                 legend.textStyle.shadowOffsetX = NULL,
                 legend.textStyle.shadowOffsetY = NULL,
                 legend.textStyle.width = NULL,
                 legend.textStyle.height = NULL,
                 legend.textStyle.textBorderColor = NULL,
                 legend.textStyle.textBorderWidth = NULL,
                 legend.textStyle.textBorderType = NULL,
                 legend.textStyle.textShadowColor = NULL,
                 legend.textStyle.textShadowBlur = NULL,
                 legend.textStyle.textShadowOffsetX = NULL,
                 legend.textStyle.textShadowOffsetY = NULL,
                 legend.pageTextStyle.color = NULL,
                 legend.pageTextStyle.fontStyle = NULL,
                 legend.pageTextStyle.fontWeight = NULL,
                 legend.pageTextStyle.fontFamily = "Segoe UI",
                 legend.pageTextStyle.fontSize = NULL,
                 legend.pageTextStyle.lineHeight = NULL,
                 legend.pageTextStyle.width = NULL,
                 legend.pageTextStyle.height = NULL,
                 legend.pageTextStyle.textBorderColor = NULL,
                 legend.pageTextStyle.textBorderWidth = NULL,
                 legend.pageTextStyle.textBorderType = NULL,
                 legend.pageTextStyle.textShadowColor = NULL,
                 legend.pageTextStyle.textShadowBlur = NULL,
                 legend.pageTextStyle.textShadowOffsetX = NULL,
                 legend.pageTextStyle.textShadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.show = NULL,
                 legend.emphasis.selectorLabel.distance = NULL,
                 legend.emphasis.selectorLabel.rotate = NULL,
                 legend.emphasis.selectorLabel.color = NULL,
                 legend.emphasis.selectorLabel.fontStyle = NULL,
                 legend.emphasis.selectorLabel.fontWeight = NULL,
                 legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                 legend.emphasis.selectorLabel.fontSize = NULL,
                 legend.emphasis.selectorLabel.align = NULL,
                 legend.emphasis.selectorLabel.verticalAlign = NULL,
                 legend.emphasis.selectorLabel.lineHeight = NULL,
                 legend.emphasis.selectorLabel.backgroundColor = NULL,
                 legend.emphasis.selectorLabel.borderColor = NULL,
                 legend.emphasis.selectorLabel.borderWidth = NULL,
                 legend.emphasis.selectorLabel.borderType = NULL,
                 legend.emphasis.selectorLabel.borderRadius = NULL,
                 legend.emphasis.selectorLabel.padding = NULL,
                 legend.emphasis.selectorLabel.shadowColor = NULL,
                 legend.emphasis.selectorLabel.shadowBlur = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.width = NULL,
                 legend.emphasis.selectorLabel.height = NULL,
                 legend.emphasis.selectorLabel.textBorderColor = NULL,
                 legend.emphasis.selectorLabel.textBorderWidth = NULL,
                 legend.emphasis.selectorLabel.textBorderType = NULL,
                 legend.emphasis.selectorLabel.textShadowColor = NULL,
                 legend.emphasis.selectorLabel.textShadowBlur = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                 tooltip.show = TRUE,
                 tooltip.trigger = "axis",
                 tooltip.backgroundColor = NULL,
                 tooltip.borderColor = NULL,
                 tooltip.borderWidth = NULL,
                 tooltip.padding = NULL,
                 tooltip.axisPointer.type = "cross",
                 tooltip.axisPointer.lineStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                 tooltip.axisPointer.shadowStyle.opacity = NULL,
                 tooltip.textStyle.color = NULL,
                 tooltip.textStyle.fontStyle = NULL,
                 tooltip.textStyle.fontWeight = NULL,
                 tooltip.textStyle.fontFamily = "Segoe UI",
                 tooltip.textStyle.lineHeight = NULL,
                 tooltip.textStyle.width = NULL,
                 tooltip.textStyle.height = NULL,
                 tooltip.textStyle.textBorderColor = NULL,
                 tooltip.textStyle.textBorderWidth = NULL,
                 tooltip.textStyle.textBorderType = NULL,
                 tooltip.textStyle.textShadowColor = NULL,
                 tooltip.textStyle.textShadowBlur = NULL,
                 tooltip.textStyle.textShadowOffsetX = NULL,
                 tooltip.textStyle.textShadowOffsetY = NULL,
                 toolbox.show = TRUE,
                 toolbox.orient = "horizontal",
                 toolbox.itemSize = 15,
                 toolbox.itemGap = 8,
                 toolbox.top = NULL,
                 toolbox.left = NULL,
                 toolbox.right = NULL,
                 toolbox.bottom = NULL,
                 toolbox.width = NULL,
                 toolbox.heigth = NULL,
                 toolbox.feature.saveAsImage.show = TRUE,
                 toolbox.feature.restore.show = TRUE,
                 toolbox.feature.dataZoom.show = TRUE,
                 toolbox.feature.magicType.show = TRUE,
                 toolbox.feature.magicType.type = c("line", "bar", "stack"),
                 toolbox.feature.dataView.show = TRUE,
                 toolbox.iconStyle.color = NULL,
                 toolbox.iconStyle.borderColor = NULL,
                 toolbox.emphasis.iconStyle.borderColor = NULL,
                 toolbox.iconStyle.shadowBlur = NULL,
                 toolbox.iconStyle.shadowColor = NULL,
                 toolbox.iconStyle.shadowOffsetX = NULL,
                 toolbox.iconStyle.shadowOffsetY = NULL,
                 Debug = FALSE) {

  if(TimeLine && length(FacetLevels) == 0L) X_Scroll <- FALSE
  if(length(GroupVar) == 0L) TimeLine <- FALSE

  # Correct args
  if(length(GroupVar) > 0L && length(XVar) == 0L) {
    XVar <- GroupVar
    GroupVar <- NULL
  }

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  # If length(YVar) > 1 and a DualYVar is supplied, dual axis take precedence
  # Throw an error instead of trimming YVar to only the first value
  if(length(YVar) > 1L && length(DualYVar) > 0) stop("When DualYVar is utilized only one DualYVar is allowed and only one YVar is allowed")
  if(length(GroupVar) > 0L && length(DualYVar) > 0) stop("When DualYVar is utilized a GroupVar is not allowed")

  # If User Supplies more than 1 YVar, then structure data to be long instead of wide
  if(length(YVar) > 1L) {
    if(length(GroupVar) > 0L) {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = c(XVar,GroupVar), measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      dt1[, GroupVars := paste0(Measures, GroupVar)]
      dt1[, Measures := NULL]
      dt1[, eval(GroupVar) := NULL]
      GroupVar <- "GroupVars"
      YVar <- "Values"
    } else {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = XVar, measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      GroupVar <- "Measures"
      YVar <- "Values"
    }
  } else {
    dt1 <- data.table::copy(dt)
  }

  # Subset columns
  Ncols <- ncol(dt1)
  if(Ncols > 2L && length(GroupVar) == 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar)]
  } else if(length(GroupVar) > 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar, GroupVar[1L])]
    if(length(FacetLevels) > 0) {
      dt1 <- dt1[get(GroupVar[1L]) %in% eval(FacetLevels)]
    }
  }

  # Minimize data before moving on
  if(!PreAgg) {

    # Define Aggregation function
    if(Debug) print("Line # Define Aggregation function")
    aggFunc <- SummaryFunction(AggMethod)

    # Aggregate data
    if(length(GroupVar) > 0L) {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar,GroupVar[1L])]
      data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))
    } else {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar)]
      data.table::setorderv(x = dt1, cols = XVar, 1L)
    }
  }

  # Transformation
  if(YVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data
  }
  if(length(DualYVar > 0L) && DualYVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = DualYVar, Methods = DualYVarTrans)$Data
  }

  # Group Variable Case
  if(length(GroupVar) > 0L) {

    # Prepare Data
    if(Debug) print("Line() Build 1")
    gv <- GroupVar[1L]
    if(PreAgg) data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))

    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Build base plot depending on GroupVar availability
    p1 <- echarts4r::e_charts_(
      data = dt1 |> dplyr::group_by(get(gv)),
      x = XVar,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      timeline = TimeLine, dispose = TRUE, width = Width, height = Height)

    # Finalize Plot Build
    if(ShowLabels) {
      p1 <- echarts4r::e_line_(
        e = p1,
        serie = YVar,
        smooth = Smooth,
        showSymbol = ShowSymbol,
        label = list(
          show = TRUE
        )
      )
    } else {
      p1 <- echarts4r::e_line_(e = p1, serie = YVar, smooth = Smooth, showSymbol = ShowSymbol)
    }

    if(Debug) print("Line() Build Echarts 4 1")
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    if(Debug) print("Line() Build Echarts 5")
    p1 <- echarts4r::e_theme(e = p1, name = Theme)
    if(Debug) print("Line() Build Echarts 6")

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    if(Debug) print("Line() Build Echarts 6")
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if(Debug) print("Line() Build Echarts 8")
    if((FacetRows > 1L || FacetCols > 1) && length(FacetLevels) > 0L) {
      if(Debug) print("Line() Build Echarts 8 2")
      p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    }

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  } else {

    # Plot
    data.table::setorderv(x = dt1, cols = XVar, 1L)
    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Build base plot depending on GroupVar availability
    if(Debug) print("Line no group Echarts")
    p1 <- echarts4r::e_charts_(
      data = dt1,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)

    if(ShowLabels) {
      p1 <- echarts4r::e_line_(e = p1, serie = YVar, smooth = Smooth, showSymbol = ShowSymbol, label = list(show = TRUE))
    } else {
      p1 <- echarts4r::e_line_(e = p1, serie = YVar, smooth = Smooth, showSymbol = ShowSymbol)
    }

    # DualYVar
    if(length(DualYVar) > 0L) {
      if(ShowLabels) {
        p1 <- echarts4r::e_line_(e = p1, serie = DualYVar, smooth = Smooth, showSymbol = ShowSymbol, label = list(show = TRUE), x_index = 1, y_index = 1)
      } else {
        p1 <- echarts4r::e_line_(e = p1, serie = DualYVar, smooth = Smooth, showSymbol = ShowSymbol, x_index = 1, y_index = 1)
      }
    }

    if (length(lineStyle.color) > 0 && length(DualYVar) == 0) {
      visualMap.min <- min(dt1[[YVar]])
      visualMap.max <- max(dt1[[YVar]])
      p1 <- e_visual_map_full(
        e = p1,
        serie = YVar,
        visualMap.show = FALSE,
        visualMap.min = visualMap.min,
        visualMap.max = visualMap.max,
        visualMap.InRange.color = lineStyle.color)
    }

    # Finalize Plot Build
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  }
  return(p1)
}

#' @title Area
#'
#' @description This function automatically builds calibration plots and calibration boxplots for model evaluation using regression, stats::quantile regression, and binary and multinomial classification
#'
#' @author Adrian Antico
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param AggMethod character
#' @param YVar Y-Axis variable name. You can supply multiple YVars
#' @param DualYVar Secondary Y-Axis variables. Leave NULL for no secondary axis. Only one variable is allowed and when this is set only one YVar is allowed. An error will be thrown if those conditions are not met
#' @param XVar X-Axis variable name
#' @param GroupVar One Grouping Variable
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param DualYVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme Provide an "Echarts" theme
#' @param TimeLine Logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param Smooth = TRUE
#' @param ShowSymbol = FALSE
#' @param areaStyle.color Fill color
#' @param areaStyle.opacity transparency
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param ContainLabel TRUE
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 1000)
#'
#' # Build plot
#' AutoPlots::Area(
#'   dt = data,
#'   PreAgg = TRUE,
#'   AggMethod = "mean",
#'   XVar = "DateTime",
#'   YVar = "Independent_Variable3",
#'   YVarTrans = "Identity",
#'   DualYVar = "Independent_Variable6",
#'   DualYVarTrans = "Identity",
#'   GroupVar = NULL,
#'   Theme = "macarons")
#'
#' @return plot
#' @export
Area <- function(dt = NULL,
                 AggMethod = "mean",
                 PreAgg = TRUE,
                 XVar = NULL,
                 YVar = NULL,
                 DualYVar = NULL,
                 GroupVar = NULL,
                 YVarTrans = "Identity",
                 DualYVarTrans = "Identity",
                 XVarTrans = "Identity",
                 FacetRows = 1,
                 FacetCols = 1,
                 FacetLevels = NULL,
                 Height = NULL,
                 Width = NULL,
                 ShowLabels = FALSE,
                 Smooth = TRUE,
                 ShowSymbol = FALSE,
                 areaStyle.color = NULL,
                 areaStyle.opacity = NULL,
                 Theme = "dark",
                 MouseScroll = FALSE,
                 TimeLine = FALSE,
                 ContainLabel = TRUE,
                 title.text = "Area Plot",
                 title.subtext = NULL,
                 title.link = NULL,
                 title.sublink = NULL,
                 title.Align = NULL,
                 title.top = NULL,
                 title.left = NULL,
                 title.right = NULL,
                 title.bottom = NULL,
                 title.padding = NULL,
                 title.itemGap = NULL,
                 title.backgroundColor = NULL,
                 title.borderColor = NULL,
                 title.borderWidth = NULL,
                 title.borderRadius = NULL,
                 title.shadowColor = NULL,
                 title.shadowBlur = NULL,
                 title.shadowOffsetX = NULL,
                 title.shadowOffsetY = NULL,
                 title.textStyle.color = NULL,
                 title.textStyle.fontStyle = NULL,
                 title.textStyle.fontWeight = NULL,
                 title.textStyle.fontFamily = "Segoe UI",
                 title.textStyle.fontSize = NULL,
                 title.textStyle.lineHeight = NULL,
                 title.textStyle.width = NULL,
                 title.textStyle.height = NULL,
                 title.textStyle.textBorderColor = NULL,
                 title.textStyle.textBorderWidth = NULL,
                 title.textStyle.textBorderType = NULL,
                 title.textStyle.textBorderDashOffset = NULL,
                 title.textStyle.textShadowColor = NULL,
                 title.textStyle.textShadowBlur = NULL,
                 title.textStyle.textShadowOffsetX = NULL,
                 title.textStyle.textShadowOffsetY = NULL,
                 title.subtextStyle.color = NULL,
                 title.subtextStyle.align = NULL,
                 title.subtextStyle.fontStyle = NULL,
                 title.subtextStyle.fontWeight = NULL,
                 title.subtextStyle.fontFamily = "Segoe UI",
                 title.subtextStyle.fontSize = NULL,
                 title.subtextStyle.lineHeight = NULL,
                 title.subtextStyle.width = NULL,
                 title.subtextStyle.height = NULL,
                 title.subtextStyle.textBorderColor = NULL,
                 title.subtextStyle.textBorderWidth = NULL,
                 title.subtextStyle.textBorderType = NULL,
                 title.subtextStyle.textBorderDashOffset = NULL,
                 title.subtextStyle.textShadowColor = NULL,
                 title.subtextStyle.textShadowBlur = NULL,
                 title.subtextStyle.textShadowOffsetX = NULL,
                 title.subtextStyle.textShadowOffsetY = NULL,
                 xAxis.title = NULL,
                 xAxis.min = NULL,
                 xAxis.max = NULL,
                 xAxis.position = "bottom",
                 xAxis.nameLocation = 'center',
                 xAxis.axisTick.customValues = NULL,
                 xAxis.nameTextStyle.color = NULL,
                 xAxis.nameTextStyle.padding = 25,
                 xAxis.nameTextStyle.align = NULL,
                 xAxis.nameTextStyle.fontStyle = NULL,
                 xAxis.nameTextStyle.fontWeight = "bolder",
                 xAxis.nameTextStyle.fontSize = NULL,
                 xAxis.nameTextStyle.fontFamily = "Segoe UI",
                 xAxis.splitNumber = NULL,
                 xAxis.axisLabel.rotate = NULL,
                 xAxis.axisLabel.margin = NULL,
                 xAxis.axisLabel.color = NULL,
                 xAxis.axisLabel.fontStyle = NULL,
                 xAxis.axisLabel.fontWeight = NULL,
                 xAxis.axisLabel.fontFamily = "Segoe UI",
                 xAxis.axisLabel.fontSize = 14,
                 xAxis.axisLabel.align = NULL,
                 xAxis.axisLabel.verticalAlign = NULL,
                 xAxis.axisLabel.backgroundColor = NULL,
                 xAxis.axisLabel.borderColor = NULL,
                 xAxis.axisLabel.borderWidth = NULL,
                 xAxis.axisLabel.borderType = NULL,
                 xAxis.axisLabel.borderRadius = NULL,
                 xAxis.axisLabel.padding = NULL,
                 xAxis.axisLabel.shadowColor = NULL,
                 xAxis.axisLabel.shadowBlur = NULL,
                 xAxis.axisLabel.shadowOffsetX = NULL,
                 xAxis.axisLabel.shadowOffsetY = NULL,
                 xAxis.axisLabel.textBorderColor = NULL,
                 xAxis.axisLabel.textBorderWidth = NULL,
                 xAxis.axisLabel.textBorderType = NULL,
                 xAxis.axisLabel.textShadowColor = NULL,
                 xAxis.axisLabel.textShadowBlur = NULL,
                 xAxis.axisLabel.textShadowOffsetX = NULL,
                 xAxis.axisLabel.textShadowOffsetY = NULL,
                 xAxis.axisLabel.overflow = NULL,
                 yAxis.title = NULL,
                 yAxis.min = NULL,
                 yAxis.max = NULL,
                 yAxis.position = "center",
                 yAxis.nameLocation = 'center',
                 yAxis.axisTick.customValues = NULL,
                 yAxis.nameTextStyle.color = NULL,
                 yAxis.nameTextStyle.padding = 25,
                 yAxis.nameTextStyle.align = NULL,
                 yAxis.nameTextStyle.fontStyle = NULL,
                 yAxis.nameTextStyle.fontWeight = "bolder",
                 yAxis.nameTextStyle.fontSize = NULL,
                 yAxis.nameTextStyle.fontFamily = "Segoe UI",
                 yAxis.splitNumber = NULL,
                 yAxis.axisLabel.rotate = NULL,
                 yAxis.axisLabel.margin = NULL,
                 yAxis.axisLabel.color = NULL,
                 yAxis.axisLabel.fontStyle = NULL,
                 yAxis.axisLabel.fontWeight = NULL,
                 yAxis.axisLabel.fontFamily = "Segoe UI",
                 yAxis.axisLabel.fontSize = 14,
                 yAxis.axisLabel.align = NULL,
                 yAxis.axisLabel.verticalAlign = NULL,
                 yAxis.axisLabel.backgroundColor = NULL,
                 yAxis.axisLabel.borderColor = NULL,
                 yAxis.axisLabel.borderWidth = NULL,
                 yAxis.axisLabel.borderType = NULL,
                 yAxis.axisLabel.borderRadius = NULL,
                 yAxis.axisLabel.padding = NULL,
                 yAxis.axisLabel.shadowColor = NULL,
                 yAxis.axisLabel.shadowBlur = NULL,
                 yAxis.axisLabel.shadowOffsetX = NULL,
                 yAxis.axisLabel.shadowOffsetY = NULL,
                 yAxis.axisLabel.textBorderColor = NULL,
                 yAxis.axisLabel.textBorderWidth = NULL,
                 yAxis.axisLabel.textBorderType = NULL,
                 yAxis.axisLabel.textShadowColor = NULL,
                 yAxis.axisLabel.textShadowBlur = NULL,
                 yAxis.axisLabel.textShadowOffsetX = NULL,
                 yAxis.axisLabel.textShadowOffsetY = NULL,
                 yAxis.axisLabel.overflow = NULL,
                 legend.show = TRUE,
                 legend.type = "scroll",
                 legend.selector = NULL,
                 legend.icon = NULL,
                 legend.align = NULL,
                 legend.padding = NULL,
                 legend.itemGap = NULL,
                 legend.itemWidth = NULL,
                 legend.orient = "vertical",
                 legend.width = NULL,
                 legend.height = NULL,
                 legend.left = NULL,
                 legend.right = 10,
                 legend.top = 30,
                 legend.bottom = NULL,
                 legend.backgroundColor = NULL,
                 legend.borderColor = NULL,
                 legend.borderWidth = NULL,
                 legend.borderRadius = NULL,
                 legend.shadowBlur = NULL,
                 legend.shadowColor = NULL,
                 legend.shadowOffsetX = NULL,
                 legend.shadowOffsetY = NULL,
                 legend.itemStyle.color = NULL,
                 legend.itemStyle.borderColor = NULL,
                 legend.itemStyle.borderWidth = NULL,
                 legend.itemStyle.borderType = NULL,
                 legend.itemStyle.shadowBlur = NULL,
                 legend.itemStyle.shadowColor = NULL,
                 legend.itemStyle.shadowOffsetX = NULL,
                 legend.itemStyle.shadowOffsetY = NULL,
                 legend.itemStyle.opacity = NULL,
                 legend.lineStyle.color = NULL,
                 legend.lineStyle.width = NULL,
                 legend.lineStyle.type = NULL,
                 legend.lineStyle.shadowBlur = NULL,
                 legend.lineStyle.shadowColor = NULL,
                 legend.lineStyle.shadowOffsetX = NULL,
                 legend.lineStyle.shadowOffsetY = NULL,
                 legend.lineStyle.opacity = NULL,
                 legend.lineStyle.inactiveColor = NULL,
                 legend.lineStyle.inactiveWidth = NULL,
                 legend.textStyle.color = NULL,
                 legend.textStyle.fontStyle = NULL,
                 legend.textStyle.fontWeight = NULL,
                 legend.textStyle.fontFamily = "Segoe UI",
                 legend.textStyle.fontSize = NULL,
                 legend.textStyle.backgroundColor = NULL,
                 legend.textStyle.borderColor = NULL,
                 legend.textStyle.borderWidth = NULL,
                 legend.textStyle.borderType = NULL,
                 legend.textStyle.borderRadius = NULL,
                 legend.textStyle.padding = NULL,
                 legend.textStyle.shadowColor = NULL,
                 legend.textStyle.shadowBlur = NULL,
                 legend.textStyle.shadowOffsetX = NULL,
                 legend.textStyle.shadowOffsetY = NULL,
                 legend.textStyle.width = NULL,
                 legend.textStyle.height = NULL,
                 legend.textStyle.textBorderColor = NULL,
                 legend.textStyle.textBorderWidth = NULL,
                 legend.textStyle.textBorderType = NULL,
                 legend.textStyle.textShadowColor = NULL,
                 legend.textStyle.textShadowBlur = NULL,
                 legend.textStyle.textShadowOffsetX = NULL,
                 legend.textStyle.textShadowOffsetY = NULL,
                 legend.pageTextStyle.color = NULL,
                 legend.pageTextStyle.fontStyle = NULL,
                 legend.pageTextStyle.fontWeight = NULL,
                 legend.pageTextStyle.fontFamily = "Segoe UI",
                 legend.pageTextStyle.fontSize = NULL,
                 legend.pageTextStyle.lineHeight = NULL,
                 legend.pageTextStyle.width = NULL,
                 legend.pageTextStyle.height = NULL,
                 legend.pageTextStyle.textBorderColor = NULL,
                 legend.pageTextStyle.textBorderWidth = NULL,
                 legend.pageTextStyle.textBorderType = NULL,
                 legend.pageTextStyle.textShadowColor = NULL,
                 legend.pageTextStyle.textShadowBlur = NULL,
                 legend.pageTextStyle.textShadowOffsetX = NULL,
                 legend.pageTextStyle.textShadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.show = NULL,
                 legend.emphasis.selectorLabel.distance = NULL,
                 legend.emphasis.selectorLabel.rotate = NULL,
                 legend.emphasis.selectorLabel.color = NULL,
                 legend.emphasis.selectorLabel.fontStyle = NULL,
                 legend.emphasis.selectorLabel.fontWeight = NULL,
                 legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                 legend.emphasis.selectorLabel.fontSize = NULL,
                 legend.emphasis.selectorLabel.align = NULL,
                 legend.emphasis.selectorLabel.verticalAlign = NULL,
                 legend.emphasis.selectorLabel.lineHeight = NULL,
                 legend.emphasis.selectorLabel.backgroundColor = NULL,
                 legend.emphasis.selectorLabel.borderColor = NULL,
                 legend.emphasis.selectorLabel.borderWidth = NULL,
                 legend.emphasis.selectorLabel.borderType = NULL,
                 legend.emphasis.selectorLabel.borderRadius = NULL,
                 legend.emphasis.selectorLabel.padding = NULL,
                 legend.emphasis.selectorLabel.shadowColor = NULL,
                 legend.emphasis.selectorLabel.shadowBlur = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.width = NULL,
                 legend.emphasis.selectorLabel.height = NULL,
                 legend.emphasis.selectorLabel.textBorderColor = NULL,
                 legend.emphasis.selectorLabel.textBorderWidth = NULL,
                 legend.emphasis.selectorLabel.textBorderType = NULL,
                 legend.emphasis.selectorLabel.textShadowColor = NULL,
                 legend.emphasis.selectorLabel.textShadowBlur = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                 tooltip.show = TRUE,
                 tooltip.trigger = "axis",
                 tooltip.backgroundColor = NULL,
                 tooltip.borderColor = NULL,
                 tooltip.borderWidth = NULL,
                 tooltip.padding = NULL,
                 tooltip.axisPointer.type = "cross",
                 tooltip.axisPointer.lineStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                 tooltip.axisPointer.shadowStyle.opacity = NULL,
                 tooltip.textStyle.color = NULL,
                 tooltip.textStyle.fontStyle = NULL,
                 tooltip.textStyle.fontWeight = NULL,
                 tooltip.textStyle.fontFamily = "Segoe UI",
                 tooltip.textStyle.lineHeight = NULL,
                 tooltip.textStyle.width = NULL,
                 tooltip.textStyle.height = NULL,
                 tooltip.textStyle.textBorderColor = NULL,
                 tooltip.textStyle.textBorderWidth = NULL,
                 tooltip.textStyle.textBorderType = NULL,
                 tooltip.textStyle.textShadowColor = NULL,
                 tooltip.textStyle.textShadowBlur = NULL,
                 tooltip.textStyle.textShadowOffsetX = NULL,
                 tooltip.textStyle.textShadowOffsetY = NULL,
                 toolbox.show = TRUE,
                 toolbox.orient = "horizontal",
                 toolbox.itemSize = 15,
                 toolbox.itemGap = 8,
                 toolbox.top = NULL,
                 toolbox.left = NULL,
                 toolbox.right = NULL,
                 toolbox.bottom = NULL,
                 toolbox.width = NULL,
                 toolbox.heigth = NULL,
                 toolbox.feature.saveAsImage.show = TRUE,
                 toolbox.feature.restore.show = TRUE,
                 toolbox.feature.dataZoom.show = TRUE,
                 toolbox.feature.magicType.show = TRUE,
                 toolbox.feature.magicType.type = c("line", "bar", "stack"),
                 toolbox.feature.dataView.show = TRUE,
                 toolbox.iconStyle.color = NULL,
                 toolbox.iconStyle.borderColor = NULL,
                 toolbox.emphasis.iconStyle.borderColor = NULL,
                 toolbox.iconStyle.shadowBlur = NULL,
                 toolbox.iconStyle.shadowColor = NULL,
                 toolbox.iconStyle.shadowOffsetX = NULL,
                 toolbox.iconStyle.shadowOffsetY = NULL,
                 Debug = FALSE) {

  if(length(GroupVar) == 0L) TimeLine <- FALSE
  if(TimeLine && length(FacetLevels) > 0) X_Scroll <- FALSE

  # Correct args
  if(length(GroupVar) > 0L && length(XVar) == 0L) {
    XVar <- GroupVar
    GroupVar <- NULL
  }

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  # If length(YVar) > 1 and a DualYVar is supplied, dual axis take precedence
  # Throw an error instead of trimming YVar to only the first value
  if(length(YVar) > 1L && length(DualYVar) > 0) stop("When DualYVar is utilized only one DualYVar is allowed and only one YVar is allowed")
  if(length(GroupVar) > 0L && length(DualYVar) > 0) stop("When DualYVar is utilized a GroupVar is not allowed")

  # If User Supplies more than 1 YVar, then structure data to be long instead of wide
  if(length(YVar) > 1L) {
    if(length(GroupVar) > 0L) {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = c(XVar,GroupVar), measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      dt1[, GroupVars := paste0(Measures, GroupVar)]
      dt1[, Measures := NULL]
      dt1[, eval(GroupVar) := NULL]
      GroupVar <- "GroupVars"
      YVar <- "Values"
    } else {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = XVar, measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      GroupVar <- "Measures"
      YVar <- "Values"
    }
  } else {
    dt1 <- data.table::copy(dt)
  }

  # Subset columns
  Ncols <- ncol(dt1)
  if(Ncols > 2L && length(GroupVar) == 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar)]
  } else if(length(GroupVar) > 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar, GroupVar[1L])]
    if(length(FacetLevels) > 0) {
      dt1 <- dt1[get(GroupVar[1L]) %in% eval(FacetLevels)]
    }
  }

  # Minimize data before moving on
  if(!PreAgg) {

    # Define Aggregation function
    if(Debug) print("Plot.Calibration.Line # Define Aggregation function")
    aggFunc <- SummaryFunction(AggMethod)

    # Aggregate data
    if(length(GroupVar) > 0L) {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar,GroupVar[1L])]
      data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))
    } else {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar)]
      data.table::setorderv(x = dt1, cols = XVar, 1L)
    }
  }

  # Transformation
  if(YVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data
  }
  if(length(DualYVar > 0L) && DualYVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = DualYVar, Methods = DualYVarTrans)$Data
  }

  # Group Variable Case
  if(length(GroupVar) > 0L) {

    # Prepare Data
    if(Debug) print("Line() Build 1")
    gv <- GroupVar[1L]
    if(PreAgg) data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))

    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Plot
    if(Debug) print("Line() Build Echarts 1")

    # Build base plot depending on GroupVar availability
    if(Debug) print(paste0("Line TimeLine = ", TimeLine))
    p1 <- echarts4r::e_charts_(
      data = dt1 |> dplyr::group_by(get(gv)),
      x = XVar,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      timeline = TimeLine,
      dispose = TRUE,
      width = Width,
      height = Height)

    # Finalize Plot Build
    if(Debug) print("Line() Build Echarts 4")
    p1 <- e_area_full(
      e = p1,
      serie = YVar,
      smooth = Smooth,
      showSymbol = ShowSymbol,
      areaStyle.opacity = areaStyle.opacity)

    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if((FacetRows > 1L || FacetCols > 1) && length(FacetLevels) > 0L) {
      p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    }

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  } else {

    # Plot
    data.table::setorderv(x = dt1, cols = XVar, 1L)
    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Build base plot depending on GroupVar availability
    if(Debug) print("Line no group Echarts")
    p1 <- echarts4r::e_charts_(
      data = dt1,
      x = XVar,
      darkMode = TRUE,
      dispose = TRUE,
      width = Width,
      height = Height)

    p1 <- e_area_full(
      e = p1,
      serie = YVar,
      smooth = Smooth,
      showSymbol = ShowSymbol,
      areaStyle.color = areaStyle.color,
      areaStyle.opacity = areaStyle.opacity)

    if(length(DualYVar) > 0L) {
      p1 <- e_area_full(
        e = p1,
        serie = DualYVar,
        smooth = Smooth,
        showSymbol = ShowSymbol,
        areaStyle.opacity = areaStyle.opacity)
    }

    # Finalize Plot Build
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  }
  return(p1)
}

#' @title Step
#'
#' @description This function automatically builds calibration plots and calibration boxplots for model evaluation using regression, stats::quantile regression, and binary and multinomial classification
#'
#' @author Adrian Antico
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param AggMethod character
#' @param YVar Y-Axis variable name. You can supply multiple YVars
#' @param DualYVar Secondary Y-Axis variables. Leave NULL for no secondary axis. Only one variable is allowed and when this is set only one YVar is allowed. An error will be thrown if those conditions are not met
#' @param XVar X-Axis variable name
#' @param GroupVar One Grouping Variable
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param DualYVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme Provide an "Echarts" theme
#' @param TimeLine Logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ShowSymbol = FALSE
#' @param ContainLabel TRUE
#' @param lineStyle.color hex or name
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 1000)
#'
#' # Build plot
#' AutoPlots::Step(
#'   dt = data,
#'   PreAgg = TRUE,
#'   AggMethod = "mean",
#'   XVar = "DateTime",
#'   YVar = "Independent_Variable3",
#'   YVarTrans = "Identity",
#'   DualYVar = "Independent_Variable6",
#'   DualYVarTrans = "Identity",
#'   GroupVar = NULL,
#'   Theme = "macarons")
#'
#' @return plot
#' @export
Step <- function(dt = NULL,
                 AggMethod = "mean",
                 PreAgg = TRUE,
                 XVar = NULL,
                 YVar = NULL,
                 DualYVar = NULL,
                 GroupVar = NULL,
                 YVarTrans = "Identity",
                 DualYVarTrans = "Identity",
                 XVarTrans = "Identity",
                 FacetRows = 1,
                 FacetCols = 1,
                 FacetLevels = NULL,
                 Height = NULL,
                 Width = NULL,
                 ShowLabels = FALSE,
                 Theme = "dark",
                 MouseScroll = FALSE,
                 TimeLine = FALSE,
                 ShowSymbol = FALSE,
                 ContainLabel = TRUE,
                 lineStyle.color = NULL,
                 title.text = "Step Plot",
                 title.subtext = NULL,
                 title.link = NULL,
                 title.sublink = NULL,
                 title.Align = NULL,
                 title.top = NULL,
                 title.left = NULL,
                 title.right = NULL,
                 title.bottom = NULL,
                 title.padding = NULL,
                 title.itemGap = NULL,
                 title.backgroundColor = NULL,
                 title.borderColor = NULL,
                 title.borderWidth = NULL,
                 title.borderRadius = NULL,
                 title.shadowColor = NULL,
                 title.shadowBlur = NULL,
                 title.shadowOffsetX = NULL,
                 title.shadowOffsetY = NULL,
                 title.textStyle.color = NULL,
                 title.textStyle.fontStyle = NULL,
                 title.textStyle.fontWeight = NULL,
                 title.textStyle.fontFamily = "Segoe UI",
                 title.textStyle.fontSize = NULL,
                 title.textStyle.lineHeight = NULL,
                 title.textStyle.width = NULL,
                 title.textStyle.height = NULL,
                 title.textStyle.textBorderColor = NULL,
                 title.textStyle.textBorderWidth = NULL,
                 title.textStyle.textBorderType = NULL,
                 title.textStyle.textBorderDashOffset = NULL,
                 title.textStyle.textShadowColor = NULL,
                 title.textStyle.textShadowBlur = NULL,
                 title.textStyle.textShadowOffsetX = NULL,
                 title.textStyle.textShadowOffsetY = NULL,
                 title.subtextStyle.color = NULL,
                 title.subtextStyle.align = NULL,
                 title.subtextStyle.fontStyle = NULL,
                 title.subtextStyle.fontWeight = NULL,
                 title.subtextStyle.fontFamily = "Segoe UI",
                 title.subtextStyle.fontSize = NULL,
                 title.subtextStyle.lineHeight = NULL,
                 title.subtextStyle.width = NULL,
                 title.subtextStyle.height = NULL,
                 title.subtextStyle.textBorderColor = NULL,
                 title.subtextStyle.textBorderWidth = NULL,
                 title.subtextStyle.textBorderType = NULL,
                 title.subtextStyle.textBorderDashOffset = NULL,
                 title.subtextStyle.textShadowColor = NULL,
                 title.subtextStyle.textShadowBlur = NULL,
                 title.subtextStyle.textShadowOffsetX = NULL,
                 title.subtextStyle.textShadowOffsetY = NULL,
                 xAxis.title = NULL,
                 xAxis.min = NULL,
                 xAxis.max = NULL,
                 xAxis.position = "bottom",
                 xAxis.nameLocation = 'center',
                 xAxis.axisTick.customValues = NULL,
                 xAxis.nameTextStyle.color = NULL,
                 xAxis.nameTextStyle.padding = 25,
                 xAxis.nameTextStyle.align = NULL,
                 xAxis.nameTextStyle.fontStyle = NULL,
                 xAxis.nameTextStyle.fontWeight = "bolder",
                 xAxis.nameTextStyle.fontSize = NULL,
                 xAxis.nameTextStyle.fontFamily = "Segoe UI",
                 xAxis.splitNumber = NULL,
                 xAxis.axisLabel.rotate = NULL,
                 xAxis.axisLabel.margin = NULL,
                 xAxis.axisLabel.color = NULL,
                 xAxis.axisLabel.fontStyle = NULL,
                 xAxis.axisLabel.fontWeight = NULL,
                 xAxis.axisLabel.fontFamily = "Segoe UI",
                 xAxis.axisLabel.fontSize = 14,
                 xAxis.axisLabel.align = NULL,
                 xAxis.axisLabel.verticalAlign = NULL,
                 xAxis.axisLabel.backgroundColor = NULL,
                 xAxis.axisLabel.borderColor = NULL,
                 xAxis.axisLabel.borderWidth = NULL,
                 xAxis.axisLabel.borderType = NULL,
                 xAxis.axisLabel.borderRadius = NULL,
                 xAxis.axisLabel.padding = NULL,
                 xAxis.axisLabel.shadowColor = NULL,
                 xAxis.axisLabel.shadowBlur = NULL,
                 xAxis.axisLabel.shadowOffsetX = NULL,
                 xAxis.axisLabel.shadowOffsetY = NULL,
                 xAxis.axisLabel.textBorderColor = NULL,
                 xAxis.axisLabel.textBorderWidth = NULL,
                 xAxis.axisLabel.textBorderType = NULL,
                 xAxis.axisLabel.textShadowColor = NULL,
                 xAxis.axisLabel.textShadowBlur = NULL,
                 xAxis.axisLabel.textShadowOffsetX = NULL,
                 xAxis.axisLabel.textShadowOffsetY = NULL,
                 xAxis.axisLabel.overflow = NULL,
                 yAxis.title = NULL,
                 yAxis.min = NULL,
                 yAxis.max = NULL,
                 yAxis.position = "center",
                 yAxis.nameLocation = 'center',
                 yAxis.axisTick.customValues = NULL,
                 yAxis.nameTextStyle.color = NULL,
                 yAxis.nameTextStyle.padding = 25,
                 yAxis.nameTextStyle.align = NULL,
                 yAxis.nameTextStyle.fontStyle = NULL,
                 yAxis.nameTextStyle.fontWeight = "bolder",
                 yAxis.nameTextStyle.fontSize = NULL,
                 yAxis.nameTextStyle.fontFamily = "Segoe UI",
                 yAxis.splitNumber = NULL,
                 yAxis.axisLabel.rotate = NULL,
                 yAxis.axisLabel.margin = NULL,
                 yAxis.axisLabel.color = NULL,
                 yAxis.axisLabel.fontStyle = NULL,
                 yAxis.axisLabel.fontWeight = NULL,
                 yAxis.axisLabel.fontFamily = "Segoe UI",
                 yAxis.axisLabel.fontSize = 14,
                 yAxis.axisLabel.align = NULL,
                 yAxis.axisLabel.verticalAlign = NULL,
                 yAxis.axisLabel.backgroundColor = NULL,
                 yAxis.axisLabel.borderColor = NULL,
                 yAxis.axisLabel.borderWidth = NULL,
                 yAxis.axisLabel.borderType = NULL,
                 yAxis.axisLabel.borderRadius = NULL,
                 yAxis.axisLabel.padding = NULL,
                 yAxis.axisLabel.shadowColor = NULL,
                 yAxis.axisLabel.shadowBlur = NULL,
                 yAxis.axisLabel.shadowOffsetX = NULL,
                 yAxis.axisLabel.shadowOffsetY = NULL,
                 yAxis.axisLabel.textBorderColor = NULL,
                 yAxis.axisLabel.textBorderWidth = NULL,
                 yAxis.axisLabel.textBorderType = NULL,
                 yAxis.axisLabel.textShadowColor = NULL,
                 yAxis.axisLabel.textShadowBlur = NULL,
                 yAxis.axisLabel.textShadowOffsetX = NULL,
                 yAxis.axisLabel.textShadowOffsetY = NULL,
                 yAxis.axisLabel.overflow = NULL,
                 legend.show = TRUE,
                 legend.type = "scroll",
                 legend.selector = NULL,
                 legend.icon = NULL,
                 legend.align = NULL,
                 legend.padding = NULL,
                 legend.itemGap = NULL,
                 legend.itemWidth = NULL,
                 legend.orient = "vertical",
                 legend.width = NULL,
                 legend.height = NULL,
                 legend.left = NULL,
                 legend.right = 10,
                 legend.top = 30,
                 legend.bottom = NULL,
                 legend.backgroundColor = NULL,
                 legend.borderColor = NULL,
                 legend.borderWidth = NULL,
                 legend.borderRadius = NULL,
                 legend.shadowBlur = NULL,
                 legend.shadowColor = NULL,
                 legend.shadowOffsetX = NULL,
                 legend.shadowOffsetY = NULL,
                 legend.itemStyle.color = NULL,
                 legend.itemStyle.borderColor = NULL,
                 legend.itemStyle.borderWidth = NULL,
                 legend.itemStyle.borderType = NULL,
                 legend.itemStyle.shadowBlur = NULL,
                 legend.itemStyle.shadowColor = NULL,
                 legend.itemStyle.shadowOffsetX = NULL,
                 legend.itemStyle.shadowOffsetY = NULL,
                 legend.itemStyle.opacity = NULL,
                 legend.lineStyle.color = NULL,
                 legend.lineStyle.width = NULL,
                 legend.lineStyle.type = NULL,
                 legend.lineStyle.shadowBlur = NULL,
                 legend.lineStyle.shadowColor = NULL,
                 legend.lineStyle.shadowOffsetX = NULL,
                 legend.lineStyle.shadowOffsetY = NULL,
                 legend.lineStyle.opacity = NULL,
                 legend.lineStyle.inactiveColor = NULL,
                 legend.lineStyle.inactiveWidth = NULL,
                 legend.textStyle.color = NULL,
                 legend.textStyle.fontStyle = NULL,
                 legend.textStyle.fontWeight = NULL,
                 legend.textStyle.fontFamily = "Segoe UI",
                 legend.textStyle.fontSize = NULL,
                 legend.textStyle.backgroundColor = NULL,
                 legend.textStyle.borderColor = NULL,
                 legend.textStyle.borderWidth = NULL,
                 legend.textStyle.borderType = NULL,
                 legend.textStyle.borderRadius = NULL,
                 legend.textStyle.padding = NULL,
                 legend.textStyle.shadowColor = NULL,
                 legend.textStyle.shadowBlur = NULL,
                 legend.textStyle.shadowOffsetX = NULL,
                 legend.textStyle.shadowOffsetY = NULL,
                 legend.textStyle.width = NULL,
                 legend.textStyle.height = NULL,
                 legend.textStyle.textBorderColor = NULL,
                 legend.textStyle.textBorderWidth = NULL,
                 legend.textStyle.textBorderType = NULL,
                 legend.textStyle.textShadowColor = NULL,
                 legend.textStyle.textShadowBlur = NULL,
                 legend.textStyle.textShadowOffsetX = NULL,
                 legend.textStyle.textShadowOffsetY = NULL,
                 legend.pageTextStyle.color = NULL,
                 legend.pageTextStyle.fontStyle = NULL,
                 legend.pageTextStyle.fontWeight = NULL,
                 legend.pageTextStyle.fontFamily = "Segoe UI",
                 legend.pageTextStyle.fontSize = NULL,
                 legend.pageTextStyle.lineHeight = NULL,
                 legend.pageTextStyle.width = NULL,
                 legend.pageTextStyle.height = NULL,
                 legend.pageTextStyle.textBorderColor = NULL,
                 legend.pageTextStyle.textBorderWidth = NULL,
                 legend.pageTextStyle.textBorderType = NULL,
                 legend.pageTextStyle.textShadowColor = NULL,
                 legend.pageTextStyle.textShadowBlur = NULL,
                 legend.pageTextStyle.textShadowOffsetX = NULL,
                 legend.pageTextStyle.textShadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.show = NULL,
                 legend.emphasis.selectorLabel.distance = NULL,
                 legend.emphasis.selectorLabel.rotate = NULL,
                 legend.emphasis.selectorLabel.color = NULL,
                 legend.emphasis.selectorLabel.fontStyle = NULL,
                 legend.emphasis.selectorLabel.fontWeight = NULL,
                 legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                 legend.emphasis.selectorLabel.fontSize = NULL,
                 legend.emphasis.selectorLabel.align = NULL,
                 legend.emphasis.selectorLabel.verticalAlign = NULL,
                 legend.emphasis.selectorLabel.lineHeight = NULL,
                 legend.emphasis.selectorLabel.backgroundColor = NULL,
                 legend.emphasis.selectorLabel.borderColor = NULL,
                 legend.emphasis.selectorLabel.borderWidth = NULL,
                 legend.emphasis.selectorLabel.borderType = NULL,
                 legend.emphasis.selectorLabel.borderRadius = NULL,
                 legend.emphasis.selectorLabel.padding = NULL,
                 legend.emphasis.selectorLabel.shadowColor = NULL,
                 legend.emphasis.selectorLabel.shadowBlur = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.width = NULL,
                 legend.emphasis.selectorLabel.height = NULL,
                 legend.emphasis.selectorLabel.textBorderColor = NULL,
                 legend.emphasis.selectorLabel.textBorderWidth = NULL,
                 legend.emphasis.selectorLabel.textBorderType = NULL,
                 legend.emphasis.selectorLabel.textShadowColor = NULL,
                 legend.emphasis.selectorLabel.textShadowBlur = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                 tooltip.show = TRUE,
                 tooltip.trigger = "axis",
                 tooltip.backgroundColor = NULL,
                 tooltip.borderColor = NULL,
                 tooltip.borderWidth = NULL,
                 tooltip.padding = NULL,
                 tooltip.axisPointer.type = "cross",
                 tooltip.axisPointer.lineStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                 tooltip.axisPointer.shadowStyle.opacity = NULL,
                 tooltip.textStyle.color = NULL,
                 tooltip.textStyle.fontStyle = NULL,
                 tooltip.textStyle.fontWeight = NULL,
                 tooltip.textStyle.fontFamily = "Segoe UI",
                 tooltip.textStyle.lineHeight = NULL,
                 tooltip.textStyle.width = NULL,
                 tooltip.textStyle.height = NULL,
                 tooltip.textStyle.textBorderColor = NULL,
                 tooltip.textStyle.textBorderWidth = NULL,
                 tooltip.textStyle.textBorderType = NULL,
                 tooltip.textStyle.textShadowColor = NULL,
                 tooltip.textStyle.textShadowBlur = NULL,
                 tooltip.textStyle.textShadowOffsetX = NULL,
                 tooltip.textStyle.textShadowOffsetY = NULL,
                 toolbox.show = TRUE,
                 toolbox.orient = "horizontal",
                 toolbox.itemSize = 15,
                 toolbox.itemGap = 8,
                 toolbox.top = NULL,
                 toolbox.left = NULL,
                 toolbox.right = NULL,
                 toolbox.bottom = NULL,
                 toolbox.width = NULL,
                 toolbox.heigth = NULL,
                 toolbox.feature.saveAsImage.show = TRUE,
                 toolbox.feature.restore.show = TRUE,
                 toolbox.feature.dataZoom.show = TRUE,
                 toolbox.feature.magicType.show = TRUE,
                 toolbox.feature.magicType.type = c("line", "bar", "stack"),
                 toolbox.feature.dataView.show = TRUE,
                 toolbox.iconStyle.color = NULL,
                 toolbox.iconStyle.borderColor = NULL,
                 toolbox.emphasis.iconStyle.borderColor = NULL,
                 toolbox.iconStyle.shadowBlur = NULL,
                 toolbox.iconStyle.shadowColor = NULL,
                 toolbox.iconStyle.shadowOffsetX = NULL,
                 toolbox.iconStyle.shadowOffsetY = NULL,
                 Debug = FALSE) {

  if(length(GroupVar) == 0L) TimeLine <- FALSE
  if(TimeLine && length(FacetLevels) > 0) X_Scroll <- FALSE

  # Correct args
  if(length(GroupVar) > 0L && length(XVar) == 0L) {
    XVar <- GroupVar
    GroupVar <- NULL
  }

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }

  # If length(YVar) > 1 and a DualYVar is supplied, dual axis take precedence
  # Throw an error instead of trimming YVar to only the first value
  if(length(YVar) > 1L && length(DualYVar) > 0) stop("When DualYVar is utilized only one DualYVar is allowed and only one YVar is allowed")
  if(length(GroupVar) > 0L && length(DualYVar) > 0) stop("When DualYVar is utilized a GroupVar is not allowed")

  # If User Supplies more than 1 YVar, then structure data to be long instead of wide
  if(length(YVar) > 1L) {
    if(length(GroupVar) > 0L) {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = c(XVar,GroupVar), measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      dt1[, GroupVars := paste0(Measures, GroupVar)]
      dt1[, Measures := NULL]
      dt1[, eval(GroupVar) := NULL]
      GroupVar <- "GroupVars"
      YVar <- "Values"
    } else {
      dt1 <- data.table::melt.data.table(data = dt, id.vars = XVar, measure.vars = YVar, variable.name = "Measures", value.name = "Values")
      GroupVar <- "Measures"
      YVar <- "Values"
    }
  } else {
    dt1 <- data.table::copy(dt)
  }

  # Subset columns
  Ncols <- ncol(dt1)
  if(Ncols > 2L && length(GroupVar) == 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar)]
  } else if(length(GroupVar) > 0L) {
    dt1 <- dt1[, .SD, .SDcols = c(YVar, XVar, DualYVar, GroupVar[1L])]
    if(length(FacetLevels) > 0) {
      dt1 <- dt1[get(GroupVar[1L]) %in% eval(FacetLevels)]
    }
  }

  # Minimize data before moving on
  if(!PreAgg) {

    # Define Aggregation function
    if(Debug) print("Plot.Calibration.Line # Define Aggregation function")
    aggFunc <- SummaryFunction(AggMethod)

    # Aggregate data
    if(length(GroupVar) > 0L) {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar,GroupVar[1L])]
      data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))
    } else {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar)]
      data.table::setorderv(x = dt1, cols = XVar, 1L)
    }
  }

  # Transformation
  if(YVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data
  }
  if(length(DualYVar > 0L) && DualYVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = DualYVar, Methods = DualYVarTrans)$Data
  }

  # Group Variable Case
  if(length(GroupVar) > 0L) {

    # Prepare Data
    if(Debug) print("Line() Build 1")
    gv <- GroupVar[1L]
    if(PreAgg) data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), c(1L,1L))

    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Plot
    if(Debug) print("Line() Build Echarts 1")

    # Build base plot depending on GroupVar availability
    if(Debug) print(paste0("Line TimeLine = ", TimeLine))
    p1 <- echarts4r::e_charts_(
      data = dt1 |> dplyr::group_by(get(gv)),
      x = XVar,
      timeline = TimeLine,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      dispose = TRUE,
      width = Width,
      height = Height)

    # Finalize Plot Build
    if(Debug) print("Line() Build Echarts 4")
    if(ShowLabels) {
      p1 <- echarts4r::e_step_(e = p1, serie = YVar, showSymbol = ShowSymbol, label = list(show = TRUE))
    } else {
      p1 <- echarts4r::e_step_(e = p1, serie = YVar, showSymbol = ShowSymbol)
    }
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if((FacetRows > 1L || FacetCols > 1) && length(FacetLevels) > 0L) {
      p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    }

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  } else {

    # Plot
    data.table::setorderv(x = dt1, cols = XVar, 1L)
    cxv <- class(dt1[[XVar]])[1L]
    if(cxv %in% "IDate") {
      dt1[, eval(XVar) := as.Date(get(XVar))]
    } else if(cxv %in% "IDateTime") {
      dt1[, eval(XVar) := as.POSIXct(get(XVar))]
    }

    # Build base plot depending on GroupVar availability
    if(Debug) print("Line no group Echarts")
    p1 <- echarts4r::e_charts_(
      data = dt1,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)

    if(ShowLabels) {
      p1 <- echarts4r::e_step_(e = p1, serie = YVar, showSymbol = ShowSymbol, label = list(show = TRUE))
    } else {
      p1 <- echarts4r::e_step_(e = p1, serie = YVar, showSymbol = ShowSymbol)
    }

    if(length(DualYVar) > 0L) {
      if(ShowLabels) {
        p1 <- echarts4r::e_step_(e = p1, serie = DualYVar, showSymbol = ShowSymbol, label = list(show = TRUE), x_index = 1, y_index = 1)
      } else {
        p1 <- echarts4r::e_step_(e = p1, serie = DualYVar, showSymbol = ShowSymbol, x_index = 1, y_index = 1)
      }
    }

    if (length(lineStyle.color) > 0 && length(DualYVar) == 0) {
      visualMap.min <- min(dt1[[YVar]])
      visualMap.max <- max(dt1[[YVar]])
      p1 <- e_visual_map_full(
        e = p1,
        serie = YVar,
        visualMap.show = FALSE,
        visualMap.min = visualMap.min,
        visualMap.max = visualMap.max,
        visualMap.InRange.color = lineStyle.color)
    }

    # Finalize Plot Build
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  }
  return(p1)
}

#' @title River
#'
#' @description This function automatically builds calibration plots and calibration boxplots for model evaluation using regression, stats::quantile regression, and binary and multinomial classification
#'
#' @author Adrian Antico
#' @family Standard Plots
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param AggMethod character
#' @param YVar Y-Axis variable name. You can supply multiple YVars
#' @param XVar X-Axis variable name
#' @param GroupVar One Grouping Variable
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels character
#' @param Theme Provide an "Echarts" theme
#' @param TimeLine Logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ShowSymbol = FALSE
#' @param itemStyle.color color or hex
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
# Create fake data
#' dates <- seq.Date(Sys.Date() - 30, Sys.Date(), by = "day")
#' grps <- lapply(LETTERS[1:5], rep, 31) |> unlist()
#' dt <- data.table::data.table(
#'   dates = rep(dates, 5),
#'   groups = grps,
#'   values = runif(length(grps), 1, 50)
#' )
#'
#' # Build Plot
#' AutoPlots::River(
#'   dt = dt,
#'   PreAgg = TRUE,
#'   XVar = "dates",
#'   YVar = "values",
#'   GroupVar = "groups",
#'   legend.orient = "horizontal",
#'   itemStyle.color = c("#FF4C4C", "#00BFFF", "#FFD700", "#32CD32", "#FF69B4")
#' )
#'
#' @return plot
#' @export
River <- function(dt = NULL,
                  AggMethod = "mean",
                  PreAgg = TRUE,
                  XVar = NULL,
                  YVar = NULL,
                  GroupVar = NULL,
                  YVarTrans = "Identity",
                  XVarTrans = "Identity",
                  FacetRows = 1,
                  FacetCols = 1,
                  FacetLevels = NULL,
                  Height = NULL,
                  Width = NULL,
                  ShowLabels = FALSE,
                  Theme = "dark",
                  MouseScroll = FALSE,
                  TimeLine = FALSE,
                  ShowSymbol = FALSE,
                  itemStyle.color = NULL,
                  title.text = "River Plot",
                  title.subtext = NULL,
                  title.link = NULL,
                  title.sublink = NULL,
                  title.Align = NULL,
                  title.top = NULL,
                  title.left = NULL,
                  title.right = NULL,
                  title.bottom = NULL,
                  title.padding = NULL,
                  title.itemGap = NULL,
                  title.backgroundColor = NULL,
                  title.borderColor = NULL,
                  title.borderWidth = NULL,
                  title.borderRadius = NULL,
                  title.shadowColor = NULL,
                  title.shadowBlur = NULL,
                  title.shadowOffsetX = NULL,
                  title.shadowOffsetY = NULL,
                  title.textStyle.color = NULL,
                  title.textStyle.fontStyle = NULL,
                  title.textStyle.fontWeight = NULL,
                  title.textStyle.fontFamily = "Segoe UI",
                  title.textStyle.fontSize = NULL,
                  title.textStyle.lineHeight = NULL,
                  title.textStyle.width = NULL,
                  title.textStyle.height = NULL,
                  title.textStyle.textBorderColor = NULL,
                  title.textStyle.textBorderWidth = NULL,
                  title.textStyle.textBorderType = NULL,
                  title.textStyle.textBorderDashOffset = NULL,
                  title.textStyle.textShadowColor = NULL,
                  title.textStyle.textShadowBlur = NULL,
                  title.textStyle.textShadowOffsetX = NULL,
                  title.textStyle.textShadowOffsetY = NULL,
                  title.subtextStyle.color = NULL,
                  title.subtextStyle.align = NULL,
                  title.subtextStyle.fontStyle = NULL,
                  title.subtextStyle.fontWeight = NULL,
                  title.subtextStyle.fontFamily = "Segoe UI",
                  title.subtextStyle.fontSize = NULL,
                  title.subtextStyle.lineHeight = NULL,
                  title.subtextStyle.width = NULL,
                  title.subtextStyle.height = NULL,
                  title.subtextStyle.textBorderColor = NULL,
                  title.subtextStyle.textBorderWidth = NULL,
                  title.subtextStyle.textBorderType = NULL,
                  title.subtextStyle.textBorderDashOffset = NULL,
                  title.subtextStyle.textShadowColor = NULL,
                  title.subtextStyle.textShadowBlur = NULL,
                  title.subtextStyle.textShadowOffsetX = NULL,
                  title.subtextStyle.textShadowOffsetY = NULL,
                  legend.show = TRUE,
                  legend.type = "scroll",
                  legend.selector = NULL,
                  legend.icon = NULL,
                  legend.align = NULL,
                  legend.padding = NULL,
                  legend.itemGap = NULL,
                  legend.itemWidth = NULL,
                  legend.orient = "vertical",
                  legend.width = NULL,
                  legend.height = NULL,
                  legend.left = NULL,
                  legend.right = 10,
                  legend.top = 30,
                  legend.bottom = NULL,
                  legend.backgroundColor = NULL,
                  legend.borderColor = NULL,
                  legend.borderWidth = NULL,
                  legend.borderRadius = NULL,
                  legend.shadowBlur = NULL,
                  legend.shadowColor = NULL,
                  legend.shadowOffsetX = NULL,
                  legend.shadowOffsetY = NULL,
                  legend.itemStyle.color = NULL,
                  legend.itemStyle.borderColor = NULL,
                  legend.itemStyle.borderWidth = NULL,
                  legend.itemStyle.borderType = NULL,
                  legend.itemStyle.shadowBlur = NULL,
                  legend.itemStyle.shadowColor = NULL,
                  legend.itemStyle.shadowOffsetX = NULL,
                  legend.itemStyle.shadowOffsetY = NULL,
                  legend.itemStyle.opacity = NULL,
                  legend.lineStyle.color = NULL,
                  legend.lineStyle.width = NULL,
                  legend.lineStyle.type = NULL,
                  legend.lineStyle.shadowBlur = NULL,
                  legend.lineStyle.shadowColor = NULL,
                  legend.lineStyle.shadowOffsetX = NULL,
                  legend.lineStyle.shadowOffsetY = NULL,
                  legend.lineStyle.opacity = NULL,
                  legend.lineStyle.inactiveColor = NULL,
                  legend.lineStyle.inactiveWidth = NULL,
                  legend.textStyle.color = NULL,
                  legend.textStyle.fontStyle = NULL,
                  legend.textStyle.fontWeight = NULL,
                  legend.textStyle.fontFamily = "Segoe UI",
                  legend.textStyle.fontSize = NULL,
                  legend.textStyle.backgroundColor = NULL,
                  legend.textStyle.borderColor = NULL,
                  legend.textStyle.borderWidth = NULL,
                  legend.textStyle.borderType = NULL,
                  legend.textStyle.borderRadius = NULL,
                  legend.textStyle.padding = NULL,
                  legend.textStyle.shadowColor = NULL,
                  legend.textStyle.shadowBlur = NULL,
                  legend.textStyle.shadowOffsetX = NULL,
                  legend.textStyle.shadowOffsetY = NULL,
                  legend.textStyle.width = NULL,
                  legend.textStyle.height = NULL,
                  legend.textStyle.textBorderColor = NULL,
                  legend.textStyle.textBorderWidth = NULL,
                  legend.textStyle.textBorderType = NULL,
                  legend.textStyle.textShadowColor = NULL,
                  legend.textStyle.textShadowBlur = NULL,
                  legend.textStyle.textShadowOffsetX = NULL,
                  legend.textStyle.textShadowOffsetY = NULL,
                  legend.pageTextStyle.color = NULL,
                  legend.pageTextStyle.fontStyle = NULL,
                  legend.pageTextStyle.fontWeight = NULL,
                  legend.pageTextStyle.fontFamily = "Segoe UI",
                  legend.pageTextStyle.fontSize = NULL,
                  legend.pageTextStyle.lineHeight = NULL,
                  legend.pageTextStyle.width = NULL,
                  legend.pageTextStyle.height = NULL,
                  legend.pageTextStyle.textBorderColor = NULL,
                  legend.pageTextStyle.textBorderWidth = NULL,
                  legend.pageTextStyle.textBorderType = NULL,
                  legend.pageTextStyle.textShadowColor = NULL,
                  legend.pageTextStyle.textShadowBlur = NULL,
                  legend.pageTextStyle.textShadowOffsetX = NULL,
                  legend.pageTextStyle.textShadowOffsetY = NULL,
                  legend.emphasis.selectorLabel.show = NULL,
                  legend.emphasis.selectorLabel.distance = NULL,
                  legend.emphasis.selectorLabel.rotate = NULL,
                  legend.emphasis.selectorLabel.color = NULL,
                  legend.emphasis.selectorLabel.fontStyle = NULL,
                  legend.emphasis.selectorLabel.fontWeight = NULL,
                  legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                  legend.emphasis.selectorLabel.fontSize = NULL,
                  legend.emphasis.selectorLabel.align = NULL,
                  legend.emphasis.selectorLabel.verticalAlign = NULL,
                  legend.emphasis.selectorLabel.lineHeight = NULL,
                  legend.emphasis.selectorLabel.backgroundColor = NULL,
                  legend.emphasis.selectorLabel.borderColor = NULL,
                  legend.emphasis.selectorLabel.borderWidth = NULL,
                  legend.emphasis.selectorLabel.borderType = NULL,
                  legend.emphasis.selectorLabel.borderRadius = NULL,
                  legend.emphasis.selectorLabel.padding = NULL,
                  legend.emphasis.selectorLabel.shadowColor = NULL,
                  legend.emphasis.selectorLabel.shadowBlur = NULL,
                  legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                  legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                  legend.emphasis.selectorLabel.width = NULL,
                  legend.emphasis.selectorLabel.height = NULL,
                  legend.emphasis.selectorLabel.textBorderColor = NULL,
                  legend.emphasis.selectorLabel.textBorderWidth = NULL,
                  legend.emphasis.selectorLabel.textBorderType = NULL,
                  legend.emphasis.selectorLabel.textShadowColor = NULL,
                  legend.emphasis.selectorLabel.textShadowBlur = NULL,
                  legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                  legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                  tooltip.show = TRUE,
                  tooltip.trigger = "axis",
                  tooltip.backgroundColor = NULL,
                  tooltip.borderColor = NULL,
                  tooltip.borderWidth = NULL,
                  tooltip.padding = NULL,
                  tooltip.axisPointer.type = "cross",
                  tooltip.axisPointer.lineStyle.color = NULL,
                  tooltip.axisPointer.shadowStyle.color = NULL,
                  tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                  tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                  tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                  tooltip.axisPointer.shadowStyle.opacity = NULL,
                  tooltip.textStyle.color = NULL,
                  tooltip.textStyle.fontStyle = NULL,
                  tooltip.textStyle.fontWeight = NULL,
                  tooltip.textStyle.fontFamily = "Segoe UI",
                  tooltip.textStyle.lineHeight = NULL,
                  tooltip.textStyle.width = NULL,
                  tooltip.textStyle.height = NULL,
                  tooltip.textStyle.textBorderColor = NULL,
                  tooltip.textStyle.textBorderWidth = NULL,
                  tooltip.textStyle.textBorderType = NULL,
                  tooltip.textStyle.textShadowColor = NULL,
                  tooltip.textStyle.textShadowBlur = NULL,
                  tooltip.textStyle.textShadowOffsetX = NULL,
                  tooltip.textStyle.textShadowOffsetY = NULL,
                  toolbox.show = TRUE,
                  toolbox.orient = "horizontal",
                  toolbox.itemSize = 15,
                  toolbox.itemGap = 8,
                  toolbox.top = NULL,
                  toolbox.left = NULL,
                  toolbox.right = NULL,
                  toolbox.bottom = NULL,
                  toolbox.width = NULL,
                  toolbox.heigth = NULL,
                  toolbox.feature.saveAsImage.show = TRUE,
                  toolbox.feature.restore.show = FALSE,
                  toolbox.feature.dataZoom.show = FALSE,
                  toolbox.feature.magicType.show = FALSE,
                  toolbox.feature.magicType.type = NULL,
                  toolbox.feature.dataView.show = TRUE,
                  toolbox.iconStyle.color = NULL,
                  toolbox.iconStyle.borderColor = NULL,
                  toolbox.emphasis.iconStyle.borderColor = NULL,
                  toolbox.iconStyle.shadowBlur = NULL,
                  toolbox.iconStyle.shadowColor = NULL,
                  toolbox.iconStyle.shadowOffsetX = NULL,
                  toolbox.iconStyle.shadowOffsetY = NULL,
                  Debug = FALSE) {

  if(length(GroupVar) == 0L) TimeLine <- FALSE
  if(length(GroupVar) == 0L && length(YVar) <= 1L) {
    if(Debug) print("if(length(GroupVar) == 0L && length(YVar) <= 1L) return(NULL)")
    return(NULL)
  }
  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })
  Ncols <- ncol(dt)
  if(length(FacetLevels) > 1L) {
    dt1 <- data.table::copy(dt[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(YVar, XVar, GroupVar)])
  } else {
    dt1 <- data.table::copy(dt[, .SD, .SDcols = c(YVar, XVar, GroupVar)])
  }

  if(Debug) print("Plot.River 3")

  # Minimize data before moving on
  if(!PreAgg) {

    if(Debug) print("Plot.River 4")

    # DCast -> redefine YVar -> Proceed as normal
    if(length(YVar) > 1L && length(GroupVar) == 0L) {
      dt1 <- data.table::melt.data.table(
        data = dt1,
        id.vars = XVar,
        measure.vars = YVar,
        variable.name = "Group",
        value.name = "Measures")
      YVar <- "Measures"
      GroupVar <- "Group"
    } else {
      dt1 <- data.table::copy(dt)
    }

    # Define Aggregation function
    if(Debug) print("Plot.Calibration.Line # Define Aggregation function")
    aggFunc <- SummaryFunction(AggMethod)

    # Aggregate data
    if(length(GroupVar) > 0L) {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar,GroupVar[1L])]
      data.table::setorderv(x = dt1, cols = c(GroupVar[1L], XVar), rep(1L, length(c(GroupVar[1L], XVar))))
    } else {
      dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), by = c(XVar)]
      data.table::setorderv(x = dt1, cols = XVar, 1L)
    }
  }

  # Transformation
  for(yvart in YVarTrans) {
    if(YVarTrans != "Identity") {
      dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = yvart, Methods = YVarTrans)$Data
    }
  }

  if(Debug) print("Plot.River 6b")

  # Plot
  data.table::setorderv(x = dt1, cols = XVar, 1L)
  cxv <- class(dt1[[XVar]])[1L]
  if(cxv %in% "IDate") {
    dt1[, eval(XVar) := as.Date(get(XVar))]
  } else if(cxv %in% "IDateTime") {
    dt1[, eval(XVar) := as.POSIXct(get(XVar))]
  }

  if(Debug) print("Plot.River 7b")

  # Build base plot depending on GroupVar availability
  if(Debug) print("Line no group Echarts")
  if (length(GroupVar) == 0) {
    p1 <- echarts4r::e_charts_(
      data = dt1,
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)
  } else {
    p1 <- echarts4r::e_charts_(
      data = dt1 |> dplyr::group_by(get(GroupVar)),
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      width = Width,
      height = Height)
  }


  p1 <- echarts4r::e_river_(
    e = p1,
    serie = YVar
  )
  if (length(itemStyle.color) == length(as.character(unique(dt1[[GroupVar]])))) {
    p1 <- echarts4r::e_color(e = p1, color = itemStyle.color)
  }

  if(Debug) print("Plot.River 8b")
  p1 <- echarts4r::e_theme(e = p1, name = Theme)

  p1 <- e_tooltip_full(
    e = p1,
    tooltip.show = tooltip.show,
    tooltip.trigger = tooltip.trigger,
    tooltip.backgroundColor = tooltip.backgroundColor,
    tooltip.borderColor = tooltip.borderColor,
    tooltip.borderWidth = tooltip.borderWidth,
    tooltip.padding = tooltip.padding,
    tooltip.axisPointer.type = tooltip.axisPointer.type,
    tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
    tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
    tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
    tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
    tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
    tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
    tooltip.textStyle.color = tooltip.textStyle.color,
    tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
    tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
    tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
    tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
    tooltip.textStyle.width = tooltip.textStyle.width,
    tooltip.textStyle.height = tooltip.textStyle.height,
    tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
    tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
    tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
    tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
    tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
    tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
    tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

  p1 <- e_toolbox_full(
    e = p1,
    toolbox.show = toolbox.show,
    toolbox.orient = toolbox.orient,
    toolbox.itemSize = toolbox.itemSize,
    toolbox.itemGap = toolbox.itemGap,
    toolbox.top = toolbox.top,
    toolbox.left = toolbox.left,
    toolbox.right = toolbox.right,
    toolbox.bottom = toolbox.bottom,
    toolbox.width = toolbox.width,
    toolbox.heigth = toolbox.heigth,
    toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
    toolbox.feature.restore.show = toolbox.feature.restore.show,
    toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
    toolbox.feature.magicType.show = toolbox.feature.magicType.show,
    toolbox.feature.magicType.type = toolbox.feature.magicType.type,
    toolbox.feature.dataView.show = toolbox.feature.dataView.show,
    toolbox.iconStyle.color = toolbox.iconStyle.color,
    toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
    toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
    toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
    toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
    toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
    toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

  p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
  p1 <- echarts4r::e_brush(e = p1)

  p1 <- e_legend_full(
    e = p1,
    legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
    legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
    legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
    legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
    legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
    legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
    legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
    legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
    legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
    legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
    legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
    legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
    legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
    legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
    legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
    legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
    legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
    legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
    legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
    legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
    legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
    legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
    legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
    legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
    legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
    legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
    legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
    legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
    legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
    legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
    legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
    legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
    legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
    legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
    legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
    legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
    legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
    legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
    legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
    legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
    legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
    legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
    legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
    legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
    legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
    legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
    legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
    legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
    legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
    legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
    legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
    legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
    legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

  p1 <- e_title_full(
    e = p1,
    title.text = title.text,
    title.subtext = title.subtext,
    title.link = title.link,
    title.sublink = title.sublink,
    title.Align = title.Align,
    title.top = title.top,
    title.left = title.left,
    title.right = title.right,
    title.bottom = title.bottom,
    title.padding = title.padding,
    title.itemGap = title.itemGap,
    title.backgroundColor = title.backgroundColor,
    title.borderColor = title.borderColor,
    title.borderWidth = title.borderWidth,
    title.borderRadius = title.borderRadius,
    title.shadowColor = title.shadowColor,
    title.shadowBlur = title.shadowBlur,
    title.shadowOffsetX = title.shadowOffsetX,
    title.shadowOffsetY = title.shadowOffsetY,
    title.textStyle.color = title.textStyle.color,
    title.textStyle.fontStyle = title.textStyle.fontStyle,
    title.textStyle.fontWeight = title.textStyle.fontWeight,
    title.textStyle.fontFamily = title.textStyle.fontFamily,
    title.textStyle.fontSize = title.textStyle.fontSize,
    title.textStyle.lineHeight = title.textStyle.lineHeight,
    title.textStyle.width = title.textStyle.width,
    title.textStyle.height = title.textStyle.height,
    title.textStyle.textBorderColor = title.textStyle.textBorderColor,
    title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
    title.textStyle.textBorderType = title.textStyle.textBorderType,
    title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
    title.textStyle.textShadowColor = title.textStyle.textShadowColor,
    title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
    title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
    title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
    title.subtextStyle.color = title.subtextStyle.color,
    title.subtextStyle.align = title.subtextStyle.align,
    title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
    title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
    title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
    title.subtextStyle.fontSize = title.subtextStyle.fontSize,
    title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
    title.subtextStyle.width = title.subtextStyle.width,
    title.subtextStyle.height = title.subtextStyle.height,
    title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
    title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
    title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
    title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
    title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
    title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
    title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
    title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

  return(p1)
}

#' @title Bar
#'
#' @description Build a bar plot by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param LabelValues A vector of values. Requires PreAgg to be set to TRUE and you'll need to ensure LabelValues are ordered the same as dt. If NULL and ShowLabels is TRUE, then bar values will be displayed
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param ShowLabels logical
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ContainLabel TRUE
#' @param backgroundStyle.color vector of colors
#' @param backgroundStyle.opacity vector of opacities
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 10000)
#'
#' # Echarts Bar Chart
#' AutoPlots::Bar(
#'   dt = data,
#'   PreAgg = TRUE,
#'   XVar = "Factor_1",
#'   YVar = "Adrian",
#'   GroupVar = NULL,
#'   LabelValues = NULL,
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   AggMethod = 'mean',
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   TimeLine = FALSE,
#'   ContainLabel = TRUE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
Bar <- function(dt = NULL,
                PreAgg = TRUE,
                XVar = NULL,
                YVar = NULL,
                GroupVar = NULL,
                LabelValues = NULL,
                YVarTrans = "Identity",
                XVarTrans = "Identity",
                FacetRows = 1,
                FacetCols = 1,
                FacetLevels = NULL,
                AggMethod = 'mean',
                Height = NULL,
                Width = NULL,
                ShowLabels = FALSE,
                Theme = "dark",
                MouseScroll = FALSE,
                TimeLine = FALSE,
                ContainLabel = TRUE,
                backgroundStyle.color = NULL,
                backgroundStyle.opacity = NULL,
                title.text = "Bar Plot",
                title.subtext = NULL,
                title.link = NULL,
                title.sublink = NULL,
                title.Align = NULL,
                title.top = NULL,
                title.left = NULL,
                title.right = NULL,
                title.bottom = NULL,
                title.padding = NULL,
                title.itemGap = NULL,
                title.backgroundColor = NULL,
                title.borderColor = NULL,
                title.borderWidth = NULL,
                title.borderRadius = NULL,
                title.shadowColor = NULL,
                title.shadowBlur = NULL,
                title.shadowOffsetX = NULL,
                title.shadowOffsetY = NULL,
                title.textStyle.color = NULL,
                title.textStyle.fontStyle = NULL,
                title.textStyle.fontWeight = NULL,
                title.textStyle.fontFamily = "Segoe UI",
                title.textStyle.fontSize = NULL,
                title.textStyle.lineHeight = NULL,
                title.textStyle.width = NULL,
                title.textStyle.height = NULL,
                title.textStyle.textBorderColor = NULL,
                title.textStyle.textBorderWidth = NULL,
                title.textStyle.textBorderType = NULL,
                title.textStyle.textBorderDashOffset = NULL,
                title.textStyle.textShadowColor = NULL,
                title.textStyle.textShadowBlur = NULL,
                title.textStyle.textShadowOffsetX = NULL,
                title.textStyle.textShadowOffsetY = NULL,
                title.subtextStyle.color = NULL,
                title.subtextStyle.align = NULL,
                title.subtextStyle.fontStyle = NULL,
                title.subtextStyle.fontWeight = NULL,
                title.subtextStyle.fontFamily = "Segoe UI",
                title.subtextStyle.fontSize = NULL,
                title.subtextStyle.lineHeight = NULL,
                title.subtextStyle.width = NULL,
                title.subtextStyle.height = NULL,
                title.subtextStyle.textBorderColor = NULL,
                title.subtextStyle.textBorderWidth = NULL,
                title.subtextStyle.textBorderType = NULL,
                title.subtextStyle.textBorderDashOffset = NULL,
                title.subtextStyle.textShadowColor = NULL,
                title.subtextStyle.textShadowBlur = NULL,
                title.subtextStyle.textShadowOffsetX = NULL,
                title.subtextStyle.textShadowOffsetY = NULL,
                xAxis.title = NULL,
                xAxis.min = NULL,
                xAxis.max = NULL,
                xAxis.position = "bottom",
                xAxis.nameLocation = 'center',
                xAxis.axisTick.customValues = NULL,
                xAxis.nameTextStyle.color = NULL,
                xAxis.nameTextStyle.padding = 25,
                xAxis.nameTextStyle.align = NULL,
                xAxis.nameTextStyle.fontStyle = NULL,
                xAxis.nameTextStyle.fontWeight = "bolder",
                xAxis.nameTextStyle.fontSize = NULL,
                xAxis.nameTextStyle.fontFamily = "Segoe UI",
                xAxis.splitNumber = NULL,
                xAxis.axisLabel.rotate = NULL,
                xAxis.axisLabel.margin = NULL,
                xAxis.axisLabel.color = NULL,
                xAxis.axisLabel.fontStyle = NULL,
                xAxis.axisLabel.fontWeight = NULL,
                xAxis.axisLabel.fontFamily = "Segoe UI",
                xAxis.axisLabel.fontSize = 14,
                xAxis.axisLabel.align = NULL,
                xAxis.axisLabel.verticalAlign = NULL,
                xAxis.axisLabel.backgroundColor = NULL,
                xAxis.axisLabel.borderColor = NULL,
                xAxis.axisLabel.borderWidth = NULL,
                xAxis.axisLabel.borderType = NULL,
                xAxis.axisLabel.borderRadius = NULL,
                xAxis.axisLabel.padding = NULL,
                xAxis.axisLabel.shadowColor = NULL,
                xAxis.axisLabel.shadowBlur = NULL,
                xAxis.axisLabel.shadowOffsetX = NULL,
                xAxis.axisLabel.shadowOffsetY = NULL,
                xAxis.axisLabel.textBorderColor = NULL,
                xAxis.axisLabel.textBorderWidth = NULL,
                xAxis.axisLabel.textBorderType = NULL,
                xAxis.axisLabel.textShadowColor = NULL,
                xAxis.axisLabel.textShadowBlur = NULL,
                xAxis.axisLabel.textShadowOffsetX = NULL,
                xAxis.axisLabel.textShadowOffsetY = NULL,
                xAxis.axisLabel.overflow = NULL,
                yAxis.title = NULL,
                yAxis.min = NULL,
                yAxis.max = NULL,
                yAxis.position = "center",
                yAxis.nameLocation = 'center',
                yAxis.axisTick.customValues = NULL,
                yAxis.nameTextStyle.color = NULL,
                yAxis.nameTextStyle.padding = 25,
                yAxis.nameTextStyle.align = NULL,
                yAxis.nameTextStyle.fontStyle = NULL,
                yAxis.nameTextStyle.fontWeight = "bolder",
                yAxis.nameTextStyle.fontSize = NULL,
                yAxis.nameTextStyle.fontFamily = "Segoe UI",
                yAxis.splitNumber = NULL,
                yAxis.axisLabel.rotate = NULL,
                yAxis.axisLabel.margin = NULL,
                yAxis.axisLabel.color = NULL,
                yAxis.axisLabel.fontStyle = NULL,
                yAxis.axisLabel.fontWeight = NULL,
                yAxis.axisLabel.fontFamily = "Segoe UI",
                yAxis.axisLabel.fontSize = 14,
                yAxis.axisLabel.align = NULL,
                yAxis.axisLabel.verticalAlign = NULL,
                yAxis.axisLabel.backgroundColor = NULL,
                yAxis.axisLabel.borderColor = NULL,
                yAxis.axisLabel.borderWidth = NULL,
                yAxis.axisLabel.borderType = NULL,
                yAxis.axisLabel.borderRadius = NULL,
                yAxis.axisLabel.padding = NULL,
                yAxis.axisLabel.shadowColor = NULL,
                yAxis.axisLabel.shadowBlur = NULL,
                yAxis.axisLabel.shadowOffsetX = NULL,
                yAxis.axisLabel.shadowOffsetY = NULL,
                yAxis.axisLabel.textBorderColor = NULL,
                yAxis.axisLabel.textBorderWidth = NULL,
                yAxis.axisLabel.textBorderType = NULL,
                yAxis.axisLabel.textShadowColor = NULL,
                yAxis.axisLabel.textShadowBlur = NULL,
                yAxis.axisLabel.textShadowOffsetX = NULL,
                yAxis.axisLabel.textShadowOffsetY = NULL,
                yAxis.axisLabel.overflow = NULL,
                legend.show = TRUE,
                legend.type = "scroll",
                legend.selector = NULL,
                legend.icon = NULL,
                legend.align = NULL,
                legend.padding = NULL,
                legend.itemGap = NULL,
                legend.itemWidth = NULL,
                legend.orient = "vertical",
                legend.width = NULL,
                legend.height = NULL,
                legend.left = NULL,
                legend.right = 10,
                legend.top = 30,
                legend.bottom = NULL,
                legend.backgroundColor = NULL,
                legend.borderColor = NULL,
                legend.borderWidth = NULL,
                legend.borderRadius = NULL,
                legend.shadowBlur = NULL,
                legend.shadowColor = NULL,
                legend.shadowOffsetX = NULL,
                legend.shadowOffsetY = NULL,
                legend.itemStyle.color = NULL,
                legend.itemStyle.borderColor = NULL,
                legend.itemStyle.borderWidth = NULL,
                legend.itemStyle.borderType = NULL,
                legend.itemStyle.shadowBlur = NULL,
                legend.itemStyle.shadowColor = NULL,
                legend.itemStyle.shadowOffsetX = NULL,
                legend.itemStyle.shadowOffsetY = NULL,
                legend.itemStyle.opacity = NULL,
                legend.lineStyle.color = NULL,
                legend.lineStyle.width = NULL,
                legend.lineStyle.type = NULL,
                legend.lineStyle.shadowBlur = NULL,
                legend.lineStyle.shadowColor = NULL,
                legend.lineStyle.shadowOffsetX = NULL,
                legend.lineStyle.shadowOffsetY = NULL,
                legend.lineStyle.opacity = NULL,
                legend.lineStyle.inactiveColor = NULL,
                legend.lineStyle.inactiveWidth = NULL,
                legend.textStyle.color = NULL,
                legend.textStyle.fontStyle = NULL,
                legend.textStyle.fontWeight = NULL,
                legend.textStyle.fontFamily = "Segoe UI",
                legend.textStyle.fontSize = NULL,
                legend.textStyle.backgroundColor = NULL,
                legend.textStyle.borderColor = NULL,
                legend.textStyle.borderWidth = NULL,
                legend.textStyle.borderType = NULL,
                legend.textStyle.borderRadius = NULL,
                legend.textStyle.padding = NULL,
                legend.textStyle.shadowColor = NULL,
                legend.textStyle.shadowBlur = NULL,
                legend.textStyle.shadowOffsetX = NULL,
                legend.textStyle.shadowOffsetY = NULL,
                legend.textStyle.width = NULL,
                legend.textStyle.height = NULL,
                legend.textStyle.textBorderColor = NULL,
                legend.textStyle.textBorderWidth = NULL,
                legend.textStyle.textBorderType = NULL,
                legend.textStyle.textShadowColor = NULL,
                legend.textStyle.textShadowBlur = NULL,
                legend.textStyle.textShadowOffsetX = NULL,
                legend.textStyle.textShadowOffsetY = NULL,
                legend.pageTextStyle.color = NULL,
                legend.pageTextStyle.fontStyle = NULL,
                legend.pageTextStyle.fontWeight = NULL,
                legend.pageTextStyle.fontFamily = "Segoe UI",
                legend.pageTextStyle.fontSize = NULL,
                legend.pageTextStyle.lineHeight = NULL,
                legend.pageTextStyle.width = NULL,
                legend.pageTextStyle.height = NULL,
                legend.pageTextStyle.textBorderColor = NULL,
                legend.pageTextStyle.textBorderWidth = NULL,
                legend.pageTextStyle.textBorderType = NULL,
                legend.pageTextStyle.textShadowColor = NULL,
                legend.pageTextStyle.textShadowBlur = NULL,
                legend.pageTextStyle.textShadowOffsetX = NULL,
                legend.pageTextStyle.textShadowOffsetY = NULL,
                legend.emphasis.selectorLabel.show = NULL,
                legend.emphasis.selectorLabel.distance = NULL,
                legend.emphasis.selectorLabel.rotate = NULL,
                legend.emphasis.selectorLabel.color = NULL,
                legend.emphasis.selectorLabel.fontStyle = NULL,
                legend.emphasis.selectorLabel.fontWeight = NULL,
                legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                legend.emphasis.selectorLabel.fontSize = NULL,
                legend.emphasis.selectorLabel.align = NULL,
                legend.emphasis.selectorLabel.verticalAlign = NULL,
                legend.emphasis.selectorLabel.lineHeight = NULL,
                legend.emphasis.selectorLabel.backgroundColor = NULL,
                legend.emphasis.selectorLabel.borderColor = NULL,
                legend.emphasis.selectorLabel.borderWidth = NULL,
                legend.emphasis.selectorLabel.borderType = NULL,
                legend.emphasis.selectorLabel.borderRadius = NULL,
                legend.emphasis.selectorLabel.padding = NULL,
                legend.emphasis.selectorLabel.shadowColor = NULL,
                legend.emphasis.selectorLabel.shadowBlur = NULL,
                legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                legend.emphasis.selectorLabel.width = NULL,
                legend.emphasis.selectorLabel.height = NULL,
                legend.emphasis.selectorLabel.textBorderColor = NULL,
                legend.emphasis.selectorLabel.textBorderWidth = NULL,
                legend.emphasis.selectorLabel.textBorderType = NULL,
                legend.emphasis.selectorLabel.textShadowColor = NULL,
                legend.emphasis.selectorLabel.textShadowBlur = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                tooltip.show = TRUE,
                tooltip.trigger = "axis",
                tooltip.backgroundColor = NULL,
                tooltip.borderColor = NULL,
                tooltip.borderWidth = NULL,
                tooltip.padding = NULL,
                tooltip.axisPointer.type = "cross",
                tooltip.axisPointer.lineStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                tooltip.axisPointer.shadowStyle.opacity = NULL,
                tooltip.textStyle.color = NULL,
                tooltip.textStyle.fontStyle = NULL,
                tooltip.textStyle.fontWeight = NULL,
                tooltip.textStyle.fontFamily = "Segoe UI",
                tooltip.textStyle.lineHeight = NULL,
                tooltip.textStyle.width = NULL,
                tooltip.textStyle.height = NULL,
                tooltip.textStyle.textBorderColor = NULL,
                tooltip.textStyle.textBorderWidth = NULL,
                tooltip.textStyle.textBorderType = NULL,
                tooltip.textStyle.textShadowColor = NULL,
                tooltip.textStyle.textShadowBlur = NULL,
                tooltip.textStyle.textShadowOffsetX = NULL,
                tooltip.textStyle.textShadowOffsetY = NULL,
                toolbox.show = TRUE,
                toolbox.orient = "horizontal",
                toolbox.itemSize = 15,
                toolbox.itemGap = 8,
                toolbox.top = NULL,
                toolbox.left = NULL,
                toolbox.right = NULL,
                toolbox.bottom = NULL,
                toolbox.width = NULL,
                toolbox.heigth = NULL,
                toolbox.feature.saveAsImage.show = TRUE,
                toolbox.feature.restore.show = TRUE,
                toolbox.feature.dataZoom.show = TRUE,
                toolbox.feature.magicType.show = TRUE,
                toolbox.feature.magicType.type = c("line", "bar", "stack"),
                toolbox.feature.dataView.show = TRUE,
                toolbox.iconStyle.color = NULL,
                toolbox.iconStyle.borderColor = NULL,
                toolbox.emphasis.iconStyle.borderColor = NULL,
                toolbox.iconStyle.shadowBlur = NULL,
                toolbox.iconStyle.shadowColor = NULL,
                toolbox.iconStyle.shadowOffsetX = NULL,
                toolbox.iconStyle.shadowOffsetY = NULL,
                Debug = FALSE) {

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  if(length(GroupVar) == 0L) TimeLine <- FALSE

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] == "factor") {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }
  if(length(XVar) > 0L && class(dt[[XVar]])[1L] == "factor") {
    dt[, eval(XVar) := as.character(get(XVar))]
  }
  if(length(YVar) > 0L && class(dt[[YVar]])[1L] == "factor") {
    dt[, eval(YVar) := as.character(get(YVar))]
  }

  # Used multiple times
  check1 <- length(XVar) != 0 && length(YVar) != 0
  check2 <- length(XVar) == 0 && length(YVar) != 0
  check3 <- length(XVar) != 0 && length(YVar) == 0

  # Define Aggregation function
  if(!PreAgg) {
    aggFunc <- SummaryFunction(AggMethod)
  }

  # Create base plot object
  numvars <- c()
  byvars <- c()
  if(check1) {
    if(length(GroupVar) != 0L) {
      if(!PreAgg) {

        if(length(FacetLevels) > 0L) {
          dt <- dt[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(YVar,XVar,GroupVar)]
        }

        if(any(tryCatch({class(dt[[eval(YVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
          numvars <- unique(c(numvars, YVar))
        } else {
          byvars <- unique(c(byvars, YVar))
        }
        if(any(tryCatch({class(dt[[eval(XVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
          if(length(numvars) > 0) {
            x <- length(unique(dt[[XVar]]))
            y <- length(unique(dt[[YVar]]))
            if(x > y) {
              byvars <- unique(c(byvars, YVar))
              numvars[1L] <- XVar
            } else {
              byvars <- unique(c(byvars, XVar))
            }
          } else {
            numvars <- unique(c(numvars, XVar))
          }
        } else {
          byvars <- unique(c(byvars, XVar))
        }
        if(any(tryCatch({class(dt[[eval(GroupVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
          dt[, eval(GroupVar) := as.character(get(GroupVar))]
          byvars <- unique(c(byvars, GroupVar))
        } else {
          byvars <- unique(c(byvars, GroupVar))
        }
        if(!is.null(byvars)) {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
          for(i in byvars) {
            if(class(temp[[i]])[1] %in% c('numeric','integer')) {
              temp[, eval(i) := as.character(get(i))]
            }
          }
        } else {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
        }
      } else {
        temp <- data.table::copy(dt)
        numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
        byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
      }

      # Transformation
      if(length(XVar) > 0L && class(temp[[XVar]])[1L] %in% c("numeric","integer")) {
        YVarTrans <- XVarTrans
      }
      if(YVarTrans != "Identity") {
        temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
      }

      # Plot
      p1 <- echarts4r::e_charts_(
        temp |> dplyr::group_by(get(GroupVar[1L])),
        x = XVar,
        darkMode = TRUE,
        emphasis = list(focus = "series"),
        dispose = TRUE,
        width = Width,
        height = Height)

      p1 <- e_bar_full(
        e = p1,
        YVar,
        label = ShowLabels,
        backgroundStyle.color = backgroundStyle.color,
        backgroundStyle.opacity = backgroundStyle.opacity)

      if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
      } else {
        p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
        p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
      }
      p1 <- echarts4r::e_theme(e = p1, name = Theme)

      p1 <- e_tooltip_full(
        e = p1,
        tooltip.show = tooltip.show,
        tooltip.trigger = tooltip.trigger,
        tooltip.backgroundColor = tooltip.backgroundColor,
        tooltip.borderColor = tooltip.borderColor,
        tooltip.borderWidth = tooltip.borderWidth,
        tooltip.padding = tooltip.padding,
        tooltip.axisPointer.type = tooltip.axisPointer.type,
        tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
        tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
        tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
        tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
        tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
        tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
        tooltip.textStyle.color = tooltip.textStyle.color,
        tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
        tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
        tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
        tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
        tooltip.textStyle.width = tooltip.textStyle.width,
        tooltip.textStyle.height = tooltip.textStyle.height,
        tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
        tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
        tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
        tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
        tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
        tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
        tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

      p1 <- e_toolbox_full(
        e = p1,
        toolbox.show = toolbox.show,
        toolbox.orient = toolbox.orient,
        toolbox.itemSize = toolbox.itemSize,
        toolbox.itemGap = toolbox.itemGap,
        toolbox.top = toolbox.top,
        toolbox.left = toolbox.left,
        toolbox.right = toolbox.right,
        toolbox.bottom = toolbox.bottom,
        toolbox.width = toolbox.width,
        toolbox.heigth = toolbox.heigth,
        toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
        toolbox.feature.restore.show = toolbox.feature.restore.show,
        toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
        toolbox.feature.magicType.show = toolbox.feature.magicType.show,
        toolbox.feature.magicType.type = toolbox.feature.magicType.type,
        toolbox.feature.dataView.show = toolbox.feature.dataView.show,
        toolbox.iconStyle.color = toolbox.iconStyle.color,
        toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
        toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
        toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
        toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
        toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
        toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

      p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
      p1 <- echarts4r::e_brush(e = p1)
      p1 <- e_title_full(
        e = p1,
        title.text = title.text,
        title.subtext = title.subtext,
        title.link = title.link,
        title.sublink = title.sublink,
        title.Align = title.Align,
        title.top = title.top,
        title.left = title.left,
        title.right = title.right,
        title.bottom = title.bottom,
        title.padding = title.padding,
        title.itemGap = title.itemGap,
        title.backgroundColor = title.backgroundColor,
        title.borderColor = title.borderColor,
        title.borderWidth = title.borderWidth,
        title.borderRadius = title.borderRadius,
        title.shadowColor = title.shadowColor,
        title.shadowBlur = title.shadowBlur,
        title.shadowOffsetX = title.shadowOffsetX,
        title.shadowOffsetY = title.shadowOffsetY,
        title.textStyle.color = title.textStyle.color,
        title.textStyle.fontStyle = title.textStyle.fontStyle,
        title.textStyle.fontWeight = title.textStyle.fontWeight,
        title.textStyle.fontFamily = title.textStyle.fontFamily,
        title.textStyle.fontSize = title.textStyle.fontSize,
        title.textStyle.lineHeight = title.textStyle.lineHeight,
        title.textStyle.width = title.textStyle.width,
        title.textStyle.height = title.textStyle.height,
        title.textStyle.textBorderColor = title.textStyle.textBorderColor,
        title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
        title.textStyle.textBorderType = title.textStyle.textBorderType,
        title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
        title.textStyle.textShadowColor = title.textStyle.textShadowColor,
        title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
        title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
        title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
        title.subtextStyle.color = title.subtextStyle.color,
        title.subtextStyle.align = title.subtextStyle.align,
        title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
        title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
        title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
        title.subtextStyle.fontSize = title.subtextStyle.fontSize,
        title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
        title.subtextStyle.width = title.subtextStyle.width,
        title.subtextStyle.height = title.subtextStyle.height,
        title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
        title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
        title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
        title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
        title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
        title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
        title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
        title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

      if(FacetRows > 1L || FacetCols > 1L) p1 <- echarts4r::e_facet(
        e = p1,
        rows = FacetRows,
        cols = FacetCols,
        legend_space = 16,
        legend_pos = "top")

      p1 <- e_x_axis_full(
        e = p1,
        serie = NULL,
        axis = "x",
        xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
        xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
        xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
        xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
        xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
        xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
        xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
        xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
        xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
        xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
        xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
        xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
        xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
        xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
        xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
        xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
        xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
        xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
        xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

      p1 <- e_y_axis_full(
        e = p1,
        serie = NULL,
        axis = "y",
        yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
        yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
        yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
        yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
        yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
        yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
        yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
        yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
        yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
        yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
        yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
        yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
        yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
        yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
        yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
        yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
        yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
        yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
        yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

      p1 <- e_legend_full(
        e = p1,
        legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
        legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
        legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
        legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
        legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
        legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
        legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
        legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
        legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
        legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
        legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
        legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
        legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
        legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
        legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
        legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
        legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
        legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
        legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
        legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
        legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
        legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
        legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
        legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
        legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
        legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
        legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
        legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
        legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
        legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
        legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
        legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
        legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
        legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
        legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
        legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
        legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
        legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
        legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
        legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
        legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
        legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
        legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
        legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
        legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
        legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
        legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
        legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
        legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
        legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
        legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
        legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
        legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

      return(p1)

    } else {

      if(Debug) {
        print("BarPlot 2.b")
        print(PreAgg)
      }

      if(!PreAgg) {
        if(tryCatch({class(dt[[eval(YVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
          numvars <- unique(c(numvars, YVar))
        } else {
          byvars <- unique(c(byvars, YVar))
        }
        if(tryCatch({class(dt[[eval(XVar)]])[1L]}, error = function(x) "bla") %in% c('numeric','integer')) {
          if(length(numvars) > 0) {
            x <- length(unique(dt[[XVar]]))
            y <- length(unique(dt[[YVar]]))
            if(x > y) {
              byvars <- unique(c(byvars, YVar))
              numvars[1L] <- XVar
            } else {
              byvars <- unique(c(byvars, XVar))
            }
          } else {
            numvars <- unique(c(numvars, XVar))
          }
        } else {
          byvars <- unique(c(byvars, XVar))
        }
        if(!is.null(byvars)) {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
          for(i in byvars) {
            if(class(temp[[i]])[1L] %in% c('numeric','integer')) {
              temp[, eval(i) := as.character(get(i))]
            }
          }
        } else {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
        }
      } else {
        temp <- data.table::copy(dt)
        if(Debug) print("BarPlot 2.bb")
        numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
        byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
      }

      if(Debug) print("BarPlot 2.bbb")

      # Transformation
      if(length(XVar) > 0L && class(temp[[XVar]])[1L] %in% c("numeric","integer")) {
        YVarTrans <- XVarTrans
      }

      if(Debug) print("BarPlot 2.bbbb")

      if(YVarTrans != "Identity") {
        temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
      }

      if(Debug) print("BarPlot 2.bbbbb")

      # Plot
      if(XVar == "Importance" && YVar == "Variable") {
        XVar <- "Variable"
        YVar <- "Importance"
      }

      if(Debug) print("BarPlot 2.bbbbbb")

      p1 <- echarts4r::e_charts_(
        temp,
        x = XVar,
        dispose = TRUE,
        darkMode = TRUE,
        width = Width,
        height = Height)

      if(Debug) print("BarPlot 2.c")

      p1 <- e_bar_full(
        e = p1,
        YVar,
        label = ShowLabels,
        backgroundStyle.color = backgroundStyle.color,
        backgroundStyle.opacity = backgroundStyle.opacity)

      if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
      } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
      } else {
        p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
        p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
      }
      if(Debug) print("BarPlot 2.cccc")
      p1 <- echarts4r::e_theme(e = p1, name = Theme)

      p1 <- e_tooltip_full(
        e = p1,
        tooltip.show = tooltip.show,
        tooltip.trigger = tooltip.trigger,
        tooltip.backgroundColor = tooltip.backgroundColor,
        tooltip.borderColor = tooltip.borderColor,
        tooltip.borderWidth = tooltip.borderWidth,
        tooltip.padding = tooltip.padding,
        tooltip.axisPointer.type = tooltip.axisPointer.type,
        tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
        tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
        tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
        tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
        tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
        tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
        tooltip.textStyle.color = tooltip.textStyle.color,
        tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
        tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
        tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
        tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
        tooltip.textStyle.width = tooltip.textStyle.width,
        tooltip.textStyle.height = tooltip.textStyle.height,
        tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
        tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
        tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
        tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
        tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
        tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
        tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

      p1 <- e_toolbox_full(
        e = p1,
        toolbox.show = toolbox.show,
        toolbox.orient = toolbox.orient,
        toolbox.itemSize = toolbox.itemSize,
        toolbox.itemGap = toolbox.itemGap,
        toolbox.top = toolbox.top,
        toolbox.left = toolbox.left,
        toolbox.right = toolbox.right,
        toolbox.bottom = toolbox.bottom,
        toolbox.width = toolbox.width,
        toolbox.heigth = toolbox.heigth,
        toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
        toolbox.feature.restore.show = toolbox.feature.restore.show,
        toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
        toolbox.feature.magicType.show = toolbox.feature.magicType.show,
        toolbox.feature.magicType.type = toolbox.feature.magicType.type,
        toolbox.feature.dataView.show = toolbox.feature.dataView.show,
        toolbox.iconStyle.color = toolbox.iconStyle.color,
        toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
        toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
        toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
        toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
        toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
        toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

      p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

      if(Debug) print("BarPlot 2.d")
      p1 <- e_x_axis_full(
        e = p1,
        serie = NULL,
        axis = "x",
        xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
        xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
        xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
        xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
        xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
        xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
        xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
        xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
        xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
        xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
        xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
        xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
        xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
        xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
        xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
        xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
        xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
        xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
        xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

      p1 <- e_y_axis_full(
        e = p1,
        serie = NULL,
        axis = "y",
        yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
        yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
        yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
        yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
        yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
        yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
        yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
        yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
        yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
        yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
        yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
        yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
        yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
        yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
        yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
        yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
        yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
        yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
        yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

      if(Debug) print("BarPlot 2.f")
      p1 <- echarts4r::e_brush(e = p1)
      p1 <- e_title_full(
        e = p1,
        title.text = title.text,
        title.subtext = title.subtext,
        title.link = title.link,
        title.sublink = title.sublink,
        title.Align = title.Align,
        title.top = title.top,
        title.left = title.left,
        title.right = title.right,
        title.bottom = title.bottom,
        title.padding = title.padding,
        title.itemGap = title.itemGap,
        title.backgroundColor = title.backgroundColor,
        title.borderColor = title.borderColor,
        title.borderWidth = title.borderWidth,
        title.borderRadius = title.borderRadius,
        title.shadowColor = title.shadowColor,
        title.shadowBlur = title.shadowBlur,
        title.shadowOffsetX = title.shadowOffsetX,
        title.shadowOffsetY = title.shadowOffsetY,
        title.textStyle.color = title.textStyle.color,
        title.textStyle.fontStyle = title.textStyle.fontStyle,
        title.textStyle.fontWeight = title.textStyle.fontWeight,
        title.textStyle.fontFamily = title.textStyle.fontFamily,
        title.textStyle.fontSize = title.textStyle.fontSize,
        title.textStyle.lineHeight = title.textStyle.lineHeight,
        title.textStyle.width = title.textStyle.width,
        title.textStyle.height = title.textStyle.height,
        title.textStyle.textBorderColor = title.textStyle.textBorderColor,
        title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
        title.textStyle.textBorderType = title.textStyle.textBorderType,
        title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
        title.textStyle.textShadowColor = title.textStyle.textShadowColor,
        title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
        title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
        title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
        title.subtextStyle.color = title.subtextStyle.color,
        title.subtextStyle.align = title.subtextStyle.align,
        title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
        title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
        title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
        title.subtextStyle.fontSize = title.subtextStyle.fontSize,
        title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
        title.subtextStyle.width = title.subtextStyle.width,
        title.subtextStyle.height = title.subtextStyle.height,
        title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
        title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
        title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
        title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
        title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
        title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
        title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
        title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

      if(Debug) print("BarPlot 2.g")
      if(FacetRows > 1L || FacetCols > 1L) {
        p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
      }

      p1 <- e_legend_full(
        e = p1,
        legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
        legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
        legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
        legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
        legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
        legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
        legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
        legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
        legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
        legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
        legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
        legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
        legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
        legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
        legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
        legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
        legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
        legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
        legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
        legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
        legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
        legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
        legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
        legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
        legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
        legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
        legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
        legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
        legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
        legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
        legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
        legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
        legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
        legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
        legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
        legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
        legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
        legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
        legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
        legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
        legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
        legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
        legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
        legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
        legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
        legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
        legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
        legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
        legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
        legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
        legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
        legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
        legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

      return(p1)
    }
  }

  if(check2) {

    if(length(GroupVar) != 0) {
      if(!PreAgg) {
        if(any(tryCatch({class(dt[[eval(YVar)]])[1]}, error = function(x) "bla") %in% c('numeric','integer'))) {
          numvars <- unique(c(numvars, YVar))
        } else {
          byvars <- unique(c(byvars, YVar))
        }
        if(any(tryCatch({class(dt[[eval(GroupVar)]])[1]}, error = function(x) "bla") %in% c('numeric','integer'))) {
          numvars <- unique(c(numvars, GroupVar))
        } else {
          byvars <- unique(c(byvars, GroupVar))
        }
        if(!is.null(byvars)) {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        } else {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
        }
      } else {
        temp <- data.table::copy(dt)
        numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
        byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
      }

      # Transformation
      if(length(XVar) > 0L && class(temp[[XVar]])[1L] %in% c("numeric","integer")) {
        YVarTrans <- XVarTrans
      }
      if(YVarTrans != "Identity") {
        temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
      }

      p1 <- echarts4r::e_charts_(
        temp, x = GroupVar[1L],
        dispose = TRUE,
        darkMode = TRUE,
        width = Width,
        height = Height)

      p1 <- e_bar_full(
        e = p1,
        YVar,
        label = ShowLabels,
        backgroundStyle.color = backgroundStyle.color,
        backgroundStyle.opacity = backgroundStyle.opacity)

      if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
      } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
      } else {
        p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
        p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
      }
      p1 <- echarts4r::e_theme(e = p1, name = Theme)

      p1 <- e_tooltip_full(
        e = p1,
        tooltip.show = tooltip.show,
        tooltip.trigger = tooltip.trigger,
        tooltip.backgroundColor = tooltip.backgroundColor,
        tooltip.borderColor = tooltip.borderColor,
        tooltip.borderWidth = tooltip.borderWidth,
        tooltip.padding = tooltip.padding,
        tooltip.axisPointer.type = tooltip.axisPointer.type,
        tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
        tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
        tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
        tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
        tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
        tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
        tooltip.textStyle.color = tooltip.textStyle.color,
        tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
        tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
        tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
        tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
        tooltip.textStyle.width = tooltip.textStyle.width,
        tooltip.textStyle.height = tooltip.textStyle.height,
        tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
        tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
        tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
        tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
        tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
        tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
        tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

      p1 <- e_toolbox_full(
        e = p1,
        toolbox.show = toolbox.show,
        toolbox.orient = toolbox.orient,
        toolbox.itemSize = toolbox.itemSize,
        toolbox.itemGap = toolbox.itemGap,
        toolbox.top = toolbox.top,
        toolbox.left = toolbox.left,
        toolbox.right = toolbox.right,
        toolbox.bottom = toolbox.bottom,
        toolbox.width = toolbox.width,
        toolbox.heigth = toolbox.heigth,
        toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
        toolbox.feature.restore.show = toolbox.feature.restore.show,
        toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
        toolbox.feature.magicType.show = toolbox.feature.magicType.show,
        toolbox.feature.magicType.type = toolbox.feature.magicType.type,
        toolbox.feature.dataView.show = toolbox.feature.dataView.show,
        toolbox.iconStyle.color = toolbox.iconStyle.color,
        toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
        toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
        toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
        toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
        toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
        toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

      p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

      p1 <- e_x_axis_full(
        e = p1,
        serie = NULL,
        axis = "x",
        xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
        xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
        xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
        xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
        xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
        xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
        xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
        xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
        xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
        xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
        xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
        xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
        xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
        xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
        xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
        xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
        xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
        xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
        xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

      p1 <- e_y_axis_full(
        e = p1,
        serie = NULL,
        axis = "y",
        yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
        yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
        yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
        yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
        yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
        yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
        yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
        yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
        yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
        yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
        yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
        yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
        yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
        yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
        yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
        yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
        yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
        yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
        yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

      p1 <- echarts4r::e_brush(e = p1)
      p1 <- e_title_full(
        e = p1,
        title.text = title.text,
        title.subtext = title.subtext,
        title.link = title.link,
        title.sublink = title.sublink,
        title.Align = title.Align,
        title.top = title.top,
        title.left = title.left,
        title.right = title.right,
        title.bottom = title.bottom,
        title.padding = title.padding,
        title.itemGap = title.itemGap,
        title.backgroundColor = title.backgroundColor,
        title.borderColor = title.borderColor,
        title.borderWidth = title.borderWidth,
        title.borderRadius = title.borderRadius,
        title.shadowColor = title.shadowColor,
        title.shadowBlur = title.shadowBlur,
        title.shadowOffsetX = title.shadowOffsetX,
        title.shadowOffsetY = title.shadowOffsetY,
        title.textStyle.color = title.textStyle.color,
        title.textStyle.fontStyle = title.textStyle.fontStyle,
        title.textStyle.fontWeight = title.textStyle.fontWeight,
        title.textStyle.fontFamily = title.textStyle.fontFamily,
        title.textStyle.fontSize = title.textStyle.fontSize,
        title.textStyle.lineHeight = title.textStyle.lineHeight,
        title.textStyle.width = title.textStyle.width,
        title.textStyle.height = title.textStyle.height,
        title.textStyle.textBorderColor = title.textStyle.textBorderColor,
        title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
        title.textStyle.textBorderType = title.textStyle.textBorderType,
        title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
        title.textStyle.textShadowColor = title.textStyle.textShadowColor,
        title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
        title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
        title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
        title.subtextStyle.color = title.subtextStyle.color,
        title.subtextStyle.align = title.subtextStyle.align,
        title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
        title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
        title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
        title.subtextStyle.fontSize = title.subtextStyle.fontSize,
        title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
        title.subtextStyle.width = title.subtextStyle.width,
        title.subtextStyle.height = title.subtextStyle.height,
        title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
        title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
        title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
        title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
        title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
        title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
        title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
        title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

      if(FacetRows > 1L || FacetCols > 1L) {
        p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
      }

      p1 <- e_legend_full(
        e = p1,
        legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
        legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
        legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
        legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
        legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
        legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
        legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
        legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
        legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
        legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
        legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
        legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
        legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
        legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
        legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
        legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
        legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
        legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
        legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
        legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
        legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
        legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
        legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
        legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
        legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
        legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
        legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
        legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
        legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
        legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
        legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
        legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
        legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
        legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
        legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
        legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
        legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
        legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
        legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
        legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
        legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
        legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
        legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
        legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
        legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
        legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
        legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
        legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
        legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
        legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
        legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
        legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
        legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

      return(p1)
    } else {
      return(NULL)
    }
  }

  if(check3) {

    if(length(GroupVar) != 0) {
      if(!PreAgg) {
        if(any(tryCatch({class(dt[[eval(XVar)]])[1]}, error = function(x) "bla") %in% c('numeric','integer'))) {
          numvars <- unique(c(numvars, XVar))
        } else {
          byvars <- unique(c(byvars, XVar))
        }
        if(any(tryCatch({class(dt[[eval(GroupVar)]])[1]}, error = function(x) "bla") %in% c('numeric','integer'))) {
          numvars <- unique(c(numvars, GroupVar))
        } else {
          byvars <- unique(c(byvars, GroupVar))
        }
        if(!is.null(byvars)) {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        } else {
          temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
        }
      } else {
        temp <- data.table::copy(dt)
        numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
        byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
      }

      # Transformation
      if(length(XVar) > 0L && class(temp[[XVar]])[1L] %in% c("numeric","integer")) {
        YVarTrans <- XVarTrans
      }
      if(YVarTrans != "Identity") {
        temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
      }

      # Plot
      p1 <- echarts4r::e_charts_(
        temp,
        x = GroupVar[1L],
        dispose = TRUE,
        darkMode = TRUE,
        width = Width,
        height = Height)

      p1 <- e_bar_full(
        e = p1,
        YVar,
        label = ShowLabels,
        backgroundStyle.color = backgroundStyle.color,
        backgroundStyle.opacity = backgroundStyle.opacity)

      if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
      } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
        p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
      } else {
        p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
        p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
      }
      p1 <- echarts4r::e_theme(e = p1, name = Theme)

      p1 <- e_tooltip_full(
        e = p1,
        tooltip.show = tooltip.show,
        tooltip.trigger = tooltip.trigger,
        tooltip.backgroundColor = tooltip.backgroundColor,
        tooltip.borderColor = tooltip.borderColor,
        tooltip.borderWidth = tooltip.borderWidth,
        tooltip.padding = tooltip.padding,
        tooltip.axisPointer.type = tooltip.axisPointer.type,
        tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
        tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
        tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
        tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
        tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
        tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
        tooltip.textStyle.color = tooltip.textStyle.color,
        tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
        tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
        tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
        tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
        tooltip.textStyle.width = tooltip.textStyle.width,
        tooltip.textStyle.height = tooltip.textStyle.height,
        tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
        tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
        tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
        tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
        tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
        tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
        tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

      p1 <- e_toolbox_full(
        e = p1,
        toolbox.show = toolbox.show,
        toolbox.orient = toolbox.orient,
        toolbox.itemSize = toolbox.itemSize,
        toolbox.itemGap = toolbox.itemGap,
        toolbox.top = toolbox.top,
        toolbox.left = toolbox.left,
        toolbox.right = toolbox.right,
        toolbox.bottom = toolbox.bottom,
        toolbox.width = toolbox.width,
        toolbox.heigth = toolbox.heigth,
        toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
        toolbox.feature.restore.show = toolbox.feature.restore.show,
        toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
        toolbox.feature.magicType.show = toolbox.feature.magicType.show,
        toolbox.feature.magicType.type = toolbox.feature.magicType.type,
        toolbox.feature.dataView.show = toolbox.feature.dataView.show,
        toolbox.iconStyle.color = toolbox.iconStyle.color,
        toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
        toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
        toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
        toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
        toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
        toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

      p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

      p1 <- e_x_axis_full(
        e = p1,
        serie = NULL,
        axis = "x",
        xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
        xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
        xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
        xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
        xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
        xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
        xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
        xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
        xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
        xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
        xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
        xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
        xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
        xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
        xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
        xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
        xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
        xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
        xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

      p1 <- e_y_axis_full(
        e = p1,
        serie = NULL,
        axis = "y",
        yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
        yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
        yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
        yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
        yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
        yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
        yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
        yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
        yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
        yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
        yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
        yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
        yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
        yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
        yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
        yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
        yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
        yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
        yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

      p1 <- echarts4r::e_brush(e = p1)
      p1 <- e_title_full(
        e = p1,
        title.text = title.text,
        title.subtext = title.subtext,
        title.link = title.link,
        title.sublink = title.sublink,
        title.Align = title.Align,
        title.top = title.top,
        title.left = title.left,
        title.right = title.right,
        title.bottom = title.bottom,
        title.padding = title.padding,
        title.itemGap = title.itemGap,
        title.backgroundColor = title.backgroundColor,
        title.borderColor = title.borderColor,
        title.borderWidth = title.borderWidth,
        title.borderRadius = title.borderRadius,
        title.shadowColor = title.shadowColor,
        title.shadowBlur = title.shadowBlur,
        title.shadowOffsetX = title.shadowOffsetX,
        title.shadowOffsetY = title.shadowOffsetY,
        title.textStyle.color = title.textStyle.color,
        title.textStyle.fontStyle = title.textStyle.fontStyle,
        title.textStyle.fontWeight = title.textStyle.fontWeight,
        title.textStyle.fontFamily = title.textStyle.fontFamily,
        title.textStyle.fontSize = title.textStyle.fontSize,
        title.textStyle.lineHeight = title.textStyle.lineHeight,
        title.textStyle.width = title.textStyle.width,
        title.textStyle.height = title.textStyle.height,
        title.textStyle.textBorderColor = title.textStyle.textBorderColor,
        title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
        title.textStyle.textBorderType = title.textStyle.textBorderType,
        title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
        title.textStyle.textShadowColor = title.textStyle.textShadowColor,
        title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
        title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
        title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
        title.subtextStyle.color = title.subtextStyle.color,
        title.subtextStyle.align = title.subtextStyle.align,
        title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
        title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
        title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
        title.subtextStyle.fontSize = title.subtextStyle.fontSize,
        title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
        title.subtextStyle.width = title.subtextStyle.width,
        title.subtextStyle.height = title.subtextStyle.height,
        title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
        title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
        title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
        title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
        title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
        title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
        title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
        title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

      p1 <- e_legend_full(
        e = p1,
        legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
        legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
        legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
        legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
        legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
        legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
        legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
        legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
        legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
        legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
        legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
        legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
        legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
        legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
        legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
        legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
        legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
        legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
        legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
        legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
        legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
        legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
        legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
        legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
        legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
        legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
        legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
        legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
        legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
        legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
        legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
        legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
        legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
        legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
        legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
        legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
        legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
        legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
        legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
        legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
        legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
        legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
        legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
        legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
        legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
        legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
        legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
        legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
        legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
        legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
        legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
        legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
        legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

      if(FacetRows > 1L || FacetCols > 1L) p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")

      return(p1)
    } else {
      return(NULL)
    }
  }

  if(!check1 && !check2 && !check3) return(NULL)

  # Return plot
  return(p1)
}

#' @title ACF
#'
#' @description Build an autocorrelation plot by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param YVar Y-Axis variable name
#' @param DateVar Date column in data
#' @param TimeUnit Select from "hour", "day", "week", "month", "quarter", "year"
#' @param MaxLags Max lag values to test
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param ContainLabel TRUE
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#' @return plot
#' @export
ACF <- function(dt = NULL,
                YVar = NULL,
                DateVar = NULL,
                TimeUnit = NULL,
                MaxLags = 50,
                YVarTrans = "Identity",
                AggMethod = 'sum',
                Height = NULL,
                Width = NULL,
                Theme = "dark",
                ContainLabel = TRUE,
                title.text = "Autocorrelation Plot",
                title.subtext = NULL,
                title.link = NULL,
                title.sublink = NULL,
                title.Align = NULL,
                title.top = NULL,
                title.left = NULL,
                title.right = NULL,
                title.bottom = NULL,
                title.padding = NULL,
                title.itemGap = NULL,
                title.backgroundColor = NULL,
                title.borderColor = NULL,
                title.borderWidth = NULL,
                title.borderRadius = NULL,
                title.shadowColor = NULL,
                title.shadowBlur = NULL,
                title.shadowOffsetX = NULL,
                title.shadowOffsetY = NULL,
                title.textStyle.color = NULL,
                title.textStyle.fontStyle = NULL,
                title.textStyle.fontWeight = NULL,
                title.textStyle.fontFamily = "Segoe UI",
                title.textStyle.fontSize = NULL,
                title.textStyle.lineHeight = NULL,
                title.textStyle.width = NULL,
                title.textStyle.height = NULL,
                title.textStyle.textBorderColor = NULL,
                title.textStyle.textBorderWidth = NULL,
                title.textStyle.textBorderType = NULL,
                title.textStyle.textBorderDashOffset = NULL,
                title.textStyle.textShadowColor = NULL,
                title.textStyle.textShadowBlur = NULL,
                title.textStyle.textShadowOffsetX = NULL,
                title.textStyle.textShadowOffsetY = NULL,
                title.subtextStyle.color = NULL,
                title.subtextStyle.align = NULL,
                title.subtextStyle.fontStyle = NULL,
                title.subtextStyle.fontWeight = NULL,
                title.subtextStyle.fontFamily = "Segoe UI",
                title.subtextStyle.fontSize = NULL,
                title.subtextStyle.lineHeight = NULL,
                title.subtextStyle.width = NULL,
                title.subtextStyle.height = NULL,
                title.subtextStyle.textBorderColor = NULL,
                title.subtextStyle.textBorderWidth = NULL,
                title.subtextStyle.textBorderType = NULL,
                title.subtextStyle.textBorderDashOffset = NULL,
                title.subtextStyle.textShadowColor = NULL,
                title.subtextStyle.textShadowBlur = NULL,
                title.subtextStyle.textShadowOffsetX = NULL,
                title.subtextStyle.textShadowOffsetY = NULL,
                xAxis.title = NULL,
                xAxis.min = NULL,
                xAxis.max = NULL,
                xAxis.position = "bottom",
                xAxis.nameLocation = 'center',
                xAxis.axisTick.customValues = NULL,
                xAxis.nameTextStyle.color = NULL,
                xAxis.nameTextStyle.padding = 25,
                xAxis.nameTextStyle.align = NULL,
                xAxis.nameTextStyle.fontStyle = NULL,
                xAxis.nameTextStyle.fontWeight = "bolder",
                xAxis.nameTextStyle.fontSize = NULL,
                xAxis.nameTextStyle.fontFamily = "Segoe UI",
                xAxis.splitNumber = NULL,
                xAxis.axisLabel.rotate = NULL,
                xAxis.axisLabel.margin = NULL,
                xAxis.axisLabel.color = NULL,
                xAxis.axisLabel.fontStyle = NULL,
                xAxis.axisLabel.fontWeight = NULL,
                xAxis.axisLabel.fontFamily = "Segoe UI",
                xAxis.axisLabel.fontSize = 14,
                xAxis.axisLabel.align = NULL,
                xAxis.axisLabel.verticalAlign = NULL,
                xAxis.axisLabel.backgroundColor = NULL,
                xAxis.axisLabel.borderColor = NULL,
                xAxis.axisLabel.borderWidth = NULL,
                xAxis.axisLabel.borderType = NULL,
                xAxis.axisLabel.borderRadius = NULL,
                xAxis.axisLabel.padding = NULL,
                xAxis.axisLabel.shadowColor = NULL,
                xAxis.axisLabel.shadowBlur = NULL,
                xAxis.axisLabel.shadowOffsetX = NULL,
                xAxis.axisLabel.shadowOffsetY = NULL,
                xAxis.axisLabel.textBorderColor = NULL,
                xAxis.axisLabel.textBorderWidth = NULL,
                xAxis.axisLabel.textBorderType = NULL,
                xAxis.axisLabel.textShadowColor = NULL,
                xAxis.axisLabel.textShadowBlur = NULL,
                xAxis.axisLabel.textShadowOffsetX = NULL,
                xAxis.axisLabel.textShadowOffsetY = NULL,
                xAxis.axisLabel.overflow = NULL,
                yAxis.title = NULL,
                yAxis.min = NULL,
                yAxis.max = NULL,
                yAxis.position = "center",
                yAxis.nameLocation = 'center',
                yAxis.axisTick.customValues = NULL,
                yAxis.nameTextStyle.color = NULL,
                yAxis.nameTextStyle.padding = 25,
                yAxis.nameTextStyle.align = NULL,
                yAxis.nameTextStyle.fontStyle = NULL,
                yAxis.nameTextStyle.fontWeight = "bolder",
                yAxis.nameTextStyle.fontSize = NULL,
                yAxis.nameTextStyle.fontFamily = "Segoe UI",
                yAxis.splitNumber = NULL,
                yAxis.axisLabel.rotate = NULL,
                yAxis.axisLabel.margin = NULL,
                yAxis.axisLabel.color = NULL,
                yAxis.axisLabel.fontStyle = NULL,
                yAxis.axisLabel.fontWeight = NULL,
                yAxis.axisLabel.fontFamily = "Segoe UI",
                yAxis.axisLabel.fontSize = 14,
                yAxis.axisLabel.align = NULL,
                yAxis.axisLabel.verticalAlign = NULL,
                yAxis.axisLabel.backgroundColor = NULL,
                yAxis.axisLabel.borderColor = NULL,
                yAxis.axisLabel.borderWidth = NULL,
                yAxis.axisLabel.borderType = NULL,
                yAxis.axisLabel.borderRadius = NULL,
                yAxis.axisLabel.padding = NULL,
                yAxis.axisLabel.shadowColor = NULL,
                yAxis.axisLabel.shadowBlur = NULL,
                yAxis.axisLabel.shadowOffsetX = NULL,
                yAxis.axisLabel.shadowOffsetY = NULL,
                yAxis.axisLabel.textBorderColor = NULL,
                yAxis.axisLabel.textBorderWidth = NULL,
                yAxis.axisLabel.textBorderType = NULL,
                yAxis.axisLabel.textShadowColor = NULL,
                yAxis.axisLabel.textShadowBlur = NULL,
                yAxis.axisLabel.textShadowOffsetX = NULL,
                yAxis.axisLabel.textShadowOffsetY = NULL,
                yAxis.axisLabel.overflow = NULL,
                legend.show = TRUE,
                legend.type = "scroll",
                legend.selector = NULL,
                legend.icon = NULL,
                legend.align = NULL,
                legend.padding = NULL,
                legend.itemGap = NULL,
                legend.itemWidth = NULL,
                legend.orient = "vertical",
                legend.width = NULL,
                legend.height = NULL,
                legend.left = NULL,
                legend.right = 10,
                legend.top = 30,
                legend.bottom = NULL,
                legend.backgroundColor = NULL,
                legend.borderColor = NULL,
                legend.borderWidth = NULL,
                legend.borderRadius = NULL,
                legend.shadowBlur = NULL,
                legend.shadowColor = NULL,
                legend.shadowOffsetX = NULL,
                legend.shadowOffsetY = NULL,
                legend.itemStyle.color = NULL,
                legend.itemStyle.borderColor = NULL,
                legend.itemStyle.borderWidth = NULL,
                legend.itemStyle.borderType = NULL,
                legend.itemStyle.shadowBlur = NULL,
                legend.itemStyle.shadowColor = NULL,
                legend.itemStyle.shadowOffsetX = NULL,
                legend.itemStyle.shadowOffsetY = NULL,
                legend.itemStyle.opacity = NULL,
                legend.lineStyle.color = NULL,
                legend.lineStyle.width = NULL,
                legend.lineStyle.type = NULL,
                legend.lineStyle.shadowBlur = NULL,
                legend.lineStyle.shadowColor = NULL,
                legend.lineStyle.shadowOffsetX = NULL,
                legend.lineStyle.shadowOffsetY = NULL,
                legend.lineStyle.opacity = NULL,
                legend.lineStyle.inactiveColor = NULL,
                legend.lineStyle.inactiveWidth = NULL,
                legend.textStyle.color = NULL,
                legend.textStyle.fontStyle = NULL,
                legend.textStyle.fontWeight = NULL,
                legend.textStyle.fontFamily = "Segoe UI",
                legend.textStyle.fontSize = NULL,
                legend.textStyle.backgroundColor = NULL,
                legend.textStyle.borderColor = NULL,
                legend.textStyle.borderWidth = NULL,
                legend.textStyle.borderType = NULL,
                legend.textStyle.borderRadius = NULL,
                legend.textStyle.padding = NULL,
                legend.textStyle.shadowColor = NULL,
                legend.textStyle.shadowBlur = NULL,
                legend.textStyle.shadowOffsetX = NULL,
                legend.textStyle.shadowOffsetY = NULL,
                legend.textStyle.width = NULL,
                legend.textStyle.height = NULL,
                legend.textStyle.textBorderColor = NULL,
                legend.textStyle.textBorderWidth = NULL,
                legend.textStyle.textBorderType = NULL,
                legend.textStyle.textShadowColor = NULL,
                legend.textStyle.textShadowBlur = NULL,
                legend.textStyle.textShadowOffsetX = NULL,
                legend.textStyle.textShadowOffsetY = NULL,
                legend.pageTextStyle.color = NULL,
                legend.pageTextStyle.fontStyle = NULL,
                legend.pageTextStyle.fontWeight = NULL,
                legend.pageTextStyle.fontFamily = "Segoe UI",
                legend.pageTextStyle.fontSize = NULL,
                legend.pageTextStyle.lineHeight = NULL,
                legend.pageTextStyle.width = NULL,
                legend.pageTextStyle.height = NULL,
                legend.pageTextStyle.textBorderColor = NULL,
                legend.pageTextStyle.textBorderWidth = NULL,
                legend.pageTextStyle.textBorderType = NULL,
                legend.pageTextStyle.textShadowColor = NULL,
                legend.pageTextStyle.textShadowBlur = NULL,
                legend.pageTextStyle.textShadowOffsetX = NULL,
                legend.pageTextStyle.textShadowOffsetY = NULL,
                legend.emphasis.selectorLabel.show = NULL,
                legend.emphasis.selectorLabel.distance = NULL,
                legend.emphasis.selectorLabel.rotate = NULL,
                legend.emphasis.selectorLabel.color = NULL,
                legend.emphasis.selectorLabel.fontStyle = NULL,
                legend.emphasis.selectorLabel.fontWeight = NULL,
                legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                legend.emphasis.selectorLabel.fontSize = NULL,
                legend.emphasis.selectorLabel.align = NULL,
                legend.emphasis.selectorLabel.verticalAlign = NULL,
                legend.emphasis.selectorLabel.lineHeight = NULL,
                legend.emphasis.selectorLabel.backgroundColor = NULL,
                legend.emphasis.selectorLabel.borderColor = NULL,
                legend.emphasis.selectorLabel.borderWidth = NULL,
                legend.emphasis.selectorLabel.borderType = NULL,
                legend.emphasis.selectorLabel.borderRadius = NULL,
                legend.emphasis.selectorLabel.padding = NULL,
                legend.emphasis.selectorLabel.shadowColor = NULL,
                legend.emphasis.selectorLabel.shadowBlur = NULL,
                legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                legend.emphasis.selectorLabel.width = NULL,
                legend.emphasis.selectorLabel.height = NULL,
                legend.emphasis.selectorLabel.textBorderColor = NULL,
                legend.emphasis.selectorLabel.textBorderWidth = NULL,
                legend.emphasis.selectorLabel.textBorderType = NULL,
                legend.emphasis.selectorLabel.textShadowColor = NULL,
                legend.emphasis.selectorLabel.textShadowBlur = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                tooltip.show = TRUE,
                tooltip.trigger = "axis",
                tooltip.backgroundColor = NULL,
                tooltip.borderColor = NULL,
                tooltip.borderWidth = NULL,
                tooltip.padding = NULL,
                tooltip.axisPointer.type = "cross",
                tooltip.axisPointer.lineStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.color = NULL,
                tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                tooltip.axisPointer.shadowStyle.opacity = NULL,
                tooltip.textStyle.color = NULL,
                tooltip.textStyle.fontStyle = NULL,
                tooltip.textStyle.fontWeight = NULL,
                tooltip.textStyle.fontFamily = "Segoe UI",
                tooltip.textStyle.lineHeight = NULL,
                tooltip.textStyle.width = NULL,
                tooltip.textStyle.height = NULL,
                tooltip.textStyle.textBorderColor = NULL,
                tooltip.textStyle.textBorderWidth = NULL,
                tooltip.textStyle.textBorderType = NULL,
                tooltip.textStyle.textShadowColor = NULL,
                tooltip.textStyle.textShadowBlur = NULL,
                tooltip.textStyle.textShadowOffsetX = NULL,
                tooltip.textStyle.textShadowOffsetY = NULL,
                toolbox.show = TRUE,
                toolbox.orient = "horizontal",
                toolbox.itemSize = 15,
                toolbox.itemGap = 8,
                toolbox.top = NULL,
                toolbox.left = NULL,
                toolbox.right = NULL,
                toolbox.bottom = NULL,
                toolbox.width = NULL,
                toolbox.heigth = NULL,
                toolbox.feature.saveAsImage.show = TRUE,
                toolbox.feature.restore.show = TRUE,
                toolbox.feature.dataZoom.show = TRUE,
                toolbox.feature.magicType.show = TRUE,
                toolbox.feature.magicType.type = c("line", "bar", "stack"),
                toolbox.feature.dataView.show = TRUE,
                toolbox.iconStyle.color = NULL,
                toolbox.iconStyle.borderColor = NULL,
                toolbox.emphasis.iconStyle.borderColor = NULL,
                toolbox.iconStyle.shadowBlur = NULL,
                toolbox.iconStyle.shadowColor = NULL,
                toolbox.iconStyle.shadowOffsetX = NULL,
                toolbox.iconStyle.shadowOffsetY = NULL,
                Debug = FALSE) {

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  dt1 <- data.table::copy(dt)

  # Convert factor to character
  if(length(YVar) > 0L && class(dt1[[YVar]])[1L] == "factor") {
    return(NULL)
  }

  # Define Aggregation function
  if(Debug) print("Plot.ACH 1")
  aggFunc <- SummaryFunction(AggMethod)

  if(Debug) print("Plot.ACH 2")

  # Transformation
  if(YVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data
  }

  if(Debug) print("Plot.ACH 3")

  # Aggregate dt1
  dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), .SDcols = c(YVar), keyby = c(DateVar)]

  if(Debug) print("Plot.ACH 3.5")

  dt1 <- AutoLagRollStats(
    data = dt1,
    DateColumn = DateVar,
    Targets = YVar,
    TimeUnitAgg = TimeUnit,
    TimeGroups = TimeUnit,
    TimeUnit = TimeUnit,
    RollOnLag1 = TRUE,
    Type = "Lag",
    SimpleImpute = TRUE,
    Lags = seq_len(MaxLags))

  if(Debug) print("Plot.ACH 4")

  # Autocorrelation data creation
  ACF_Data <- data.table::data.table(Lag = 1:50, Cor = 0.0, `Lower 95th` = 0.0, `Upper 95th` = 0.0)
  if(Debug) print("Plot.ACH 5")
  for(i in seq_len(MaxLags)) {# i = 1
    lagCol <- names(dt1)[which(grepl(pattern = paste0("_LAG_",i,"_"), x = names(dt1)))]
    lag_test <- stats::cor.test(x = dt1[[YVar]], y = dt1[[lagCol]])
    data.table::set(ACF_Data, i = i, j = "Lag", value = i)
    data.table::set(ACF_Data, i = i, j = "Cor", value = lag_test$estimate)
    data.table::set(ACF_Data, i = i, j = "Lower 95th", value = lag_test$conf.int[1L])
    data.table::set(ACF_Data, i = i, j = "Upper 95th", value = lag_test$conf.int[2L])
  }

  if(Debug) print("Plot.ACH 6")

  # Plot
  p1 <- echarts4r::e_charts_(
    ACF_Data,
    x = "Lag",
    dispose = TRUE,
    darkMode = TRUE,
    width = Width,
    height = Height)

  if(Debug) print("Plot.ACH 7")
  p1 <- echarts4r::e_bar_(e = p1, "Cor")

  if(Debug) print("Plot.ACH 8")

  # MAX Band is not working currently so plot looks stupid with this
  # p1 <- echarts4r::e_band_(
  #   e = p1,
  #   min = "Lower 95th", max = "Upper 95th", stack = "confidence-band",
  #   areaStyle = list(list(color = "#54535387"), list(color = "#54535387"))
  # )

  # Alternative bands: just lines but they are correct
  p1 <- echarts4r::e_line_(e = p1, "Lower 95th", smooth = TRUE)
  p1 <- echarts4r::e_line_(e = p1, "Upper 95th", smooth = TRUE)

  # Extras
  if(Debug) print("Plot.ACH 10")
  p1 <- echarts4r::e_theme(e = p1, name = Theme)

  p1 <- e_tooltip_full(
    e = p1,
    tooltip.show = tooltip.show,
    tooltip.trigger = tooltip.trigger,
    tooltip.backgroundColor = tooltip.backgroundColor,
    tooltip.borderColor = tooltip.borderColor,
    tooltip.borderWidth = tooltip.borderWidth,
    tooltip.padding = tooltip.padding,
    tooltip.axisPointer.type = tooltip.axisPointer.type,
    tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
    tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
    tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
    tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
    tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
    tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
    tooltip.textStyle.color = tooltip.textStyle.color,
    tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
    tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
    tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
    tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
    tooltip.textStyle.width = tooltip.textStyle.width,
    tooltip.textStyle.height = tooltip.textStyle.height,
    tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
    tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
    tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
    tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
    tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
    tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
    tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

  p1 <- e_toolbox_full(
    e = p1,
    toolbox.show = toolbox.show,
    toolbox.orient = toolbox.orient,
    toolbox.itemSize = toolbox.itemSize,
    toolbox.itemGap = toolbox.itemGap,
    toolbox.top = toolbox.top,
    toolbox.left = toolbox.left,
    toolbox.right = toolbox.right,
    toolbox.bottom = toolbox.bottom,
    toolbox.width = toolbox.width,
    toolbox.heigth = toolbox.heigth,
    toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
    toolbox.feature.restore.show = toolbox.feature.restore.show,
    toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
    toolbox.feature.magicType.show = toolbox.feature.magicType.show,
    toolbox.feature.magicType.type = toolbox.feature.magicType.type,
    toolbox.feature.dataView.show = toolbox.feature.dataView.show,
    toolbox.iconStyle.color = toolbox.iconStyle.color,
    toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
    toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
    toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
    toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
    toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
    toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

  p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
  p1 <- e_x_axis_full(
    e = p1,
    serie = NULL,
    axis = "x",
    xAxis.title = "Lag", xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
    xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
    xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
    xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
    xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
    xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
    xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
    xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
    xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
    xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
    xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
    xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
    xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
    xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
    xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
    xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
    xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
    xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
    xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

  p1 <- e_y_axis_full(
    e = p1,
    serie = NULL,
    axis = "y",
    yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
    yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
    yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
    yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
    yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
    yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
    yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
    yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
    yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
    yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
    yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
    yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
    yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
    yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
    yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
    yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
    yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
    yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
    yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

  p1 <- echarts4r::e_brush(e = p1)
  p1 <- e_title_full(
    e = p1,
    title.text = title.text,
    title.subtext = title.subtext,
    title.link = title.link,
    title.sublink = title.sublink,
    title.Align = title.Align,
    title.top = title.top,
    title.left = title.left,
    title.right = title.right,
    title.bottom = title.bottom,
    title.padding = title.padding,
    title.itemGap = title.itemGap,
    title.backgroundColor = title.backgroundColor,
    title.borderColor = title.borderColor,
    title.borderWidth = title.borderWidth,
    title.borderRadius = title.borderRadius,
    title.shadowColor = title.shadowColor,
    title.shadowBlur = title.shadowBlur,
    title.shadowOffsetX = title.shadowOffsetX,
    title.shadowOffsetY = title.shadowOffsetY,
    title.textStyle.color = title.textStyle.color,
    title.textStyle.fontStyle = title.textStyle.fontStyle,
    title.textStyle.fontWeight = title.textStyle.fontWeight,
    title.textStyle.fontFamily = title.textStyle.fontFamily,
    title.textStyle.fontSize = title.textStyle.fontSize,
    title.textStyle.lineHeight = title.textStyle.lineHeight,
    title.textStyle.width = title.textStyle.width,
    title.textStyle.height = title.textStyle.height,
    title.textStyle.textBorderColor = title.textStyle.textBorderColor,
    title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
    title.textStyle.textBorderType = title.textStyle.textBorderType,
    title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
    title.textStyle.textShadowColor = title.textStyle.textShadowColor,
    title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
    title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
    title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
    title.subtextStyle.color = title.subtextStyle.color,
    title.subtextStyle.align = title.subtextStyle.align,
    title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
    title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
    title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
    title.subtextStyle.fontSize = title.subtextStyle.fontSize,
    title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
    title.subtextStyle.width = title.subtextStyle.width,
    title.subtextStyle.height = title.subtextStyle.height,
    title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
    title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
    title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
    title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
    title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
    title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
    title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
    title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

  return(p1)
}

#' @title PACF
#'
#' @description Build a partial autocorrelation plot by simply passing arguments to a single function
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param YVar Y-Axis variable name
#' @param DateVar Date column in data
#' @param MaxLags Max value for lags to test
#' @param TimeUnit Select from "hour", "day", "week", "month", "quarter", "year"
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height "400px"
#' @param Width "200px"
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param ContainLabel TRUE
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#' @return plot
#' @export
PACF <- function(dt = NULL,
                 YVar = NULL,
                 DateVar = NULL,
                 TimeUnit = NULL,
                 MaxLags = 50,
                 YVarTrans = "Identity",
                 AggMethod = 'sum',
                 Height = NULL,
                 Width = NULL,
                 Theme = "dark",
                 ContainLabel = TRUE,
                 title.text = "Partial Autocorrelation Plot",
                 title.subtext = NULL,
                 title.link = NULL,
                 title.sublink = NULL,
                 title.Align = NULL,
                 title.top = NULL,
                 title.left = NULL,
                 title.right = NULL,
                 title.bottom = NULL,
                 title.padding = NULL,
                 title.itemGap = NULL,
                 title.backgroundColor = NULL,
                 title.borderColor = NULL,
                 title.borderWidth = NULL,
                 title.borderRadius = NULL,
                 title.shadowColor = NULL,
                 title.shadowBlur = NULL,
                 title.shadowOffsetX = NULL,
                 title.shadowOffsetY = NULL,
                 title.textStyle.color = NULL,
                 title.textStyle.fontStyle = NULL,
                 title.textStyle.fontWeight = NULL,
                 title.textStyle.fontFamily = "Segoe UI",
                 title.textStyle.fontSize = NULL,
                 title.textStyle.lineHeight = NULL,
                 title.textStyle.width = NULL,
                 title.textStyle.height = NULL,
                 title.textStyle.textBorderColor = NULL,
                 title.textStyle.textBorderWidth = NULL,
                 title.textStyle.textBorderType = NULL,
                 title.textStyle.textBorderDashOffset = NULL,
                 title.textStyle.textShadowColor = NULL,
                 title.textStyle.textShadowBlur = NULL,
                 title.textStyle.textShadowOffsetX = NULL,
                 title.textStyle.textShadowOffsetY = NULL,
                 title.subtextStyle.color = NULL,
                 title.subtextStyle.align = NULL,
                 title.subtextStyle.fontStyle = NULL,
                 title.subtextStyle.fontWeight = NULL,
                 title.subtextStyle.fontFamily = "Segoe UI",
                 title.subtextStyle.fontSize = NULL,
                 title.subtextStyle.lineHeight = NULL,
                 title.subtextStyle.width = NULL,
                 title.subtextStyle.height = NULL,
                 title.subtextStyle.textBorderColor = NULL,
                 title.subtextStyle.textBorderWidth = NULL,
                 title.subtextStyle.textBorderType = NULL,
                 title.subtextStyle.textBorderDashOffset = NULL,
                 title.subtextStyle.textShadowColor = NULL,
                 title.subtextStyle.textShadowBlur = NULL,
                 title.subtextStyle.textShadowOffsetX = NULL,
                 title.subtextStyle.textShadowOffsetY = NULL,
                 xAxis.title = NULL,
                 xAxis.min = NULL,
                 xAxis.max = NULL,
                 xAxis.position = "bottom",
                 xAxis.nameLocation = 'center',
                 xAxis.axisTick.customValues = NULL,
                 xAxis.nameTextStyle.color = NULL,
                 xAxis.nameTextStyle.padding = 25,
                 xAxis.nameTextStyle.align = NULL,
                 xAxis.nameTextStyle.fontStyle = NULL,
                 xAxis.nameTextStyle.fontWeight = "bolder",
                 xAxis.nameTextStyle.fontSize = NULL,
                 xAxis.nameTextStyle.fontFamily = "Segoe UI",
                 xAxis.splitNumber = NULL,
                 xAxis.axisLabel.rotate = NULL,
                 xAxis.axisLabel.margin = NULL,
                 xAxis.axisLabel.color = NULL,
                 xAxis.axisLabel.fontStyle = NULL,
                 xAxis.axisLabel.fontWeight = NULL,
                 xAxis.axisLabel.fontFamily = "Segoe UI",
                 xAxis.axisLabel.fontSize = 14,
                 xAxis.axisLabel.align = NULL,
                 xAxis.axisLabel.verticalAlign = NULL,
                 xAxis.axisLabel.backgroundColor = NULL,
                 xAxis.axisLabel.borderColor = NULL,
                 xAxis.axisLabel.borderWidth = NULL,
                 xAxis.axisLabel.borderType = NULL,
                 xAxis.axisLabel.borderRadius = NULL,
                 xAxis.axisLabel.padding = NULL,
                 xAxis.axisLabel.shadowColor = NULL,
                 xAxis.axisLabel.shadowBlur = NULL,
                 xAxis.axisLabel.shadowOffsetX = NULL,
                 xAxis.axisLabel.shadowOffsetY = NULL,
                 xAxis.axisLabel.textBorderColor = NULL,
                 xAxis.axisLabel.textBorderWidth = NULL,
                 xAxis.axisLabel.textBorderType = NULL,
                 xAxis.axisLabel.textShadowColor = NULL,
                 xAxis.axisLabel.textShadowBlur = NULL,
                 xAxis.axisLabel.textShadowOffsetX = NULL,
                 xAxis.axisLabel.textShadowOffsetY = NULL,
                 xAxis.axisLabel.overflow = NULL,
                 yAxis.title = NULL,
                 yAxis.min = NULL,
                 yAxis.max = NULL,
                 yAxis.position = "center",
                 yAxis.nameLocation = 'center',
                 yAxis.axisTick.customValues = NULL,
                 yAxis.nameTextStyle.color = NULL,
                 yAxis.nameTextStyle.padding = 25,
                 yAxis.nameTextStyle.align = NULL,
                 yAxis.nameTextStyle.fontStyle = NULL,
                 yAxis.nameTextStyle.fontWeight = "bolder",
                 yAxis.nameTextStyle.fontSize = NULL,
                 yAxis.nameTextStyle.fontFamily = "Segoe UI",
                 yAxis.splitNumber = NULL,
                 yAxis.axisLabel.rotate = NULL,
                 yAxis.axisLabel.margin = NULL,
                 yAxis.axisLabel.color = NULL,
                 yAxis.axisLabel.fontStyle = NULL,
                 yAxis.axisLabel.fontWeight = NULL,
                 yAxis.axisLabel.fontFamily = "Segoe UI",
                 yAxis.axisLabel.fontSize = 14,
                 yAxis.axisLabel.align = NULL,
                 yAxis.axisLabel.verticalAlign = NULL,
                 yAxis.axisLabel.backgroundColor = NULL,
                 yAxis.axisLabel.borderColor = NULL,
                 yAxis.axisLabel.borderWidth = NULL,
                 yAxis.axisLabel.borderType = NULL,
                 yAxis.axisLabel.borderRadius = NULL,
                 yAxis.axisLabel.padding = NULL,
                 yAxis.axisLabel.shadowColor = NULL,
                 yAxis.axisLabel.shadowBlur = NULL,
                 yAxis.axisLabel.shadowOffsetX = NULL,
                 yAxis.axisLabel.shadowOffsetY = NULL,
                 yAxis.axisLabel.textBorderColor = NULL,
                 yAxis.axisLabel.textBorderWidth = NULL,
                 yAxis.axisLabel.textBorderType = NULL,
                 yAxis.axisLabel.textShadowColor = NULL,
                 yAxis.axisLabel.textShadowBlur = NULL,
                 yAxis.axisLabel.textShadowOffsetX = NULL,
                 yAxis.axisLabel.textShadowOffsetY = NULL,
                 yAxis.axisLabel.overflow = NULL,
                 legend.show = TRUE,
                 legend.type = "scroll",
                 legend.selector = NULL,
                 legend.icon = NULL,
                 legend.align = NULL,
                 legend.padding = NULL,
                 legend.itemGap = NULL,
                 legend.itemWidth = NULL,
                 legend.orient = "vertical",
                 legend.width = NULL,
                 legend.height = NULL,
                 legend.left = NULL,
                 legend.right = 10,
                 legend.top = 30,
                 legend.bottom = NULL,
                 legend.backgroundColor = NULL,
                 legend.borderColor = NULL,
                 legend.borderWidth = NULL,
                 legend.borderRadius = NULL,
                 legend.shadowBlur = NULL,
                 legend.shadowColor = NULL,
                 legend.shadowOffsetX = NULL,
                 legend.shadowOffsetY = NULL,
                 legend.itemStyle.color = NULL,
                 legend.itemStyle.borderColor = NULL,
                 legend.itemStyle.borderWidth = NULL,
                 legend.itemStyle.borderType = NULL,
                 legend.itemStyle.shadowBlur = NULL,
                 legend.itemStyle.shadowColor = NULL,
                 legend.itemStyle.shadowOffsetX = NULL,
                 legend.itemStyle.shadowOffsetY = NULL,
                 legend.itemStyle.opacity = NULL,
                 legend.lineStyle.color = NULL,
                 legend.lineStyle.width = NULL,
                 legend.lineStyle.type = NULL,
                 legend.lineStyle.shadowBlur = NULL,
                 legend.lineStyle.shadowColor = NULL,
                 legend.lineStyle.shadowOffsetX = NULL,
                 legend.lineStyle.shadowOffsetY = NULL,
                 legend.lineStyle.opacity = NULL,
                 legend.lineStyle.inactiveColor = NULL,
                 legend.lineStyle.inactiveWidth = NULL,
                 legend.textStyle.color = NULL,
                 legend.textStyle.fontStyle = NULL,
                 legend.textStyle.fontWeight = NULL,
                 legend.textStyle.fontFamily = "Segoe UI",
                 legend.textStyle.fontSize = NULL,
                 legend.textStyle.backgroundColor = NULL,
                 legend.textStyle.borderColor = NULL,
                 legend.textStyle.borderWidth = NULL,
                 legend.textStyle.borderType = NULL,
                 legend.textStyle.borderRadius = NULL,
                 legend.textStyle.padding = NULL,
                 legend.textStyle.shadowColor = NULL,
                 legend.textStyle.shadowBlur = NULL,
                 legend.textStyle.shadowOffsetX = NULL,
                 legend.textStyle.shadowOffsetY = NULL,
                 legend.textStyle.width = NULL,
                 legend.textStyle.height = NULL,
                 legend.textStyle.textBorderColor = NULL,
                 legend.textStyle.textBorderWidth = NULL,
                 legend.textStyle.textBorderType = NULL,
                 legend.textStyle.textShadowColor = NULL,
                 legend.textStyle.textShadowBlur = NULL,
                 legend.textStyle.textShadowOffsetX = NULL,
                 legend.textStyle.textShadowOffsetY = NULL,
                 legend.pageTextStyle.color = NULL,
                 legend.pageTextStyle.fontStyle = NULL,
                 legend.pageTextStyle.fontWeight = NULL,
                 legend.pageTextStyle.fontFamily = "Segoe UI",
                 legend.pageTextStyle.fontSize = NULL,
                 legend.pageTextStyle.lineHeight = NULL,
                 legend.pageTextStyle.width = NULL,
                 legend.pageTextStyle.height = NULL,
                 legend.pageTextStyle.textBorderColor = NULL,
                 legend.pageTextStyle.textBorderWidth = NULL,
                 legend.pageTextStyle.textBorderType = NULL,
                 legend.pageTextStyle.textShadowColor = NULL,
                 legend.pageTextStyle.textShadowBlur = NULL,
                 legend.pageTextStyle.textShadowOffsetX = NULL,
                 legend.pageTextStyle.textShadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.show = NULL,
                 legend.emphasis.selectorLabel.distance = NULL,
                 legend.emphasis.selectorLabel.rotate = NULL,
                 legend.emphasis.selectorLabel.color = NULL,
                 legend.emphasis.selectorLabel.fontStyle = NULL,
                 legend.emphasis.selectorLabel.fontWeight = NULL,
                 legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                 legend.emphasis.selectorLabel.fontSize = NULL,
                 legend.emphasis.selectorLabel.align = NULL,
                 legend.emphasis.selectorLabel.verticalAlign = NULL,
                 legend.emphasis.selectorLabel.lineHeight = NULL,
                 legend.emphasis.selectorLabel.backgroundColor = NULL,
                 legend.emphasis.selectorLabel.borderColor = NULL,
                 legend.emphasis.selectorLabel.borderWidth = NULL,
                 legend.emphasis.selectorLabel.borderType = NULL,
                 legend.emphasis.selectorLabel.borderRadius = NULL,
                 legend.emphasis.selectorLabel.padding = NULL,
                 legend.emphasis.selectorLabel.shadowColor = NULL,
                 legend.emphasis.selectorLabel.shadowBlur = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                 legend.emphasis.selectorLabel.width = NULL,
                 legend.emphasis.selectorLabel.height = NULL,
                 legend.emphasis.selectorLabel.textBorderColor = NULL,
                 legend.emphasis.selectorLabel.textBorderWidth = NULL,
                 legend.emphasis.selectorLabel.textBorderType = NULL,
                 legend.emphasis.selectorLabel.textShadowColor = NULL,
                 legend.emphasis.selectorLabel.textShadowBlur = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                 legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                 tooltip.show = TRUE,
                 tooltip.trigger = "axis",
                 tooltip.backgroundColor = NULL,
                 tooltip.borderColor = NULL,
                 tooltip.borderWidth = NULL,
                 tooltip.padding = NULL,
                 tooltip.axisPointer.type = "cross",
                 tooltip.axisPointer.lineStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.color = NULL,
                 tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                 tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                 tooltip.axisPointer.shadowStyle.opacity = NULL,
                 tooltip.textStyle.color = NULL,
                 tooltip.textStyle.fontStyle = NULL,
                 tooltip.textStyle.fontWeight = NULL,
                 tooltip.textStyle.fontFamily = "Segoe UI",
                 tooltip.textStyle.lineHeight = NULL,
                 tooltip.textStyle.width = NULL,
                 tooltip.textStyle.height = NULL,
                 tooltip.textStyle.textBorderColor = NULL,
                 tooltip.textStyle.textBorderWidth = NULL,
                 tooltip.textStyle.textBorderType = NULL,
                 tooltip.textStyle.textShadowColor = NULL,
                 tooltip.textStyle.textShadowBlur = NULL,
                 tooltip.textStyle.textShadowOffsetX = NULL,
                 tooltip.textStyle.textShadowOffsetY = NULL,
                 toolbox.show = TRUE,
                 toolbox.orient = "horizontal",
                 toolbox.itemSize = 15,
                 toolbox.itemGap = 8,
                 toolbox.top = NULL,
                 toolbox.left = NULL,
                 toolbox.right = NULL,
                 toolbox.bottom = NULL,
                 toolbox.width = NULL,
                 toolbox.heigth = NULL,
                 toolbox.feature.saveAsImage.show = TRUE,
                 toolbox.feature.restore.show = TRUE,
                 toolbox.feature.dataZoom.show = TRUE,
                 toolbox.feature.magicType.show = TRUE,
                 toolbox.feature.magicType.type = c("line", "bar", "stack"),
                 toolbox.feature.dataView.show = TRUE,
                 toolbox.iconStyle.color = NULL,
                 toolbox.iconStyle.borderColor = NULL,
                 toolbox.emphasis.iconStyle.borderColor = NULL,
                 toolbox.iconStyle.shadowBlur = NULL,
                 toolbox.iconStyle.shadowColor = NULL,
                 toolbox.iconStyle.shadowOffsetX = NULL,
                 toolbox.iconStyle.shadowOffsetY = NULL,
                 Debug = FALSE) {

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  dt1 <- data.table::copy(dt)

  if(grepl(" ", YVar)) {
    data.table::setnames(x = dt1, old = YVar, new = gsub(pattern = " ", replacement = ".", x = YVar))
    YVar <- gsub(pattern = " ", replacement = ".", x = YVar)
  }

  # Convert factor to character
  if(length(YVar) > 0L && class(dt1[[YVar]])[1L] == "factor") {
    return(NULL)
  }

  # Define Aggregation function
  if(Debug) print("Plot.PACH 1")
  aggFunc <- SummaryFunction(AggMethod)

  if(Debug) print("Plot.PACH 2")

  # Transformation
  if(YVarTrans != "Identity") {
    dt1 <- AutoTransformationCreate(data = dt1, ColumnNames = YVar, Methods = YVarTrans)$Data
  }

  if(Debug) print("Plot.PACH 3")

  # Aggregate dt1
  dt1 <- dt1[, lapply(.SD, noquote(aggFunc)), .SDcols = c(YVar), keyby = c(DateVar)]

  if(Debug) print("Plot.PACH 3.5")

  dt1 <- AutoLagRollStats(
    data = dt1,
    DateColumn = DateVar,
    Targets = YVar,
    TimeUnitAgg = TimeUnit,
    TimeGroups = TimeUnit,
    TimeUnit = TimeUnit,
    RollOnLag1 = TRUE,
    Type = "Lag",
    SimpleImpute = TRUE,
    Lags = seq_len(MaxLags))

  if(Debug) print("Plot.PACH 4")

  # Autocorrelation data creation
  PACF_Data <- data.table::data.table(Lag = 1:50, Cor = 0.0, `Lower 95th` = 0.0, `Upper 95th` = 0.0)
  LagCols <- c()
  if(Debug) print("Plot.ACH 5")
  for(i in seq_len(MaxLags)) {# i = 1L  i = 2L
    LagCols[i] <- names(dt1)[which(grepl(pattern = paste0("_LAG_",i,"_"), x = names(dt1)))]
    if(i == 1L) {
      lag_test <- stats::cor.test(x = dt1[[YVar]], y = dt1[[LagCols]])
      data.table::set(PACF_Data, i = i, j = "Lag", value = i)
      data.table::set(PACF_Data, i = i, j = "Cor", value = lag_test$estimate)
      data.table::set(PACF_Data, i = i, j = "Lower 95th", value = lag_test$conf.int[1L])
      data.table::set(PACF_Data, i = i, j = "Upper 95th", value = lag_test$conf.int[2L])
    } else {
      x <- as.vector(stats::lm(formula = stats::as.formula(paste0(YVar, " ~ ", paste0(LagCols, collapse = " + "))), data = dt1)$residuals)
      lag_test <- stats::cor.test(x = x, y = dt1[[LagCols[i]]])
      data.table::set(PACF_Data, i = i, j = "Lag", value = i)
      data.table::set(PACF_Data, i = i, j = "Cor", value = lag_test$estimate)
      data.table::set(PACF_Data, i = i, j = "Lower 95th", value = lag_test$conf.int[1L])
      data.table::set(PACF_Data, i = i, j = "Upper 95th", value = lag_test$conf.int[2L])
    }
  }

  if(Debug) print("Plot.PACH 6")

  # Plot
  p1 <- echarts4r::e_charts_(
    PACF_Data,
    x = "Lag",
    dispose = TRUE,
    darkMode = TRUE,
    width = Width,
    height = Height)

  if(Debug) print("Plot.PACH 7")
  p1 <- echarts4r::e_bar_(e = p1, "Cor")

  if(Debug) print("Plot.PACH 8")

  p1 <- echarts4r::e_line_(e = p1, "Lower 95th", smooth = TRUE)

  if(Debug) print("Plot.PACH 9")

  p1 <- echarts4r::e_line_(e = p1, "Upper 95th", smooth = TRUE)

  # Extras
  if(Debug) print("Plot.PACH 10")
  p1 <- echarts4r::e_theme(e = p1, name = Theme)

  p1 <- e_tooltip_full(
    e = p1,
    tooltip.show = tooltip.show,
    tooltip.trigger = tooltip.trigger,
    tooltip.backgroundColor = tooltip.backgroundColor,
    tooltip.borderColor = tooltip.borderColor,
    tooltip.borderWidth = tooltip.borderWidth,
    tooltip.padding = tooltip.padding,
    tooltip.axisPointer.type = tooltip.axisPointer.type,
    tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
    tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
    tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
    tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
    tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
    tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
    tooltip.textStyle.color = tooltip.textStyle.color,
    tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
    tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
    tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
    tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
    tooltip.textStyle.width = tooltip.textStyle.width,
    tooltip.textStyle.height = tooltip.textStyle.height,
    tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
    tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
    tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
    tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
    tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
    tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
    tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

  p1 <- e_toolbox_full(
    e = p1,
    toolbox.show = toolbox.show,
    toolbox.orient = toolbox.orient,
    toolbox.itemSize = toolbox.itemSize,
    toolbox.itemGap = toolbox.itemGap,
    toolbox.top = toolbox.top,
    toolbox.left = toolbox.left,
    toolbox.right = toolbox.right,
    toolbox.bottom = toolbox.bottom,
    toolbox.width = toolbox.width,
    toolbox.heigth = toolbox.heigth,
    toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
    toolbox.feature.restore.show = toolbox.feature.restore.show,
    toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
    toolbox.feature.magicType.show = toolbox.feature.magicType.show,
    toolbox.feature.magicType.type = toolbox.feature.magicType.type,
    toolbox.feature.dataView.show = toolbox.feature.dataView.show,
    toolbox.iconStyle.color = toolbox.iconStyle.color,
    toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
    toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
    toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
    toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
    toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
    toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

  p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")
  p1 <- e_x_axis_full(
    e = p1,
    serie = NULL,
    axis = "x",
    xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else "Lags", xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
    xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
    xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
    xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
    xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
    xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
    xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
    xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
    xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
    xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
    xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
    xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
    xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
    xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
    xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
    xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
    xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
    xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
    xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

  p1 <- e_y_axis_full(
    e = p1,
    serie = NULL,
    axis = "y",
    yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
    yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
    yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
    yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
    yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
    yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
    yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
    yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
    yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
    yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
    yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
    yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
    yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
    yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
    yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
    yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
    yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
    yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
    yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

  p1 <- echarts4r::e_brush(e = p1)
  p1 <- e_title_full(
    e = p1,
    title.text = title.text,
    title.subtext = title.subtext,
    title.link = title.link,
    title.sublink = title.sublink,
    title.Align = title.Align,
    title.top = title.top,
    title.left = title.left,
    title.right = title.right,
    title.bottom = title.bottom,
    title.padding = title.padding,
    title.itemGap = title.itemGap,
    title.backgroundColor = title.backgroundColor,
    title.borderColor = title.borderColor,
    title.borderWidth = title.borderWidth,
    title.borderRadius = title.borderRadius,
    title.shadowColor = title.shadowColor,
    title.shadowBlur = title.shadowBlur,
    title.shadowOffsetX = title.shadowOffsetX,
    title.shadowOffsetY = title.shadowOffsetY,
    title.textStyle.color = title.textStyle.color,
    title.textStyle.fontStyle = title.textStyle.fontStyle,
    title.textStyle.fontWeight = title.textStyle.fontWeight,
    title.textStyle.fontFamily = title.textStyle.fontFamily,
    title.textStyle.fontSize = title.textStyle.fontSize,
    title.textStyle.lineHeight = title.textStyle.lineHeight,
    title.textStyle.width = title.textStyle.width,
    title.textStyle.height = title.textStyle.height,
    title.textStyle.textBorderColor = title.textStyle.textBorderColor,
    title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
    title.textStyle.textBorderType = title.textStyle.textBorderType,
    title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
    title.textStyle.textShadowColor = title.textStyle.textShadowColor,
    title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
    title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
    title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
    title.subtextStyle.color = title.subtextStyle.color,
    title.subtextStyle.align = title.subtextStyle.align,
    title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
    title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
    title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
    title.subtextStyle.fontSize = title.subtextStyle.fontSize,
    title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
    title.subtextStyle.width = title.subtextStyle.width,
    title.subtextStyle.height = title.subtextStyle.height,
    title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
    title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
    title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
    title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
    title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
    title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
    title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
    title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

  return(p1)
}

#' @title StackedBar
#'
#' @description Build a stacked bar plot vs a grouped bar plot
#'
#' @family Standard Plots
#'
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param GroupVar Column name of Group Variable for distinct colored histograms by group levels
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param AggMethod Choose from 'mean', 'sum', 'sd', and 'median'
#' @param Height NULL
#' @param Width NULL
#' @param ShowLabels logical
#' @param Theme "auritus","azul","bee-inspired","blue","caravan","carp","chalk","cool","dark-bold","dark","eduardo", #' "essos","forest","fresh-cut","fruit","gray","green","halloween","helianthus","infographic","inspired", #' "jazz","london","dark","macarons","macarons2","mint","purple-passion","red-velvet","red","roma","royal", #' "sakura","shine","tech-blue","vintage","walden","wef","weforum","westeros","wonderland"
#' @param TimeLine logical
#' @param MouseScroll logical, zoom via mouse scroll
#' @param ContainLabel TRUE
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 10000)
#'
#' # Echarts Stacked Bar Chart
#' AutoPlots::StackedBar(
#'   dt = data,
#'   PreAgg = TRUE,
#'   XVar = "Factor_1",
#'   YVar = "Adrian",
#'   GroupVar = "Factor_2",
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   AggMethod = 'mean',
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   TimeLine = FALSE,
#'   ContainLabel = TRUE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
StackedBar <- function(dt = NULL,
                       PreAgg = TRUE,
                       XVar = NULL,
                       YVar = NULL,
                       GroupVar = NULL,
                       YVarTrans = "Identity",
                       XVarTrans = "Identity",
                       FacetRows = 1,
                       FacetCols = 1,
                       FacetLevels = NULL,
                       AggMethod = 'mean',
                       Height = NULL,
                       Width = NULL,
                       ShowLabels = FALSE,
                       Theme = "dark",
                       MouseScroll = FALSE,
                       TimeLine = FALSE,
                       ContainLabel = TRUE,
                       title.text = "Stacked Bar Plot",
                       title.subtext = NULL,
                       title.link = NULL,
                       title.sublink = NULL,
                       title.Align = NULL,
                       title.top = NULL,
                       title.left = NULL,
                       title.right = NULL,
                       title.bottom = NULL,
                       title.padding = NULL,
                       title.itemGap = NULL,
                       title.backgroundColor = NULL,
                       title.borderColor = NULL,
                       title.borderWidth = NULL,
                       title.borderRadius = NULL,
                       title.shadowColor = NULL,
                       title.shadowBlur = NULL,
                       title.shadowOffsetX = NULL,
                       title.shadowOffsetY = NULL,
                       title.textStyle.color = NULL,
                       title.textStyle.fontStyle = NULL,
                       title.textStyle.fontWeight = NULL,
                       title.textStyle.fontFamily = "Segoe UI",
                       title.textStyle.fontSize = NULL,
                       title.textStyle.lineHeight = NULL,
                       title.textStyle.width = NULL,
                       title.textStyle.height = NULL,
                       title.textStyle.textBorderColor = NULL,
                       title.textStyle.textBorderWidth = NULL,
                       title.textStyle.textBorderType = NULL,
                       title.textStyle.textBorderDashOffset = NULL,
                       title.textStyle.textShadowColor = NULL,
                       title.textStyle.textShadowBlur = NULL,
                       title.textStyle.textShadowOffsetX = NULL,
                       title.textStyle.textShadowOffsetY = NULL,
                       title.subtextStyle.color = NULL,
                       title.subtextStyle.align = NULL,
                       title.subtextStyle.fontStyle = NULL,
                       title.subtextStyle.fontWeight = NULL,
                       title.subtextStyle.fontFamily = "Segoe UI",
                       title.subtextStyle.fontSize = NULL,
                       title.subtextStyle.lineHeight = NULL,
                       title.subtextStyle.width = NULL,
                       title.subtextStyle.height = NULL,
                       title.subtextStyle.textBorderColor = NULL,
                       title.subtextStyle.textBorderWidth = NULL,
                       title.subtextStyle.textBorderType = NULL,
                       title.subtextStyle.textBorderDashOffset = NULL,
                       title.subtextStyle.textShadowColor = NULL,
                       title.subtextStyle.textShadowBlur = NULL,
                       title.subtextStyle.textShadowOffsetX = NULL,
                       title.subtextStyle.textShadowOffsetY = NULL,
                       xAxis.title = NULL,
                       xAxis.min = NULL,
                       xAxis.max = NULL,
                       xAxis.position = "bottom",
                       xAxis.nameLocation = 'center',
                       xAxis.axisTick.customValues = NULL,
                       xAxis.nameTextStyle.color = NULL,
                       xAxis.nameTextStyle.padding = 25,
                       xAxis.nameTextStyle.align = NULL,
                       xAxis.nameTextStyle.fontStyle = NULL,
                       xAxis.nameTextStyle.fontWeight = "bolder",
                       xAxis.nameTextStyle.fontSize = NULL,
                       xAxis.nameTextStyle.fontFamily = "Segoe UI",
                       xAxis.splitNumber = NULL,
                       xAxis.axisLabel.rotate = NULL,
                       xAxis.axisLabel.margin = NULL,
                       xAxis.axisLabel.color = NULL,
                       xAxis.axisLabel.fontStyle = NULL,
                       xAxis.axisLabel.fontWeight = NULL,
                       xAxis.axisLabel.fontFamily = "Segoe UI",
                       xAxis.axisLabel.fontSize = 14,
                       xAxis.axisLabel.align = NULL,
                       xAxis.axisLabel.verticalAlign = NULL,
                       xAxis.axisLabel.backgroundColor = NULL,
                       xAxis.axisLabel.borderColor = NULL,
                       xAxis.axisLabel.borderWidth = NULL,
                       xAxis.axisLabel.borderType = NULL,
                       xAxis.axisLabel.borderRadius = NULL,
                       xAxis.axisLabel.padding = NULL,
                       xAxis.axisLabel.shadowColor = NULL,
                       xAxis.axisLabel.shadowBlur = NULL,
                       xAxis.axisLabel.shadowOffsetX = NULL,
                       xAxis.axisLabel.shadowOffsetY = NULL,
                       xAxis.axisLabel.textBorderColor = NULL,
                       xAxis.axisLabel.textBorderWidth = NULL,
                       xAxis.axisLabel.textBorderType = NULL,
                       xAxis.axisLabel.textShadowColor = NULL,
                       xAxis.axisLabel.textShadowBlur = NULL,
                       xAxis.axisLabel.textShadowOffsetX = NULL,
                       xAxis.axisLabel.textShadowOffsetY = NULL,
                       xAxis.axisLabel.overflow = NULL,
                       yAxis.title = NULL,
                       yAxis.min = NULL,
                       yAxis.max = NULL,
                       yAxis.position = "center",
                       yAxis.nameLocation = 'center',
                       yAxis.axisTick.customValues = NULL,
                       yAxis.nameTextStyle.color = NULL,
                       yAxis.nameTextStyle.padding = 25,
                       yAxis.nameTextStyle.align = NULL,
                       yAxis.nameTextStyle.fontStyle = NULL,
                       yAxis.nameTextStyle.fontWeight = "bolder",
                       yAxis.nameTextStyle.fontSize = NULL,
                       yAxis.nameTextStyle.fontFamily = "Segoe UI",
                       yAxis.splitNumber = NULL,
                       yAxis.axisLabel.rotate = NULL,
                       yAxis.axisLabel.margin = NULL,
                       yAxis.axisLabel.color = NULL,
                       yAxis.axisLabel.fontStyle = NULL,
                       yAxis.axisLabel.fontWeight = NULL,
                       yAxis.axisLabel.fontFamily = "Segoe UI",
                       yAxis.axisLabel.fontSize = 14,
                       yAxis.axisLabel.align = NULL,
                       yAxis.axisLabel.verticalAlign = NULL,
                       yAxis.axisLabel.backgroundColor = NULL,
                       yAxis.axisLabel.borderColor = NULL,
                       yAxis.axisLabel.borderWidth = NULL,
                       yAxis.axisLabel.borderType = NULL,
                       yAxis.axisLabel.borderRadius = NULL,
                       yAxis.axisLabel.padding = NULL,
                       yAxis.axisLabel.shadowColor = NULL,
                       yAxis.axisLabel.shadowBlur = NULL,
                       yAxis.axisLabel.shadowOffsetX = NULL,
                       yAxis.axisLabel.shadowOffsetY = NULL,
                       yAxis.axisLabel.textBorderColor = NULL,
                       yAxis.axisLabel.textBorderWidth = NULL,
                       yAxis.axisLabel.textBorderType = NULL,
                       yAxis.axisLabel.textShadowColor = NULL,
                       yAxis.axisLabel.textShadowBlur = NULL,
                       yAxis.axisLabel.textShadowOffsetX = NULL,
                       yAxis.axisLabel.textShadowOffsetY = NULL,
                       yAxis.axisLabel.overflow = NULL,
                       legend.show = TRUE,
                       legend.type = "scroll",
                       legend.selector = NULL,
                       legend.icon = NULL,
                       legend.align = NULL,
                       legend.padding = NULL,
                       legend.itemGap = NULL,
                       legend.itemWidth = NULL,
                       legend.orient = "vertical",
                       legend.width = NULL,
                       legend.height = NULL,
                       legend.left = NULL,
                       legend.right = 10,
                       legend.top = 30,
                       legend.bottom = NULL,
                       legend.backgroundColor = NULL,
                       legend.borderColor = NULL,
                       legend.borderWidth = NULL,
                       legend.borderRadius = NULL,
                       legend.shadowBlur = NULL,
                       legend.shadowColor = NULL,
                       legend.shadowOffsetX = NULL,
                       legend.shadowOffsetY = NULL,
                       legend.itemStyle.color = NULL,
                       legend.itemStyle.borderColor = NULL,
                       legend.itemStyle.borderWidth = NULL,
                       legend.itemStyle.borderType = NULL,
                       legend.itemStyle.shadowBlur = NULL,
                       legend.itemStyle.shadowColor = NULL,
                       legend.itemStyle.shadowOffsetX = NULL,
                       legend.itemStyle.shadowOffsetY = NULL,
                       legend.itemStyle.opacity = NULL,
                       legend.lineStyle.color = NULL,
                       legend.lineStyle.width = NULL,
                       legend.lineStyle.type = NULL,
                       legend.lineStyle.shadowBlur = NULL,
                       legend.lineStyle.shadowColor = NULL,
                       legend.lineStyle.shadowOffsetX = NULL,
                       legend.lineStyle.shadowOffsetY = NULL,
                       legend.lineStyle.opacity = NULL,
                       legend.lineStyle.inactiveColor = NULL,
                       legend.lineStyle.inactiveWidth = NULL,
                       legend.textStyle.color = NULL,
                       legend.textStyle.fontStyle = NULL,
                       legend.textStyle.fontWeight = NULL,
                       legend.textStyle.fontFamily = "Segoe UI",
                       legend.textStyle.fontSize = NULL,
                       legend.textStyle.backgroundColor = NULL,
                       legend.textStyle.borderColor = NULL,
                       legend.textStyle.borderWidth = NULL,
                       legend.textStyle.borderType = NULL,
                       legend.textStyle.borderRadius = NULL,
                       legend.textStyle.padding = NULL,
                       legend.textStyle.shadowColor = NULL,
                       legend.textStyle.shadowBlur = NULL,
                       legend.textStyle.shadowOffsetX = NULL,
                       legend.textStyle.shadowOffsetY = NULL,
                       legend.textStyle.width = NULL,
                       legend.textStyle.height = NULL,
                       legend.textStyle.textBorderColor = NULL,
                       legend.textStyle.textBorderWidth = NULL,
                       legend.textStyle.textBorderType = NULL,
                       legend.textStyle.textShadowColor = NULL,
                       legend.textStyle.textShadowBlur = NULL,
                       legend.textStyle.textShadowOffsetX = NULL,
                       legend.textStyle.textShadowOffsetY = NULL,
                       legend.pageTextStyle.color = NULL,
                       legend.pageTextStyle.fontStyle = NULL,
                       legend.pageTextStyle.fontWeight = NULL,
                       legend.pageTextStyle.fontFamily = "Segoe UI",
                       legend.pageTextStyle.fontSize = NULL,
                       legend.pageTextStyle.lineHeight = NULL,
                       legend.pageTextStyle.width = NULL,
                       legend.pageTextStyle.height = NULL,
                       legend.pageTextStyle.textBorderColor = NULL,
                       legend.pageTextStyle.textBorderWidth = NULL,
                       legend.pageTextStyle.textBorderType = NULL,
                       legend.pageTextStyle.textShadowColor = NULL,
                       legend.pageTextStyle.textShadowBlur = NULL,
                       legend.pageTextStyle.textShadowOffsetX = NULL,
                       legend.pageTextStyle.textShadowOffsetY = NULL,
                       legend.emphasis.selectorLabel.show = NULL,
                       legend.emphasis.selectorLabel.distance = NULL,
                       legend.emphasis.selectorLabel.rotate = NULL,
                       legend.emphasis.selectorLabel.color = NULL,
                       legend.emphasis.selectorLabel.fontStyle = NULL,
                       legend.emphasis.selectorLabel.fontWeight = NULL,
                       legend.emphasis.selectorLabel.fontFamily = "Segoe UI",
                       legend.emphasis.selectorLabel.fontSize = NULL,
                       legend.emphasis.selectorLabel.align = NULL,
                       legend.emphasis.selectorLabel.verticalAlign = NULL,
                       legend.emphasis.selectorLabel.lineHeight = NULL,
                       legend.emphasis.selectorLabel.backgroundColor = NULL,
                       legend.emphasis.selectorLabel.borderColor = NULL,
                       legend.emphasis.selectorLabel.borderWidth = NULL,
                       legend.emphasis.selectorLabel.borderType = NULL,
                       legend.emphasis.selectorLabel.borderRadius = NULL,
                       legend.emphasis.selectorLabel.padding = NULL,
                       legend.emphasis.selectorLabel.shadowColor = NULL,
                       legend.emphasis.selectorLabel.shadowBlur = NULL,
                       legend.emphasis.selectorLabel.shadowOffsetX = NULL,
                       legend.emphasis.selectorLabel.shadowOffsetY = NULL,
                       legend.emphasis.selectorLabel.width = NULL,
                       legend.emphasis.selectorLabel.height = NULL,
                       legend.emphasis.selectorLabel.textBorderColor = NULL,
                       legend.emphasis.selectorLabel.textBorderWidth = NULL,
                       legend.emphasis.selectorLabel.textBorderType = NULL,
                       legend.emphasis.selectorLabel.textShadowColor = NULL,
                       legend.emphasis.selectorLabel.textShadowBlur = NULL,
                       legend.emphasis.selectorLabel.textShadowOffsetX = NULL,
                       legend.emphasis.selectorLabel.textShadowOffsetY = NULL,
                       tooltip.show = TRUE,
                       tooltip.trigger = "axis",
                       tooltip.backgroundColor = NULL,
                       tooltip.borderColor = NULL,
                       tooltip.borderWidth = NULL,
                       tooltip.padding = NULL,
                       tooltip.axisPointer.type = "cross",
                       tooltip.axisPointer.lineStyle.color = NULL,
                       tooltip.axisPointer.shadowStyle.color = NULL,
                       tooltip.axisPointer.shadowStyle.shadowBlur = NULL,
                       tooltip.axisPointer.shadowStyle.shadowOffsetX = NULL,
                       tooltip.axisPointer.shadowStyle.shadowOffsetY = NULL,
                       tooltip.axisPointer.shadowStyle.opacity = NULL,
                       tooltip.textStyle.color = NULL,
                       tooltip.textStyle.fontStyle = NULL,
                       tooltip.textStyle.fontWeight = NULL,
                       tooltip.textStyle.fontFamily = "Segoe UI",
                       tooltip.textStyle.lineHeight = NULL,
                       tooltip.textStyle.width = NULL,
                       tooltip.textStyle.height = NULL,
                       tooltip.textStyle.textBorderColor = NULL,
                       tooltip.textStyle.textBorderWidth = NULL,
                       tooltip.textStyle.textBorderType = NULL,
                       tooltip.textStyle.textShadowColor = NULL,
                       tooltip.textStyle.textShadowBlur = NULL,
                       tooltip.textStyle.textShadowOffsetX = NULL,
                       tooltip.textStyle.textShadowOffsetY = NULL,
                       toolbox.show = TRUE,
                       toolbox.orient = "horizontal",
                       toolbox.itemSize = 15,
                       toolbox.itemGap = 8,
                       toolbox.top = NULL,
                       toolbox.left = NULL,
                       toolbox.right = NULL,
                       toolbox.bottom = NULL,
                       toolbox.width = NULL,
                       toolbox.heigth = NULL,
                       toolbox.feature.saveAsImage.show = TRUE,
                       toolbox.feature.restore.show = TRUE,
                       toolbox.feature.dataZoom.show = TRUE,
                       toolbox.feature.magicType.show = TRUE,
                       toolbox.feature.magicType.type = c("line", "bar", "stack"),
                       toolbox.feature.dataView.show = TRUE,
                       toolbox.iconStyle.color = NULL,
                       toolbox.iconStyle.borderColor = NULL,
                       toolbox.emphasis.iconStyle.borderColor = NULL,
                       toolbox.iconStyle.shadowBlur = NULL,
                       toolbox.iconStyle.shadowColor = NULL,
                       toolbox.iconStyle.shadowOffsetX = NULL,
                       toolbox.iconStyle.shadowOffsetY = NULL,
                       Debug = FALSE) {

  if(length(XVar) == 0L) return(NULL)
  if(length(YVar) == 0L) return(NULL)
  if(length(GroupVar) == 0L) return(NULL)

  if(!data.table::is.data.table(dt)) tryCatch({data.table::setDT(dt)}, error = function(x) {
    dt <- data.table::as.data.table(dt)
  })

  # Convert factor to character
  if(length(GroupVar) > 0L && class(dt[[GroupVar]])[1L] %in% c("factor","integer","numeric")) {
    dt[, eval(GroupVar) := as.character(get(GroupVar))]
  }
  if(length(XVar) > 0L && class(dt[[XVar]])[1L] %in% c("factor","integer","numeric")) {
    dt[, eval(XVar) := as.character(get(XVar))]
  }
  if(length(YVar) > 0L && class(dt[[YVar]])[1L] == "factor") {
    dt[, eval(YVar) := as.character(get(YVar))]
  }

  if(class(dt[[YVar]])[1L] %in% c("character","factor") && class(dt[[XVar]])[1L] %in% c("numeric","integer")) {
    l <- YVar
    YVar <- XVar
    XVar <- l
    rm(l)
  }

  if(length(GroupVar) == 0L) TimeLine <- FALSE

  # Used multiple times
  check1 <- length(XVar) != 0 && length(YVar) != 0 && length(GroupVar) > 0L

  if(!PreAgg) {
    aggFunc <- SummaryFunction(AggMethod)
  }

  # Create base plot object
  numvars <- c()
  byvars <- c()
  if(check1) {
    if(!PreAgg) {

      if(length(FacetLevels) > 0L) {
        dt <- dt[get(GroupVar) %in% c(eval(FacetLevels)), .SD, .SDcols = c(YVar,XVar,GroupVar)]
      }

      if(any(tryCatch({class(dt[[eval(YVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
        numvars <- unique(c(numvars, YVar))
      } else {
        byvars <- unique(c(byvars, YVar))
      }
      if(any(tryCatch({class(dt[[eval(XVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
        if(length(numvars) > 0) {
          x <- length(unique(dt[[XVar]]))
          y <- length(unique(dt[[YVar]]))
          if(x > y) {
            byvars <- unique(c(byvars, YVar))
            numvars[1L] <- XVar
          } else {
            byvars <- unique(c(byvars, XVar))
          }
        } else {
          numvars <- unique(c(numvars, XVar))
        }
      } else {
        byvars <- unique(c(byvars, XVar))
      }
      if(any(tryCatch({class(dt[[eval(GroupVar)]])}, error = function(x) "bla") %in% c('numeric','integer'))) {
        dt[, eval(GroupVar) := as.character(get(GroupVar))]
        byvars <- unique(c(byvars, GroupVar))
      } else {
        byvars <- unique(c(byvars, GroupVar))
      }
      if(!is.null(byvars)) {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars), keyby = c(byvars)]
        for(i in byvars) {
          if(class(temp[[i]]) %in% c('numeric','integer')) {
            temp[, eval(i) := as.character(get(i))]
          }
        }
      } else {
        temp <- dt[, lapply(.SD, noquote(aggFunc)), .SDcols = c(numvars)]
      }
    } else {
      temp <- data.table::copy(dt)
      numvars <- ColNameFilter(data = temp, Types = 'numeric')[[1L]]
      byvars <- unlist(ColNameFilter(data = temp, Types = "character"))
    }

    # Transformation
    if(length(XVar) > 0L && class(temp[[XVar]])[1L] %in% c("numeric","integer")) {
      YVarTrans <- XVarTrans
    }
    if(YVarTrans != "Identity") {
      temp <- AutoTransformationCreate(data = temp, ColumnNames = numvars, Methods = YVarTrans)$Data
    }

    p1 <- echarts4r::e_charts_(
      data = temp |> dplyr::group_by(get(GroupVar[1L])),
      x = XVar,
      dispose = TRUE,
      darkMode = TRUE,
      emphasis = list(focus = "series"),
      width = Width,
      height = Height)
    if(ShowLabels) {
      p1 <- echarts4r::e_bar_(
        e = p1,
        YVar,
        stack = XVar,
        label = list(show = TRUE))
    } else {
      p1 <- echarts4r::e_bar_(
        e = p1,
        YVar,
        stack = XVar)
    }
    if(MouseScroll && FacetRows == 1L && FacetCols == 1L) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = c(0,1))
    } else if(MouseScroll && (FacetRows > 1L || FacetCols > 1L)) {
      p1 <- echarts4r::e_datazoom(e = p1, type = "inside", x_index = seq(0, FacetRows * FacetCols - 1, 1))
    } else {
      p1 <- echarts4r::e_datazoom(e = p1, x_index = c(0,1))
      p1 <- echarts4r::e_datazoom(e = p1, y_index = c(0,1))
    }
    p1 <- echarts4r::e_theme(e = p1, name = Theme)

    p1 <- e_tooltip_full(
      e = p1,
      tooltip.show = tooltip.show,
      tooltip.trigger = tooltip.trigger,
      tooltip.backgroundColor = tooltip.backgroundColor,
      tooltip.borderColor = tooltip.borderColor,
      tooltip.borderWidth = tooltip.borderWidth,
      tooltip.padding = tooltip.padding,
      tooltip.axisPointer.type = tooltip.axisPointer.type,
      tooltip.axisPointer.lineStyle.color = tooltip.axisPointer.lineStyle.color,
      tooltip.axisPointer.shadowStyle.color = tooltip.axisPointer.shadowStyle.color,
      tooltip.axisPointer.shadowStyle.shadowBlur = tooltip.axisPointer.shadowStyle.shadowBlur,
      tooltip.axisPointer.shadowStyle.shadowOffsetX = tooltip.axisPointer.shadowStyle.shadowOffsetX,
      tooltip.axisPointer.shadowStyle.shadowOffsetY = tooltip.axisPointer.shadowStyle.shadowOffsetY,
      tooltip.axisPointer.shadowStyle.opacity = tooltip.axisPointer.shadowStyle.opacity,
      tooltip.textStyle.color = tooltip.textStyle.color,
      tooltip.textStyle.fontStyle = tooltip.textStyle.fontStyle,
      tooltip.textStyle.fontWeight = tooltip.textStyle.fontWeight,
      tooltip.textStyle.fontFamily = tooltip.textStyle.fontFamily,
      tooltip.textStyle.lineHeight = tooltip.textStyle.lineHeight,
      tooltip.textStyle.width = tooltip.textStyle.width,
      tooltip.textStyle.height = tooltip.textStyle.height,
      tooltip.textStyle.textBorderColor = tooltip.textStyle.textBorderColor,
      tooltip.textStyle.textBorderWidth = tooltip.textStyle.textBorderWidth,
      tooltip.textStyle.textBorderType = tooltip.textStyle.textBorderType,
      tooltip.textStyle.textShadowColor = tooltip.textStyle.textShadowColor,
      tooltip.textStyle.textShadowBlur = tooltip.textStyle.textShadowBlur,
      tooltip.textStyle.textShadowOffsetX = tooltip.textStyle.textShadowOffsetX,
      tooltip.textStyle.textShadowOffsetY = tooltip.textStyle.textShadowOffsetY)

    p1 <- e_toolbox_full(
      e = p1,
      toolbox.show = toolbox.show,
      toolbox.orient = toolbox.orient,
      toolbox.itemSize = toolbox.itemSize,
      toolbox.itemGap = toolbox.itemGap,
      toolbox.top = toolbox.top,
      toolbox.left = toolbox.left,
      toolbox.right = toolbox.right,
      toolbox.bottom = toolbox.bottom,
      toolbox.width = toolbox.width,
      toolbox.heigth = toolbox.heigth,
      toolbox.feature.saveAsImage.show = toolbox.feature.saveAsImage.show,
      toolbox.feature.restore.show = toolbox.feature.restore.show,
      toolbox.feature.dataZoom.show = toolbox.feature.dataZoom.show,
      toolbox.feature.magicType.show = toolbox.feature.magicType.show,
      toolbox.feature.magicType.type = toolbox.feature.magicType.type,
      toolbox.feature.dataView.show = toolbox.feature.dataView.show,
      toolbox.iconStyle.color = toolbox.iconStyle.color,
      toolbox.iconStyle.borderColor = toolbox.iconStyle.borderColor,
      toolbox.emphasis.iconStyle.borderColor = toolbox.emphasis.iconStyle.borderColor,
      toolbox.iconStyle.shadowBlur = toolbox.iconStyle.shadowBlur,
      toolbox.iconStyle.shadowColor = toolbox.iconStyle.shadowColor,
      toolbox.iconStyle.shadowOffsetX = toolbox.iconStyle.shadowOffsetX,
      toolbox.iconStyle.shadowOffsetY = toolbox.iconStyle.shadowOffsetY)

    p1 <- echarts4r::e_show_loading(e = p1, hide_overlay = TRUE, text = "Calculating...", color = "#000", text_color = "white", mask_color = "#000")

    p1 <- e_x_axis_full(
      e = p1,
      serie = NULL,
      axis = "x",
      xAxis.title = if(length(xAxis.title) > 0L) xAxis.title else XVar, xAxis.nameLocation = xAxis.nameLocation, xAxis.axisTick.customValues = xAxis.axisTick.customValues,
      xAxis.position = xAxis.position, xAxis.nameTextStyle.color = xAxis.nameTextStyle.color,
      xAxis.nameTextStyle.padding = xAxis.nameTextStyle.padding, xAxis.nameTextStyle.align = xAxis.nameTextStyle.align,
      xAxis.nameTextStyle.fontStyle = xAxis.nameTextStyle.fontStyle, xAxis.nameTextStyle.fontWeight = xAxis.nameTextStyle.fontWeight,
      xAxis.nameTextStyle.fontSize = xAxis.nameTextStyle.fontSize, xAxis.nameTextStyle.fontFamily = xAxis.nameTextStyle.fontFamily, xAxis.min = xAxis.min,
      xAxis.max = xAxis.max, xAxis.splitNumber = xAxis.splitNumber, xAxis.axisLabel.rotate = xAxis.axisLabel.rotate,
      xAxis.axisLabel.margin = xAxis.axisLabel.margin, xAxis.axisLabel.color = xAxis.axisLabel.color,
      xAxis.axisLabel.fontStyle = xAxis.axisLabel.fontStyle, xAxis.axisLabel.fontWeight = xAxis.axisLabel.fontWeight,
      xAxis.axisLabel.fontFamily = xAxis.axisLabel.fontFamily, xAxis.axisLabel.fontSize = xAxis.axisLabel.fontSize,
      xAxis.axisLabel.align = xAxis.axisLabel.align, xAxis.axisLabel.verticalAlign = xAxis.axisLabel.verticalAlign,
      xAxis.axisLabel.backgroundColor = xAxis.axisLabel.backgroundColor, xAxis.axisLabel.borderColor = xAxis.axisLabel.borderColor,
      xAxis.axisLabel.borderWidth = xAxis.axisLabel.borderWidth, xAxis.axisLabel.borderType = xAxis.axisLabel.borderType,
      xAxis.axisLabel.borderRadius = xAxis.axisLabel.borderRadius, xAxis.axisLabel.padding = xAxis.axisLabel.padding,
      xAxis.axisLabel.shadowColor = xAxis.axisLabel.shadowColor, xAxis.axisLabel.shadowBlur = xAxis.axisLabel.shadowBlur,
      xAxis.axisLabel.shadowOffsetX = xAxis.axisLabel.shadowOffsetX, xAxis.axisLabel.shadowOffsetY = xAxis.axisLabel.shadowOffsetY,
      xAxis.axisLabel.textBorderColor = xAxis.axisLabel.textBorderColor, xAxis.axisLabel.textBorderWidth = xAxis.axisLabel.textBorderWidth,
      xAxis.axisLabel.textBorderType = xAxis.axisLabel.textBorderType, xAxis.axisLabel.textShadowColor = xAxis.axisLabel.textShadowColor,
      xAxis.axisLabel.textShadowBlur = xAxis.axisLabel.textShadowBlur, xAxis.axisLabel.textShadowOffsetX = xAxis.axisLabel.textShadowOffsetX,
      xAxis.axisLabel.textShadowOffsetY = xAxis.axisLabel.textShadowOffsetY, xAxis.axisLabel.overflow = xAxis.axisLabel.overflow)

    p1 <- e_y_axis_full(
      e = p1,
      serie = NULL,
      axis = "y",
      yAxis.title = if(length(yAxis.title) > 0L) yAxis.title else YVar, yAxis.nameLocation = yAxis.nameLocation,  yAxis.axisTick.customValues = yAxis.axisTick.customValues,
      yAxis.position = yAxis.position, yAxis.nameTextStyle.color = yAxis.nameTextStyle.color,
      yAxis.nameTextStyle.padding = yAxis.nameTextStyle.padding, yAxis.nameTextStyle.align = yAxis.nameTextStyle.align,
      yAxis.nameTextStyle.fontStyle = yAxis.nameTextStyle.fontStyle, yAxis.nameTextStyle.fontWeight = yAxis.nameTextStyle.fontWeight,
      yAxis.nameTextStyle.fontSize = yAxis.nameTextStyle.fontSize, yAxis.nameTextStyle.fontFamily = yAxis.nameTextStyle.fontFamily, yAxis.min = yAxis.min,
      yAxis.max = yAxis.max, yAxis.splitNumber = yAxis.splitNumber, yAxis.axisLabel.rotate = yAxis.axisLabel.rotate,
      yAxis.axisLabel.margin = yAxis.axisLabel.margin, yAxis.axisLabel.color = yAxis.axisLabel.color,
      yAxis.axisLabel.fontStyle = yAxis.axisLabel.fontStyle, yAxis.axisLabel.fontWeight = yAxis.axisLabel.fontWeight,
      yAxis.axisLabel.fontFamily = yAxis.axisLabel.fontFamily, yAxis.axisLabel.fontSize = yAxis.axisLabel.fontSize,
      yAxis.axisLabel.align = yAxis.axisLabel.align, yAxis.axisLabel.verticalAlign = yAxis.axisLabel.verticalAlign,
      yAxis.axisLabel.backgroundColor = yAxis.axisLabel.backgroundColor, yAxis.axisLabel.borderColor = yAxis.axisLabel.borderColor,
      yAxis.axisLabel.borderWidth = yAxis.axisLabel.borderWidth, yAxis.axisLabel.borderType = yAxis.axisLabel.borderType,
      yAxis.axisLabel.borderRadius = yAxis.axisLabel.borderRadius, yAxis.axisLabel.padding = yAxis.axisLabel.padding,
      yAxis.axisLabel.shadowColor = yAxis.axisLabel.shadowColor, yAxis.axisLabel.shadowBlur = yAxis.axisLabel.shadowBlur,
      yAxis.axisLabel.shadowOffsetX = yAxis.axisLabel.shadowOffsetX, yAxis.axisLabel.shadowOffsetY = yAxis.axisLabel.shadowOffsetY,
      yAxis.axisLabel.textBorderColor = yAxis.axisLabel.textBorderColor, yAxis.axisLabel.textBorderWidth = yAxis.axisLabel.textBorderWidth,
      yAxis.axisLabel.textBorderType = yAxis.axisLabel.textBorderType, yAxis.axisLabel.textShadowColor = yAxis.axisLabel.textShadowColor,
      yAxis.axisLabel.textShadowBlur = yAxis.axisLabel.textShadowBlur, yAxis.axisLabel.textShadowOffsetX = yAxis.axisLabel.textShadowOffsetX,
      yAxis.axisLabel.textShadowOffsetY = yAxis.axisLabel.textShadowOffsetY, yAxis.axisLabel.overflow = yAxis.axisLabel.overflow)

    p1 <- echarts4r::e_brush(e = p1)
    p1 <- e_title_full(
      e = p1,
      title.text = title.text,
      title.subtext = title.subtext,
      title.link = title.link,
      title.sublink = title.sublink,
      title.Align = title.Align,
      title.top = title.top,
      title.left = title.left,
      title.right = title.right,
      title.bottom = title.bottom,
      title.padding = title.padding,
      title.itemGap = title.itemGap,
      title.backgroundColor = title.backgroundColor,
      title.borderColor = title.borderColor,
      title.borderWidth = title.borderWidth,
      title.borderRadius = title.borderRadius,
      title.shadowColor = title.shadowColor,
      title.shadowBlur = title.shadowBlur,
      title.shadowOffsetX = title.shadowOffsetX,
      title.shadowOffsetY = title.shadowOffsetY,
      title.textStyle.color = title.textStyle.color,
      title.textStyle.fontStyle = title.textStyle.fontStyle,
      title.textStyle.fontWeight = title.textStyle.fontWeight,
      title.textStyle.fontFamily = title.textStyle.fontFamily,
      title.textStyle.fontSize = title.textStyle.fontSize,
      title.textStyle.lineHeight = title.textStyle.lineHeight,
      title.textStyle.width = title.textStyle.width,
      title.textStyle.height = title.textStyle.height,
      title.textStyle.textBorderColor = title.textStyle.textBorderColor,
      title.textStyle.textBorderWidth = title.textStyle.textBorderWidth,
      title.textStyle.textBorderType = title.textStyle.textBorderType,
      title.textStyle.textBorderDashOffset = title.textStyle.textBorderDashOffset,
      title.textStyle.textShadowColor = title.textStyle.textShadowColor,
      title.textStyle.textShadowBlur = title.textStyle.textShadowBlur,
      title.textStyle.textShadowOffsetX = title.textStyle.textShadowOffsetX,
      title.textStyle.textShadowOffsetY = title.textStyle.textShadowOffsetY,
      title.subtextStyle.color = title.subtextStyle.color,
      title.subtextStyle.align = title.subtextStyle.align,
      title.subtextStyle.fontStyle = title.subtextStyle.fontStyle,
      title.subtextStyle.fontWeight = title.subtextStyle.fontWeight,
      title.subtextStyle.fontFamily = title.subtextStyle.fontFamily,
      title.subtextStyle.fontSize = title.subtextStyle.fontSize,
      title.subtextStyle.lineHeight = title.subtextStyle.lineHeight,
      title.subtextStyle.width = title.subtextStyle.width,
      title.subtextStyle.height = title.subtextStyle.height,
      title.subtextStyle.textBorderColor = title.subtextStyle.textBorderColor,
      title.subtextStyle.textBorderWidth = title.subtextStyle.textBorderWidth,
      title.subtextStyle.textBorderType = title.subtextStyle.textBorderType,
      title.subtextStyle.textBorderDashOffset = title.subtextStyle.textBorderDashOffset,
      title.subtextStyle.textShadowColor = title.subtextStyle.textShadowColor,
      title.subtextStyle.textShadowBlur = title.subtextStyle.textShadowBlur,
      title.subtextStyle.textShadowOffsetX = title.subtextStyle.textShadowOffsetX,
      title.subtextStyle.textShadowOffsetY = title.subtextStyle.textShadowOffsetY)

    if(FacetRows > 1L || FacetCols > 1L) {
      p1 <- echarts4r::e_facet(e = p1, rows = FacetRows, cols = FacetCols, legend_space = 16, legend_pos = "top")
    }

    p1 <- e_legend_full(
      e = p1,
      legend.show = legend.show, legend.type = legend.type, legend.selector = legend.selector,
      legend.icon = legend.icon, legend.align = legend.align, legend.padding = legend.padding,
      legend.itemGap = legend.itemGap, legend.itemWidth = legend.itemWidth, legend.orient = legend.orient,
      legend.width = legend.width, legend.height = legend.height, legend.left = legend.left,
      legend.right = legend.right, legend.top = legend.top, legend.bottom = legend.bottom,
      legend.backgroundColor = legend.backgroundColor, legend.borderColor = legend.borderColor,
      legend.borderWidth = legend.borderWidth, legend.borderRadius = legend.borderRadius,
      legend.shadowBlur = legend.shadowBlur, legend.shadowColor = legend.shadowColor,
      legend.shadowOffsetX = legend.shadowOffsetX, legend.shadowOffsetY = legend.shadowOffsetY,
      legend.itemStyle.color = legend.itemStyle.color, legend.itemStyle.borderColor = legend.itemStyle.borderColor,
      legend.itemStyle.borderWidth = legend.itemStyle.borderWidth, legend.itemStyle.borderType = legend.itemStyle.borderType,
      legend.itemStyle.shadowBlur = legend.itemStyle.shadowBlur, legend.itemStyle.shadowColor = legend.itemStyle.shadowColor,
      legend.itemStyle.shadowOffsetX = legend.itemStyle.shadowOffsetX, legend.itemStyle.shadowOffsetY = legend.itemStyle.shadowOffsetY,
      legend.itemStyle.opacity = legend.itemStyle.opacity, legend.lineStyle.color = legend.lineStyle.color,
      legend.lineStyle.width = legend.lineStyle.width, legend.lineStyle.type = legend.lineStyle.type,
      legend.lineStyle.shadowBlur = legend.lineStyle.shadowBlur, legend.lineStyle.shadowColor = legend.lineStyle.shadowColor,
      legend.lineStyle.shadowOffsetX = legend.lineStyle.shadowOffsetX, legend.lineStyle.shadowOffsetY = legend.lineStyle.shadowOffsetY,
      legend.lineStyle.opacity = legend.lineStyle.opacity, legend.lineStyle.inactiveColor = legend.lineStyle.inactiveColor,
      legend.lineStyle.inactiveWidth = legend.lineStyle.inactiveWidth, legend.textStyle.color = legend.textStyle.color,
      legend.textStyle.fontStyle = legend.textStyle.fontStyle, legend.textStyle.fontWeight = legend.textStyle.fontWeight,
      legend.textStyle.fontFamily = legend.textStyle.fontFamily, legend.textStyle.fontSize = legend.textStyle.fontSize,
      legend.textStyle.backgroundColor = legend.textStyle.backgroundColor, legend.textStyle.borderColor = legend.textStyle.borderColor,
      legend.textStyle.borderWidth = legend.textStyle.borderWidth, legend.textStyle.borderType = legend.textStyle.borderType,
      legend.textStyle.borderRadius = legend.textStyle.borderRadius, legend.textStyle.padding = legend.textStyle.padding,
      legend.textStyle.shadowColor = legend.textStyle.shadowColor, legend.textStyle.shadowBlur = legend.textStyle.shadowBlur,
      legend.textStyle.shadowOffsetX = legend.textStyle.shadowOffsetX, legend.textStyle.shadowOffsetY = legend.textStyle.shadowOffsetY,
      legend.textStyle.width = legend.textStyle.width, legend.textStyle.height = legend.textStyle.height,
      legend.textStyle.textBorderColor = legend.textStyle.textBorderColor, legend.textStyle.textBorderWidth = legend.textStyle.textBorderWidth,
      legend.textStyle.textBorderType = legend.textStyle.textBorderType, legend.textStyle.textShadowColor = legend.textStyle.textShadowColor,
      legend.textStyle.textShadowBlur = legend.textStyle.textShadowBlur, legend.textStyle.textShadowOffsetX = legend.textStyle.textShadowOffsetX,
      legend.textStyle.textShadowOffsetY = legend.textStyle.textShadowOffsetY, legend.pageTextStyle.color = legend.pageTextStyle.color,
      legend.pageTextStyle.fontStyle = legend.pageTextStyle.fontStyle, legend.pageTextStyle.fontWeight = legend.pageTextStyle.fontWeight,
      legend.pageTextStyle.fontFamily = legend.pageTextStyle.fontFamily, legend.pageTextStyle.fontSize = legend.pageTextStyle.fontSize,
      legend.pageTextStyle.lineHeight = legend.pageTextStyle.lineHeight, legend.pageTextStyle.width = legend.pageTextStyle.width,
      legend.pageTextStyle.height = legend.pageTextStyle.height, legend.pageTextStyle.textBorderColor = legend.pageTextStyle.textBorderColor,
      legend.pageTextStyle.textBorderWidth = legend.pageTextStyle.textBorderWidth, legend.pageTextStyle.textBorderType = legend.pageTextStyle.textBorderType,
      legend.pageTextStyle.textShadowColor = legend.pageTextStyle.textShadowColor, legend.pageTextStyle.textShadowBlur = legend.pageTextStyle.textShadowBlur,
      legend.pageTextStyle.textShadowOffsetX = legend.pageTextStyle.textShadowOffsetX, legend.pageTextStyle.textShadowOffsetY = legend.pageTextStyle.textShadowOffsetY,
      legend.emphasis.selectorLabel.show = legend.emphasis.selectorLabel.show, legend.emphasis.selectorLabel.distance = legend.emphasis.selectorLabel.distance,
      legend.emphasis.selectorLabel.rotate = legend.emphasis.selectorLabel.rotate, legend.emphasis.selectorLabel.color = legend.emphasis.selectorLabel.color,
      legend.emphasis.selectorLabel.fontStyle = legend.emphasis.selectorLabel.fontStyle, legend.emphasis.selectorLabel.fontWeight = legend.emphasis.selectorLabel.fontWeight,
      legend.emphasis.selectorLabel.fontFamily = legend.emphasis.selectorLabel.fontFamily, legend.emphasis.selectorLabel.fontSize = legend.emphasis.selectorLabel.fontSize,
      legend.emphasis.selectorLabel.align = legend.emphasis.selectorLabel.align, legend.emphasis.selectorLabel.verticalAlign = legend.emphasis.selectorLabel.verticalAlign,
      legend.emphasis.selectorLabel.lineHeight = legend.emphasis.selectorLabel.lineHeight, legend.emphasis.selectorLabel.backgroundColor = legend.emphasis.selectorLabel.backgroundColor,
      legend.emphasis.selectorLabel.borderColor = legend.emphasis.selectorLabel.borderColor, legend.emphasis.selectorLabel.borderWidth = legend.emphasis.selectorLabel.borderWidth,
      legend.emphasis.selectorLabel.borderType = legend.emphasis.selectorLabel.borderType, legend.emphasis.selectorLabel.borderRadius = legend.emphasis.selectorLabel.borderRadius,
      legend.emphasis.selectorLabel.padding = legend.emphasis.selectorLabel.padding, legend.emphasis.selectorLabel.shadowColor = legend.emphasis.selectorLabel.shadowColor,
      legend.emphasis.selectorLabel.shadowBlur = legend.emphasis.selectorLabel.shadowBlur, legend.emphasis.selectorLabel.shadowOffsetX = legend.emphasis.selectorLabel.shadowOffsetX,
      legend.emphasis.selectorLabel.shadowOffsetY = legend.emphasis.selectorLabel.shadowOffsetY, legend.emphasis.selectorLabel.width = legend.emphasis.selectorLabel.width,
      legend.emphasis.selectorLabel.height = legend.emphasis.selectorLabel.height, legend.emphasis.selectorLabel.textBorderColor = legend.emphasis.selectorLabel.textBorderColor,
      legend.emphasis.selectorLabel.textBorderWidth = legend.emphasis.selectorLabel.textBorderWidth, legend.emphasis.selectorLabel.textBorderType = legend.emphasis.selectorLabel.textBorderType,
      legend.emphasis.selectorLabel.textShadowColor = legend.emphasis.selectorLabel.textShadowColor, legend.emphasis.selectorLabel.textShadowBlur = legend.emphasis.selectorLabel.textShadowBlur,
      legend.emphasis.selectorLabel.textShadowOffsetX = legend.emphasis.selectorLabel.textShadowOffsetX, legend.emphasis.selectorLabel.textShadowOffsetY = legend.emphasis.selectorLabel.textShadowOffsetY)

    return(p1)

  } else {
    if(Debug) print("XVar, YVar, and GroupVar need to have length > 0")
  }
}

#' @title BarPlot3D
#'
#' @description Build a 3D Bar Plot
#'
#' @family Standard Plots
#' @author Adrian Antico
#'
#' @param dt source data.table
#' @param PreAgg logical. Is your data pre aggregated
#' @param YVar Y-Axis variable name
#' @param XVar X-Axis variable name
#' @param ZVar Z-Axis variable name
#' @param YVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param XVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param ZVarTrans "Asinh", "Log", "LogPlus1", "Sqrt", "Asin", "Logit", "PercRank", "Standardize", "BoxCox", "YeoJohnson"
#' @param FacetRows Defaults to 1 which causes no faceting to occur vertically. Otherwise, supply a numeric value for the number of output grid rows
#' @param FacetCols Defaults to 1 which causes no faceting to occur horizontally. Otherwise, supply a numeric value for the number of output grid columns
#' @param FacetLevels Faceting rows x columns is the max number of levels allowed in a grid. If your GroupVar has more you can supply the levels to display.
#' @param Height "400px"
#' @param Width "200px"
#' @param MouseScroll logical, zoom via mouse scroll
#' @param Theme "dark-blue"
#' @param AggMethod 'mean', 'median', 'sum', 'sd', 'coeffvar', 'count'
#' @param NumberBins = 21
#' @param NumLevels_Y = 20
#' @param NumLevels_X = 20
#' @param ShowLabels character
#' @param ContainLabel TRUE
#' @param title.text Title name
#' @param title.subtext Subtitle name
#' @param title.link Title as a link
#' @param title.sublink Subtitle as a link
#' @param title.Align 'auto' 'left' 'right' 'center'
#' @param title.top 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.left distance between title and left side of container
#' @param title.right distance between title and right side of container
#' @param title.bottom 'auto' '20' 'top' 'middle' 'bottom'
#' @param title.padding numeric
#' @param title.itemGap space between title and subtitle
#' @param title.backgroundColor hex or name
#' @param title.borderColor hex or name
#' @param title.borderWidth numeric
#' @param title.borderRadius numeric
#' @param title.shadowColor hex or name
#' @param title.shadowBlur numeric
#' @param title.shadowOffsetX numeric
#' @param title.shadowOffsetY numeric
#' @param title.textStyle.color hex or name
#' @param title.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.textStyle.fontSize numeric
#' @param title.textStyle.lineHeight numeric
#' @param title.textStyle.width numeric
#' @param title.textStyle.height numeric
#' @param title.textStyle.textBorderColor hex or name
#' @param title.textStyle.textBorderWidth numeric
#' @param title.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.textStyle.textBorderDashOffset numeric
#' @param title.textStyle.textShadowColor hex or name
#' @param title.textStyle.textShadowBlur numeric
#' @param title.textStyle.textShadowOffsetX numeric
#' @param title.textStyle.textShadowOffsetY numeric
#' @param title.subtextStyle.color hex or name
#' @param title.subtextStyle.align 'auto' 'left' 'right' 'center'
#' @param title.subtextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param title.subtextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param title.subtextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param title.subtextStyle.fontSize numeric
#' @param title.subtextStyle.lineHeight numeric
#' @param title.subtextStyle.width numeric
#' @param title.subtextStyle.height numeric
#' @param title.subtextStyle.textBorderColor hex or name
#' @param title.subtextStyle.textBorderWidth numeric
#' @param title.subtextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param title.subtextStyle.textBorderDashOffset numeric
#' @param title.subtextStyle.textShadowColor numeric
#' @param title.subtextStyle.textShadowBlur numeric
#' @param title.subtextStyle.textShadowOffsetX numeric
#' @param title.subtextStyle.textShadowOffsetY numeric
#' @param xAxis.title Axis title
#' @param xAxis.min Min value
#' @param xAxis.max Max value
#' @param xAxis.position 'bottom', 'top'
#' @param xAxis.nameLocation 'center', 'start', 'middle'
#' @param xAxis.axisTick.customValues Custom values for the x-axis
#' @param xAxis.nameTextStyle.color hex or named color
#' @param xAxis.nameTextStyle.padding numeric
#' @param xAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param xAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.nameTextStyle.fontSize numeric
#' @param xAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.splitNumber numeric. Increments for numeric axis labels
#' @param xAxis.axisLabel.rotate numeric
#' @param xAxis.axisLabel.margin numeric
#' @param xAxis.axisLabel.color hex or named
#' @param xAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param xAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param xAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param xAxis.axisLabel.fontSize numeric
#' @param xAxis.axisLabel.align 'center', 'left', 'right'
#' @param xAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param xAxis.axisLabel.backgroundColor hex or named
#' @param xAxis.axisLabel.borderColor hex or named
#' @param xAxis.axisLabel.borderWidth numeric
#' @param xAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.borderRadius numeric
#' @param xAxis.axisLabel.padding numeric
#' @param xAxis.axisLabel.shadowColor hex or named
#' @param xAxis.axisLabel.shadowBlur numeric
#' @param xAxis.axisLabel.shadowOffsetX numeric
#' @param xAxis.axisLabel.shadowOffsetY numeric
#' @param xAxis.axisLabel.textBorderColor hex or named
#' @param xAxis.axisLabel.textBorderWidth numeric
#' @param xAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param xAxis.axisLabel.textShadowColor hex or named
#' @param xAxis.axisLabel.textShadowBlur numeric
#' @param xAxis.axisLabel.textShadowOffsetX numeric
#' @param xAxis.axisLabel.textShadowOffsetY numeric
#' @param xAxis.axisLabel.overflow 'truncate'
#' @param yAxis.title Axis title
#' @param yAxis.min Min value
#' @param yAxis.max Max value
#' @param yAxis.position 'bottom', 'top'
#' @param yAxis.axisTick.customValues Custom values for the y-axis
#' @param yAxis.nameLocation 'center', 'start', 'middle'
#' @param yAxis.nameTextStyle.color hex or named color
#' @param yAxis.nameTextStyle.padding numeric
#' @param yAxis.nameTextStyle.align 'center', 'left', 'right'
#' @param yAxis.nameTextStyle.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.nameTextStyle.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.nameTextStyle.fontSize numeric
#' @param yAxis.nameTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.splitNumber numeric. Increments for numeric axis labels
#' @param yAxis.axisLabel.rotate numeric
#' @param yAxis.axisLabel.margin numeric
#' @param yAxis.axisLabel.color hex or named
#' @param yAxis.axisLabel.fontStyle 'italic', 'normal' 'oblique'
#' @param yAxis.axisLabel.fontWeight 'normal', 'bold', 'bolder', 'lighter'
#' @param yAxis.axisLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param yAxis.axisLabel.fontSize numeric
#' @param yAxis.axisLabel.align 'center', 'left', 'right'
#' @param yAxis.axisLabel.verticalAlign 'top', 'bottom'
#' @param yAxis.axisLabel.backgroundColor hex or named
#' @param yAxis.axisLabel.borderColor hex or named
#' @param yAxis.axisLabel.borderWidth numeric
#' @param yAxis.axisLabel.borderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.borderRadius numeric
#' @param yAxis.axisLabel.padding numeric
#' @param yAxis.axisLabel.shadowColor hex or named
#' @param yAxis.axisLabel.shadowBlur numeric
#' @param yAxis.axisLabel.shadowOffsetX numeric
#' @param yAxis.axisLabel.shadowOffsetY numeric
#' @param yAxis.axisLabel.textBorderColor hex or named
#' @param yAxis.axisLabel.textBorderWidth numeric
#' @param yAxis.axisLabel.textBorderType 'solid', 'dashed', 'dotted'
#' @param yAxis.axisLabel.textShadowColor hex or named
#' @param yAxis.axisLabel.textShadowBlur numeric
#' @param yAxis.axisLabel.textShadowOffsetX numeric
#' @param yAxis.axisLabel.textShadowOffsetY numeric
#' @param yAxis.axisLabel.overflow 'truncate'
#' @param legend.show logical
#' @param legend.type 'scroll' 'plain'
#' @param legend.selector logical
#' @param legend.icon 'circle', 'rect', 'roundRect', 'triangle', 'diamond', 'pin', 'arrow', 'none'
#' @param legend.align 'auto' 'left' 'right'
#' @param legend.padding numeric
#' @param legend.itemGap numeric
#' @param legend.itemWidth numeric
#' @param legend.orient 'vertical' 'horizontal'
#' @param legend.width numeric
#' @param legend.height numeric
#' @param legend.left numeric
#' @param legend.right numeric
#' @param legend.top numeric
#' @param legend.bottom numeric
#' @param legend.backgroundColor hex or color name
#' @param legend.borderColor hex or color name
#' @param legend.borderWidth numeric
#' @param legend.borderRadius numeric
#' @param legend.shadowBlur numeric
#' @param legend.shadowColor hex or color name
#' @param legend.shadowOffsetX numeric
#' @param legend.shadowOffsetY numeric
#' @param legend.itemStyle.color hex or color name
#' @param legend.itemStyle.borderColor hex or color name
#' @param legend.itemStyle.borderWidth numeric
#' @param legend.itemStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.itemStyle.shadowBlur numeric
#' @param legend.itemStyle.shadowColor hex or color name
#' @param legend.itemStyle.shadowOffsetX numeric
#' @param legend.itemStyle.shadowOffsetY numeric
#' @param legend.itemStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.color hex or color name
#' @param legend.lineStyle.width numeric
#' @param legend.lineStyle.type 'solid' 'dashed' 'dotted'
#' @param legend.lineStyle.shadowBlur numeric
#' @param legend.lineStyle.shadowColor hex or color name
#' @param legend.lineStyle.shadowOffsetX numeric
#' @param legend.lineStyle.shadowOffsetY numeric
#' @param legend.lineStyle.opacity numeric 0 to 1
#' @param legend.lineStyle.inactiveColor hex or color name
#' @param legend.lineStyle.inactiveWidth numeric
#' @param legend.textStyle.color hex or color name
#' @param legend.textStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.textStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.textStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.textStyle.fontSize numeric
#' @param legend.textStyle.backgroundColor hex or color name
#' @param legend.textStyle.borderColor hex or color name
#' @param legend.textStyle.borderWidth numeric
#' @param legend.textStyle.borderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.borderRadius numeric
#' @param legend.textStyle.padding numeric
#' @param legend.textStyle.shadowColor hex or color name
#' @param legend.textStyle.shadowBlur numeric
#' @param legend.textStyle.shadowOffsetX numeric
#' @param legend.textStyle.shadowOffsetY numeric
#' @param legend.textStyle.width numeric
#' @param legend.textStyle.height numeric
#' @param legend.textStyle.textBorderColor hex or color name
#' @param legend.textStyle.textBorderWidth numeric
#' @param legend.textStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.textStyle.textShadowColor hex or color name
#' @param legend.textStyle.textShadowBlur numeric
#' @param legend.textStyle.textShadowOffsetX numeric
#' @param legend.textStyle.textShadowOffsetY numeric
#' @param legend.pageTextStyle.color hex or color name
#' @param legend.pageTextStyle.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.pageTextStyle.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.pageTextStyle.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.pageTextStyle.fontSize numeric
#' @param legend.pageTextStyle.lineHeight numeric
#' @param legend.pageTextStyle.width numeric
#' @param legend.pageTextStyle.height numeric
#' @param legend.pageTextStyle.textBorderColor hex or color name
#' @param legend.pageTextStyle.textBorderWidth numeric
#' @param legend.pageTextStyle.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.pageTextStyle.textShadowColor hex or color name
#' @param legend.pageTextStyle.textShadowBlur numeric
#' @param legend.pageTextStyle.textShadowOffsetX numeric
#' @param legend.pageTextStyle.textShadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.show logical
#' @param legend.emphasis.selectorLabel.distance numeric
#' @param legend.emphasis.selectorLabel.rotate numeric
#' @param legend.emphasis.selectorLabel.color hex or color name
#' @param legend.emphasis.selectorLabel.fontStyle 'normal' 'italic' 'oblique'
#' @param legend.emphasis.selectorLabel.fontWeight 'normal' 'bold' 'bolder' 'lighter'
#' @param legend.emphasis.selectorLabel.fontFamily 'sans-serif', 'serif', 'monospace', 'Arial', 'Times New Roman', 'Roboto', 'Open Sans', 'Lato', 'Helvetica', 'Georgia', 'Verdana', 'Arial', 'Tahoma', 'Courier New'
#' @param legend.emphasis.selectorLabel.fontSize numeric
#' @param legend.emphasis.selectorLabel.align 'left' 'center' 'right'
#' @param legend.emphasis.selectorLabel.verticalAlign 'top' 'middle' 'bottom'
#' @param legend.emphasis.selectorLabel.lineHeight numeric
#' @param legend.emphasis.selectorLabel.backgroundColor hex or color name
#' @param legend.emphasis.selectorLabel.borderColor hex or color name
#' @param legend.emphasis.selectorLabel.borderWidth numeric
#' @param legend.emphasis.selectorLabel.borderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.borderRadius numeric
#' @param legend.emphasis.selectorLabel.padding numeric
#' @param legend.emphasis.selectorLabel.shadowColor hex or color name
#' @param legend.emphasis.selectorLabel.shadowBlur numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.shadowOffsetY numeric
#' @param legend.emphasis.selectorLabel.width numeric
#' @param legend.emphasis.selectorLabel.height numeric
#' @param legend.emphasis.selectorLabel.textBorderColor hex or color name
#' @param legend.emphasis.selectorLabel.textBorderWidth numeric
#' @param legend.emphasis.selectorLabel.textBorderType 'solid' 'dashed' 'dotted'
#' @param legend.emphasis.selectorLabel.textShadowColor hex or color name
#' @param legend.emphasis.selectorLabel.textShadowBlur numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetX numeric
#' @param legend.emphasis.selectorLabel.textShadowOffsetY numeric
#' @param tooltip.show logical
#' @param tooltip.trigger "axis" "item" "none"
#' @param tooltip.backgroundColor hex or name
#' @param tooltip.borderColor numeric
#' @param tooltip.borderWidth numeric
#' @param tooltip.padding numeric
#' @param tooltip.axisPointer.type "line" or "shadow"
#' @param tooltip.axisPointer.lineStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.color hex or name
#' @param tooltip.axisPointer.shadowStyle.shadowBlur numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetX numeric
#' @param tooltip.axisPointer.shadowStyle.shadowOffsetY numeric
#' @param tooltip.axisPointer.shadowStyle.opacity numeric between 0 and 1
#' @param tooltip.textStyle.color hex or name
#' @param tooltip.textStyle.fontStyle "normal" "italic" "oblique"
#' @param tooltip.textStyle.fontWeight "normal" "bold" "bolder" "lighter"
#' @param tooltip.textStyle.fontFamily valid family name
#' @param tooltip.textStyle.lineHeight numeric
#' @param tooltip.textStyle.width numeric
#' @param tooltip.textStyle.height numeric
#' @param tooltip.textStyle.textBorderColor hex or name
#' @param tooltip.textStyle.textBorderWidth numeric
#' @param tooltip.textStyle.textBorderType "solid" "dashed" "dotted"
#' @param tooltip.textStyle.textShadowColor hex or name
#' @param tooltip.textStyle.textShadowBlur numeric
#' @param tooltip.textStyle.textShadowOffsetX numeric
#' @param tooltip.textStyle.textShadowOffsetY numeric
#' @param toolbox.show logical
#' @param toolbox.orient "horizontal" or "vertical"
#' @param toolbox.itemSize Default 15
#' @param toolbox.itemGap Default 8
#' @param toolbox.top numeric
#' @param toolbox.left numeric
#' @param toolbox.right numeric
#' @param toolbox.bottom numeric
#' @param toolbox.width numeric
#' @param toolbox.heigth numeric
#' @param toolbox.feature.saveAsImage.show logical
#' @param toolbox.feature.restore.show logical
#' @param toolbox.feature.dataZoom.show logical
#' @param toolbox.feature.magicType.show logical
#' @param toolbox.feature.magicType.type 'bar' 'line' 'stack'
#' @param toolbox.feature.dataView.show logical
#' @param toolbox.iconStyle.color hex
#' @param toolbox.iconStyle.borderColor hex
#' @param toolbox.emphasis.iconStyle.borderColor hex
#' @param toolbox.iconStyle.shadowBlur numeric
#' @param toolbox.iconStyle.shadowColor hex
#' @param toolbox.iconStyle.shadowOffsetX numeric
#' @param toolbox.iconStyle.shadowOffsetY numeric
#' @param Debug Debugging purposes
#'
#' @examples
#' # Create fake data
#' data <- AutoPlots::FakeDataGenerator(N = 10000)
#'
#' # Echarts 3D Bar Chart
#' AutoPlots::BarPlot3D(
#'   dt = data,
#'   PreAgg = TRUE,
#'   AggMethod = 'mean',
#'   XVar = "Factor_1",
#'   YVar = "Factor_2",
#'   ZVar = "Adrian",
#'   YVarTrans = "Identity",
#'   XVarTrans = "Identity",
#'   ZVarTrans = "Identity",
#'   FacetRows = 1,
#'   FacetCols = 1,
#'   FacetLevels = NULL,
#'   NumberBins = 21,
#'   NumLevels_Y = 33,
#'   NumLevels_X = 33,
#'   Height = NULL,
#'   Width = NULL,
#'   ShowLabels = FALSE,
#'   Theme = "dark",
#'   MouseScroll = FALSE,
#'   ContainLabel = TRUE,
#'   Debug = FALSE)
#'
#' @return plot
#' @export
BarPlot3D <- function(dt,
                      PreAgg = TRUE,
                      AggMethod = 'mean',
                      XVar = NULL,
                      YVar = NULL,
                      ZVar = NULL,
                      YVarTrans = "Identity",
                      XVarTrans = "Identity",
                      ZVarTrans = "Identity",
                      FacetRows = 1,
                      FacetCols = 1,
                      FacetLevels = NULL,
                      NumberBins = 21,
                      NumLevels_Y = 33,
                      NumLevels_X = 33,
                      Height = NULL,
                      Width = NULL,
                      ShowLabels = FALSE,
                      Theme = "dark",
                      MouseScroll = FALSE,
                      ContainLabel = TRUE,
                      title.text = "3D Bar Plot",
                      title.subtext = NULL,
                      title.link = NULL,
                      title.sublink = NULL,
                      title.Align = NULL,
                      title.top = NULL,
                      title.left = NULL,
                      title.right = NULL,
                      title.bottom = NULL,
                      title.padding = NULL,
                      title.itemGap = NULL,
                      title.backgroundColor = NULL,
                      title.borderColor = NULL,
                      title.borderWidth = NULL,
                      title.borderRadius = NULL,
                      title.shadowColor = NULL,
                      title.shadowBlur = NULL,
                      title.shadowOffsetX = NULL,
                      title.shadowOffsetY = NULL,
                      title.textStyle.color = NULL,
                      title.textStyle.fontStyle = NULL,
                      title.textStyle.fontWeight = NULL,
                      title.textStyle.fontFamily = "Segoe UI",
                      title.textStyle.fontSize = NULL,
                      title.textStyle.lineHeight = NULL,
                      title.textStyle.width = NULL,
                      title.textStyle.height = NULL,
                      title.textStyle.textBorderColor = NULL,
                      title.textStyle.textBorderWidth = NULL,
                      title.textStyle.textBorderType = NULL,
                      title.textStyle.textBorderDashOffset = NULL,
                      title.textStyle.textShadowColor = NULL,
                      title.textStyle.textShadowBlur = NULL,
                      title.textStyle.textShadowOffsetX = NULL,
                      title.textStyle.textShadowOffsetY = NULL,
                      title.subtextStyle.color = NULL,
                      title.subtextStyle.align = NULL,
                      title.subtextStyle.fontStyle = NULL,
                      title.subtextStyle.fontWeight = NULL,
                      title.subtextStyle.fontFamily = "Segoe UI",
                      title.subtextStyle.fontSize = NULL,
                      title.subtextStyle.lineHeight = NULL,
                      title.subtextStyle.width = NULL,
                      title.subtextStyle.height = NULL,
                      title.subtextStyle.textBorderColor = NULL,
                      title.subtextStyle.textBorderWidth = NULL,
                      title.subtextStyle.textBorderType = NULL,
                      title.subtextStyle.textBorderDashOffset = NULL,
                      title.subtextStyle.textShadowColor = NULL,
                      title.subtextStyle.textShadowBlur = NULL,
                      title.subtextStyle.textShadowOffsetX = NULL,
                      title.subtextStyle.textShadowOffsetY = NULL,
                      xAxis.title = NULL,
                      xAxis.min = NULL,
                      xAxis.max = NULL,
                      xAxis.position = "bottom",
                      xAxis.nameLocation = 'center',
                      xAxis.axisTick.customValues = NULL,
                      xAxis.nameTextStyle.color = NULL,
                      xAxis.nameTextStyle.padding = 25,
                      xAxis.nameTextStyle.align = NULL,
                      xAxis.nameTextStyle.fontStyle = NULL,
                      xAxis.nameTextStyle.fontWeight = "bolder",
                      xAxis.nameTextStyle.fontSize = NULL,
                      xAxis.nameTextStyle.fontFamily = "Segoe UI",
                      xAxis.splitNumber = NULL,
                      xAxis.axisLabel.rotate = NULL,
                      xAxis.axisLabel.margin = NULL,
                      xAxis.axisLabel.color = NULL,
                      xAxis.axisLabel.fontStyle = NULL,
                      xAxis.axisLabel.fontWeight = NULL,
                      xAxis.axisLabel.fontFamily = "Segoe UI",
                      xAxis.axisLabel.fontSize = 14,
                      xAxis.axisLabel.align = NULL,
                      xAxis.axisLabel.verticalAlign = NULL,
                      xAxis.axisLabel.backgroundColor = NULL,
                      xAxis.axisLabel.borderColor = NULL,
                      xAxis.axisLabel.borderWidth = NULL,
                      xAxis.axisLabel.borderType = NULL,
                      xAxis.axisLabel.borderRadius = NULL,
                      xAxis.axisLabel.padding = NULL,
                      xAxis.axisLabel.shadowColor = NULL,
                      xAxis.axisLabel.shadowBlur = NULL,
                      xAxis.axisLabel.shadowOffsetX = NULL,
                      xAxis.axisLabel.shadowOffsetY = NULL,
                      xAxis.axisLabel.textBorderColor = NULL,
                      xAxis.axisLabel.textBorderWidth = NULL,
                      xAxis.axisLabel.textBorderType = NULL,
                      xAxis.axisLabel.textShadowColor = NULL,
                      xAxis.axisLabel.textShadowBlur = NULL,
                      xAxis.axisLabel.textShadowOffsetX = NULL,
                      xAxis.axisLabel.textShadowOffsetY = NULL,
                      xAxis.axisLabel.overflow = NULL,
                      yAxis.title = NULL,
                      yAxis.min = NULL,
                      yAxis.max = NULL,
                      yAxis.position = "center",
                      yAxis.nameLocation = 'center',
                      yAxis.axisTick.customValues = NULL,
                      yAxis.nameTextStyle.color = NULL,
                      yAxis.nameTextStyle.padding = 25,
                      yAxis.nameTextStyle.align = NULL,
                      yAxis.nameTextStyle.fontStyle = NULL,
                      yAxis.nameTextStyle.fontWeight = "bolder",
                      yAxis.nameTextStyle.fontSize = NULL,
                      yAxis.nameTextStyle.fontFamily = "Segoe UI",
                      yAxis.splitNumber = NULL,
                      yAxis.axisLabel.rotate = NULL,
                      yAxis.axisLabel.margin = NULL,
                      yAxis.axisLabel.color = NULL,
                      yAxis.axisLabel.fontStyle = NULL,
                      yAxis.axisLabel.fontWeight = NULL,
                      yAxis.axisLabel.fontFamily = "Segoe UI",
                      yAxis.axisLabel.fontSize = 14,
                      yAxis.axisLabel.align = NULL,
                      yAxis.axisLabel.verticalAlign = NULL,
                      yAxis.axisLabel.backgroundColor = NULL,
                      yAxis.axisLabel.borderColor = NULL,
                      yAxis.axisLabel.borderWidth = NULL,
                      yAxis.axisLabel.borderType = NULL,
                      yAxis.axisLabel.borderRadius = NULL,
                      yAxis.axisLabel.padding = NULL,
                      yAxis.axisLabel.shadowColor = NULL,
                      yAxis.axisLabel.shadowBlur = NULL,
                      yAxis.axisLabel.shadowOffsetX = NULL,
                      yAxis.axisLabel.shadowOffsetY = NULL,
                      yAxis.axisLabel.textBorderColor = NULL,
                      yAxis.axisLabel.textBorderWidth = NULL,
                      yAxis.axisLabel.textBorderType = NULL,
                      yAxis.axisLabel.textShadowColor = NULL,
                      yAxis.axisLabel.textShadowBlur = NULL,
                      yAxis.axisLabel.textShadowOffsetX = NULL,
                      yAxis.axisLabel.textShadowOffsetY = NULL,
                      yAxis.axisLabel.overflow = NULL,
                      legend.show = TRUE,
                      legend.type = "scroll",
                      legend.selector = NULL,
                      legend.icon = NULL,
                      legend.align = NULL,
                      legend.padding = NULL,
                      legend.itemGap = NULL,
                      legend.itemWidth = NULL,
                      legend.orient = "vertical",
                      legend.width = NULL,
                      legend.height = NULL,
                      legend.left = NULL,
                      legend.right = 10,
                      legend.top = 30,
                      legend.bottom = NULL,
                      legend.backgroundColor = NULL,
                      legend.borderColor = NULL,
                      legend.borderWidth = NULL,
                      legend.borderRadius = NULL,
                      legend.shadowBlur = NULL,
                      legend.shadowColor = NULL,
                      legend.shadowOffsetX = NULL,
                      legend.shadowOffsetY = NULL,
                      legend.itemStyle.color = NULL,
                      legend.itemStyle.borderColor = NULL,
                      legend.itemStyle.borderWidth = NULL,
                      legend.itemStyle.borderType = NULL,
                      legend.itemStyle.shadowBlur = NULL,
                      legend.itemStyle.shadowColor = NULL,
                      legend.itemStyle.shadowOffsetX = NULL,
                      legend.itemStyle.shadowOffsetY = NULL,
                      legend.itemStyle.opacity = NULL,
                      legend.lineStyle.color = NULL,
                      legend.lineStyle.width = NULL,
                      legend.lineStyle.type = NULL,
                      legend.lineStyle.shadowBlur = NULL,
                      legend.lineStyle.shadowColor = NULL,
                      legend.lineStyle.shadowOffsetX = NULL,
                      legend.lineStyle.shadowOffsetY = NULL,
                      legend.lineStyle.opacity = NULL,
                      legend.lineStyle.inactiveColor = NULL,
                      legend.lineStyle.inactiveWidth = NULL,
                      legend.textStyle.color = NULL,
                      legend.textStyle.fontStyle = NULL,
                      legend.textStyle.fontWeight = NULL,
                      legend.textStyle.fontFamily = "Segoe UI",
                      legend.textStyle.fontSize = NULL,
                      legend.textStyle.backgroundColor = NULL,
                      legend.textStyle.borderColor = NULL,
                      legend.textStyle.borderWidth = NULL,
                      legend.textStyle.borderType = NULL,
                      legend.textStyle.borderRadius = NULL,
                      legend.textStyle.padding = NULL,
                      legend.textStyle.shadowColor = NULL,
                      legend.textStyle.shadowBlur = NULL,
                      legend.textStyle.shadowOffsetX = NULL,
                      legend.textStyle.shadowOffsetY = NULL,
                      legend.textStyle.width = NULL,
                      legend.textStyle.height = NULL,
                      legend.textStyle.textBorderColor = NULL,
                      legend.textStyle.textBorderWidth = NULL,
                      legend.textStyle.textBorderType = NULL,
                      legend.textStyle.textShadowColor = NULL,
                      legend.textStyle.textShadowBlur = NULL,
                      legend.textStyle.textShadowOffsetX = NULL,
                      legend.textStyle.textShadowOffsetY = NULL,
                      legend.pageTextStyle.color = NULL,
                      legend.p