% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ford.R
\name{ford}
\alias{ford}
\title{Variable selection by the FORD algorithm}
\usage{
ford(
  Y,
  X,
  dist.type.X = "continuous",
  num_features = NULL,
  stop = TRUE,
  na.rm = TRUE,
  standardize = "scale",
  numCores = parallel::detectCores(),
  parPlat = "none",
  printIntermed = TRUE
)
}
\arguments{
\item{Y}{Vector of responses (length n)}

\item{X}{Matrix of predictors (n by p)}

\item{dist.type.X}{A string specifying the distribution type of X: either "continuous" or "discrete". Default is "continuous".}

\item{num_features}{Number of variables to be selected, cannot be larger than p. The default value is NULL and in that
case it will be set equal to p. If stop == TRUE (see below), then num_features is irrelevant.}

\item{stop}{Stops at the first instance of negative irdc, if TRUE.}

\item{na.rm}{Removes NAs if TRUE.}

\item{standardize}{Standardize covariates if set equal to "scale" or "bounded". Otherwise will use the raw inputs.
The default value is "scale" and normalizes each column of X to have mean zero and variance 1. If set equal to "bounded"
map the values of each column of X to [0, 1].}

\item{numCores}{Number of cores that are going to be used for
parallelizing the variable selection process.}

\item{parPlat}{Specifies the parallel platform to chunk data by rows.
It can take three values:
1- The default value is set to 'none', in which case no row chunking
is done;
2- the \code{parallel} cluster to be used for row chunking;
3- "locThreads", specifying that row chunking will be done via
threads on the host machine.}

\item{printIntermed}{The default value is TRUE, in which case print intermediate results from the cluster nodes before final processing.}
}
\value{
An object of class "ford", with attributes
\code{selectedVar}, showing the selected variables in decreasing
order of predictive power, and \code{step_nu}, listing
the 'irdc' values. Typically the latter will begin to level off at
some point, with additional marginal improvements being small.
}
\description{
FORD is a variable selection algorithm based on  Integrated R square dependence coefficient \code{\link{irdc}}.
}
\details{
ford is a forward stepwise algorithm that uses the conditional dependence coefficient (\code{\link{irdc}})
at each step, instead of the multiple correlation coefficient
as in ordinary forward stepwise. If \code{stop} == TRUE, the process is stopped at the first instance of
nonpositive irdc, thereby selecting a subset of variables. Otherwise, a set of covariates of size
\code{num_features}, ordered according to predictive power (as measured by irdc) is produced.

\emph{Parallel computation:}

The computation can be lengthy, so the package offers two kinds of
parallel computation.

The first, controlled by the argument \code{numCores},
specifies the number of cores to be used on the host
machine. If at a given step there are k candidate variables
under consideration for inclusion, these k tasks are assigned
to the various cores.

The second approach, controlled by the argument \code{parPlat}
("parallel platform"), involves the user first setting up a cluster via
the \pkg{parallel} package. The data are divided into chunks by rows,
with each cluster node applying ford to its data chunk.  The
union of the results is then formed, and fed through ford one more
time to adjust the discrepancies. The idea is that that last step
will not be too lengthy, as the number of candidate variables has
already been reduced.  A cluster size of r may actually
produce a speedup factor of more than r (Matloff 2016).

Potentially the best speedup is achieved by using the two approaches
together.

The first approach cannot be used on Windows platforms, as
\code{parallel::mcapply} has no effect.  Windows users should thus
use the second approach only.

In addition to speed, the second approach is useful for diagnostics, as
the results from the different chunks gives the user an
idea of the degree of sampling variability in the
ford results.

In the second approach, a random permutation is applied to the
rows of the dataset, as many datasets are sorted by one or more
columns.

Note that if a certain value of a feature is rare in the
full dataset, it may be absent entirely in some chunk.
}
\examples{
# Example 1
n = 500
p = 10
x <- matrix(rnorm(n * p), nrow = n)
colnames(x) = paste0(rep("x", p), seq(1, p))
y <- x[, 1] * x[, 8] + x[, 10]^2
# with num_features equal to 3 and stop equal to FALSE, ford will give a list of
# three selected features
result1 = ford(y, x, num_features = 3, stop = FALSE, numCores = 1)
result1
# Example 2
# same example, but stop according to the stopping rule
result2 = ford(y, x, numCores = 1)
result2
}
\references{
Azadkia, M. and Roudaki, P. (2025). A New Measure Of Dependence: Integrated R2
\url{http://arxiv.org/abs/2505.18146}.

Matloff, N. (2016). Software Alchemy: Turning Complex
Statistical Computations into Embarrassingly-Parallel Ones.
\emph{J. of Stat. Software.}
}
\seealso{
\code{\link{irdc}}, \code{\link[FOCI]{foci}}, \code{\link[KPC]{KFOCI}}
}
\author{
Mona Azadkia, Pouya Roudaki
}
