#' Exports an object of FSK class as an .fskx file
#'
#' @param fsk_object The instance of FSK2R to be exported.
#' @param out_path Path where the file is to be saved.
#' @param check Whether checks are made. TRUE by default.
#'
#' @importFrom utils zip
#' @importFrom R.utils isAbsolutePath
#'
#' @export
#'
#' @return None
#'
#' @examples
#' \donttest{
#'  path_example <- system.file("extdata", "ToyModelv4.fskx", package = "FSK2R")
#'  my_fsk <- import_fsk(path_example)
#'  class(my_fsk)
#'  export_fsk(my_fsk, out_path=file.path(tempdir(), "out.fskx"))
#' }
#'
export_fsk <- function(fsk_object, out_path, check = TRUE) {

    tmp_folder <- tempdir()  # tempdir used for zipping
    # my_tempdir <- "temp_fsk"  # For debugging, it is just easier to know where the tempdir is

    my_tempdir <- file.path(tmp_folder, paste0("/FSK_", sample(1:9999, 1)))

    if (dir.exists(my_tempdir)) {
        file.remove(my_tempdir)
    }

    dir.create(my_tempdir)

    ## Write every file

    export_R_model(fsk_object, my_tempdir, check)
    export_visualization(fsk_object, my_tempdir, check)
    export_readme(fsk_object, my_tempdir, check)
    export_metadata(fsk_object, my_tempdir, check)
    export_packages(fsk_object, my_tempdir, check)
    export_manifest(fsk_object, my_tempdir, check)
    export_modelmetadata(fsk_object, my_tempdir, check)
    export_sbmlModel(fsk_object, my_tempdir, check)
    export_simulation(fsk_object, my_tempdir, check)
    export_otherfiles(fsk_object, my_tempdir, check)
    
    # Always create an empty workspace.RData file to comply with FSKX standard
    # (This will be overwritten if a real workspace exists in other_files)
    workspace_file <- file.path(my_tempdir, "workspace.RData")
    if (!file.exists(workspace_file)) {
        # Create empty workspace
        save(list = character(0), file = workspace_file)
    }

    ## Zip and move to out_path

    old_wd <- getwd()
    on.exit(setwd(old_wd))

    setwd(my_tempdir)

    if (isAbsolutePath(out_path)) {

        out_path_whole <- out_path

    } else {

        out_path_whole <- file.path(old_wd, out_path)

    }

    # Include all files and subdirectories recursively, preserving directory structure
    all_files <- list.files(recursive = TRUE, all.files = FALSE, include.dirs = FALSE)
    if (length(all_files) > 0) {
        # Use zip instead of zipr to properly handle directory structure
        # zipr doesn't preserve directory structure properly
        zip(zipfile = out_path_whole, files = all_files, flags = "-r")
    } else {
        # Create empty zip if no files
        zip(zipfile = out_path_whole, files = character(0))
    }

}

#' Export other files
#'
#' @inheritParams export_fsk
#'
export_otherfiles <- function(fsk_object, out_path, check = FALSE) {

    if (length(fsk_object$other_files) == 0) {
        return()
    }
    
    # Preserve directory structure by using the names (original paths) as destinations
    for (i in seq_along(fsk_object$other_files)) {
        # Get original path from names (e.g., "./simulations/data.csv")
        original_path <- names(fsk_object$other_files)[i]
        # Get source file path (temp directory location)
        source_path <- fsk_object$other_files[i]
        
        # Skip if source file doesn't exist
        if (!file.exists(source_path)) {
            if (check) {
                warning(paste("Source file not found:", source_path))
            }
            next
        }
        
        # Clean the original path (remove "./" prefix if present)
        clean_path <- gsub("^\\./", "", original_path)
        
        # Create destination path preserving directory structure
        dest_path <- file.path(out_path, clean_path)
        dest_dir <- dirname(dest_path)
        
        # Create subdirectories if needed
        if (!dir.exists(dest_dir)) {
            dir.create(dest_dir, recursive = TRUE)
        }
        
        # Copy file to preserve directory structure
        file.copy(source_path, dest_path, overwrite = TRUE)
        
        # If R script, normalize to LF
        ext <- tolower(tools::file_ext(dest_path))
        if (identical(ext, "r")) {
            lines <- tryCatch(readLines(dest_path, warn = FALSE), error = function(e) NULL)
            if (!is.null(lines)) {
                write_text_lf(dest_path, lines)
            }
        }
    }

}

#' Export the model.sbml
#'
#' @importFrom xml2 as_xml_document write_xml
#' @importFrom dplyr %>%
#'
#' @inheritParams export_fsk
#'
export_sbmlModel <- function(fsk_object, out_path, check = FALSE) {

    con <- file(file.path(out_path, "model.sbml"), open = "wb")
    on.exit(close(con))
    fsk_object$model %>% as_xml_document() %>% write_xml(con)

}

#' Export the sim.sedml
#'
#' @importFrom xml2 as_xml_document write_xml
#' @importFrom dplyr %>%
#'
#' @inheritParams export_fsk
#'
export_simulation <- function(fsk_object, out_path, check = FALSE) {

    con <- file(file.path(out_path, "sim.sedml"), open = "wb")
    on.exit(close(con))
    fsk_object$simulation %>% as_xml_document() %>% write_xml(con)

}

#' Functions for exporting the R model of an FSK2R object
#'
#' @inheritParams export_fsk
#'
export_R_model <- function(fsk_object, out_path, check = FALSE) {

    # Extract actual script filename from RDF metadata
    script_files <- extract_script_filenames_from_rdf(fsk_object$model_metadata)
    model_filename <- if (!is.null(script_files$modelScript)) {
        script_files$modelScript
    } else {
        "model.R"  # default fallback
    }

    write_text_lf(file.path(out_path, model_filename), fsk_object$R_model)

}
##' Functions for exporting the README of an FSK2R object
#'
#' @inheritParams export_fsk
#'
#'
export_readme <- function(fsk_object, out_path, check = FALSE) {

    write_text_lf(file.path(out_path, "README.txt"), fsk_object$readme)

}

#' Functions for exporting the visualization script of an FSK2R object
#'
#' @inheritParams export_fsk
#'
#'
export_visualization <- function(fsk_object, out_path, check = FALSE) {

    # Extract actual script filename from RDF metadata
    script_files <- extract_script_filenames_from_rdf(fsk_object$model_metadata)
    visualization_filename <- if (!is.null(script_files$visualizationScript)) {
        script_files$visualizationScript
    } else {
        "visualization.R"  # default fallback
    }

    write_text_lf(file.path(out_path, visualization_filename), fsk_object$visualization)

}

#' Functions for exporting the workspace of an FSK2R object
#'
#' Exports simulation environment variables as workspace.r file and 
#' workspace.RData for easy loading in R sessions.
#'
#' @inheritParams export_fsk
#' @param simulation_env Environment containing simulation results (optional)
#' @importFrom utils capture.output
#'
export_workspace <- function(fsk_object, out_path, check = FALSE, simulation_env = NULL) {
    
    if (!is.null(simulation_env)) {
        # Export workspace.r file (R script to recreate variables)
        # Get all variables from simulation environment
        all_vars <- ls(simulation_env)
        
        # Filter out functions and system variables
        user_vars <- all_vars[sapply(all_vars, function(var_name) {
            obj <- get(var_name, envir = simulation_env)
            !is.function(obj) && !is.environment(obj) && 
            !var_name %in% c("last.warning", ".Last.value", ".Random.seed")
        })]
        
        # Create R script content
        workspace_lines <- c(
            "# FSK2R Simulation Workspace",
            "# This file contains all variables generated during simulation",
            paste("# Generated on:", Sys.time()),
            ""
        )
        
        # Add variable assignments
        for (var_name in user_vars) {
            obj <- get(var_name, envir = simulation_env)
            tryCatch({
                # Use dput to create reproducible R code
                var_lines <- capture.output(dput(obj))
                workspace_lines <- c(workspace_lines,
                                   paste0("# Variable: ", var_name),
                                   paste0(var_name, " <- ", paste(var_lines, collapse = "")),
                                   "")
            }, error = function(e) {
                # For objects that can't be dput, add a comment
                workspace_lines <<- c(workspace_lines,
                                    paste0("# Variable: ", var_name, " (could not export: ", e$message, ")"),
                                    "")
            })
        }
        
        write_text_lf(file.path(out_path, "workspace.r"), workspace_lines)
        
        # Also save as workspace.RData for direct loading
        workspace_data_path <- file.path(out_path, "workspace.RData")
        if (length(user_vars) > 0) {
            save(list = user_vars, envir = simulation_env, file = workspace_data_path)
        }
    }
}

#' Function for exporting the metadata of an FSK2R object
#'
#' @inheritParams export_fsk
#'
#' @importFrom jsonlite toJSON
#'
export_metadata <- function(fsk_object, out_path, check = FALSE) {

    # Serialize first with proper scalar/array handling
    json_output <- jsonlite::toJSON(fsk_object$metadata, auto_unbox = TRUE, pretty = TRUE)
    
    # Post-process JSON string to remove empty arrays and null values
    clean_json <- clean_json_string(json_output)
    write_text_lf(file.path(out_path, "metaData.json"), clean_json)
    
    if (check) {
        cat("Used JSON string post-processing to remove empty values\n")
    }

}

#' Clean empty values from JSON string
#'
#' @param json_string JSON string to clean
#' @return Cleaned JSON string with empty arrays and null values removed
clean_json_string <- function(json_string) {
    
    # Define all fields that should be arrays of strings (based on FSKX schema)
    # NOTE: country and region are excluded here due to context-specific handling
    string_array_fields <- c(
        # modelCategory
        "modelSubClass", "basicProcess",
        # scope  
        "spatialInformation",
        # scope.product[]
        "method", "packaging", "treatment", 
        # scope.populationGroup[] (country and region handled separately)
        "populationSpan", "populationDescription", "populationAge", "bmi", 
        "specialDietGroups", "patternConsumption", "populationRiskFactor", "season",
        # dietaryAssessmentMethod[]
        "numberOfFoodItems", "recordTypes", "foodDescriptors",
        # laboratory[]
        "accreditation",
        # modelMath.modelEquation[]
        "modelHypothesis",
        # modelMath.exposure[]
        "treatment", "contamination", "scenario",
        # modelMath
        "event"
    )
    
    # Generic fix: convert string to array for all specified fields
    # "fieldName": "some string" -> "fieldName": ["some string"]
    field_pattern <- paste(string_array_fields, collapse = "|")
    pattern <- sprintf('"(%s)":\\s*"([^"]*)"', field_pattern)
    replacement <- '"\\1": ["\\2"]'
    json_string <- gsub(pattern, replacement, json_string, perl = TRUE)
    
    # Context-aware fix for country and region:
    # Only convert to arrays when inside populationGroup objects (not in author/creator/laboratory)
    
    # Simpler approach: Use gsubfn or a manual approach
    # Look for populationGroup sections and convert country/region within them
    
    # First, let's try a more targeted approach using temporary markers
    # Mark populationGroup sections for processing
    json_string <- gsub(
        '("populationGroup":\\s*\\[)',
        '\\1__POPGROUP_START__',
        json_string,
        perl = TRUE
    )
    
    # Mark the end of populationGroup arrays
    # This is tricky with nested structures, so let's use a different approach
    # Convert country/region to arrays only when they follow populationGroup context
    json_string <- gsub(
        '(__POPGROUP_START__[^\\]]*?)"(country|region)":\\s*"([^"]*)"',
        '\\1"\\2": ["\\3"]',
        json_string,
        perl = TRUE
    )
    
    # Clean up markers
    json_string <- gsub('__POPGROUP_START__', '', json_string, perl = TRUE)
    
    # Remove empty arrays: "field": []
    json_string <- gsub(',\\s*"[^"]+":\\s*\\[\\s*\\]', '', json_string, perl = TRUE)
    json_string <- gsub('"[^"]+":\\s*\\[\\s*\\],?', '', json_string, perl = TRUE)
    
    # Remove null values: "field": null
    json_string <- gsub(',\\s*"[^"]+":\\s*null', '', json_string, perl = TRUE)
    json_string <- gsub('"[^"]+":\\s*null,?', '', json_string, perl = TRUE)
    
    # Remove empty strings: "field": ""
    json_string <- gsub(',\\s*"[^"]+":\\s*""', '', json_string, perl = TRUE)
    json_string <- gsub('"[^"]+":\\s*"",?', '', json_string, perl = TRUE)
    
    # Remove "NULL" string values: "field": "NULL"
    json_string <- gsub(',\\s*"[^"]+":\\s*"NULL"', '', json_string, perl = TRUE)
    json_string <- gsub('"[^"]+":\\s*"NULL",?', '', json_string, perl = TRUE)
    
    # Clean up trailing commas that might have been left behind
    json_string <- gsub(',\\s*}', '}', json_string, perl = TRUE)
    json_string <- gsub(',\\s*]', ']', json_string, perl = TRUE)
    
    # Clean up leading commas in objects/arrays
    json_string <- gsub('\\{\\s*,', '{', json_string, perl = TRUE)
    json_string <- gsub('\\[\\s*,', '[', json_string, perl = TRUE)
    
    return(json_string)
}

#' Functions for exporting the packages of an FSK2R object
#'
#' @inheritParams export_fsk
#'
#' @importFrom tidyr %>%
#' @importFrom jsonlite toJSON
#'
export_packages <- function(fsk_object, out_path, check = FALSE) {

    json_str <- jsonlite::toJSON(fsk_object$packages, pretty = TRUE)
    write_text_lf(file.path(out_path, "packages.json"), json_str)

}



#' Functions for exporting the manifest of an FSK2R object
#'
#' @inheritParams export_fsk
#'
#' @importFrom rlang .data
#'
export_manifest <- function(fsk_object, out_path, check = FALSE) {

    if (is.null(fsk_object$manifest)) {

        fsk_object <- update_manifest(fsk_object)

    }

    data <- fsk_object$manifest %>%
        mutate(out = paste0('<content location ="', .data$location,
                            '" format ="', .data$format, '" />'))

    write_text_lf(
        file.path(out_path, "manifest.xml"),
        c('<?xml version="1.0" encoding="UTF-8"?>',
          '<omexManifest xmlns="http://identifiers.org/combine.specifications/omex-manifest">',
          data$out,
          "</omexManifest>"
        )
    )
}

#' Updates the manifest file
#'
#' @importFrom tibble tribble
#' @importFrom tools file_ext
#'
#' @param fsk_object An instance of FSK2R.
#'
#'
update_manifest <- function(fsk_object) {

    data_types <- tribble(
        ~format, ~type,
        "https://www.iana.org/assignments/media-types/text/csv", ".csv",
        "http://purl.org/NET/mediatypes/application/zip", ".zip",
        "https://www.iana.org/assignments/media-types/application/json", ".json",
        "http://purl.org/NET/mediatypes/text-xplain", ".txt",
        "http://purl.org/NET/mediatypes/application/r", ".R",
        "http://purl.org/NET/mediatypes/application/r", ".r",
        "https://www.iana.org/assignments/media-types/application/xml", ".xml",
        "https://www.iana.org/assignments/media-types/text/plain", ".dat"
    )

    # Extract actual script filenames from RDF metadata
    script_files <- extract_script_filenames_from_rdf(fsk_object$model_metadata)
    
    # Use extracted filenames or fall back to defaults
    model_script_name <- if (!is.null(script_files$modelScript)) {
        paste0("./", script_files$modelScript)
    } else {
        "./model.R"  # default fallback
    }
    
    visualization_script_name <- if (!is.null(script_files$visualizationScript)) {
        paste0("./", script_files$visualizationScript)
    } else {
        "./visualization.R"  # default fallback
    }

    # Standard FSK files with dynamic script names
    manifest <- tribble(
        ~format, ~location,
        "http://identifiers.org/combine.specifications/omex", ".",
        "https://www.iana.org/assignments/media-types/application/json", "./packages.json",
        "http://purl.org/NET/mediatypes/text-xplain", "./README.txt",
        "http://identifiers.org/combine.specifications/omex-metadata", "./metadata.rdf",
        "http://identifiers.org/combine.specifications/omex-manifest", "./manifest.xml",
        "http://purl.org/NET/mediatypes/application/sbml+xml", "./model.sbml",
        "http://purl.org/NET/mediatypes/application/r",  visualization_script_name,
        "http://identifiers.org/combine.specifications/sed-ml", "./sim.sedml",
        "http://purl.org/NET/mediatypes/application/r", model_script_name,
        "https://www.iana.org/assignments/media-types/application/json", "./metaData.json"
    )
    
    # Add other files with their preserved directory structure
    if (length(fsk_object$other_files) > 0) {
        other_entries <- tibble()
        
        for (i in seq_along(fsk_object$other_files)) {
            # Get original path (this preserves directory structure)
            original_path <- names(fsk_object$other_files)[i]
            
            # Ensure path starts with "./" for manifest format
            if (!startsWith(original_path, "./")) {
                manifest_path <- paste0("./", original_path)
            } else {
                manifest_path <- original_path
            }
            
            # Determine file format based on extension
            file_extension <- paste0(".", file_ext(original_path))
            format_match <- data_types[data_types$type == file_extension, ]
            
            if (nrow(format_match) > 0) {
                file_format <- format_match$format[1]
            } else {
                # Default format for unknown extensions - use FSK-specific resourceFile format
                file_format <- "https://knime.bfr.berlin/mediatypes/resourceFile"
            }
            
            # Add to other entries
            other_entry <- tribble(~format, ~location, file_format, manifest_path)
            other_entries <- rbind(other_entries, other_entry)
        }
        
        # Combine standard manifest with other files
        manifest <- rbind(manifest, other_entries)
    }

    fsk_object$manifest <- manifest

    return(fsk_object)

}


#' Functions for exporting the model metadata of an FSK2R object
#'
#' @inheritParams export_fsk
#'
export_modelmetadata <- function(fsk_object, out_path, check = FALSE) {

    # Extract actual script filenames from RDF metadata
    script_files <- extract_script_filenames_from_rdf(fsk_object$model_metadata)
    
    # Use extracted filenames or fall back to defaults
    model_script_name <- if (!is.null(script_files$modelScript)) {
        paste0("/", script_files$modelScript)
    } else {
        "/model.R"  # default fallback
    }
    
    visualization_script_name <- if (!is.null(script_files$visualizationScript)) {
        paste0("/", script_files$visualizationScript)
    } else {
        "/visualization.R"  # default fallback
    }

    my_lines <- c('<?xml version="1.0" encoding="UTF-8"?>',
                  '<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#" xmlns:dcterms="http://purl.org/dc/terms/" xmlns:vCard="http://www.w3.org/2006/vcard/ns#">',
                  '<rdf:Description rdf:about=".">',
                  '<dcterms:conformsTo>2.0</dcterms:conformsTo>',
                  '</rdf:Description>',
                  paste0('<rdf:Description rdf:about="', visualization_script_name, '">'),
                  '<dc:type xmlns:dc="http://purl.org/dc/elements/1.1/">visualizationScript</dc:type>',
                  '</rdf:Description>',
                  '<rdf:Description rdf:about="/workspace.RData">',
                  '<dc:type xmlns:dc="http://purl.org/dc/elements/1.1/">workspace</dc:type>',
                  '</rdf:Description>',
                  paste0('<rdf:Description rdf:about="', model_script_name, '">'),
                  '<dc:type xmlns:dc="http://purl.org/dc/elements/1.1/">modelScript</dc:type>',
                  '</rdf:Description>',
                  '<rdf:Description rdf:about="/README.txt">',
                  '<dc:type xmlns:dc="http://purl.org/dc/elements/1.1/">readme</dc:type>',
                  '</rdf:Description>',
                  '</rdf:RDF>'
                  )

    write_text_lf(file.path(out_path, "metadata.rdf"), my_lines)

}
