#' Algorithm for Decoding Hidden Markov Models (global)
#' 
#' The function decodes a trainded hidden Markov model into a most likely sequence of 
#' hidden states. Different to the \code{\link{local_decoding_algorithm}}, 
#' this algorithm determines the sequence of most likely hidden states for all 
#' time points simultaneously.  See MacDonald & Zucchini (2009, Paragraph 5.3.2) 
#' for further details.
#'
#' @param x a vector object containing the time-series of observations that are assumed to 
#'          be realizations of the (hidden Markov state dependent) observation 
#'          process of the model.
#' @param m    a (finite) number of states in the hidden Markov chain.
#' @param delta a vector object containing values for the marginal probability 
#'    distribution of the \code{m} states of the Markov chain at the time point \code{t=1}.
#' @param gamma  a matrix (\code{ncol=nrow=m}) containing values for the transition 
#'    matrix of the hidden Markov chain.
#' @param distribution_class a single character string object with the abbreviated name of 
#'    the \code{m} observation distributions of the Markov dependent observation process.  
#'    The following distributions are supported by this algorithm: Poisson (\code{pois}); 
#'    generalized Poisson (\code{genpois}); normal (\code{norm}); geometric (\code{geom}).
#' @param distribution_theta a list object containing the parameter values for the 
#'    \code{m} observation distributions that are dependent on the hidden Markov state.
#' @param discr_logL a logical object. It is \code{TRUE} if the discrete log-likelihood 
#'    shall be calculated (for \code{distribution_class="norm"} instead of the general 
#'    log-likelihood).  Default is \code{FALSE}.
#' @param discr_logL_eps a single numerical value to approximately determine the discrete 
#'    log-likelihood for a hidden Markov model based on nomal distributions 
#'    (for \code{"norm"}).  The default value is \code{0.5}.
#'
#' @return
#' The \code{Viterbi_algorithm} returns a list containing the following two components:
#' \describe{
#' \item{omega}{a (T,m)-matrix (when T indicates the length/size of the observation 
#'      time-series and m the number of states of the HMM) containing probabilities 
#'      (maximum probability to generate the first t members (t=1,...,T) of the given 
#'      time-series x with the HMM and to stop in state i=1,...,m) calculated by the 
#'      algorithm. See MacDonald & Zucchini (2009, Paragraph 5.3.2) for further details.}
#' \item{decoding}{a numerical vector containing the globally most likely sequence of 
#'      hidden states as decoded by the Viterbi algorithm.}
#'}
#' 
#' @references 
#' MacDonald, I. L.,  Zucchini, W. (2009) \emph{Hidden Markov Models for Time Series: 
#' An Introduction Using R}, Boca Raton: Chapman & Hall.
#' 
#' Forney, G.D. (1973). The Viterbi algorithm. Proceeding of the IEE, vol. 
#' \bold{61}(3), 268--278.
#' 
#' Viterbi, A.J. (1967). Error Bounds for concolutional codes and an asymptotically 
#' optimal decoding algorithm. Information Theory, IEEE Transactions on, 
#' vol. \bold{13}(2), 260--269.
#' 
#' @author The basic algorithm for a Poisson-HMM can be found in MacDonald & Zucchini 
#'    (2009, Paragraph A.2.4).  Extension and implementation by Vitali Witowski (2013).
#'    
#' @seealso \code{\link{local_decoding_algorithm}},	\code{\link{HMM_decoding}}
#' @export
#'
#' @examples
#' x <- c(1,16,19,34,22,6,3,5,6,3,4,1,4,3,5,7,9,8,11,11,
#'   14,16,13,11,11,10,12,19,23,25,24,23,20,21,22,22,18,7,
#'   5,3,4,3,2,3,4,5,4,2,1,3,4,5,4,5,3,5,6,4,3,6,4,8,9,12,
#'   9,14,17,15,25,23,25,35,29,36,34,36,29,41,42,39,40,43,
#'   37,36,20,20,21,22,23,26,27,28,25,28,24,21,25,21,20,21,
#'   11,18,19,20,21,13,19,18,20,7,18,8,15,17,16,13,10,4,9,
#'   7,8,10,9,11,9,11,10,12,12,5,13,4,6,6,13,8,9,10,13,13,
#'   11,10,5,3,3,4,9,6,8,3,5,3,2,2,1,3,5,11,2,3,5,6,9,8,5,
#'   2,5,3,4,6,4,8,15,12,16,20,18,23,18,19,24,23,24,21,26,
#'   36,38,37,39,45,42,41,37,38,38,35,37,35,31,32,30,20,39,
#'   40,33,32,35,34,36,34,32,33,27,28,25,22,17,18,16,10,9,
#'   5,12,7,8,8,9,19,21,24,20,23,19,17,18,17,22,11,12,3,9,
#'   10,4,5,13,3,5,6,3,5,4,2,5,1,2,4,4,3,2,1) 
#'   
#' # Train hidden Markov model for m = 4 -----
#' \donttest{
#'   m_trained_HMM <- 
#'      HMM_training(x = x, 
#'               min_m = 4, 
#'               max_m = 4, 
#'  distribution_class = "pois")$trained_HMM_with_selected_m
#'  
#'  # Decode the trained HMM using the Viterbi algorithm to get 
#'  # the globally most likely sequence of hidden states for 
#'  # the time-series of observations
#'  global_decoding <- 
#'  Viterbi_algorithm(x = x, 
#'                    m = m_trained_HMM$m, 
#'                delta = m_trained_HMM$delta, 
#'                gamma = m_trained_HMM$gamma, 
#'      distribution_class = m_trained_HMM$distribution_class, 
#'      distribution_theta = m_trained_HMM$distribution_theta)
#'      
#' # Most likely sequence of hidden states
#' 
#' print(global_decoding$decoding)
#' plot(global_decoding$decoding)
#' }  
#' 
Viterbi_algorithm <- function(x, m, delta, gamma, distribution_class, distribution_theta, 
                              discr_logL = FALSE, discr_logL_eps = 0.5)
{
  svtpoue <- small_value_to_prevent_overflow_and_underflow_errors <- 4.940656e-142
  
  size <- length(x)
  
  function_discr_log_L_p_norm <- function(x, mean, sd, discr_logL_eps)
  {
    foo <- pnorm((x + discr_logL_eps), mean = mean, sd = sd) - pnorm((x - discr_logL_eps), mean = mean, sd = sd)	
  return(foo)
  }    
    
# Calcualtion of probabilities -----
  
  if (distribution_class == "pois") 
  {
    distribution_means <- distribution_theta$lambda		
    probabilities <- outer(X = x, Y = distribution_theta$lambda, FUN = dpois)
  }
  
  if (distribution_class == "geom") 
  {
    distribution_means <- 1 / distribution_theta$prob$prob				
	  probabilities <- matrix(x, ncol = m, nrow = size)
    probabilities <- t(apply(X = probabilities, MARGIN = 1, FUN = dgeom, prob = distribution_theta$prob))	
  }
  
  if (distribution_class == "genpois") 
  {	 
    distribution_means <- distribution_theta$lambda1 / (1 - distribution_theta$lambda2) 
    probabilities <- matrix(x, ncol = m, nrow = size)
    probabilities <- t(apply(X = probabilities, MARGIN = 1, FUN = dgenpois, 
                             lambda1 = distribution_theta$lambda1, 
                             lambda2 = distribution_theta$lambda2))
  }
  
  if (distribution_class == "norm" & discr_logL == FALSE)
  {	
    distribution_means <- distribution_theta$mean
    probabilities <- matrix(x, ncol = m, nrow = size)
    probabilities <- t(apply(X = probabilities, MARGIN = 1, FUN = dnorm, 
                             mean = distribution_theta$mean, sd = distribution_theta$sd))
  }
  
  if (distribution_class == "norm" & discr_logL == TRUE)
  {	
    distribution_means <- distribution_theta$mean
    probabilities <- matrix(x, ncol = m, nrow = size)
    probabilities <- t(apply(X = probabilities, MARGIN = 1, 
                             FUN = function_discr_log_L_p_norm,
                             mean = distribution_theta$mean, 
                             sd = distribution_theta$sd, 
                             discr_logL_eps = discr_logL_eps))
  }

    probabilities <- ifelse(!is.na(probabilities), probabilities, svtpoue)
    probabilities <- ifelse(!probabilities <= 0, probabilities, svtpoue) 		
    probabilities <- ifelse(!probabilities == Inf, probabilities, svtpoue)	
    probabilities <- ifelse(!probabilities == -Inf, probabilities, svtpoue) 

  omega <- matrix(svtpoue, ncol = m, nrow = size)
  foo <- delta * probabilities[1,]
  omega[1,] <- foo / sum(foo)
  for(i in 2:size)
  {
    foo <- apply(omega[i-1,] * gamma, 2, max) * probabilities[i,]
    omega[i,] <- foo / sum(foo)
  }
  
  decoding <- numeric(size)
  decoding[size] <- which.max(omega[size,])
  for(i in (size - 1):1)
  {
    decoding[i] <- which.max(gamma[,decoding[i+1]] * omega[i,])
  }

 
return(list(omega=omega, 
            decoding=decoding))
}
