% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MR.quantile.r
\name{MR.quantile}
\alias{MR.quantile}
\title{Multiply Robust Estimation of the Marginal Quantile}
\usage{
MR.quantile(response, tau = 0.5, imp.model = NULL, mis.model = NULL,
  moment = NULL, order = 1, L = 30, data, bootstrap = FALSE,
  bootstrap.size = 300, alpha = 0.05)
}
\arguments{
\item{response}{The response variable of interest whose marginal quantile is to be estimated.}

\item{tau}{A numeric value in (0,1). The quantile to be estimated.}

\item{imp.model}{A list of imputation models defined by \code{\link{glm.work}}.}

\item{mis.model}{A list of missingness probability models defined by \code{\link{glm.work}}. The dependent variable is always specified as \code{R}.}

\item{moment}{A vector of auxiliary variables whose moments are to be calibrated.}

\item{order}{A numeric value. The order of moments up to which to be calibrated.}

\item{L}{Number of imputations.}

\item{data}{A data frame with missing data encoded as \code{NA}.}

\item{bootstrap}{Logical. If \code{bootstrap = TRUE}, the bootstrap will be applied to calculate the standard error and construct a Wald confidence interval.}

\item{bootstrap.size}{A numeric value. Number of bootstrap resamples generated if \code{bootstrap = TRUE}.}

\item{alpha}{Significance level used to construct the 100(1 - \code{alpha})\% Wald confidence interval.}
}
\value{
\item{\code{q}}{The estimated value of the marginal quantile.}
\item{\code{SE}}{The bootstrap standard error of \code{q} when \code{bootstrap = TRUE}.}
\item{\code{CI}}{A Wald-type confidence interval based on \code{q} and \code{SE} when \code{bootstrap = TRUE}.}
\item{\code{weights}}{The calibration weights if any \code{imp.model}, \code{mis.model} or \code{moment} is specified.}
}
\description{
\code{MR.quantile()} is used to estimate the marginal quantile of a variable which is subject to missingness. Multiple missingness probability models and imputation models are allowed.
}
\examples{
# Simulated data set
set.seed(123)
n <- 400
gamma0 <- c(1, 2, 3)
alpha0 <- c(-0.8, -0.5, 0.3)
X <- runif(n, min = -2.5, max = 2.5)
p.mis <- 1 / (1 + exp(alpha0[1] + alpha0[2] * X + alpha0[3] * X ^ 2))
R <- rbinom(n, size = 1, prob = 1 - p.mis)
a.x <- gamma0[1] + gamma0[2] * X + gamma0[3] * exp(X)
Y <- rnorm(n, a.x, sd = sqrt(4 * X ^ 2 + 2))
dat <- data.frame(X, Y)
dat[R == 0, 2] <- NA

# Define the outcome regression models and missingness probability models
imp1 <- glm.work(formula = Y ~ X + exp(X), family = gaussian)
imp2 <- glm.work(formula = Y ~ X + I(X ^ 2), family = gaussian)
mis1 <- glm.work(formula = R ~ X + I(X ^ 2), family = binomial(link = logit))
mis2 <- glm.work(formula = R ~ X + exp(X), family = binomial(link = cloglog))
MR.quantile(response = Y, tau = 0.25, imp.model = list(imp1, imp2),
            mis.model = list(mis1, mis2), L = 10, data = dat)
MR.quantile(response = Y, tau = 0.25, moment = c(X), order = 2, data = dat)
}
\references{
Han, P., Kong, L., Zhao, J. and Zhou, X. (2019). A general framework for quantile estimation with incomplete data. \emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)}. \strong{81}(2), 305--333.
}
