% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NPT_discrete.R
\name{NPT_discrete}
\alias{NPT_discrete}
\title{Discrete Niche Classification Based on Periodic Table of Niches}
\usage{
NPT_discrete(data, dimension, clustering_method = "CART", k_max = 6)
}
\arguments{
\item{data}{A data frame containing species and their functional trait measurements.
Each row represents a species and columns contain trait values.}

\item{dimension}{A named list where each element represents a niche dimension
(e.g., "Growth", "Survival", "Reproduction") and contains a character vector of
column names corresponding to traits associated with that dimension.}

\item{clustering_method}{Character string specifying the clustering method to use.
Options are:
\itemize{
\item \code{"CART"} - Classification and Regression Trees (default)
\item \code{"kmeans"} - K-means clustering with automatic k selection
}}

\item{k_max}{Integer specifying the maximum number of clusters allowed for k-means
clustering. Default is \code{6}. This parameter is only used when
\code{clustering_method = "kmeans"}. The optimal k value will be selected using
the elbow method, constrained by this maximum value.}
}
\value{
A list containing two elements:
\item{niche_classification}{A data frame with species names, cluster assignments
for each dimension, and comprehensive niche codes}
\item{summary}{A summary data frame showing unique niche codes, the number of
species in each niche, and lists of species names}
}
\description{
This function implements a discrete niche classification scheme based on the
periodic table of niches concept. It performs Principal Component Analysis (PCA)
on functional traits grouped by niche dimensions, followed by clustering to
create hierarchical niche classifications.
}
\details{
The function implements the methodology described in Winemiller et al. (2015)
for creating discrete niche classification schemes. The approach follows three
main steps:

\strong{Step 1: PCA Analysis by Dimension}

Separate Principal Component Analysis is performed on trait data for each niche
dimension. This dimensional approach prevents functionally unrelated traits from
masking important ecological patterns. The first two principal components are
retained for each dimension.

\strong{Step 2: Clustering Methods}

Two clustering approaches are available:
\itemize{
\item \strong{CART}: Uses the Euclidean distance from the origin in PCA space
as the response variable and original trait values as predictors to create
regression trees. The tree is pruned using cross-validation.
\item \strong{k-means}: Performs clustering on the two-dimensional PCA space
with automatic optimal k selection using the elbow method. The optimal k
is constrained by the \code{k_max} parameter (default maximum = 6).
}

\strong{Step 3: Hierarchical Niche Classification}

The function combines clustering results from all niche dimensions to create
a comprehensive niche classification scheme. Each species receives a niche code
representing its cluster membership across all dimensions (e.g., "1,2,1" for
cluster 1 in dimension 1, cluster 2 in dimension 2, and cluster 1 in dimension 3).

The function also calculates niche occupancy statistics, comparing the number
of realized niches to the total number of potential niche combinations.
}
\note{
\itemize{
\item Missing values should be handled prior to using this function.
\item The function prints diagnostic information during execution, including
variance explained by PCs and clustering results.
\item For k-means clustering, setting a lower \code{k_max} value will force
simpler clustering solutions, while higher values allow for more complex
niche subdivisions.
\item \strong{Randomness warning:} The \code{kmeans} method involves
random initialization of cluster centers, so results may vary
between runs. For reproducibility, set a random seed using
\code{set.seed()} before running this function. CART results may
also differ slightly if predictor splitting involves tie-breaking.
}
}
\examples{
\dontrun{
# Load and prepare data
data(PFF)
rownames(PFF) <- PFF$species
PFF_traits <- PFF[, c("SLA", "SRL", "Leaf_Nmass", "Root_Nmass","Height",
                      "Leaf_CN", "Root_CN","SeedMass", "FltDate", "FltDur")]
# Perform log transformation of data and remove missing values
PFF_traits <- log(na.omit(PFF_traits))
head(PFF_traits)
# Define trait dimensions
dimension <- list(Grow = c("SLA", "SRL", "Leaf_Nmass", "Root_Nmass"),
                  Survive = c("Height", "Leaf_CN", "Root_CN"),
                  Reproductive = c("SeedMass", "FltDate", "FltDur"))

set.seed(123)
discrete_result <- NPT_discrete(data = PFF_traits, dimension = dimension)
head(discrete_result$niche_classification)
}

}
\references{
\enumerate{
\item Winemiller, K. O., Fitzgerald, D. B., Bower, L. M., & Pianka, E. R. (2015).
Functional traits, convergent evolution, and periodic tables of niches.
Ecology letters, 18(8), 737-751.
\item Pianka, E. R., Vitt, L. J., Pelegrin, N., Fitzgerald, D. B., & Winemiller, K. O. (2017).
Toward a periodic table of niches, or exploring the lizard niche hypervolume.
The American Naturalist, 190(5), 601-616.
}
}
