# HCInvNHE.R
#' Half-Cauchy Inverse NHE Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Inverse NHE distribution.
#'
#' The Half-Cauchy Inverse NHE distribution is parameterized by the parameters
#' \eqn{\beta > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy Inverse NHE distribution has CDF:
#'
#' \deqn{
#' F(x; \beta, \lambda, \theta) =
#' \quad 1 - \frac{2}{\pi }\arctan \left[ { - \frac{1}{\theta }\left\{ {1 - {{\left( 
#' {1 + \frac{\lambda }{x}} \right)}^\beta }} \right\}} \right] \quad ;\;x > 0.
#' }
#'
#' where \eqn{\beta}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' Included functions are:
#' \itemize{
#'   \item \code{dhc.inv.NHE()} — Density function
#'   \item \code{phc.inv.NHE()} — Distribution function
#'   \item \code{qhc.inv.NHE()} — Quantile function
#'   \item \code{rhc.inv.NHE()} — Random generation
#'   \item \code{hhc.inv.NHE()} — Hazard function
#' }
#'
#' @usage
#' dhc.inv.NHE(x, beta, lambda, theta, log = FALSE)
#' phc.inv.NHE(q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.inv.NHE(p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.inv.NHE(n, beta, lambda, theta)
#' hhc.inv.NHE(x, beta, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.inv.NHE}: numeric vector of (log-)densities
#'   \item \code{phc.inv.NHE}: numeric vector of probabilities
#'   \item \code{qhc.inv.NHE}: numeric vector of quantiles
#'   \item \code{rhc.inv.NHE}: numeric vector of random variates
#'   \item \code{hhc.inv.NHE}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary, A.K., Telee, L.B.S. & Kumar,V. (2022). 
#' Half-Cauchy Inverse NHE Distribution: Properties and Applications.
#' \emph{ Nepal Journal of Mathematical Sciences (NJMS)}, \bold{3(2)}, 
#' 1--12.  \doi{10.3126/njmathsci.v3i2.49198}
#'
#' Chaudhary, A. K., Sapkota, L. P., & Kumar, V. (2022). 
#' Some properties and applications of half Cauchy extended exponential 
#' distribution.\emph{ Int. J. Stat. Appl. Math.}, \bold{7(4)}, 226--235. 
#' \doi{10.22271/maths.2022.v7.i4c.866}
#'
#' Chaudhary, A.K., & Kumar, V. (2022). 
#' Half Cauchy-Modified Exponential Distribution: Properties and Applications. 
#' \emph{ Nepal Journal of Mathematical Sciences (NJMS)}, \bold{3(1)}, 
#' 47--58. \doi{10.3126/njmathsci.v3i1.44125}
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dhc.inv.NHE(x, 2.0, 0.5, 2.5)
#' phc.inv.NHE(x, 2.0, 0.5, 2.5)
#' qhc.inv.NHE(0.5, 2.0, 0.5, 2.5)
#' rhc.inv.NHE(10, 2.0, 0.5, 2.5)
#' hhc.inv.NHE(x, 2.0, 0.5, 2.5)
#' 
#' # Data
#' x <- relief
#' # ML estimates
#' params = list(beta=79.7799, lambda=0.1129, theta=154.1769)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.inv.NHE, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.inv.NHE, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params,
#'              dfun = dhc.inv.NHE, pfun=phc.inv.NHE, plot=FALSE)
#' print.gofic(res)
#' 
#' @name HCInvNHE
#' @aliases dhc.inv.NHE phc.inv.NHE qhc.inv.NHE rhc.inv.NHE hhc.inv.NHE
#' @export
NULL

#' @rdname HCInvNHE
#' @usage NULL
#' @export   
dhc.inv.NHE <- function (x, beta, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, beta, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(beta, lambda, theta) <= 0)) {
        stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- (1+(lambda/x[valid]))	
	z <- -(1 - u ^ beta)/theta	
    pdf[valid] <- (2/pi)*(beta*lambda/theta)* (u^(beta-1)) * (x[valid]^-2) * (1/(1+ (z^2)))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCInvNHE
#' @usage NULL
#' @export
phc.inv.NHE <- function (q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, beta, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(beta, lambda, theta) <= 0)) {
		stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -(1/theta)* (1-(1+(lambda/q[valid]))^beta)
    cdf[valid] <- 1 - (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCInvNHE
#' @usage NULL
#' @export
qhc.inv.NHE <- function (p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <-  lambda/((1+ theta*tan((1-p[valid]) * pi *0.5)) ^ (1/beta) -1)	   
	}    
    return(qtl)  
 }  

#' @rdname HCInvNHE
#' @usage NULL
#' @export
rhc.inv.NHE <- function(n, beta, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(beta, lambda, theta), is.numeric)))
		stop("'beta', 'lambda', and 'theta' must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.inv.NHE(u, beta, lambda, theta)
    return(q)
}

#' @rdname HCInvNHE
#' @usage NULL
#' @export
hhc.inv.NHE <- function (x, beta, lambda, theta)
{
	if (!all(sapply(list(x, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")    
    nume <- dhc.inv.NHE(x, beta, lambda, theta)
	surv <- 1.0 - phc.inv.NHE(x, beta, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf) 
}




