# LindleyGIE.R
#' Lindley Generalized Inverted Exponential(LGIE) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the LGIE distribution.
#'
#' The LGIE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The LGIE distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' 1-\left(1-e^{-\lambda / x}\right)^{\alpha \theta}\left[1-\left(\frac{\theta}
#' {\theta+1}\right) \ln \left(1-e^{-\lambda / x}\right)^\alpha\right]  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlind.ginv.exp()} — Density function
#'   \item \code{plind.ginv.exp()} — Distribution function
#'   \item \code{qlind.ginv.exp()} — Quantile function
#'   \item \code{rlind.ginv.exp()} — Random generation
#'   \item \code{hlind.ginv.exp()} — Hazard function
#' }
#'
#' @usage
#' dlind.ginv.exp(x, alpha, lambda, theta, log = FALSE)
#' plind.ginv.exp(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlind.ginv.exp(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlind.ginv.exp(n, alpha, lambda, theta)
#' hlind.ginv.exp(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlind.ginv.exp}: numeric vector of (log-)densities
#'   \item \code{plind.ginv.exp}: numeric vector of probabilities
#'   \item \code{qlind.ginv.exp}: numeric vector of quantiles
#'   \item \code{rlind.ginv.exp}: numeric vector of random variates
#'   \item \code{hlind.ginv.exp}: numeric vector of hazard values
#' }
#'
#' @references 
#' Telee, L. B. S., & Kumar, V. (2021).
#' Lindley Generalized Inverted Exponential Distribution: Model and Applications.
#' \emph{Pravaha}, \bold{27(1)}, 61--72. \doi{10.3126/pravaha.v27i1.50616}   
#'
#' Yadav, R.S., & Kumar, V. (2020). Arctan Generalized Inverted Exponential Distribution. 
#' \emph{J. Nat. Acad. Math.}, \bold{34}, 71--92.
#'
#' @examples
#' x <- seq(5, 10, 0.2)
#' dlind.ginv.exp(x, 5.0, 1.5, 0.5)
#' plind.ginv.exp(x, 5.0, 1.5, 0.5)
#' qlind.ginv.exp(0.5, 5.0, 1.5, 0.5)
#' rlind.ginv.exp(10, 5.0, 1.5, 0.5)
#' hlind.ginv.exp(x, 5.0, 1.5, 0.5)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=97.0105, lambda=29.9324, theta=0.9028)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plind.ginv.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlind.ginv.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlind.ginv.exp, pfun=plind.ginv.exp, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyGIE
#' @aliases dlind.ginv.exp plind.ginv.exp qlind.ginv.exp rlind.ginv.exp hlind.ginv.exp
#' @export
NULL

#' @rdname LindleyGIE
#' @usage NULL
#' @export   
dlind.ginv.exp  <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- 1-exp(-lambda/x[valid])
    pdf[valid] <- alpha*lambda*theta^2/(x^2*(theta+1)) *exp(-lambda/x[valid])*u^(alpha*theta-1)*(1-log(u^alpha))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyGIE
#' @usage NULL
#' @export 
plind.ginv.exp  <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
		u <- 1-exp(-lambda/q[valid])
       cdf[valid] <- 1-u^(alpha*theta)*(1-(theta/(theta+1))*log(u^alpha))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyGIE
#' @usage NULL
#' @export 
qlind.ginv.exp  <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Parameters 'alpha', 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	((1-exp(-lambda/x))^(alpha*theta))*(1-(theta/(theta+1))*log((1-exp(-lambda/x))^alpha))+wi-1   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LindleyGIE
#' @usage NULL
#' @export 
rlind.ginv.exp  <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qlind.ginv.exp (u, alpha, lambda, theta)
    return(q)
}

#' @rdname LindleyGIE
#' @usage NULL
#' @export 
hlind.ginv.exp  <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
		
	nume <- dlind.ginv.exp (x, alpha, lambda, theta) 
	surv <- 1.0 - plind.ginv.exp (x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
	
    return(hrf)   
}
