# ModUbd.R
#' Modified UBD (MUBD) Distribution
#'
#' Density, distribution function, quantile function, random generation,
#' and hazard rate function for the Modified UBD (MUBD) distribution.
#'
#' @details
#' The Modified UBD (MUBD) distribution is a flexible lifetime distribution
#' with positive shape parameters \eqn{\alpha > 0}, \eqn{\beta > 0}
#' and scale parameter \eqn{\lambda > 0}.
#'
#' The MUDB distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\exp \left\{1-\left(1+x^\beta e^{-\lambda / x}\right)
#' ^\alpha\right\}  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' @usage
#' dmod.ubd(x, alpha, beta, lambda, log = FALSE)
#' pmod.ubd(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qmod.ubd(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rmod.ubd(n, alpha, beta, lambda)
#' hmod.ubd(x, alpha, beta, lambda)
#' 
#' @param x Vector of positive quantiles.
#' @param q Vector of positive quantiles.
#' @param p Vector of probabilities.
#' @param n Number of random observations. Must be a positive integer.
#' @param alpha Shape parameter (\eqn{\alpha > 0}).
#' @param beta Shape parameter (\eqn{\beta > 0}).
#' @param lambda Scale parameter (\eqn{\lambda > 0}).
#' @param log Logical; if TRUE, returns the log-density.
#' @param lower.tail Logical; if TRUE (default), returns \eqn{P(X \le x)}.
#' @param log.p Logical; if TRUE, probabilities are returned on the log scale.
#'
#' @return
#' \code{dmod.ubd} returns the probability density function.  
#' \code{pmod.ubd} returns the cumulative distribution function.  
#' \code{qmod.ubd} returns the quantile function.  
#' \code{rmod.ubd} generates random variates.  
#' \code{hmod.ubd} returns the hazard rate function.
#'
#' @references
#' Chaudhary, A.K., Telee, L. B. S., & Kumar, V. (2023).
#' Modified Upside Down Bathtub-Shaped Hazard Function Distribution:
#' Properties and Applications. \emph{Journal of Econometrics and Statistics},
#' \bold{3(1)}, 107--120.
#'
#' @aliases dmod.ubd pmod.ubd qmod.ubd rmod.ubd hmod.ubd
#'
#' @examples
#' x <- seq(0.1, 1, by=0.1)
#' dmod.ubd(x, alpha = 1.5, beta = 1.2, lambda = 2)
#' pmod.ubd(x, alpha = 1.5, beta = 1.2, lambda = 2)
#' qmod.ubd(0.5, alpha = 1.5, beta = 1.2, lambda = 2)
#' rmod.ubd(10, alpha = 1.5, beta = 1.2, lambda = 2)
#' hmod.ubd(x, alpha = 1.5, beta = 1.2, lambda = 2)
#'
#' # Data
#' x <- fibers69
#' # ML estimates
#' params = list(alpha=0.8559, beta=3.0133, lambda=7.0336)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pmod.ubd, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qmod.ubd, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dmod.ubd, pfun=pmod.ubd, plot=TRUE)
#' print.gofic(out)
#'
#' @name ModUbd
#' @rdname ModUbd
#' @export
NULL

#' @rdname ModUbd
#' @usage NULL
#' @export
dmod.ubd <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- lambda/x[valid]	
	w <- exp(-u)
	z <- 1 + (x[valid] ^ beta) * w
    pdf[valid] <- alpha*(beta+u)*(x[valid]^(beta-1)) * w * (z ^ (alpha-1)) * exp(1-(z^ alpha))  
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname ModUbd
#' @usage NULL
#' @export
pmod.ubd <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	z <- 1 + (q[valid] ^ beta) * exp(-lambda/q[valid])
    cdf[valid] <- 1.0- exp(1.0 - z ^ alpha)   
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname ModUbd
#' @usage NULL
#' @export
qmod.ubd <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	beta*log(x)-(lambda/x)-log((1-log(1-wi)) ^(1/alpha) -1)   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname ModUbd
#' @usage NULL
#' @export
rmod.ubd <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qmod.ubd(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ModUbd
#' @usage NULL
#' @export
hmod.ubd <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")     
    nume <- dmod.ubd(x, alpha, beta, lambda)
	surv <- 1.0 - pmod.ubd(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
