source(paste(DSE.HOME,"/data/tf1dec93.dat", sep=""))
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



############################################################################

#    functions for data analysis      <<<<<<<<<<<<<

############################################################################
phase.plots <- function(data, max.lag=5,diff=F){
# data is a matrix with a variable in each column.(each column is a
# time series), or an object of class TSdata, inwhich case output.data(data) is used.
# trace plots of data and lagged (difference) data (phase space).
# Non-linearities may show up as a non-linear surface, but this is
#   a projection so, for example, a spherical space would not show up.
#  Some sort of cross-section window would show this but require even more plots.
# A good statistical test would be better!

par(mfcol=c(5,5))  #,mar=c(2.1,4.1,4.1,0.1) )
if (data == "TSdata") data <- output.data(data)
Time <- dim(data)[1]
p <-dim(data)[2]
d <- array(NA,c(Time,p,1+max.lag))
d[,,1] <- data
for (l in 1:max.lag) 
  if (diff) {d[(1+l):Time,,l+1] <- d[(1+l):Time,,l]-d[l:(Time-1),,l]}
  else      {d[(1+l):Time,,l+1] <- d[l:(Time-1),,l] } #lag
for (l in 0:max.lag)
   for (l2 in 0:max.lag)
      for(i in 1:p) 
         for(i2 in 1:p) 
           {plot(d[,i,(l+1)],d[,i2,(l2+1)],type="l",xlab="",ylb="")
            title(main=paste("[",i,"L",l,",",i2,"L",l2,"]"))
           }
invisible()
}


############################################################################

#  functions for model estimation (see also VARX in dse.s) and reduction   <<<<<<<<<<<<<

############################################################################

est.wt.variables <-function(data, variable.weights,
                        estimation="est.VARX.ls", estimation.args=NULL)
{# estimation.args should be NULL if no args are needed.
 # scale each output variable in data according to variable.weights, estimate,
 # then unscale estimated model.
 if (is.matrix(variable.weights))
    {if (any(svd(variable.weights)$d == 0))  
       stop("variable.weights transformation must be non singular.")
    }
 else
   {if (any(variable.weights == 0))  stop("variable.weights must be non zero.")
    variable.weights <- diag(variable.weights)
   }
 inv.wts <- solve(variable.weights)
 dimnames(inv.wts)          <-list(NULL, dimnames(output.data(data))[[2]])
 dimnames(variable.weights) <-list(NULL, dimnames(output.data(data))[[2]])
 scaled.model <- do.call(estimation, append(list(
           TSdata(scale(data, list(output=inv.wts)))), estimation.args))
 model <-scale(scaled.model$model, list(output=variable.weights))
 model$description <- 
    paste("model estimated by est.wt.variables with", estimation)
 l(model, data)
}

est.min.sqerror <- function(data, init.model, error.weights=1, ...) 
{# error.weights indicates the weight to place on periods in the forecast.
 minimize.TSestModel( l(init.model, data), obj.func=sum.sqerror, obj.func.args=list(error.weights=error.weights), ...)
}

est.max.like <- function(data, init.model, ...) 
  {max.like( l(init.model, data), ...)}

max.like <- function(obj,...)UseMethod("minimize")

minimize.TSestModel <- function(emodel, algorithm="nlmin",
      max.iter=20, ftol=1e-5, gtol=1e-3,
      dfunc=numerical.grad, line.search="nlmin", 
      obj.func=like, obj.func.args=NULL) 
{# maximum likelihood estimation...
 # emodel is an object of class TSestModel (with initial parameter values and data). 
 # max.iter is an integer indicating the maximum number of iterations.
 # The value returned is an object of class TSestModel with  additional
 #  elements $converged, which is T or F indicating convergence, 
 #  and $dfpMin.results or $nlmin.results.
 # If this function calls dfp the Hessian,etc are return as  $dfpMin.results
 # If this function is called again and those results are
 # available then they are used. 
 # This could cause problems if $model is modified. If that is
 # done then $dfpMin.results should be set to NULL.
 # algorithm in {"dfpMin","nlmin", "nlsimplex"}

 if(!is.TSestModel(emodel)) TS.error.exit()
 est  <- emodel$estimates
 Shape <- emodel$model
 Data <- TSdata(emodel$data)
 assign("Obj.Func.ARGS" , append(list(model=Shape, data=Data),
        obj.func.args),  frame = 1)
 if (algorithm=="dfpMin")
    {if (is.null(emodel$dfpMin.results)) parms <- Shape$parms
     else parms <- emodel$dfpMin.results
     dfpMin.results <- dfpMin(obj.func, parms, dfunc=dfunc, 
         max.iter=max.iter, ftol=ftol, gtol=gtol, line.search=line.search) 
     Shape$parms <- dfpMin.results$parms
     emodel <- l(set.arrays(Shape),Data)
     emodel$dfpMin.results <- dfpMin.results
     emodel$converged <- dfpMin.results$converged
     emodel$model$description <- paste("Estimated with max.like/dfpMin (",
       c("not converged", "converged")[1+emodel$converged],
       ") from initial model: ", emodel$model$description)
    }
 else if (algorithm=="nlmin")
   {nlmin.results <-nlmin(obj.func,Shape$parms, max.iter=max.iter, max.fcal=5*max.iter, ckfc=0.01)
    Shape$parms <- nlmin.results$x
    emodel <- l(set.arrays(Shape),Data)
    emodel$nlmin.results <- nlmin.results
    emodel$converged <- nlmin.results$converged  # this should be improved with conv.type info
    emodel$model$description <- paste("Estimated with max.like/nlmin (",
       c("not converged", "converged")[1+emodel$converged],
       ") from initial model: ", emodel$model$description)
   }
 else if (algorithm=="nlsimplex")
   {results <-nlsimplex(obj.func,Shape$parms, max.iter=max.iter)
    Shape$parms <- results$x
    emodel <- l(set.arrays(Shape),Data)
    emodel$nlsimplex.results <- results
    emodel$converged <- results$converged  # this should be improved with conv.type info
    emodel$model$description <- paste("Estimated with max.like/nlsimplex (",
       c("not converged", "converged")[1+emodel$converged],
       ") from initial model: ", emodel$model$description)
   }
  else stop(paste("Minimization method ", algorithm, " not supported."))
# remove(c("Obj.Func.ARGS"), where=1)
 emodel
}    

est.black.box <- function(data,...)
{# call current best black box technique.
  est.black.box4(data, ...)
}

est.black.box1 <- function(data,estimation="est.VARX.ls", reduction="reduction.Mittnik", 
        criterion="taic", trend=F, subtract.means=F, verbose=T, max.lag=6)
{if ((estimation!="est.VARX.ls") && (trend) )
     {cat("Trend estimation only support with est.VARX.ls.\n")
      cat("Proceeding using est.VARX.ls.\n")
      estimation<-"est.VARX.ls"
     }

 if(estimation=="est.VARX.ls")
     model <- est.VARX.ls(data,trend=trend, subtract.means=subtract.means, 
                          max.lag=max.lag)
 else if(estimation=="est.VARX.ar")
     model <- est.VARX.ar(data, subtract.means=subtract.means, max.lag=max.lag)
 else if(estimation=="est.VARX")
     model <- est.VARX(data,trend=trend, subtract.means=subtract.means, 
                        max.lag=max.lag)
 else if(estimation=="est.SS.Mittnik")
     model <- est.SS.Mittnik(data,max.lag=max.lag, 
                             subtract.means=subtract.means, normalize=F)
 else
   stop("estimation technique not supported.")
 if (verbose) 
   cat("First VAR model,              lags= ", dim(model$model$A)[1]-1,
       ", -log likelihood = ", model$estimates$like[1], "\n")
 model <- l(to.SS(model),data)
 n <- dim(model$model$F)[1]
 if (verbose) cat("Equivalent    state space model, n= ", n,
                  ", -log likelihood = ", model$estimates$like[1], "\n")
 if (1 < n)
   {model <- eval(call(reduction,model,criterion=criterion, verbose=verbose))
    if (verbose) 
       cat("Final reduced state space model, n= ", dim(model$model$F)[1],
           ", -log likelihood = ", model$estimates$like[1], "\n")
   }
  if (verbose && exists.graphics.device()) check.residuals(model)
 model
}

est.SS.Mittnik <- function(data, max.lag=6, n=NULL, subtract.means=F, normalize=F)
{#  estimate a nested-balanced state space model by svd from least squares
 # estimate of markov parameters a la Mittnik p1195 Comp.Math Appl.v17,1989.
 # The quality of the estimate seems to be quite sensitive to max.lag, 
 #   and this is not properly resolved yet.
 # If n is not supplied the svd criteria will be printed and n prompted for.
 # If subtract.means=T then the sample mean is subtracted. 
 # If normalize is T the lsfit estimation is done with outputs normalize to cov=I
 # (There still seems to be something wrong here!!).
 # The model is then re-transformed to the original scale.
 
  data <- TSdata(data)
  m <- ncol(input.data(data))
  if(is.null(m))  m <- 0
  p <- ncol(output.data(data))
  N <- nrow(output.data(data))
  if (subtract.means)
    {if(m!=0)input.data(data)<-input.data(data)-t(matrix(apply(input.data(data),2, mean), m,N))
     output.data(data)<- output.data(data) - t(matrix(apply(output.data(data),2, mean), p,N))
    }
  if (normalize)
    {svd.cov <- svd(var(output.data(data)))
     output.data(data) <- output.data(data) %*% svd.cov$u %*% diag(1/svd.cov$d^.5)
    }
      # shift input to give feedthrough in one period
  if (m != 0) {z <- cbind(input.data(data)[2:N,],output.data(data)[1:(N-1),])}
  else z <- output.data(data)
  Past <- matrix(NA,N-1-max.lag,(p+m)*(1+max.lag))
  for (i in 0:max.lag) 
    Past[,(1+(m+p)*i):((m+p)*(1+i))] <-z[(1+max.lag-i):(N-1-i),]
  M <- t(lsfit(Past,output.data(data)[(max.lag+2):N,],intercept=F)$coef)
  if (normalize && (m!=0))  # correct exogenous blocks for normalization
    {Tinv <- diag(svd.cov$d^.5)%*%svd.cov$u
     for (i in 0:max.lag) 
       M[,(1+(m+p)*i):(m+(m+p)*i)] <- Tinv %*% M[,(1+(m+p)*i):(m+(m+p)*i),drop=F]
    }
  if (p==1) M <-matrix(M,1,length(M))
#  browser()
  model <-balance.Mittnik.svd( M, m=m, n=n )$model
  z <-"nested-balanced model from least sq. estimates of markov parameters a la Mittnik"
  if(subtract.means) z <-paste(z," - means subtracted")
  if(normalize)      z <-paste(z," - outputs normalized")
  model$description  <-z
  series.names(model)  <- series.names(data)
  KF(model, data)
}

reduction.Mittnik <- function(model, data=NULL, criterion=NULL, verbose=T)
{# nested-balanced state space model reduction by svd of Hankel generated from a model
# If a state space model is supplied the max. state dimension for the result is
#  taken from the model. If an ARMA model is supplied then singular values 
#  will be printed and the program prompts for the max. state dimension.

  if(!is.TSm.or.em(model)) TS.error.exit()
  if (is.TSestModel(model)) 
    {data <-model$data
     model <- model$model
    }
  else if(is.null(data))
    stop("Reduction requires data to calculate criteria (balancing does not).")
  if(is.SS(model)) nMax <- dim(model$F)[2]
  else nMax <-NULL
  reduction.Mittnik.from.Hankel( markov.parms(model), nMax=nMax, data=data, 
        criterion=criterion, verbose=verbose)
}

reduction.Mittnik.from.Hankel<- function(M, data=NULL, nMax=NULL, criterion=NULL, verbose=T, spawn=.SPAWN)
{# Select a reduced state space model by svd a la Mittnik.
 #  Models and several criteria for all state dimensions up to the max.  
 #  state dim. specified are calculated. (If nMax is not supplied then svd
 #  criteria are printed and the program prompts for nMax). 
 #  If criteria is not specified then the program prompts for the 
 #   dimension (n) to use for the returned model.
 #  The program requires data to calculate selection criteria. (The program
 #  balance.Mittnik calculates svd criteria only and can be used for reduction
 #  without data.)
  # M is a matrix with p x (m+p)  blocks giving the markov parameters,
  # that is, the first row of the Hankel matrix. It can be generated from the
  # model as in the function markov.parms, or from the data, as in the function
  # est.SS.Mittnik.
  # The output dimension p is taken from nrow(M).
  # data is necessary only if criteria (AIC,etc) are to be calculated.
  # REFERENCES
  #   - S.Mittnik, Multivariate Time Series Analysis With State Space
  #     Models, Computers Math Appl. Vol 17, No 8/9, pp1189-1201, 1989.
  #
  #   - S.Mittnik, Macroeconomic Forecasting Experience With Balance State
  #     Space Models,  International Journal Of Forecasting, Vol 6,
  #     pp337-348, 1990.
  #
  #   - S.Mittnik, Forecasting With Balanced State Space Representations
  #     of Multivariate Distributed Lag Models.  Journal Of Forecasting,
  #     Vol.9, 207-218, 1990.  
 
   data <- TSdata(data)
   m <-ncol(input.data(data))      # dim of input series
   if(is.null(m))m<-0
   z <-balance.Mittnik.svd(M, m, nMax)
   largeModel <- z$model
   svd.crit    <-z$crit
   n <- dim(largeModel$F)[1]
   if (!spawn)
     {# The more complicated For loop used below is to avoid S memory problems
      #  with the more straight forward version:
      values <- NULL 
      for (i in 1:n) 
        {if(m!=0) z <-largeModel$G[1:i,,drop=F]
         else     z <-NULL
         z <-list(F=largeModel$F[1:i,1:i,drop=F],G=z,
                  H=largeModel$H[,1:i,drop=F],K= largeModel$K[1:i,,drop=F])
         class(z) <-c("innov","SS","TSmodel")
         z <- KF(set.parameters(z),data)
         z <-information.tests.calculations(z)
         values <-rbind(values, z)
         if (verbose) cat(".")
        }
      }
   else 
     {if (verbose) cat("Spawning processes to calculate criteria test for state dimension 1 to ",n)
      forloop <- function(largeModel, data)
           {if(!is.null(largeModel$G)) z <-largeModel$G[1:forloop.i,,drop=F]
            else                       z <-NULL
            z <-list(F=largeModel$F[1:forloop.i,1:forloop.i,drop=F],G=z,
                     H=largeModel$H[  , 1:forloop.i, drop=F],
                     K=largeModel$K[1:forloop.i,,drop=F])
            information.tests.calculations(KF(set.parameters(TSmodel(z)),data))
           }
       assign("balance.forloop", forloop, where=1)
       assign("forloop.n", n,   where=1 )
       assign("forloop.values", matrix(NA,n,12),   where=1 )
       assign("forloop.largeModel", largeModel, where=1)
       assign("forloop.data", data,   where=1 )
       on.exit(remove(c("balance.forloop", "forloop.i", "forloop.n", 
          "forloop.values", "forloop.largeModel", "forloop.data"),where=1))
       For (forloop.i=1:forloop.n,
         forloop.values[forloop.i,]<-balance.forloop(forloop.largeModel, 
         forloop.data), sync=T)
       values <-forloop.values
      }
    dimnames(values) <- list(NULL,c("port","like","aic","bic", 
          "gvc","rice","fpe","taic","tbic","tgvc","trice","tfpe")) 
    if (verbose) cat("\n")
    opt <-apply(values,2,order)[1,]  # minimum
    if (verbose | is.null(criterion))
      {zz<-criteria.table.nheading()
       options(width=120)
       print(values,digits=4)
       cat("opt     ")
       for (i in 1:length(opt)) cat(opt[i],"    ")
       cat("\n")
       zz<-criteria.table.legend()
      }
    if (is.null(criterion))
      {n <- eval(parse(prompt="Enter the state dimension (enter 0 to stop): "))
       if( n<1) stop("TERMINATED! STATE DIMENSION MUST BE GREATER THAN 0!")
      }
    else { n <- opt[criterion == dimnames(values)[[2]]]  }
    if(m==0) z <-NULL 
      else   z <-largeModel$G[1:n,,drop=F]
    z <- list(description="nested model a la Mittnik",
          F=largeModel$F[1:n,1:n,drop=F],G=z,
          H=largeModel$H[,1:n,drop=F],K= largeModel$K[1:n,,drop=F], 
          names=series.names(data))
    KF(set.parameters(TSmodel(z)),data)          
}


############################################################################

#    functions for model analysis   <<<<<<<<<<<<<

############################################################################

shock.decomposition <- function(model, horizon=30, shock=rep(1,horizon))
{ if(!is.TSm.or.em(model)) TS.error.exit()

  if (is.TSestModel(model)) 
     {data <- model$data   # just for $input
      model <- model$model
     }
  else 
     {m <- 1   # this needs more thought
      data <- list(input=matrix(0,horizon,m))
      class(data) <- "TSdata"
     }
   model$z0 <- NULL    # zero initial conditions
   m <-dim(model$G)[2]
   if (is.null(m)) m <-0
   p <-dim(model$H)[1]
   par(mfrow = c(p, p) , mar = c(2.1, 4.1,3.1, 0.1) )
   for (i in 1:p)
     {output.data(data) <- matrix(0, horizon, p)
      output.data(data)[,i] <- shock
      z <- l(model,data)
      graph(z, reset.screen=F)
     }
invisible()
}

############################################################################

#    functions for forecasting    <<<<<<<<<<<<<

# Class "feather.forecasts" has a forecast path from multiple starting points
#  in the data (so the graph may look like a feather).
# In the simplest case it would start from the end of the data 
#  and give the path out to a horizon.

############################################################################
############################################################################

#    methods for forecast       <<<<<<<<<<<<<

############################################################################
is.forecast <-function(obj) inherits(obj,"forecast")

forecast <-function(obj, ...) 
     UseMethod("forecast")
forecast.TSestModel <- function(model, ...)
     {forecast.TSmodel(model$model, model$data, ...)}
forecast.TSdata <- function(data, model, ...)
    {forecast.TSmodel(model, data, ...)}

forecast.TSmodel <- function(model, data,  horizon=36, conditioning.inputs=NULL, conditioning.inputs.forecasts=NULL, percent=NULL)
{# Calculate (multiple) forecasts from the end of data to a horizon determined
 # either from supplied input data or the argument horizon (more details below).
 # In  the case of a model with no inputs the horizon is determined by the
 #   argument horizon.
 # In the case of models with inputs, on which the forecasts
 #  are conditioned, the argument horizon is ignored (except when percent is
 #  specified) and the actual horizon is determined by the inputs in the 
 #  following way:
 # If inputs are not specified by optional arguments (as below) then the default
 #  will be to use input.data(data). This will be the same as the function l() unless
 #  input.data(data) is longer (after NAs are trimmed from each separately) than
 #  output.data(data).
 # Otherwise, if conditioning.inputs is specified it is used for input.data(data).
 #    It must be a time series matrix or a list of time series matrices each
 #    of which is used in turn as input.data(data). The default above is the same as
 #        forecast(model, trim.na(data), conditioning.inputs=trim.na(input.data(data)) )
 # Otherwise, if conditioning.inputs.forecasts is specified it is appended 
 #   to input.data(data). It must be a time series  
 #   matrix or a list of time series matrices each of which is 
 #   appended to input.data(data) and the concatenation used as conditioning.inputs.
 #   Both conditioning.inputs and conditioning.inputs.forecasts should not be
 #   specified.
 # Otherwise, if percent is specified then conditioning.inputs.forecasts are set
 #    to percent/100 times the value of input corresponding to the last period
 #    of output.data(data) and used for horizon periods. percent can be a vector, 
 #    in which case each value is applied in turn. ie c(90,100,110) would would 
 #    give results for conditioning.input.forcasts 10 percent above and below 
 #    the last value of input.

 # The result is an object of class forecast which is a list with 
 #   elements model, horizon, conditioning.inputs, percent, and forecast.
 #   forecast is a list with TSdata objects as elements, one for each element 
 #   in the list conditioning.inputs.

 if ((!is.null(conditioning.inputs))& (!is.null(conditioning.inputs.forecasts)))
       warning(paste("conditioning.inputs and conditioning.inputs.forecasts",
        " should not both be supplied. conditioning.inputs are being used."))

 if ((!is.null(conditioning.inputs))& (!is.null(percent)))
       warning(paste("conditioning.inputs and percent",
         " should not both be supplied. conditioning.inputs are being used."))

 if ((!is.null(percent))& (!is.null(conditioning.inputs.forecasts)))
    warning(paste("percent and conditioning.inputs.forecasts should not",
          " both be supplied. conditioning.inputs.forecasts are being used."))

 output <- trim.na(output.data(data))
 sampleT <- dim(output)[1]
 if (is.null(input.data(data))) 
     {pr <- l(model, data, sampleT=sampleT, 
                   predictT=dim(output)[1]+horizon)$estimates$pred
      pred <- window(pr, end=end(output), warn=F)
      pr <- window(pr, start=c(0,1)+end(output), warn=F)
    #  pr[1:(sampleT-1),] <- NA
    #  pr[sampleT,] <- output[sampleT,]
      proj <- list(pr)
     }
 else
  {if (!is.null(conditioning.inputs)) {} # do nothing
   else if (!is.null(conditioning.inputs.forecasts))
         {if (is.matrix(conditioning.inputs.forecasts)) 
            conditioning.inputs.forecasts <-list(conditioning.inputs.forecasts)
          conditioning.inputs <- list()
          for (i in 1:length(conditioning.inputs.forecasts) )
            {inp <-tframed(rbind(input.data(data),conditioning.inputs.forecasts[[i]]), 
              list(start=start(input.data(data)),
                   frequency=frequency(input.data(data))))
             conditioning.inputs <- append(conditioning.inputs, list(inp))
         }  }  
   else if (!is.null(percent))   
        {last.in <- input.data(data)[sampleT,]
         for (i in 1:length(percent) )
           {pol <- t(matrix(last.in*percent[i]/100, length(last.in), horizon))
            inp <-ts(rbind(input.data(data)[seq(sampleT),,drop=F],pol), 
                     start=start(input.data(data)),
                     frequency=frequency(input.data(data)))
            conditioning.inputs <- append(conditioning.inputs, list(inp))
        }  }  
   else conditioning.inputs <- trim.na(input.data(data))

   if (is.matrix(conditioning.inputs))
          conditioning.inputs <- list(conditioning.inputs)

   proj <- NULL
   for (policy in  conditioning.inputs)
        {pdata <- list(input=policy, output=output)
         class(pdata) <- "TSdata"
         if(2 != length(start(pdata)))
            stop("input and output data must have the same starting period (after NAs are removed).")
      predictT <- dim(policy)[1]
      if (sampleT > predictT) 
         stop("input series must be at least as long as output series (after NAs are removed).")
      pr <- l(model, pdata, sampleT=sampleT, predictT=predictT)$estimates$pred
#    The following lines sometimes cause problems if output is output[...,]
#    See comments in dse2.function.tests
        if (0 == length(proj)) pred <- window(pr, end=end(output), warn=F)
        if(all(end(pr)==end(output)))
          {pr <- NULL
           warning("Input is not longer than output data. No forecasts produced.") 
          }
        else pr <- window(pr, start=c(0,1)+end(output), warn=F)
     #    pr[1:(sampleT-1),] <- NA
     #    pr[sampleT,] <- output[sampleT,]  # so plots show first step
         proj <- append(proj, list(pr))
        }
   }
 proj <- list(model=model, data=data, 
                horizon=horizon, percent=percent,
                conditioning.inputs=conditioning.inputs,
                conditioning.inputs.forecasts=conditioning.inputs.forecasts,
                forecast=proj,pred=pred) 
 class(proj) <- "forecast"
 invisible(proj)
}

test.equal.forecast <-function(obj1, obj2, fuzz=1e-14)
{# N.B. models are not compared (so equivalent models will compare true)
 # inputs are not compared, so they may be provided differently.
 r <- all(class(obj1) == class(obj2))
 if (r) r <- all(output.data(obj1$data) == output.data(obj2$data))
 if (r) r <- all(obj1$horizon == obj2$horizon)
 if (r) r <- fuzz > max(abs(obj1$pred - obj2$pred))
 if (r) r <- length(obj1$forecast)==length(obj2$forecast)
 for (i in seq(length(obj1$forecast)))
   if (r) r <- fuzz > max(abs(obj1$forecast[[i]] - obj2$forecast[[i]]))
 r
}

plot.forecast <- function(obj, Start=NULL, End=NULL, select.series=NULL,
        names=output.series.names(obj$data))
{#The default starting point (Start) for plots is the start data.
 #The default ending point (End) for plots is the end of forecast.
   output <-trim.na(output.data(obj$data))
   if(is.null(names)) names <- rep(" ", dim(output)[2])
   if (is.null(select.series)) select.series <- 1: dim(output)[2]
   old.par <-par(mfcol = c(length(select.series), 1), mar= c(5.1,6.1,4.1,2.1))
   on.exit(par(old.par))
   H <- 0
   for (t in 1:length(obj$conditioning.inputs))
      H <- max(H, dim(obj$forecast[[t]])[1])
   tf <-expand(tframe(output), add.end=H)
   for(i in select.series) 
        {z <- c(output[,i], rep(NA,H))
         for (t in 1:length(obj$conditioning.inputs))
            {zz <- c(rep(NA,periods(output)),obj$forecast[[t]][,i],
                     rep(NA,H-dim(obj$forecast[[t]])[1]))
             zz[periods(output) ] <- output[periods(output), i] #so line joins last data to first forecast
             z <- cbind(z,zz)
            }
         tframe(z) <- tf
         if (!is.null(Start)) z <- window(z,start=Start, warn=F)
         if (!is.null(End))   z <- window(z,end=End, warn=F)
         plot(z, ylab = names[i]) # tsplot
         if(i == select.series[1]) 
             title(main = "Predictions (dotted) and actual data (solid)")
        }
   invisible()
}

output.series.names.forecast <-function(obj)
   {m <- output.series.names(obj$model)
    d <- output.series.names(obj$data)
    if(!all(m == d))
       warning("data and model names do not correspond. Model names returned.")
    m
   }

input.series.names.forecast <-function(obj)
   {m <- input.series.names(obj$model)
    d <- input.series.names(obj$data)
    if(!all(m == d))
       warning("data and model names do not correspond. Model names returned.")
    m
   }

############################################################################

#    methods for feather.forecasts        <<<<<<<<<<<<<

############################################################################
is.feather.forecasts <-function(obj) inherits(obj, "feather.forecasts")

output.series.names.feather.forecasts <-function(x) output.series.names(x$data)
 input.series.names.feather.forecasts <-function(x)  input.series.names(x$data)

feather.forecasts <-function(obj, ...) UseMethod("feather.forecasts")

feather.forecasts.TSestModel <- function(model, ...)
     {feather.forecasts.TSmodel(model$model, model$data, ...)}

feather.forecasts.TSdata <- function(data, model, ...)
    {feather.forecasts.TSmodel(model, data, ...)}

feather.forecasts.TSmodel <- function(model, data, horizon=36,
             from.periods =NULL, ...)
  {if(!is.TSmodel(model)) TS.error.exit(clss="TSmodel")
   if(!is.TSdata(data)) TS.error.exit(clss="TSdata")
   if (is.null(from.periods))
     {if(is.null(input.data(data))) from.periods <-10*seq(floor(periods(data)))
      else from.periods <-10*seq(floor(min(periods(data),
                                           input.periods(data)-horizon)/10))
     }
   # periods.TSPADIdata returns NA rather than fetching data.
   if ((!is.na(periods(data))) && (max(from.periods) > periods(data) ))
     stop("from.periods cannot exceed available output data.")
   if (!is.null(input.data(data)))
     if ((!is.na(input.periods(data))) && 
        ((max(from.periods)+horizon) > input.periods(data) ))
       stop("forecasts cannot exceed available input data.")
   shf <- start.shift(model,data,y0=NULL)  # ? y0=y0)
   proj <- NULL
   for (sampleT in from.periods)
     {pr <-l(model, data, sampleT=sampleT, 
              predictT=sampleT+horizon, result="pred", ...)
      pr[1:(sampleT-1),] <- NA
      # make period before prediction = data so graphics look correct.
      #following if is kludge to prevent retrieving data
      if (!is.TSPADIdata(data))  
        pr[sampleT,] <- output.data(data)[sampleT+shf$shift*shf$lags,]
      proj <- append(proj, list(pr))
     }
   # names are available from  data or model
   proj <- list(model=model, data=data, from.periods=from.periods, 
                horizon=horizon, feather.forecasts=proj)
   class(proj) <- "feather.forecasts"
   invisible(proj)
}

plot.feather.forecasts <- function(obj, Start=NULL, End=NULL, select.series=NULL, graphs.per.page=5, reset.screen=T)
{#The default starting point (Start) for plots is the start of data.
 #The default ending point (End) for plots is the end of forecasts.
   p <- dim(obj$feather.forecasts[[1]])[2]
   freq <- frequency(obj$data)
   names <- output.series.names(obj)
   if(is.null(names)) names <- paste("output", 1:p)
   if (is.null(Start)) Start <- start(obj$data)
   if (is.null(End))   End   <- add.date(end(output.data(obj$data)),
                                   max(obj$horizon), frequency(obj$data))
   if (is.null(select.series)) select.series <- 1:p
   if (!is.numeric(select.series)) select.series <- match(select.series, names)
   if(reset.screen) 
     {Ngraphs <- length(select.series)
      Ngraphs <- min(Ngraphs, graphs.per.page)
      old.par <- par(mfcol = c(Ngraphs, 1), mar= c(5.1,6.1,4.1,2.1)) 
      on.exit(par(old.par))
     }
   # if below is a kludge to skip getting TSPADI data.  
   for(i in select.series) 
        {if (is.TSPADIdata(obj$data)) 
           {zz <- NULL # kludge
            ltys <- rep(2,length(obj$from.periods))
           }
         else 
           {zz <- window(output.data(obj$data,series=i), start=Start,warn=F)
            ltys <- c(1,rep(2,length(obj$from.periods)))
           }
         for (t in 1:length(obj$from.periods))
            {zz <- tbind(zz,
                     tframed(obj$feather.forecasts[[t]][,i], 
                       list(start=start(obj$feather.forecasts[[t]]),freq=freq)))
            }
         plot(window(zz,start=Start,end=End, warn=F), ylab = names[i],
              lty= ltys) #tsplot
         if(i == select.series[1]) 
             title(main = "Predictions (dotted) and actual data (solid)")
        }
   invisible()
}



############################################################################
#
#       procedure for testing functions   <<<<<<<<<<<<<
#
############################################################################


dse2.function.tests <- function(verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-8, graphics=T)
{max.error <- NA
 if (synopsis & !verbose) cat("All dse2 tests ...") 
 if (verbose) cat("dse2 test 0 ... ")
  z <- example.BOC.93.4.data.all
  z$input <- NULL
  mod1 <- TSmodel(est.VARX.ar(z, re.add.means=F, warn=F))
  ok <- is.TSmodel(mod1)
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse2 test 1 ... ")
  z <- est.black.box1(example.BOC.93.4.data.all, verbose=F, max.lag=2)
  error <- max(abs(z$estimates$like[1]+4025.943051342767))
  ok <- is.TSestModel(z) &  (fuzz.large > error )
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("dse2 test 2 ... ")
  z <- est.wt.variables(example.BOC.93.4.data.all, c(1,10,10),
                        estimation="est.VARX.ls")
  error <- max(abs(z$estimates$like[1]+4125.05572604540066)) 
  ok <- is.TSestModel(z) &   (fuzz.large > error)
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("dse2 test 3 ... ")
  z <- est.SS.Mittnik(example.BOC.93.4.data.all, max.lag=2, n=3)
  error <- max(abs(z$estimates$like[1]+3794.0394069904219))
  ok <- is.SS(z$model) &   (fuzz.large > error )
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("dse2 test 4 ... ")
  z <- l( reduction.Mittnik(z, criterion="taic", verbose=F), 
         example.BOC.93.4.data.all)
  error <- max(abs(z$estimates$like[1]+3795.6760513068380)) 
  ok <- is.SS(z$model)  &  (fuzz.large > error )
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!! (error magnitude= ", error,")\n")
    }

  modSS<-z

  if (verbose) cat("dse2 test 5 ... ")
  z <- feather.forecasts( modSS,  from.periods=c(250,300))
  error <- max(abs
       (c(z$feather.forecasts[[1]][286,],z$feather.forecasts[[2]][336,])
       -c(-0.00092229286770808757701, -0.0086020067525247358164, 
           0.0043454851777852505565,  -0.0066741302949233430319,
          -0.0089398331205012854933,   0.0021769124280658046222)))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("dse2 test 6 ... ")
  output.data(modSS$data) <- window(output.data(modSS$data), end=c(1969,6))
  # it should be possible to do the following instead, but tsp seems to
  # sometimes get mixed up in forecast and cause System terminating: bad address
  # output.data(modSS$data) <- output.data(modSS$data)[1:100,]
  z <- forecast(modSS, percent=c(90,100,110))

# previously 136 below
  error <- max(abs(
    c(z$forecast[[1]][36,],z$forecast[[2]][36,], z$forecast[[3]][36,])
     -c(-0.00310702417651131587, -0.00604105559321206804,0.00214657444656118738,
      -0.00345224972784219028, -0.00671228396225603124,0.00238508249578931863,
      -0.00379747527917305948, -0.00738351233129999531,0.00262359054501745074)))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

if (graphics) {
  if (verbose) cat("dse2 test 7 (graphics) ... ")
  ok <- dse2.graphics.tests(verbose=verbose, pause=T)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }
}

  if (synopsis) 
    {if (verbose) cat("All dse2 tests completed")
     if (all.ok) cat(" ok\n")
     else    
       {cat(", some failed!")
        if(max.error > fuzz.small)
            cat(" max. error magnitude= ", max.error,")")
        cat("\n")
       }
    }

  invisible(all.ok)
}

dse2.graphics.tests <- function(verbose=T, synopsis=T,  pause=F)
{# graphics tests do not do any value comparisons
  if (synopsis & !verbose) cat("dse2 graphics tests ...")
  if (verbose) cat("  dse2 graphics test 1 ...")

  # If no device is active then write to postscript file 
  if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
             {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
  if(pause) dev.ask(ask=T)

  data <- example.BOC.93.4.data.all
  mod1 <- TSmodel(est.VARX.ls(data,max.lag=3))
  modSS <- l(to.SS(mod1),data)

  z <- feather.forecasts( modSS,  from.periods=c(230,250))
  plot(z, Start=c(1980,1))
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse2 graphics test 2 ...")
  z <- forecast(modSS, percent=c(90,100,110))
  plot(z, Start=c(1985,1))
  if (verbose) cat("ok\n")

  if (synopsis) 
    {if (verbose) cat("All dse2 graphics tests completed\n")
     else cat("completed\n")
    }
      

  invisible(T)
}


############################################################################
#
#       end
#
############################################################################
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################




############################################################################

# Functions in dse3a.s and dse3b.s file are mainly for evaluating estimation techniques.

# The first group are for generating simulations (ie- generate multiple
#   stochastic simulations of a model using simulate.)
#   These are good for looking at the stochastic properties of a model,
#   but mostly these are intended
#   only for verification purposes since other functions also generate 
#   simulations and it is usually more efficient to regenerate by setting
#   the seed than it is to save data.
#   The main function in this group is monte.carlo.simulations().
#   The main object class in this group is "simulation"

# The second group are for analysing the convergence of estimators. This
#  is extended to functions of estimates (such as model roots). These
#  functions evaluate a single given estimation method with multiple
#  simulated data sets from a given "true" model.
#  The main function in this group is eval.estimation().
#  The main object classes in this group are
#    "estimation.evaluation"
#     c("roots.ee","estimation.evaluation")
#     c("TSmodel.ee","estimation.evaluation")
#     c("TSestModel.ee","estimation.evaluation")
#     c("parms.ee","estimation.evaluation")           and
#     c("roots.ee","estimation.evaluation")


# The third group applies multiple estimation techniques to a given data set.
#  This is primarily a utility for other functions 
#  The main (only) function in this group is estimate.models().
#  It returns an object of class c("estimated.models")


# The fourth group looks at the forecasts and the covariance of forecasts 
#   for multiple horizons. 
#  The simplest case horizon.forecasts() which calculates the forecast for 
#    different horizons at all periods in the sample. This is primarily
#    a utility for calculating the forecast error.
#  The next case is is estimators.horizon.forecasts() which 
#   is an extention of horizon.forecasts(). It takes specified data 
#   and estimation techniques and calculates forecasts from the estimated 
#   models.
#  The generic function forecast.cov() which considers mulitple  
#   models and calculates the cov of forecasts relative to a given data set.
#   It takes a list of models (+ trend, zero)  and calculates the cov 
#   of predictions. It uses forecast.cov.single.TSmodel()
#  The next case, forecast.cov.estimators.wrt.data() uses a list of estimation
#   methods to estimate a list of models and calculate the cov of predictions 
#   relative to one given data set.

#   The next case forecast.cov.wrt.true() takes a list of models (+ trend,
#   zero)  and calculates the cov of forecasts relative to data sets 
#   simulated with a true.model.
#   The next case, forecast.cov.estimators.wrt.data simulates data and 
#   uses a list of estimation methods to estimate a list of models, then
#   calculates the cov of predictions relative to other simulated data set.
#  The main object classes in this group are
#     c("estimators.horizon.forecasts.wrt.data") # ? "horizon.forecasts")
#     "horizon.forecasts"
#     c("multi.model.horizon.forecasts","horizon.forecasts")
#     "forecast.cov"
#     c("forecast.cov.wrt.data", "forecast.cov")
#     c("forecast.cov.wrt.true", "forecast.cov")
#     c("forecast.cov.estimators.wrt.data",  "forecast.cov")
#     c("forecast.cov.estimators.wrt.true",  "forecast.cov")


# The fifth group are some experimental estimation techniques.

############################################################################
#
#       utilities  <<<<<<<<<<
#
############################################################################

minimum.startup.lag <- function(model)UseMethod("minimum.startup.lag")
minimum.startup.lag.TSestModel <- function(model)  
    {minimum.startup.lag(model$model)}

minimum.startup.lag.ARMA <- function(model)
  {lag <- dim(model$A)[1] 
   if (!is.null(model$C)) lag <- max(lag, dim(model$C)[1])
   lag
  }

minimum.startup.lag.SS <- function(model)  { 1+dim(model$F)[2] }


############################################################################
#
#       methods for monte.carlo.simulations  <<<<<<<<<<
#
############################################################################
generate.model.SS <- function(m,n,p, stable=F)
 {#randomly generate an innov state space model. Discard models with largest root
  # greater than 1 (if stable=F) or equal to or greater than 1 if stable=T.
  repeat
    {FF <- matrix(runif(n^2, min=-1,max=1),n,n)
     if (m!=0) G <- matrix(runif(n*m, min=-1,max=1),n,m)
     else G <- NULL
     H <- matrix(runif(n*p, min=-1,max=1),p,n)
     K <- matrix(runif(n*p, min=-1,max=1),n,p)
     model <- set.parameters(list(F=FF, G=G,H=H,K=K))
     if (stable) {if (max(Mod(roots(model))) <  1.0) break()}
     else        {if (max(Mod(roots(model))) <= 1.0) break()}
    }
  model
 }


monte.carlo.simulations <-function( model, simulation.args=NULL, 
           replications=100, seed=NULL, ...)
{#Produces multiple simulations.
	UseMethod("monte.carlo.simulations")
}

monte.carlo.simulations.TSestModel <- function(model, simulation.args=NULL, 
           replications=100, seed=NULL, ...)
  {if (is.null(simulation.args$sd) & is.null(simulation.args$SIGMA)) 
     simulation.args$SIGMA <- model$estimates$cov
   monte.carlo.simulations( model$model, simulation.args=simulation.args, 
           replications=replications, seed=seed, ...)
  }

monte.carlo.simulations.estimation.evaluation  <-function(model,...)
       {monte.carlo.simulations(model$model, seed=get.seed(model), ...)}
monte.carlo.simulations.monte.carlo.simulation <-function(model,...)
       {monte.carlo.simulations(model$model, seed=get.seed(model), ...)}


monte.carlo.simulations.default <-function( model, simulation.args=NULL,
          replications=100, seed=NULL, Spawn=.SPAWN, quiet=F)
{  
 seed <- set.seed(seed)
 arglist <- append(list(model), simulation.args)
 if (Spawn)
  {if (!quiet)cat("Spawning processes to calculate ", replications, " replications.\n")
   assign("sim.forloop.n", replications, where = 1)
   assign("sim.forloop.result", list(NULL), where = 1)
 #  assign("sim.forloop.model", model, where = 1)
   assign("sim.forloop.arglist", arglist, where = 1)
   on.exit(remove(c("sim.forloop.i", "sim.forloop.n", "sim.forloop.result",
       "sim.forloop.arglist"),where = 1))
   For(sim.forloop.i = 1:sim.forloop.n, sim.forloop.result[[sim.forloop.i]] <- 
       do.call("simulate",  sim.forloop.arglist),
       first=options(warn=-1), sync = T)
   result <- array(NA, c(dim(sim.forloop.result[[1]]$output),replications))
   for (i in 1:replications) result[,,i] <- sim.forloop.result[[i]]$output
   tframe(result) <- tframe(sim.forloop.result[[1]]$output)
  }
 else
  {#r <- simulate(model, list(...))$output
   r <- do.call("simulate", arglist)$output
   result <- array(NA, c(dim(r),replications))
   tframe(result) <- tframe(r)
   result[,,1] <- r
   if (1 < replications)
     for (i in 2:replications) 
        result[,,i] <- do.call("simulate", arglist)$output
  }
# use dimnames rather than $names so select.series in plot will work with names
dimnames(result) <- list(NULL,output.series.names(model),NULL)
r<-list(simulations=result, model=model, seed=seed, version=version, 
              simulation.args=simulation.args,
              description = "data generated by monte.carlo.simulation.default")
class(r) <-  c("monte.carlo.simulations")
invisible(r)
}

is.monte.carlo.simulation <- function(obj)
{  r <- "monte.carlo.simulations"==last.class(obj)
   if(is.na(r)) r <- F
   r
}


print.monte.carlo.simulations <- function(obj, digits=4)
{cat("Simulation with seed ", obj$seed, " from model:\n")
 print(obj$model)
 invisible(obj)
}

output.series.names.monte.carlo.simulations <- function(obj)
   {dimnames(obj$simulations)[[2]]}

input.series.names.monte.carlo.simulations <- function(obj)
  {input.series.names(obj$simulation.args$data)}

test.equal.monte.carlo.simulations <-function(d1,d2, fuzz=1e-16)
 {if (length(d1$result) != length(d2$result)) r <-F
  else  r <- all(fuzz > abs(d1$simulations - d2$simulations))
  r
 }


summary.monte.carlo.simulations <- function(obj, digits = 6,
     select.series=NULL, periods=1:3)
 {cat("Object class monte.carlo.simulations\n")
  cat(obj$description, "\n")
  sim.dim <- dim(obj$simulations)
  cat("periods=",sim.dim[1],"variables=",sim.dim[2], 
      "simulations=",sim.dim[3], "\n")
  cat("seed= ", get.seed(obj), "\n")
  z <- NULL
  if (!is.null(select.series))
    {if (sim.dim[3] <20) 
        warning("SD calculation is not very good with so few simulations.")
     names <- output.series.names(obj)
     if(is.null(names)) names <- output.series.names(obj$model)
     if (!is.numeric(select.series)) select.series <-match(select.series, names)
     names <- names[select.series]
     mn <-apply(obj$simulations[periods,select.series,,drop=F], c(1,2),mean)
     sd <-apply(obj$simulations[periods,select.series,,drop=F], c(1,2),var)^ 0.5
     z <- rbind(mn,sd) 
     dimnames(z)<- list(c(paste("mean period", periods), 
                          paste("S.D. period",periods)), names)
     print(z, digits=digits)
    }
  invisible()
 }



plot.monte.carlo.simulations <- function(obj, Start=NULL, End=NULL,
    select.series=seq((dim(obj$simulations)[2])), 
    select.simulations=seq(dim(obj$simulations)[3]),
    graphs.per.page=5)
  {names <- output.series.names(obj)
   if(is.null(names)) names <- output.series.names(obj$model)
   if (is.null(Start)) Start <- start(obj$simulations)
   if (is.null(End))   End   <- end(obj$simulations)
   tf.p <- tframe(obj$simulations)
   Ngraph <- min(length(select.series), graphs.per.page)
   old.par <-par(mfcol = c(Ngraph, 1), mar= c(5.1,6.1,4.1,2.1))
   on.exit(par(old.par))
   #zz<- matrix(NA, dim(sim)[1], length(obj$simulations))
   if (!is.numeric(select.series)) select.series <- match(select.series, names)
   for(i in select.series) 
        {zz <- (obj$simulations)[,i,select.simulations]
         tframe(zz) <- tf.p
         plot(window(zz,start=Start,end=End, warn=F), ylab=names[i]) #tsplot
         if(i == select.series[1])  title(main = "Monte Carlo Simulations")
        }
   invisible()
}



distribution.monte.carlo.simulations <- function(obj,
     select.series=seq(dim(obj$simulations)[2]),
     x.sections=T, periods=1:3, graphs.per.page=5)
  {
if (dim(obj$simulations)[3] <20) 
     warning("This is not very good with so few simulations.")
   names <- output.series.names(obj)
   if(is.null(names)) names <- output.series.names(obj$model)
   if (!is.numeric(select.series)) select.series <- match(select.series, names)
   names <- names[select.series]
   Ngraph <- min(length(select.series), graphs.per.page)
   if (x.sections)
       {data <- obj$simulations[periods, select.series,,drop=F]
        old.par <-par(mfrow =c(Ngraph, length(periods)), mar=c(5.1,6.1,4.1,2.1))
       }
   else 
       {old.par <-par(mfrow =c(Ngraph, 1), mar=c(5.1,6.1,4.1,2.1))
        mn <- apply(obj$simulations[, select.series,,drop=F], c(1,2), mean)
        sd <- apply(obj$simulations[, select.series,,drop=F], c(1,2), var) ^ 0.5
        plt <- array(c(mn, mn+sd, mn-sd, mn+2*sd, mn-2*sd), c(dim(mn),5))
        tf.p <- tframe(obj$simulations)
      }
   on.exit(par(old.par))
   for (i in 1:length(select.series)) 
    {if (x.sections)
        {for (j in 1:length(periods)) 
          {if (is.Splus()) plot(ksmooth(data[j,i,], # -mean[j,i],
                              bandwidth=var(data[j,i,])^0.5, kernel="parzen"),
                        type="l",xlab=paste("Period",periods[j]),ylab=names[i])
           if (is.R()) plot(density(data[j,i,]), # -mean[j,i]
                 type="l",xlab=paste("Period",periods[j]),ylab=names[i])
           if ((i == 1) & (j ==length(periods)%/%2))
              title(main = "kernel estimate of distributions")
           }
        }
     else
        {pl <-plt[,i,]
         tframe(pl) <- tf.p
         plot(pl, type="l", lty= c(1,3,3,2,2), ylab=names[i]) #tsplot
         if (i == 1) title(main = "Simulation mean, 1 & 2 S.D. estimates")
        }
    }
  invisible()
  }

############################################################################
#
#       methods for estimation.evaluation.  <<<<<<<<<<
#
############################################################################

#e.bb.ar.100 <- eval.estimation( mod2, replications=100, 
#               estimation.args=list(estimation="est.VARX.ar", verbose=F))

#e.bb.ls.over <- eval.estimation( simple.mod, replications=100, 
#   estimation.args=list(estimation="est.VARX.ls", max.lag=6, verbose=F), 
#   criterion="parms")

TSmodel.TSestModel <- function(obj)
  {# Return a TSmodel object but also retains convergence info (if not null).
   obj$model$converged <- obj$converged
   obj$model
  }

eval.estimation <-function( model, replications=100, seed=NULL, quiet=F, 
                       simulation.args=NULL,
                       estimation="black.box", estimation.args=list(verbose=F), 
                       criterion ="parms", criterion.args =NULL, spawn=.SPAWN)
{# estimation.args and criterion.args should be NULL if no args are needed.
 # If model is an object of class "estimation.evaluation" or "simulation"
 # then the model and the seed!!! are extracted so the evaluation will be
 # based on the same generated sample.
 # criterion can be in { "parms", "roots", TSmodel", "TSestModel"}
 # With the default (parms) and $model the other criteria can be reconstructed
 #  when the estimation method gets the correct form for the model. ( This
 #  is not usually the case with the default method "black.box".)
 # This is done by via the generic functions roots, TSmodel and TSestModel.
 # If criterion = "roots" then criterion.args= list(verbose=F) is advised.
 # example simulation.args=list(sampleT=100, sd=1.5)

 if (is.estimation.evaluation(model) | is.monte.carlo.simulation(model))
   {seed <- get.seed(model)
    model<- model$model
   }
  
 truth <- do.call(criterion, append(list(model), criterion.args))
 seed <- set.seed(seed)
 if (!spawn)
   {if(!quiet) cat("Calculating ", replications, " estimates.\n")
    result <- vector("list",replications)
    for (i in 1:replications)
       {data <- do.call("simulate", append(list(model), simulation.args))
        m   <-  do.call(estimation, append(list(data),  estimation.args))
        result[[i]]<-do.call(criterion, append(list(m), criterion.args))
       }
   }
 else
    {if(!quiet)
        cat("Spawning processes to calculate ", replications, " estimates.\n")
     est.forloop <- function(estimation, estimation.args, model, 
                             simulation.args, criterion, criterion.args)
       {data <- do.call("simulate", append(list(model), simulation.args))
        m   <-  do.call(estimation, append(list(data),  estimation.args))
        do.call(criterion, append(list(m), criterion.args))
       }
     assign("est.forloop", est.forloop, where = 1)
     assign("est.forloop.n", replications, where = 1)
     assign("est.forloop.result", list(NULL), where = 1)
     assign("est.forloop.estimation", estimation, where = 1)
     assign("est.forloop.model", model, where = 1)
     assign("est.forloop.simulation.args", simulation.args, where = 1)
     assign("est.forloop.criterion", criterion, where = 1)
     assign("est.forloop.estimation.args", estimation.args, where = 1)
     assign("est.forloop.criterion.args", criterion.args, where = 1)

     on.exit(remove(c("est.forloop", "est.forloop.i", "est.forloop.n",
         "est.forloop.result",  "est.forloop.estimation","est.forloop.model",
         "est.forloop.simulation.args", "est.forloop.criterion", 
         "est.forloop.estimation.args","est.forloop.criterion.args"),where = 1))

     For(est.forloop.i = 1:est.forloop.n, est.forloop.result[[est.forloop.i ]]<-
       est.forloop(est.forloop.estimation, est.forloop.estimation.args, est.forloop.model, 
       est.forloop.simulation.args, est.forloop.criterion, est.forloop.criterion.args),
       first=options(warn=-1), sync = T)
     result<-est.forloop.result
    }
r<-list(result=result,truth=truth,model=model,
           seed=seed, version=version,
           estimation=estimation, estimation.args=estimation.args,
            criterion=criterion,   criterion.args=criterion.args, 
            simulation.args=simulation.args)


class(r) <- c(paste(criterion,".ee",sep=""), "estimation.evaluation")
invisible(r)
}

eval.estimation.set <- function(estimation, model, 
         replications=10, sampleT=100,
         eval.proc.args=list(NULL), 
         criterion = "roots", p.set=NULL)
{ # evaluate for a set of models.   NOT WORKING
 r <- NULL
   for (p in p.set)
     {m<-model
      m$parms <- p
      m <- set.arrays(m)
      r[[p]] <- estimation.test(estimation, m, replications=10, sampleT=100,eval.proc.args=list(NULL), criterion = "roots")
     }
r
}

is.estimation.evaluation <- function(obj)
{  r <- "estimation.evaluation"==last.class(obj)
   if(is.na(r)) r <- F
   r
}

test.equal.estimation.evaluation <- function(obj1,obj2)
 {all(as.character(obj1) == as.character(obj2))}

print.estimation.evaluation <- function(obj, digits=4)
{cat("Estimation evaluation with model:\n")
 print(obj$model, digits=digits)
 cat("Evaluation criterion: ",obj$criterion, "\n")
 invisible(obj)
}

summary.estimation.evaluation <-  function(obj)
{ cat("Object of class: ", class(obj),"\n")
  cat("Evaluation of `",obj$estimation,"'")
  if(!is.list((obj$estimation.args)[[1]]))
    {cat( " estimation with argument ") 
     cat(labels(obj$estimation.args),"= `",obj$estimation.args,"'")
    }
  cat("\n")
  cat("using criterion `", obj$criterion, "' with argument ")
  cat(labels(obj$criterion.args)," = `", obj$criterion.args, "'\n")
  cat(length(obj$result), " replications, seed = ", obj$seed, "\n")
  cat("true model:\n")
  print(obj$model)
  invisible()
}

distribution <- function(obj, ...)UseMethod("distribution")

distribution.estimation.evaluation <- function(obj, ...)
 {distribution(parms(obj), ...)}

############################################################################
#
#       methods for roots.ee  (estimation.evaluation)  <<<<<<<<<<
#
############################################################################

summary.roots.ee <-  function(obj, verbose=T)
{ if (verbose) NextMethod("summary")
  all.converged <- NULL
  if (!is.null(obj$result.conv)) all.converged<-all(obj$result.conv)
  N <- length(obj$result)
  p <- 0
  for (i in 1:N) p <- max(p, length((obj$result)[[i]]))
  r <- matrix(NA, N, p)
  for (i in 1:N) r[i,1:length((obj$result)[[i]])] <- (obj$result)[[i]]
  m <- apply(r,2,sum)/N
  cov <- r- t(matrix(obj$truth, p, N))
  cov <- (t(Conj(cov)) %*% cov)/(N-1)
  ecov <- r- t(matrix(m, p, N))
  ecov <- (t(Conj(ecov)) %*% ecov)/(N-1)
  if (verbose)
    {if (!is.null(obj$result.conv))
        {if(all.converged) cat("All estimates converged.\n")
         else cat(sum(!obj$result.conv)," estimates did not converge!\n")
        }
     cat("\nTrue model criterion mean: ",obj$truth,"\n")
     cat("Sampling estimate of mean: ",m,"\n")
     cat("Estimate of sampling covariance [e*Conj(t(e))] using true model:\n")
     print(cov)
     cat("\nEstimate of sampling covariance (using sample mean and not the true model):\n")
     print(ecov)
    }
  invisible(  invisible(list(truth=obj$truth, all.converged=all.converged, estimate=m, sample.cov=cov, est.sample.cov=ecov))
)
}

graph.roots.ee <- function(...)
     UseMethod("plot.roots.ee")

plot.roots.ee <- function(obj, complex.plane=T, cum=T, norm=F, bounds=T, transform=NULL, invert=F, Sort=T)
{# If complex.plane is T then all results are plotted on a complex plane and 
 #   the arguements cum and Sort do not apply. If complex.plane is F 
 #   then a sequential plot of the real and imaginary parts is produced.
 # If cum is true the cummulative average is plotted.
 # If mod is true the modulus is used, otherwise real and imaginary are separated.
 # if invert is true the reciprical is used (before cummulating).
 # if Sort is true then sort is applied (before cum but after mod) by the Re part of the root.
 #   Some grouping is usually necessary since roots are not in an obvious order
 #   but sorting by the real part of the roots could be improved upon.
   N<-length(obj$result)
   n <- 0
   for (i in 1:N) n <- max(n, length((obj$result)[[i]]))
   r <- matrix(0,N, n) 
   for (i in 1:N) r[i,1:length((obj$result)[[i]])] <- (obj$result)[[i]]
   true.lines <- c(obj$truth, rep(0,n-length(obj$truth)))
   if (invert)
         {true.lines <- 1/true.lines
          r <- 1/r
         }
   if(!is.null(transform)) 
         {r <- do.call(transform,list(r))
          true.lines <-do.call(transform,list(true.lines))
         }
   if (complex.plane)
    {plot.roots(obj$truth, pch="o")
     for (i in 1:N) add.plot.roots(r[i,], pch="*") 
     add.plot.roots(0, pch="+") # add.plot.roots(0+0i, pch="+")
    }
  else
     {if (Sort)
        {r <- t(apply(r,1,sort))
         true.lines <- sort(true.lines)
        }
      if (cum) r <- apply(r,2,cumsum)/matrix(1:N,N,ncol(r))
      else     r <- r 
      if(is.complex(r))
         { r <- cbind(Re(r), Im(r))
          true.lines <-c(Re(true.lines),Im(true.lines))
         }
      r[is.inf(r)] <- 0
      true.lines <-t(matrix(true.lines, length(true.lines),N))
      matplot(x=seq(nrow(r)), y=cbind(0,true.lines, r), type="l",
             lty=c(1,rep(3,dim(true.lines)[2]), rep(2,dim(r)[2])) )
     }
  invisible(r)
}

roots.roots.ee <- function(obj, ...)   {obj}

distribution.roots.ee <- function(obj, mod=T, invert=F, Sort=F, bandwidth=0.2, select=NULL)
{# if mod is true the modulus is used, otherwise real and imaginary are separated.
 # if invert is true the reciprical is used.
 # if Sort is true then sort is applied (before cum). This is of particular interest
 #   with estimation methods like black.box which may not return parameters
 #   of the same length or in the same order.
 # If select is not NULL then only the indicated roots are plotted. 
 #     ie - select=c(1,2)  will plot only the two largest roots
      N<-length(obj$result)
      n <- 0
      for (i in 1:N) n <- max(n, length((obj$result)[[i]]))
      r <- matrix(0,N,n)
      for (i in 1:N) r[i,] <- c((obj$result)[[i]], 
                                rep(0,n-length((obj$result)[[i]])))
      true.lines <- c(obj$truth, rep(0,n-length(obj$truth)))
      if (invert)
         {true.lines <- 1/true.lines
          r <- 1/r
         }
      if(mod) 
         {r <- Mod(r) 
          true.lines <-Mod(true.lines)
          xlab <-"Mod root "
         }
      else
         { r <- cbind(Re(r), Im(r))
          true.lines <-c(Re(true.lines),Im(true.lines))
          xlab <-"Real part root "
         }
      r[is.inf(r)] <- 0
      if (Sort)
        {r <- t(apply(r,1,sort))
         true.lines <- sort(true.lines)
        }
      if(!is.null(select)) r <- r[,select, drop=F]
      par(mfcol=c(dim(r)[2],1))
      for ( i in 1:dim(r)[2])
         {if (is.Splus()) rd <- ksmooth(r[,i], bandwidth=bandwidth) 
          if (is.R())     rd <- density(r[,i], bw= bandwidth)
          if (i > n) xlab <-"Imaginary part root "
          plot(rd, type="l", ylab="density", ylim=c(0, max(rd$y)),
               xlab=paste(xlab, n-(-i%%n)) )
          lines(rep(true.lines[i],2),c(1,0))
         }
      invisible()
}


############################################################################
#
#       methods for parms.ee (estimation.evaluation)  <<<<<<<<<<
#
############################################################################

summary.parms.ee <-  function(obj, verbose=T)
{ if (verbose) NextMethod("summary")
  all.converged <- NULL
  if (!is.null(obj$result.conv)) all.converged<-all(obj$result.conv)
  N <- length(obj$result)
  p <- 0
  for (i in 1:N) p <- max(p, length((obj$result)[[i]]))
  r <- matrix(NA, N, p)
  for (i in 1:N) r[i,1:length((obj$result)[[i]])] <- (obj$result)[[i]]
  m <- apply(r,2,sum)/N
  cov <- r- t(matrix(obj$truth, p, N))
  cov <- (t(cov) %*% cov)/(N-1)
  ecov <- r- t(matrix(m, p, N))
  ecov <- (t(ecov) %*% ecov)/(N-1)
  if (verbose)
    {if (!is.null(obj$result.conv))
        {if(all.converged) cat("All estimates converged.\n")
         else cat(sum(!obj$result.conv)," estimates did not converge!\n")
        }
     cat("\nTrue model criterion mean: ",obj$truth,"\n")
     cat("Sampling estimate of mean: ",m,"\n")
     cat("Estimate of sampling covariance using true model:\n")
     print(cov)
     cat("\nEstimate of sampling covariance (using sample mean and not the true model):\n")
     print(ecov)
    }
  invisible(list(truth=obj$truth, all.converged=all.converged, estimate=m, sample.cov=cov, est.sample.cov=ecov))
}

graph.TSmodel.ee <- function(...)
     UseMethod("plot.TSmodel.ee")

plot.parms.ee <- function(obj, cum=T, norm=F, bounds=T, invert=F, Sort=F)
{# if cum is true the cummulative average is plotted.
 # if norm is true the norm is used, each parameter is plotted.
 # if invert is true the reciprical is used (before cummulating).
 # if Sort is true then sort is applied (before cum). This is not usually
 #   recommended but of interest
 #   with estimation methods like black.box which may not return parameters
 #   of the same length or in the same order.
      N<-length(obj$result)
      n <- 0
      for (i in 1:N) n <- max(n, length((obj$result)[[i]]))
      r <- matrix(0,N,n)
      for (i in 1:N) r[i,1:length((obj$result)[[i]])] <- (obj$result)[[i]]
      true.lines <- c(obj$truth, rep(0,n-length(obj$truth)))
      if (invert)
         {true.lines <- 1/true.lines
          r <- 1/r
         }
      if(norm) 
         {r <- matrix((apply(r^2,1,sum))^.5, N,1)
          true.lines <-sum(true.lines^2)^.5
         }
      r[is.inf(r)] <- 0
      if (Sort)
        {r <- t(apply(r,1,sort))
         true.lines <- sort(true.lines)
        }
      true.lines <-t(matrix(true.lines, length(true.lines),N))
      Om <-NULL
      if (bounds)
        {z  <- r-true.lines
         Om <- t(z) %*% z/(nrow(z)-1)
         Om <- diag(Om)^.5
         Om <- t(matrix(Om, length(Om), N))
         Om <- Om/matrix((1:N)^.5 , N, ncol(Om))
         Om <- cbind(true.lines+Om, true.lines-Om)
        }
      if (cum) r<- apply(r,2,cumsum)/matrix(1:N,N,ncol(r))
      matplot(x=matrix(seq(nrow(r)),nrow(r),1), y=cbind(0,true.lines,r, Om), 
              type="l", lty=c(1,rep(3,dim(true.lines)[2]), rep(4,dim(r)[2]), 
                     rep(2,2*dim(r)[2]))) 
      invisible(r)
}

roots.parms.ee <- function(obj, criterion.args=NULL)
{# extract roots criterion 
  model <- obj$model
  truth <-do.call("roots", append(list(model), criterion.args))
  r <- NULL
  for (m in obj$result)
    {model$parms <- m
     model <- set.arrays(model)
     r <- append(r, 
           list(do.call("roots", append(list(model), criterion.args))))
    }
  ok <- T
  for (m in obj$result)
     ok <- ok & (length(model$parms) == length(m))  # not perfect but ...
  if (!ok) warning("Parameters do not all correspond to given true model.")
  obj$result<-r
  obj$truth <-truth
  obj$criterion<-"roots"
  obj$criterion.args <-criterion.args
  class(obj) <- c("roots.ee","estimation.evaluation")
  invisible(obj)
}

distribution.parms.ee <- function(obj,  Sort=F, bandwidth=0.2)
{# if Sort is true then sort is applied (before cum). This is of particular interest
 #   with estimation methods like black.box which may not return parameters
 #   of the same length or in the same order.
      N<-length(obj$result)
      n <- length(obj$truth)
      for (i in 1:N) n <- max(n, length((obj$result)[[i]]))
      r <- matrix(0,N,n)
      for (i in 1:N) r[i,1:length((obj$result)[[i]])] <- (obj$result)[[i]]
      true.lines <- c(obj$truth, rep(0,n-length(obj$truth)))
      if (Sort)
        {r <- t(apply(r,1,sort))
         true.lines <- sort(true.lines)
        }
      xlab <-"parameter "
      par(mfcol=c(dim(r)[2],1))
      for ( i in 1:dim(r)[2])
         {if (is.Splus()) rd <- ksmooth(r[,i], bandwidth=bandwidth)
          if (is.R())     rd <- density(r[,i], bw=bandwidth)
          plot(rd, type="l", ylim=c(0, max(rd$y)),
               ylab="density",  xlab=paste(xlab, i) )
          lines(rep(true.lines[i],2),c(1,0))
         }
      invisible()
}

TSmodel.parms.ee <- function(obj)
{# rebuild model from parms
  model <- obj$model
  truth <-TSmodel(model)
  r <- NULL
  for (m in obj$result)
    {model$parms <- m
     model <- set.arrays(model)
     r <- append(r, list(model))
    }
  ok <- T
  for (m in obj$result)
     ok <- ok & (length(model$parms) == length(m))  # not perfect but ...
  if (!ok) warning("Parameters do not all correspond to given true model.")
  obj$result<-r
  obj$truth <-truth
  obj$criterion<-"TSmodel"
  obj$criterion.args <-NULL
  class(obj) <- c("TSmodel.ee","estimation.evaluation")
  invisible(obj)
}

TSestModel.parms.ee <- function(obj)
{# rebuild ... 
  model <- obj$model
  truth <-l(TSmodel(model), data)   # need to regenerate data from seed
  r <- NULL
  for (m in obj$result)
    {model$parms <- m
     model <- l( set.arrays(model), data)
     r <- append(r, list(model))
    }
  ok <- T
  for (m in obj$result)
     ok <- ok & (length(model$parms) == length(m))  # not perfect but ...
  if (!ok) warning("Parameters do not all correspond to given true model.")
  obj$result<-r
  obj$truth <-truth
  obj$criterion<-"TSestModel"
  obj$criterion.args <-NULL
  class(obj) <- c("TSestModel.ee","estimation.evaluation")
  invisible(obj)
}

############################################################################
#
#       methods for TSmodel.ee (estimation.evaluation)  <<<<<<<<<<
#
############################################################################

summary.TSmodel.ee <-  function(obj)
{ cat("Object of class: ",class(obj), "\n")
  if (!is.null((obj$result)[[1]]$converged))
    {rc <- rep(NA,length(obj$result))
     for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
     if(all(rc)) cat("All estimates converged.\n")
     else
       {cat(sum(!rc)," estimates did not converge!\n")
    }  }
 # summary(parms(obj))
 #summary(roots(obj))  these are slow
  summary.default(obj)
  invisible()
}

parms.TSmodel.ee <- function(obj, criterion.args=NULL)
{# extract parameters from models in the list and 
 #   return a list of class parms.ee estimation.evaluation 
 # criterion.args is not used. It is provided only so calls from 
 #   summary.TSmodel.ee can provide this argument.
  truth <-parms(obj$truth)
  r <- NULL
  for (m in obj$result) 
     r <- append(r,list(parms(m)))
  if (! is.null((obj$result)[[1]]$converged))
    {rc <- rep(NA,length(obj$result))
     for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
     obj$result.conv<-rc
    }
  obj$result<-r
  obj$truth <-truth
  obj$criterion<-"parms"
  obj$criterion.args <-criterion.args
  class(obj) <- c("parms.ee","estimation.evaluation")
  invisible(obj)
}

roots.TSmodel.ee <- function(obj, criterion.args=list( randomize=T))
{# extract roots criterion 
  truth <-do.call("roots", append(list(obj$truth), criterion.args))
  r <- NULL
  for (m in obj$result)
     r <- append(r, 
           list(do.call("roots", append(list(m), criterion.args))))
  if (! is.null((obj$result)[[1]]$converged))
    {rc <- rep(NA,length(obj$result))
     for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
     obj$result.conv<-rc
    }
  obj$result<-r
  obj$truth <-truth
  obj$criterion<-"roots"
  obj$criterion.args <-criterion.args
  class(obj) <- c("roots.ee","estimation.evaluation")
  invisible(obj)
}

plot.TSmodel.ee <- function(...)
     UseMethod("graph.TSmodel.ee")
graph.TSmodel.ee <- function(obj, graph.args=NULL,
                       criterion ="parms", criterion.args=NULL)
{# extract criterion and pass to another method with graph.args
  r <- do.call(paste(criterion,".TSmodel.ee", sep=""), 
               append(list(obj), list(criterion.args=criterion.args)))
  do.call("graph", append(list(r), graph.args))
  invisible(r)
}

############################################################################
#
#       methods for TSestModel.ee (estimation.evaluation)   <<<<<<<<<<
#
############################################################################

summary.TSestModel.ee    <- function(obj)
{ cat("Object of class: ",class(obj), "\n")
  if (!is.null((obj$result)[[1]]$converged))
    {rc <- rep(NA,length(obj$result))
     for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
     if(all(rc)) cat("All estimates converged.\n")
     else
       {cat(sum(!rc)," estimates did not converge!\n")
    }  }
  summary.default(obj)
  invisible()
}

parms.TSestModel.ee <- function(obj, criterion.args=NULL)
{# extract parameters from models in the list and convergence info.
 #   return a list of class parms.ee estimation.evaluation
 # criterion.args is not used. It is provided only so calls from 
 #   summary.TSmodel.ee can provide this argument.
  truth <-parms(obj$truth)
  r <- NULL
  for (m in obj$result) r <- append(r,list(parms(m)))
  rc <- rep(NA,length(obj$result))
  for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
  obj$result<-r
  obj$result.conv<-rc
  obj$truth <-truth
  obj$criterion<-"parms"
  obj$criterion.args <-criterion.args
  class(obj) <- c("parms.ee","estimation.evaluation")
  invisible(obj)
}

roots.TSestModel.ee <- function(obj, criterion.args=NULL)
{# extract roots criterion 
  truth <-do.call("roots", append(list(obj$truth), criterion.args))
  r <- NULL
  for (m in obj$result)
     r <- append(r, 
             list(do.call("roots", append(list(m), criterion.args))))
  rc <- rep(NA,length(obj$result))
  for (i in 1:length(rc)) rc[i] <- (obj$result)[[i]]$converged
  obj$result<-r
  obj$result.conv<-rc
  obj$truth <-truth
  obj$criterion<-"roots"
  obj$criterion.args <-criterion.args
  class(obj) <- c("roots.ee","estimation.evaluation")
  invisible(obj)
}

plot.TSestModel.ee <- function(...)
     UseMethod("graph.TSestModel.ee")
graph.TSestModel.ee <- function(obj, graph.args=NULL,
                       criterion ="parms", criterion.args=NULL)
{# extract criterion and pass to another method with graph.args
  r <- do.call(paste(criterion,".TSestModel.ee", sep=""), 
               append(list(obj), list(criterion.args=criterion.args)))
  do.call("graph", append(list(r), graph.args))
  invisible(r)
}

############################################################################
#
#       function for generating estimated.models (and methods).   <<<<<<<<<<
#
############################################################################
estimate.models <-function(data, estimation.sample=NULL, trend=F,quiet=F,
                       estimation.methods=NULL)
{# Estimate models from data with methods indicated by estimation.methods. 

  if (!is.null(estimation.sample))
    {# is.integer in the next line does not work 
     if (0 != (estimation.sample %%1))
        stop("estimation.sample must be an integer.")
     if (estimation.sample <= 0)
        stop("estimation.sample must be a positive integer.")
     if (nrow(output.data(data)) < estimation.sample)
        stop("estimation.sample cannot be greater than the sample size.")
     output.data(data) <- output.data(data)[1:estimation.sample,, drop=F]
     if (!is.null(input.data(data)))
        input.data(data) <- input.data(data)[1:estimation.sample,, drop=F]
    }
   r <-list(estimation.methods=estimation.methods)
  if (trend) r$trend.coef <- lsfit(1:periods(data), output.data(data))$coef
  if (!is.null(estimation.methods))
    {r$multi.model <- vector("list", length(estimation.methods))
     for (j in 1:length(estimation.methods))
       {est <-  do.call(names(estimation.methods)[j], 
                  append(list(data),  estimation.methods[[j]]))
        if (!is.null(est$converged) )
              est$model$converged <- est$converged
        # $ causes problems here
        r$multi.model[[j]] <- est$model
       }
    }
  class(r) <- c("estimated.models")
  r
}

is.estimated.models <- function(obj)
{  r <- "estimated.models"==last.class(obj)
   if(is.na(r)) r <- F
   r
}

test.equal.estimated.models <- function(obj1,obj2)
 {all(as.character(obj1) == as.character(obj2))}

print.estimated.models <- function(obj, digits=4)
 {cat("Estimated models:\n")
  if (!is.null(obj$trend.coef)) cat("trend coef: ", obj$trend.coef, "\n")
  if (!is.null(obj$multi.model))
    {for (j in 1:length(obj$multi.model))
       {cat("model ", j, "\n")
        print((obj$multi.model)[[j]])
        cat("\n")
       }
    }
 invisible(obj)
}

summary.estimated.models <-  function(obj)
{ cat("Object of class: ", class(obj),"\n")
  cat("Estimated models:\n")
  if (!is.null(obj$trend.coef)) cat("trend coef: ", obj$trend.coef, "\n")
  if (!is.null(obj$multi.model))
    {for (j in 1:length(obj$multi.model))
        cat(name(obj$estimation.methods)[j], obj$estimation.methods[[j]], "\n")
    }
  invisible()
}

roots.estimated.models <- function(obj, digits=4, mod=F)
 {cat("Estimated models:\n")
  if (!is.null(obj$trend.coef)) cat("trend coef: ", obj$trend.coef, "\n")
  if (!is.null(obj$multi.model))
    {r <- vector("list",length(obj$multi.model)) 
     for (j in 1:length(obj$multi.model))
       {cat("model ", j, "\n")
        r[[j]] <- roots((obj$multi.model)[[j]])
        if (mod) r[[j]] <- Mod(r[[j]])
        print(r[[j]], digits=digits)
        cat("\n")
       }
    }
  invisible(r)
 }

############################################################################

#    functions for evaluating forecasts    <<<<<<<<<<<<<

# Class "horizon.forecasts"  has 
#  multiple horizons forecasts calculated from every data point.  
#  This is primarily used for calculating forecast errors at different horizons.

############################################################################

#    methods for horizon.forecasts        <<<<<<<<<<<<<

############################################################################
is.horizon.forecasts <-function(obj)
{r <- "horizon.forecasts" == last.class(obj)
 if(is.na(r)) r <- F
 r
}

test.equal.horizon.forecasts <-function(obj1, obj2, fuzz=1e-14)
{# N.B. models are not compared (so equivalent models will compare true)
 r <- all(class(obj1) == class(obj2))
 if (r) r <- test.equal(obj1$data, obj2$data)
 if (r) r <- all(obj1$horizons == obj2$horizons)
 if (r) r <- obj1$discard.before == obj2$discard.before
 if (r) r <- fuzz > max(abs(obj1$horizon.forecasts - obj2$horizon.forecasts))
 if (r) r <- test.equal(obj1$data, obj2$data)
 r
}

horizon.forecasts <- function(obj, ...) UseMethod("horizon.forecasts")

horizon.forecasts.TSestModel <- function(obj, ...)
{ horizon.forecasts.TSmodel(obj$model, obj$data, ...)}

horizon.forecasts.TSdata <- function(data,model, ...)
{ horizon.forecasts.TSmodel(model, data, ...)}

horizon.forecasts.TSmodel <- function(model, data, horizons=1:4, discard.before=minimum.startup.lag(model), fortran=T)
{# calculate multiple "horizon"-step ahead forecasts 
 # ie. calculate forecasts but return only those indicated by horizons.
 #     Thus, for example, the result of
 #          horizon.forecasts(model, data horizons=c(1,5))    
 #     would be the one-step ahead and five step ahead forecasts.
 # The result is a list of class horizon.forecasts with elements model (a 
 #   TSmodel), data, horizons, discard.before, and horizon.forecasts.
 # horizon.forecasts is an array with three dimension: 
 #   c(length(horizons), dim(model$data)).
 # Projections are not calculated before discard.before or after
 #   the end of output.data(data).
 # Each horizon is aligned so that horizon.forecasts[h,t,] contains the forecast
 #   for the data point output.data(data)[t,] (from horizon[h] periods prior).
 
   if(!check.consistent.dimensions(model,data)) stop("dimension error\n")
   if (fortran) proj <- horizon.forecasts.fortran(model, data, 
                           horizons=horizons, discard.before=discard.before)
   else
     {TT <-periods(data)
      proj <- array(NA,c(length(horizons),dim(output.data(data))))
      for (t in discard.before:(TT-min(horizons)) )
        {horizons <- horizons[t <= (TT-horizons)]
         z <- l(model, data, sampleT=t, predictT=TT)$estimates$pred
         for (h in 1: length(horizons) )
             proj[h,t+horizons[h],] <- z[t+horizons[h],]
        }
     }
   dimnames(proj) <- list(NULL, NULL, output.series.names(data))
   proj <- list(model=model, data=data, horizons=horizons, 
                discard.before=discard.before, horizon.forecasts=proj)
   class(proj) <- "horizon.forecasts"
   invisible(proj)
}

# pbb2.6<- horizon.forecasts(l(bb2.vs.ls2[[3]]$model,aug10.KFmonitor.data), horizons=6, discard.before=)

# zf<- horizon.forecasts(l(bb2.vs.ls2[[3]]$model,aug10.KFmonitor.data), horizons=c(3,4), discard.before=220)

horizon.forecasts.fortran <- function(obj, ...) 
   UseMethod("horizon.forecasts.fortran")

horizon.forecasts.fortran.ARMA <-function( model, data, horizons=1:4,  discard.before=minimum.startup.lag(model))
{ if (discard.before < dim(model$A)[1] )
       warning(paste("Results may be spurious. discard.before should be set higher than the AR order (=", 
                   dim(model$A)[1]-1, ")."))
 horizons <- sort(horizons)
  p <- output.dimension(data)
  TT <- periods(data)
  proj <- array(0,c(length(horizons),TT,p))
  storage.mode(proj) <- "double"
  m <- input.dimension(model)
  if (m==0)
     {C <- array(0,c(1,p,1))    # can't pass 0 length array to fortran
      u <- matrix(0,TT,1)
     }
  else
     {C <-    model$C
      u <- input.data(data)
      if (discard.before < dim(C)[1] )
        warning(paste("Results may be spurious. discard.before should be set higher than the order of C (=", 
                      dim(C)[1]-1, ")."))
     }
  TREND <- model$TREND
  if (is.null(model$TREND)) TREND <- rep(0,p)
  is  <- max(m,p)
   .Fortran("armaprj",
                  proj=proj,    
                  as.integer(discard.before), 
                  as.integer(horizons), 
                  as.integer(length(horizons)), 
                  ey= as.double(array(0,dim(output.data(data)))), 
                  as.integer( m), 
                  as.integer( p) ,      
                  as.integer( dim(model$A)[1]),  # 1+order of A  
                  as.integer( dim(model$B)[1]),  # 1+order of B  
                  as.integer( dim(C)[1]),  # 1+order of C  
                  as.integer(TT),
                  as.double(u),
                  as.double(output.data(data)),         
                  as.double(model$A),  
                  as.double(model$B),   
                  as.double(C),
                  as.double(TREND),
                  as.integer(is),  # scratch array dim
                  as.double(matrix(0,is,is)),  # scratch array
                  as.double(matrix(0,is,is)),  # scratch array
                  as.double(rep(0,is))         # scratch array
             )$proj
}

horizon.forecasts.fortran.SS <-function( model, data, horizons=1:4, discard.before=minimum.startup.lag(model))
{ horizons <- sort(horizons)
  p <- output.dimension(data)
  TT <- periods(data)
  proj <- array(0,c(length(horizons),TT,p))
  storage.mode(proj) <- "double"
     gain <- is.innov.SS(model)
     n <- dim(model$F)[2]
     if (discard.before <= n )
         warning(paste("discard.before should probably be set higher than the state dimension (=", n, ")."))
     if (is.null(model$G))
       {m<-0
        G<-matrix(0,n,1)       # can't call fortran with 0 length arrays
        u <- matrix(0,TT,1)
       }
     else
       {m <- dim(model$G)[2]
        G <-model$G
        u <- input.data(data)[1:periods(data),,drop=F]
       } 
     if (gain)     # K or Q,R can be NUll in model, which messes up fortran
       {K <-    model$K
        Q <-    matrix(0,1,1)      #not used
        R <-    matrix(0,1,1)      #not used
       }
     else
       {Q <-    model$Q
        if (ncol(Q)<n) Q <- cbind(Q,matrix(0,n,n-ncol(Q))) # Q assumed square in fortran
        R <-    model$R
        K <-    matrix(0,n,p)      # this is used
       }
     if(is.null(model$z0)) z <-rep(0,n)   # initial state
     else  z <-model$z0
     if(is.null(model$P0)) P <- diag(1,n) # initial state tracking error 
     else  P <- model$P0              # this is not used in innov. models

     .Fortran("kfprj",
                  proj= proj, 
                  as.integer(discard.before), 
                  as.integer(horizons), 
                  as.integer(length(horizons)), 
                  ey= as.double(matrix(0,TT,p)), 
                  as.integer(m), 
                  as.integer(n), 
                  as.integer(p), 
                  as.integer(TT),  
                  as.double(u), 
                  as.double(output.data(data)),  
                  as.double(model$F),   
                  as.double(G),   
                  as.double(model$H),  
                  as.double(K), 
                  as.double(Q),      
                  as.double(R),    
                  as.logical(gain),
                  as.double(z),
                  as.double(P))$proj
}


plot.horizon.forecasts <- function(obj, Start=NULL, End=NULL, select.series=NULL, names=output.series.names(obj$data))
{#If select.series is not NULL then only indicated variables are plotted
 # if Start is null it is set to the beginning of the data.
 # if End is null it is set to the end of the data.
   output <-output.data(obj$data)
   if(is.null(names)) names <- rep(" ", dim(output)[2])
   if (is.null(select.series)) select.series <- 1: dim(output)[2]
   if (is.null(Start)) Start <- start(output)
   if (is.null(End)) End <- end(output)
   old.par <-par(mfcol = c(length(select.series), 1), mar= c(5.1,6.1,4.1,2.1))
   on.exit(par(old.par))
   for(i in select.series) 
        {zz <- tbind(output[,i],t((obj$horizon.forecasts)[,,i]))
         tframe(zz) <- tframe(output)
         plot(window(zz,start=Start,end=End, warn=F), ylab = names[i]) #tsplot
         if(i == select.series[1]) title(main = "Actual data (solid)")
        }
   invisible()
}

 
############################################################################
#
#       methods for estimators.horizon.forecasts.wrt.data.   <<<<<<<<<<
#
############################################################################
estimators.horizon.forecasts <-function(data, 
                       estimation.sample=.5, horizons=1:12,quiet=F,
                       estimation.methods=NULL)
{ # estimation.sample indicates the part of the data to use for estimation.
  # If estimation.sample is less than or equal 1.0 it is
  # used to indicate the portion of points to use for estimation. 
  # Otherwise it should be an integer and is used to indicate the number
  # of points from the beginning of the sample to use for estimation. 
  
  if (is.null(estimation.methods)) stop("estimation.methods must be specified.\n")
  if (estimation.sample <= 1.0 )
     estimation.sample <- as.integer(round(estimation.sample*nrow(output.data(data))))
  r <- list(data=data, estimation.sample =estimation.sample, horizons=horizons,  
            estimation.methods=estimation.methods )
  models <-estimate.models(data, estimation.sample=estimation.sample, 
                       trend=F,quiet=quiet,
                       estimation.methods=estimation.methods)$models
  r$multi.model <- models
  r$horizon.forecasts <- vector("list", length(estimation.methods))
  for (j in 1:length(estimation.methods))
        (r$horizon.forecasts)[[j]] <- horizon.forecasts(l(models[[j]],data),
                                  horizons=horizons, discard.before=1)
  class(r) <- c("estimators.horizon.forecasts.wrt.data") # ? "horizon.forecasts")
  r
}


############################################################################
#
#       methods for forecast.cov.   (including multiple models)<<<<<<<<<<
#
############################################################################

horizon.forecasts.forecast.cov <- function(obj,horizons=NULL, discard.before=NULL)
{# Calculate forecasts of an object for which cov has been calculated.
 # In a sense this is a step backward, but is sometimes useful to look at
 # forecasts after methods have been analysed on the basis of cov. 
 if(is.null(horizons))       horizons <- obj$horizons
 if(is.null(discard.before)) discard.before <- obj$discard.before
 if (!is.null(obj$model))
   {proj <- horizon.forecasts.TSmodel(obj$model, obj$data, horizons=horizons, 
                       discard.before=discard.before)
    class(proj) <- "horizon.forecasts"
   }
 else if (!is.null(obj$multi.model))
   {proj <-vector("list", length(obj$multi.model))
    for (i in seq(length(obj$multi.model)))
      proj[[i]] <-horizon.forecasts.TSmodel(
             (obj$multi.model)[[i]], obj$data, 
             horizons=horizons, discard.before=discard.before)
    class(proj) <- c("multi.model.horizon.forecasts","horizon.forecasts")
   }
 else  stop("Object does not include a model.\n")
 invisible(proj)
}

plot.multi.model.horizon.forecasts <- function(obj, Start=NULL, End=NULL, select.series=NULL)
 {for (i in seq(length(obj)))
    {plot(obj[[i]], Start=Start, End=End, select.series=select.series)
     cat("press return to continue>");key<-scan(what="");cat("\n")
    }
  invisible()
 }
# zz<-forecast.cov(zl, discard.before=1, horizons=1:12)
# z<-forecast.cov(l(mod3,simulate(mod3)), discard.before=20, horizons=1:12)
# zz<-forecast.cov(l(mod3,simulate(mod3)), discard.before=80, horizons=1:4)
# zzz<-forecast.cov(zz$model,zz$data, discard.before=zz$discard.before, horizons=zz$horizons)

forecast.cov <- function(obj, ...)
  {# Use model and data to construct the cov of predictions at horizons.
   # Discard predictions before (but not including) discard.before to remove 
   #    initial condition problems or do out-of-sample analysis.
   #  obj can be a TSestModel or 
   #        a TSmodel, in which case the second arg must be TSdata, or
   #          TSdata,  in which case the second arg must be a TSmodel.
   UseMethod("forecast.cov")
  }

forecast.cov.TSestModel <-function( emodel, ...)
   {forecast.cov.TSmodel( (emodel$model) , ..., data=(emodel$data) )}

forecast.cov.TSdata <-function( pred, data=NULL, horizons=1:12, discard.before=1, fortran=T)
{# Use pred$output as the predictions of data and calculate forecast.cov
 # This is mainly useful for a fixed prediction like zero or trend.
 # The calculation is dominated by sample effects: more points are
 #  dropped from the end for longer prediction horizons; the trend
 #  predictions are better for the first few periods.
 # With very large samples the result should be nearly constant for 
 # all horizons.
 # The default discard.before=1 should work ok for data, but is not 
 #    consistent with the value for model forecasts. When this routine is
 #    called by other functions the value will usually be overridden.
   horizons <- sort(horizons)
   p <- output.dimension(data)
   TT  <- periods(data)
   cov <- array(0,c(length(horizons), p,p))
   N <- rep(0,length(horizons))   # the sample size used at each horizon
   err <- pred$output - output.data(data)
   if (fortran)
     {storage.mode(cov) <-"double"
      storage.mode(err) <-"double"
      r <- .Fortran("dataepr",
                  forecast.cov=cov,    
                  as.integer(discard.before), 
                  as.integer(horizons), 
                  as.integer(length(horizons)), 
                  sample.size=as.integer(rep(0, length(horizons))),
                  as.integer(p), 
                  predictT=as.integer(TT), 
                  as.double(err)) [c("forecast.cov","sample.size")]
     }
   else
     {for (t in discard.before:(TT-horizons[1]+1))
        {h <- t-1+horizons[(t-1+horizons) <= TT]
         e <- err[h,,drop=F]
         for (k in 1:length(h))
           {N[k] <- N[k]+1
            cov[k,,] <- cov[k,,]*((N[k]-1)/N[k]) + e[k,] %o% e[k,]/N[k] 
           }
        }
       r <- list( forecast.cov=cov, sample.size=N)
     }
  dimnames(r$forecast.cov) <- list(paste("horizon",as.character(horizons)),NULL,NULL)
  r$forecast.cov <- list(r$forecast.cov)
  r <- append(r, list(pred=pred, data=data, model=NULL, horizons=horizons, 
                      discard.before=discard.before))
  class(r) <- "forecast.cov"
  r
}


forecast.cov.TSmodel <-function(obj, ..., data=NULL, discard.before=NULL,
       horizons=1:12, zero=F, trend=F, estimation.sample= periods(data),
       fortran=T)
{# Calculate the forecast cov of models in list(obj, ...) with data.
 # Using obj, ... instead of just something like model.list make argument
 # matching a bit messier, but means the method gets called for a single 
 #  TSmodel (obj), or for a list of TSmodels (obj, ...), without making the 
 #  list into a class, etc.
 # This is just multiple applications of  forecast.cov.single.TSmodel
 # discard.before is an integer indicating the number of points in the
 #   beginning of forecasts to discard before calculating covariances.
 #   If it is the default, NULL, then the default (minimum.startup.lag) will
 #   be used for each model and the default (1) will be used for trend and zero.
 # If zero  is T then forecast.cov is also calculated for a forecast of zero.
 # If trend is T then forecast.cov is also calculated for a forecast of a linear
 #   trend using data to estimation.sample.
  if (is.null(data)) stop("data= must be supplied.")
  model.list <- list(obj, ...)
  r <- list(data=data, horizons=horizons, discard.before =discard.before)
  if (trend)
     {y <- output.data(data)[1:estimation.sample,]
      pred <- cbind(1,1:periods(data)) %*%
                              (lsfit(1:estimation.sample, y)$coef)
      if (is.null(discard.before))
         r$forecast.cov.trend <- (forecast.cov.TSdata(list(output=pred), data,
             horizons=horizons)$forecast.cov)[[1]]
      else
         r$forecast.cov.trend <- (forecast.cov.TSdata(list(output=pred), data,
           horizons=horizons,discard.before=discard.before)$forecast.cov)[[1]]
     }
  if (zero)
     {if (is.null(discard.before))
        r$forecast.cov.zero <- (forecast.cov.TSdata(
             list(output=array(0,dim(output.data(data)))), data, 
             horizons=horizons)$forecast.cov)[[1]]
      else
        r$forecast.cov.zero <- (forecast.cov.TSdata(
             list(output=array(0,dim(output.data(data)))), data,
           horizons=horizons,discard.before=discard.before)$forecast.cov)[[1]]
     }
  r$forecast.cov <-vector("list", length(model.list))
  if (is.TSmodel(model.list)) model.list <- list(model.list)
  i <-0  
  for (model in model.list)
      {i <- i+1
       if (is.null(discard.before))
             rn <-  forecast.cov.single.TSmodel(TSmodel(model), data, 
                           horizons=horizons, fortran=fortran)
       else  rn <-  forecast.cov.single.TSmodel(TSmodel(model), data, 
                           horizons=horizons, discard.before=discard.before, 
                           fortran=fortran)
       #  $ in the following causes problems for some reason
       r$forecast.cov[[i]] <- rn$forecast.cov
       r$sample.size   <- rn$sample.size
      }
  r$multi.model <- model.list
  class(r) <- c("forecast.cov.wrt.data", "forecast.cov")
  r
}

is.forecast.cov.wrt.data <- function(obj)
{  r <- "forecast.cov.wrt.data"==class(obj)[1]
   if(is.na(r)) r <- F
   r
}

forecast.cov.single.TSmodel <-function( model, data=NULL, horizons=1:12, 
          discard.before=minimum.startup.lag(model), fortran=T)
{ if(!check.consistent.dimensions(model,data)) stop("dimension error.")
  if (discard.before < 1) stop("discard.before cannot be less than 1.")
  horizons <- sort(horizons)
  names <- series.names(data)$output
  if (fortran) 
     r <- forecast.cov.fortran(model, data, horizons=horizons, discard.before=discard.before)
  else
    { p <- output.dimension(data)
      shf <- start.shift(model,data,y0=y0)
      TT  <-periods(data)-(shf$shift)*(shf$lags+shf$terminal.periods)
      cov <- array(0,c(length(horizons), p,p))
      N <- rep(0,length(horizons))   # the sample size used at each horizon
  # there is a problem here with troll models trying to simulate further than
  #  the database allows. (after many steps.)
      for (t in discard.before:(TT-horizons[1]+1))
        {pred <- l(model, data, sampleT=t, predictT=TT, result="pred")
         # Eliminate longer horizons as data runs out.
         # This assumes HORIZ is sorted in ascending order.
         h <- t-1+horizons[(t-1+horizons) <= TT]
         e <- pred[h,,drop=F]- output.data(data)[h,,drop=F]
         for (k in 1:length(h))
           {N[k] <- N[k]+1
            cov[k,,] <- cov[k,,]*((N[k]-1)/N[k]) + e[k,] %o% e[k,]/N[k] 
           }
        }
       r <- list( forecast.cov=cov, sample.size=N)
     }
  dimnames(r$forecast.cov) <- list(paste("horizon",as.character(horizons)),names,names)
#  old:
# The following puts the cov in a sub list. This seems unnecessary for a single
#   cov, but means the same structure can be used with multiple model covs.
#  r$forecast.cov <- list(r$forecast.cov)
#  r <- append(r, list(model=model, data=data, horizons=horizons, 
#                     discard.before=discard.before))
#  class(r) <- "forecast.cov"
 r
}

forecast.cov.fortran <- function(obj, ...) 
   {if (!exists(paste("forecast.cov.fortran.", class(obj)[1], sep="")))
stop("Fortran for this model class is not available. Try forecast.cov( ..., fortran=F)")
    UseMethod("forecast.cov.fortran")
   }

forecast.cov.fortran.ARMA <-function( model, data, horizons=1:12 , discard.before=minimum.startup.lag(model))
{ if (discard.before < dim(model$A)[1] )
       warning(paste("Results may be spurious. discard.before should be set higher than the AR order (=",
                    dim(model$A)[1]-1, ")."))
  horizons <- sort(horizons)
  p <- output.dimension(data)
  TT <- periods(data)
  cov <- array(0,c(length(horizons), p,p))
  N <- rep(0,length(horizons))   # the sample size used of each horizon
  m <- dim(model$C)[3]
  if (is.null(model$C))
     {m <- 0
      C <- array(0,c(1,p,1))    # can't pass 0 length array to fortran
      u <- matrix(0,TT,1)
     }
  else
     {C <-    model$C
      m <- dim(model$C)[3]
      u <- input.data(data)
      if (discard.before < dim(C)[1] )
        warning(paste("Results may be spurious. discard.before should be set higher than the order of C (=", dim(C)[1]-1, ")."))
     }
  TREND <- model$TREND
  if (is.null(model$TREND)) TREND <- rep(0,p)
  storage.mode(cov) <-"double"
  is  <- max(m,p)
  .Fortran("armaepr",
                  forecast.cov=cov,    
                  as.integer(discard.before), 
                  as.integer(horizons), 
                  as.integer(length(horizons)), 
                  sample.size=as.integer(rep(0, length(horizons))),
                  pred= as.double(array(0,dim(output.data(data)))), 
                  as.integer( m), 
                  as.integer( p) ,      
                  as.integer( dim(model$A)[1]),  # 1+order of A  
                  as.integer( dim(model$B)[1]),  # 1+order of B  
                  as.integer( dim(C)[1]),  # 1+order of C  
                  predictT=as.integer(TT),
                  as.integer(nrow(output.data(data))), 
                  as.double(u),
                  as.double(output.data(data)),         
                  as.double(model$A),  
                  as.double(model$B),   
                  as.double(C),
                  as.double(TREND),
                  as.integer(is),  # scratch array dim
                  as.double(matrix(0,is,is)),  # scratch array
                  as.double(matrix(0,is,is)),  # scratch array
                  as.double(rep(0,is))         # scratch array
              )[c("forecast.cov","sample.size")]
}

forecast.cov.fortran.innov <-function(obj, ...)
  {forecast.cov.fortran.SS(obj, ...)}
forecast.cov.fortran.non.innov <-function(obj, ...) 
  {forecast.cov.fortran.SS(obj, ...)}

forecast.cov.fortran.SS <-function( model, data, horizons=1:12 , discard.before=minimum.startup.lag(model))
{ horizons <- sort(horizons)
  p <- output.dimension(data)
  TT <- periods(data)
  cov <- array(0,c(length(horizons), p,p))
  N <- rep(0,length(horizons))   # the sample size used at each horizon
     gain <- is.innov.SS(model)
     n <- dim(model$F)[2]
     if (discard.before <= n )
       warning(paste("discard.before should probably be set higher than the state dimension (=", n, ")."))
     if (is.null(model$G))
       {m<-0
        G<-matrix(0,n,1)       # can't call fortran with 0 length arrays
        u <- matrix(0,TT,1)
       }
     else
       {m <- dim(model$G)[2]
        G <-model$G
        u <- input.data(data)
       } 
     if (gain)     # K or Q,R can be NUll in model, which messes up fortran
       {K <-    model$K
        Q <-    matrix(0,1,1)      #not used
        R <-    matrix(0,1,1)      #not used
       }
     else
       {Q <-    model$Q
        if (ncol(Q)<n) Q <- cbind(Q,matrix(0,n,n-ncol(Q))) # Q assumed square in fortran
        R <-    model$R
        K <-    matrix(0,n,p)      # this is used
       }
     if(is.null(model$z0)) z <-rep(0,n)   # initial state
     else  z <-model$z0
     if(is.null(model$P0)) P <- diag(1,n) # initial state tracking error 
     else  P <- model$P0              # this is not used in innov. models

     storage.mode(cov) <-"double"
     r <- .Fortran("kfepr",
                  forecast.cov=cov,    
                  as.integer(discard.before), 
                  as.integer(horizons), 
                  as.integer(length(horizons)), 
                  sample.size=as.integer(rep(0, length(horizons))),
                  pred= as.double(array(0,dim(output.data(data)))), 
                  as.integer(m), 
                  as.integer(n), 
                  as.integer(p), 
                  predictT=as.integer(TT), 
                  as.integer(nrow(output.data(data))),  
                  as.double(u), 
                  as.double(output.data(data)),  
                  as.double(model$F),   
                  as.double(G),   
                  as.double(model$H),  
                  as.double(K), 
                  as.double(Q),      
                  as.double(R),    
                  as.logical(gain),
                  as.double(z),
                  as.double(P)) [c("forecast.cov","sample.size")]
  r
}

is.forecast.cov<-function(obj)
{  r <- "forecast.cov"==last.class(obj)
   if(is.na(r)) r <- F
   r
}

print.forecast.cov <- function(obj, digits=4)
{for (i in 1:dim((obj$forecast.cov)[[1]])[3]) 
     {cat("   ",dimnames(obj$forecast.cov)[[1]][i], "\n")
      z <- NULL
      for (j in 1:length(obj$forecast.cov) )
         z <- tbind(z, (obj$forecast.cov)[[j]][,i,i])
      print(z, digits=digits)
     }
 invisible(obj)
}

summary.forecast.cov <-  function(obj, digits=4, horizons=obj$horizons, 
    select.series=seq(output.dimension(obj$data)))
{cat("class: ", class(obj),"   ")
 cat( length(obj$horizons), " horizons\n")
# summary.default(obj)
 cat(length(obj$multi.model),"models\n")
 names <- output.series.names(obj$data)
 if(!is.numeric(select.series)) select.series <- match(select.series, names)
 names <- names[select.series]
 for (i in seq(length(obj$multi.model)))
   {cat("Model", i, obj$multi.model[[i]]$description,  "\n")
    cat("   variable", names,"\n")
    z <- NULL
    for (h in seq(length(horizons))) z <- rbind(z,
             diag(obj$forecast.cov[[i]][h,select.series,select.series])^0.5)
    dimnames(z) <- list(paste("S.D.horizon", horizons), names)
    print(z, digits = digits)   
   }
 cat("\n")
 NextMethod("summary")
}

test.equal.forecast.cov <-function(obj1, obj2, fuzz=1e-14)
{if (is.null(obj1$seed)) ok <- T
 else ok <- all(obj1$seed == obj2$seed)
 if (ok & !is.null(obj1$forecast.cov.true) )
  {if (is.null(obj2$forecast.cov.true)) ok <-F
   ok <- fuzz > max(abs(obj1$forecast.cov.true-obj2$forecast.cov.true))
  }
 if (ok & !is.null(obj1$forecast.cov.zero)) 
  {if (is.null(obj2$forecast.cov.zero)) ok <-F
   else ok <- fuzz > max(abs(obj1$forecast.cov.zero-obj2$forecast.cov.zero))
  }
 if (ok & !is.null(obj1$forecast.cov.trend)) 
  {if (is.null(obj2$forecast.cov.trend)) ok <-F
   else ok <- fuzz > max(abs(obj1$forecast.cov.trend-obj2$forecast.cov.trend))
  }
 for (i in 1:length(obj1$forecast.cov))
   {if (ok & !is.null((obj1$forecast.cov)[[i]])) 
         {if (is.null((obj2$forecast.cov)[[i]])) ok <-F 
          else ok <- fuzz > 
               max(abs((obj1$forecast.cov)[[i]]-(obj2$forecast.cov)[[i]]))
      }
   }
 ok
}

total.forecast.cov <- function(obj, select=NULL)
{if (is.null(select)) select <-1:dim((obj$forecast.cov)[[1]])[2]
 N <- c( dim((obj$forecast.cov)[[1]])[1] ,1,1)
 for (j in 1:length(obj$forecast.cov) )
   {z <- apply((obj$forecast.cov)[[j]],1,diag)
    # $ causes problems
    obj$forecast.cov[[j]] <- array(apply(z[select,],2,sum), N)
   }
 if(!is.null(obj$forecast.cov.true))
   {z <- apply(obj$forecast.cov.true,1,diag)
    obj$forecast.cov.true <- array(apply(z[select,],2,sum), N)
   }
 if(!is.null(obj$forecast.cov.zero))
   {z <- apply(obj$forecast.cov.zero,1,diag)
    obj$forecast.cov.zero <- array(apply(z[select,],2,sum), N)
   }
 if(!is.null(obj$forecast.cov.trend)) 
   {z <- apply(obj$forecast.cov.trend,1,diag)
    obj$forecast.cov.trend <- array(apply(z[select,],2,sum), N)
   }
 invisible(obj)
}



plot.forecast.cov <- function(obj,
    select.series=1:dim((obj$forecast.cov)[[1]])[2],
    select.cov= 1:length(obj$forecast.cov),
    select.true=T, select.zero=T, select.trend=T,
    y.limit=NULL,
    line.labels=F,
    lty=seq(length(select.cov) +select.true+select.zero+select.trend),
    Legend=NULL, Title=NULL,
    graphs.per.page=5,  ...)
{# ... should be arguments to par().
 # select.cov indicates which covariances to display 
 #  (ie. which model or estimation method)
 # select.series, if specified, indicates which series to display.
 #  cex= can be passed as an argument to change character print size.
 p <- dim((obj$forecast.cov)[[1]])[2]
 Ngraph <- 1+min(length(select.series), graphs.per.page)
 old.par <- par(mfcol=c(Ngraph,1), mar = c(5.1, 6.1, 4.1, 2.1))
 on.exit(par(old.par))
 par(...)
 names <-dimnames((obj$forecast.cov)[[1]])[[2]]
 if (is.null(names)) names <- paste("variable",1:p)
 for (i in select.series) 
   {z <- matrix(0,length(obj$horizons),length(select.cov))
    for (j in 1:length(select.cov))
            z[,j]<-(obj$forecast.cov)[[select.cov[j] ]][,i,i]
    if(select.trend & !is.null(obj$forecast.cov.trend))
            z <- tbind((obj$forecast.cov.trend)[,i,i],z)
    if(select.zero & !is.null(obj$forecast.cov.zero)) 
            z <- tbind((obj$forecast.cov.zero)[,i,i],z)
    if(select.true & !is.null(obj$forecast.cov.true)) 
            z <- tbind((obj$forecast.cov.true)[,i,i],z)
    show <- 1:length(select.cov)
    if (!is.null(y.limit))
      {z[z>y.limit] <- NA
       show <- !apply(is.na(z),2,all) #Legend may be messed up for multi-v. case
       z <- z[,show]
      }
    if (is.character(i)) ylab <- i
    else ylab <- names[i]
    matplot(obj$horizons,z, type="l",lty=lty, xlab="horizon", ylab=ylab) #cex, mar set by par above
    if (line.labels)
      {labels <- select.cov[show]
       if (! is.null(obj$selection.index)) labels <- obj$selection.index[labels]
       text(dim(z)[1], z[dim(z)[1],], labels)
      }
    if ((i==1) & (!is.null(Title))) title(main = Title) 
   }
 if (is.null(Legend))
   {Legend <- paste("prediction covariance", select.cov[show])
    if (! is.null(obj$selection.index)) 
       Legend <- paste(Legend, "=", obj$selection.index[show])
    if(select.trend & !is.null(obj$forecast.cov.trend)) Legend  <- c("trend",Legend)
    if(select.zero  & !is.null(obj$forecast.cov.zero))  Legend  <- c("zero", Legend)
    if(select.true  & !is.null(obj$forecast.cov.true))  Legend  <- c("true", Legend)
   }
 par(mfg=c(Ngraph,1,Ngraph,1))
 legend((par()$usr)[1:2],(par()$usr)[3:4], Legend, lty=lty, bty="n")
 invisible()
}

graph.forecast.cov <- function(..., select=NULL, Legend=NULL, Title=NULL)
{# graph multiple prediction evaluations
 obj <- list(...)[[1]]
 data <- obj$data
 p <- dim((obj$forecast.cov)[[1]])[2]
 if (is.null(select)) select <-1:p
 old.par <- par(mfcol=c(1+length(select),1))
 on.exit(par(old.par))
 names <-output.series.names(data)
 if (is.null(names)) names <- rep(" ",p)
 N <- length(list(...))
 for (i in select) 
   {z <- matrix(0,dim((obj$forecast.cov)[[1]])[1],N)
    j <- 0
    for (obj in list(...) ) 
       {j <- j+1
        if(!is.forecast.cov(obj))
           { if (i == select[1])
              {mn <- as.character(substitute(list(...)))[j+1]
               cat("Warning: object ",mn, " is not of class forecast.cov!\n")
           }  }
        else z[,j] <-(obj$forecast.cov)[[j]][,i,i]
       }
    matplot(x=obj$horizons, y=z, xlab="horizon", ylab=names[i], type="l") #, cex=2*par()$cex) 
   }

 par(mfg=c(N+1,1,N+1,1))
 if(is.null(Title))
   {obj <- list(...)[[1]]
    Title <- paste("Models estimated with data from 1 to ", 
                   as.character(obj$estimation.sample) )
    Title <- paste(Title, "\nPrediction variance calculated with predictions from")
    Title <- paste(Title, as.character(obj$discard.before))
    Title <- paste(Title, "to")
    Title <- paste(Title, as.character(periods(data)))
   }
 title(main = Title)  #, cex=par()$cex)
 if (is.null(Legend))
    {Legend <-as.character(substitute(list(...)))[-1]
     for (obj in list(...) ) 
        Legend <-paste(Legend, paste(obj$estimation.method, obj$estimation.arg))
    }
 legend((par()$usr)[1:2],(par()$usr)[3:4], Legend, lty=1:N, bty="n")
 invisible()
}

test.graph.forecast.cov <- function(..., select.series=NULL, lty=1:5, Legend=NULL, Title=NULL)
{# graph multiple prediction evaluations
 # Only the first forecast.cov is used from each element of ... . Use extract
 #  first if necessary to select other elements.

 # select.series, if specified, indicates which series to display.
 # If specified select.series should be a list of vectors (one vector for each 
 # element of ...) indicating the series to be used in each element of ... .
 # This allows comparing models which only have a subset of series in common.
 # If not specified it is assumed that all elements of ... have the same
 # number of series and all will be displayed

 N <- length(list(...))
 obj <- list(...)[[1]]
 data <- obj$data
 horizon <- dim((obj$forecast.cov)[[1]])[1]
 p <- dim((obj$forecast.cov)[[1]])[2]
 if (is.null(select.series))
     select.series <-rep(1:dim((obj$forecast.cov)[[1]])[2], N)
 old.par <- par(mfcol=c(1+length(select.series),1))
 names <-dimnames(output.data(data))[[2]]
 if (is.null(names)) names <- rep(" ",p)
 for (i in seq(length(select.series[[1]]))) 
   {z <- matrix(0,horizon,N)
    j <- 0
    for (obj in list(...) ) 
       {j <- j+1
        if(!is.forecast.cov(obj))
           {mn <- as.character(substitute(list(...)))[j+1]
            cat("Warning: object ",mn, " is not of class forecast.cov!\n")
           } 
        else z[,j] <-(obj$forecast.cov)[[1]][, 
              select.series[[j]][i],select.series[[j]][i]]
       }
    matplot(x=obj$horizons, y=z, xlab="horizon", ylab=names[i], type="l") #, cex=2*par()$cex)
   }

 par(mfg=c(N+1,1,N+1,1))
 if(is.null(Title))
   {obj <- list(...)[[1]]
    Title <- paste("Models estimated with data from 1 to ", 
                   as.character(obj$estimation.sample) )
    Title <- paste(Title, "\nPrediction variance calculated with predictions from")
    Title <- paste(Title, as.character(obj$discard.before))
    Title <- paste(Title, "to")
    Title <- paste(Title, as.character(periods(data)))
   }
 title(main = Title)  #, cex=par()$cex)
 if (is.null(Legend))
    {Legend <-as.character(substitute(list(...)))[-1]
     for (obj in list(...) ) 
        Legend <-paste(Legend, paste(obj$estimation.method, obj$estimation.arg))
    }
 legend((par()$usr)[1:2],(par()$usr)[3:4], Legend, lty=1:N, bty="n")
 par(old.par)
 invisible()
}

############################################################################
#
#       end
#
############################################################################
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################




############################################################################
#
#   methods for "forecast.cov.estimators.wrt.data", "forecast.cov"  <<<<<<<<<<
#                  (multiple estimators,  given data)
#
############################################################################
#z <-out.of.sample.forecast.cov.estimators.wrt.data(zl$data,
#       estimation.methods = list(est.VARX.ar=list(max.lag=2),
#                                 est.VARX.ls=list(max.lag=2)))

#z <-out.of.sample.forecast.cov.estimators.wrt.data(data,
#             estimation.sample=.5,trend=T, zero=T,
#             estimation.methods = list(
#                   est.VARX.ls=list(max.lag=4),
#                   est.wt.variables=list(
#                       variable.weights=c(1,1,0.5,0.5,0.5,0.5,1,0.5,0.5,0.5),
#                       estimation.methods=list(est.VARX.ls=list(max.lag=4))))


out.of.sample.forecast.cov.estimators.wrt.data <-function(data, zero=F, trend=F,
                       estimation.sample=.5, horizons=1:12,quiet=F,
                       estimation.methods=NULL)
{ # estimation.sample indicates the portion of the data to use for estimation.
  #If estimation.sample is an integer then it is used to indicate the number
  # of points in the sample to use for estimation. If it is a fracton it is
  # used to indicate the portion of points to use for estimation. The remainder
  # of the sample is used for evaluating forecasts.
  
  if (estimation.sample < 1.0 )
     estimation.sample <- as.integer(round(estimation.sample*nrow(output.data(data))))
  discard.before <- 1+estimation.sample
  forecast.cov.estimators.wrt.data(data, estimation.sample, discard.before,
                       horizons=horizons, zero=zero, trend=trend, quiet=quiet,
                       estimation.methods=estimation.methods)
}


forecast.cov.estimators.wrt.data <-function(data, estimation.sample=NULL, 
                       discard.before=10,
                       horizons=1:12, zero=F, trend=F,quiet=F,
                       estimation.methods=NULL)
{# Calculate the forecasts cov of models estimated from data with estimation
 #   methods indicated by estimation.methods  (see estimate.models).
 # estimation.sample is an integer indicating the number of points in the
 #     sample to use for estimation. If it is NULL the whole sample is used.
 # discard.before is an integer indicating 1+the number of points in the
 #     beginning of forecasts to discard for calculating covariances.
 # If zero  is T then forecast.cov is also calculated for a forecast of zero.
 # If trend is T then forecast.cov is also calculated for a forecast of a linear trend.

  r <- list(data=data, estimation.sample =estimation.sample,
            horizons=horizons, discard.before =discard.before, 
            estimation.methods=estimation.methods)
  if (zero)
     {r$forecast.cov.zero <-forecast.cov.TSdata(
             list(output=array(0,dim(output.data(data)))), data, 
                discard.before=discard.before, horizons=horizons)$forecast.cov[[1]]
     }
  models <-estimate.models(data, estimation.sample=estimation.sample, 
                       trend=trend,quiet=quiet,
                       estimation.methods=estimation.methods)
  if (trend)
     {pred <- cbind(1,1:periods(data)) %*% models$trend.coef
      r$forecast.cov.trend <- forecast.cov.TSdata(list(output=pred), data, 
              discard.before=discard.before, horizons=horizons)$forecast.cov[[1]]
     }
  r$multi.model <- models$multi.model
  if (!is.null(estimation.methods))
    {r$forecast.cov <- vector("list", length(estimation.methods))
     for (j in 1:length(estimation.methods))
        {rn <-  forecast.cov.single.TSmodel(r$multi.model[[j]], data, 
                        discard.before=discard.before, horizons=horizons)
         r$forecast.cov[[j]] <- rn$forecast.cov
         r$sample.size   <- rn$sample.size
        }
    }
  class(r) <- c("forecast.cov.estimators.wrt.data", "forecast.cov")
  r
}

is.forecast.cov.estimators.wrt.data<-function(obj)
{  r <- "forecast.cov.estimators.wrt.data"==class(obj)[1]
   if(is.na(r)) r <- F
   r
}

combine.forecast.cov.estimators.wrt.data <- function(e1,e2)
  {if(! test.equal(e1$data, e2$data)) 
       warning("data is not the same. Second set suppressed.")
   if(! all(e1$estimation.sample == e2$estimation.sample)) 
       warning("estimation.sample's are not the same. Second one suppressed.")
   if(! all(e1$horizon == e2$horizon)) 
       stop("horizon's are not the same.")
   if(e1$discard.before != e2$discard.before) 
       warning("discard.before's are not the same. Second one suppressed.")
   e1$forecast.cov <- append(e1$forecast.cov, e2$forecast.cov)
   e1$estimation.methods <- append(e1$estimation.methods, e2$estimation.methods)
# fix   e1$multi.model <- append(e1$multi.model, e2$multi.model)  
   e1
}

extract.forecast.cov.estimators.wrt.data <- function(e,n)
  {# select indicated forecast.cov
   e$forecast.cov <- e$forecast.cov[[n]]
   e$estimation.methods <- e$estimation.methods[[n]]
   e$multi.model        <- e$multi.model[[n]]  
   e
}

plot.forecast.cov.estimators.wrt.data <- function(obj, 
    select.series=1:dim(obj$forecast.cov[[1]])[2], 
    select.cov=1:length(obj$forecast.cov),
    select.zero=T, select.trend=T,  lty=1:5, ...)
{# ... should be arguments to par(). See plot.forecast.cov for more details.
Legend<- paste(names(obj$estimation.methods), obj$estimation.methods)[select.cov]
 if(select.trend & !is.null(obj$forecast.cov.trend)) Legend  <- c("trend",Legend)
 if(select.zero  & !is.null(obj$forecast.cov.zero))  Legend  <- c( "zero",Legend)
 plot.forecast.cov(obj, select.series=select.series, lty=lty,
        select.cov=select.cov,
        select.zero=select.zero, select.trend=select.trend, Legend=Legend, 
        Title="Prediction variance relative to given data.",
        ...)
 invisible()
}

#graph.forecast.cov.estimators.wrt.data <- function( ..., select=NULL)
#{obj <- list( data = list(...)[[1]]$data)
# i <-0
# for (obji in list(...) )
#   {i <- i+1
#    obji$data <-NULL
#    na <- paste("obj$obj",as.character(i), sep="")
#    na <- paste(paste(na, names(obji), sep=""), " <-obji[[j]]" )
#    for (j in 1:length(obji)) eval(parse(text=na[j]))
#   }
# class(obj) <- "forecast.cov.estimators.wrt.data"
# plot.forecast.cov.estimators.wrt.data(obj, select=select)
# invisible()
#}

#date.ts <- function(x,i)
#{# date of ith position in time series x
# s <- start(x)
# tsp.x <-tsp(x)
# p <-s[2]+i
# y <- s[1]+ ((p-1) %/% tsp.x[3])
# p <- 1+(p %% tsp.x[3])
# c(y,p)
#}


############################################################################
#
#   methods for "forecast.cov.wrt.true", "forecast.cov"  <<<<<<<<<<
#    given true model, evaluate multiple estimation techniques
#    with multiple simulations for estimation and
#         multiple simulations for forecast
#
############################################################################
forecast.cov.wrt.true <-function( models, true.model, 
        pred.replications=1, simulation.args=NULL, quiet=F, 
        seed=NULL, Spawn=.SPAWN, 
        horizons=1:12, discard.before=10, trend=NULL, zero=NULL)
{# models should be a list of models
 # The true model is used to generate more
 # data and for each generated data set the forecasts of the 
 # models are evaluated against the simulated data.
 # if trend is not null it is treated as a model output (forecast) and
 # should be the same dimension as a simulation of the models with 
 # simulation.args. If zero is not null a zero forecast is also evaluated.

 seed <- set.seed(seed)
      
 if (Spawn & (pred.replications > 1))
   {if(!quiet) 
      cat("Spawning processes to calculate ", pred.replications,
            " forecast replications.\n")
    rep.forloop <- function(models, true.model, simulation.args,
                         horizons, discard.before, zero, trend)
      {data<-do.call("simulate",append(list(true.model), simulation.args))
       r <- NULL
       for (j in 1:length(models))
              {r <- c(r, forecast.cov.single.TSmodel(models[[j]],data,
                                   horizons=horizons, 
                                  discard.before=discard.before)$forecast.cov)
              }
         r.true <- forecast.cov.single.TSmodel(true.model,data, horizons=horizons,
                               discard.before=discard.before)$forecast.cov
         if (is.null(trend)) r.trend <- NULL
         else  r.trend <-forecast.cov.TSdata(list(output=trend),
            data, discard.before=discard.before,horizons=horizons)$forecast.cov[[1]]
         if(is.null(zero)) r.zero <- NULL
         else r.zero <- forecast.cov.TSdata(
              list(output=array(0,dim(output.data(data)))), data, 
                discard.before=discard.before, horizons=horizons)$forecast.cov[[1]]
         c(dim(r.true),r.true, r.zero, r.trend,r)
       }

    assign("rep.forloop", rep.forloop, where = 1)
    assign("rep.forloop.n", pred.replications, where = 1)
    assign("rep.forloop.result", 0, where = 1)
    assign("rep.forloop.true.model", true.model, where = 1)
    assign("rep.forloop.simulation.args", simulation.args, where = 1)
    assign("rep.forloop.models", models, where = 1)
    assign("rep.forloop.horizons", horizons, where = 1)
    assign("rep.forloop.discard.before", discard.before, where = 1)
    assign("rep.forloop.trend", trend, where = 1)
    assign("rep.forloop.zero", zero, where = 1)
    on.exit(remove(c("rep.forloop", "rep.forloop.i", "rep.forloop.n",
            "rep.forloop.models",  "rep.forloop.true.model",
            "rep.forloop.simulation.args","rep.forloop.result",
            "rep.forloop.horizons", "rep.forloop.discard.before", 
            "rep.forloop.trend","rep.forloop.zero"),where = 1))

    For(rep.forloop.i = 1:rep.forloop.n, 
       rep.forloop.result <- rep.forloop.result +
          rep.forloop(rep.forloop.models, rep.forloop.true.model,
          rep.forloop.simulation.args, rep.forloop.horizons,
          rep.forloop.discard.before, rep.forloop.zero, rep.forloop.trend),
       first=options(warn=-1), sync = T)

    names <- list(paste("horizon",as.character(horizons)),NULL,NULL)
    result  <- rep.forloop.result/pred.replications
    d <- result[1:3]  # this is not a very elegant way to pass this info.
    l <- prod(d)
    r.true  <- array(result[4:(3+l)], d)
    dimnames(r.true)  <- names
    lj <- 1
    if (is.null(zero)) r.zero <- NULL
    else
      {r.zero  <- array(result[(4+l):(3+2*l)], d)
       lj <-lj+1
       dimnames(r.zero)  <- names
      }
    if (is.null(trend)) r.trend <- NULL
    else
      {r.trend  <- array(result[(4+lj*l):(3+(1+lj)*l)], d)
       lj <-lj+1
       dimnames(r.trend) <- names
      }
    r <- vector("list", length(models))
    for (j in 1:length(models))
      {r[[j]] <- array(result[(4+lj*l):(3+(1+lj)*l)], d) 
       lj <-lj+1
       dimnames(r[[j]])  <- names
      }
     }
   else
     {r <-vector("list",length(models))
      r.true <- 0
      if (!is.null(zero)) r.zero <- 0
      else r.zero <- NULL
      if (!is.null(trend)) r.trend<- 0
      else r.trend <- NULL
      for (i in 1:pred.replications)
        {data<-do.call("simulate",append(list(true.model), simulation.args))
         for (j in 1:length(models))
              {fc <- forecast.cov.single.TSmodel(models[[j]],data,
                                   horizons=horizons, 
                                  discard.before=discard.before)$forecast.cov
               if (i == 1) r[[j]] <- fc
               else        r[[j]] <- r[[j]] + fc
              }
         r.true <- r.true+forecast.cov.single.TSmodel(true.model,data, horizons=horizons,
                               discard.before=discard.before)$forecast.cov
         if (!is.null(trend))
           r.trend <-r.trend+forecast.cov.TSdata(list(output=trend),
             data,discard.before=discard.before,horizons=horizons)$forecast.cov[[1]]
         if(!is.null(zero))
            r.zero <- r.zero + forecast.cov.TSdata(
                list(output=array(0,dim(output.data(data)))), data, 
                 discard.before=discard.before, horizons=horizons)$forecast.cov[[1]]
        }
      for (j in 1:length(models))  r[[j]] <- r[[j]]/pred.replications
      r.true  <-  r.true/ pred.replications
      if (!is.null(zero)) r.zero  <-  r.zero/ pred.replications
      if (!is.null(trend)) r.trend <-  r.trend/pred.replications
     }
   r<-list(forecast.cov=r, forecast.cov.true=r.true, 
           forecast.cov.zero=r.zero, forecast.cov.trend=r.trend,
           multi.model=models,
           seed=seed, version=version,
           pred.replications=pred.replications,
           horizons=horizons, discard.before=discard.before)
   class(r) <- c("forecast.cov.wrt.true", "forecast.cov")
   invisible(r)
}

forecast.cov.estimators.wrt.true <-function(true.model, Spawn=.SPAWN, seed=NULL,
                       simulation.args=NULL,
                       est.replications=2, pred.replications=2,
                       discard.before=10, horizons=1:12,quiet=F,
                       estimation.methods=NULL)
{# Calculate the forecasts cov of models estimated from simulations of 
 # true.model with estimation methods indicated by estimation.methods (see 
 #       estimate.models). 
 # discard.before is an integer indicating 1+the number of points in the
 # beginning of forecasts to discard for calculating forecast covariances.
 # The returned results has element
 #  $forecast.cov.true  $forecast.cov.zero $forecast.cov.trend containing 
 #    covariances averaged over estimation replications and simulation
 #    replications (forcasts will not change but simulated data will).
 #  $forecast.cov a list of the same length as estimation.methods with each
 #    element containing covariances averaged over estimation replications 
 #    and simulation replications.
 #  $estimated.models a list of length est.replications, with each elements as
 #    returned by estimate.models, thus each element has $multi.model as a
 #    subelement containing models for different estimation techniques.  
 #    So, eg.   $estimated.models[[2]]$multi.model[[1]]  in the result will
 #    be the model from the first estimation technique in the second replication. 
 seed <- set.seed(seed)
 
 estimated.models <- vector("list", est.replications)
 for (i in 1:est.replications)
        {data<-do.call("simulate",append(list(true.model), simulation.args))
         models <-estimate.models(data, trend=T,quiet=quiet,
                       estimation.methods=estimation.methods)
         estimated.models[[i]] <- models
         rn <- forecast.cov.wrt.true( models$multi.model, true.model, 
                    pred.replications=pred.replications, zero=T, quiet=quiet,
                    simulation.args=simulation.args, Spawn=Spawn,
                    horizons=horizons, discard.before=discard.before,
                    trend=cbind(1,1:periods(data)) %*% models$trend.coef)
         if (i==1)
           r<-rn[c("forecast.cov","forecast.cov.true",
                   "forecast.cov.zero","forecast.cov.trend")]
         else
          {for (j in 1:length(estimation.methods))
              r$forecast.cov[[j]] <- r$forecast.cov[[j]]*(i-1)/i + rn$forecast.cov[[j]]/i
           r$forecast.cov.true  <- r$forecast.cov.true *(i-1)/i + rn$forecast.cov.true/i
           r$forecast.cov.zero  <- r$forecast.cov.zero *(i-1)/i + rn$forecast.cov.zero/i
           r$forecast.cov.trend <- r$forecast.cov.trend*(i-1)/i + rn$forecast.cov.trend/i
          }
        }
  r<-append(r, list(true.model=true.model,estimation.methods=estimation.methods,
         estimated.models=estimated.models,
         seed=seed, version=version,
         horizons=horizons, 
         discard.before=discard.before, est.replications=est.replications,
         pred.replications=pred.replications, simulation.args=simulation.args))
  class(r) <- c("forecast.cov.estimators.wrt.true", "forecast.cov")
  r
}

is.forecast.cov.estimators.wrt.true<-function(obj)
{  r <- "forecast.cov.estimators.wrt.true"==class(obj)[1]
   if(is.na(r)) r <- F
   r
}


print.forecast.cov.estimators.wrt.true <- function(obj, digits=4)
{cat("forecast.cov.estimators.wrt.true\n")
 cat("essential data:", obj$essential.data, "\n")
 cat("considering:", output.series.names(obj$all.data), 
                      input.series.names(obj$all.data), "\n")
 invisible(obj)
}

summary.forecast.cov.estimators.wrt.true <- function(obj, digits = 4)
{cat("class: ", class(obj), "   ")
 cat(length(obj$horizons), " horizons\n")
 for (i in seq(length(obj$estimated.models)))
   {if (!is.null(obj$estimated.models[[i]]$multi.model[[1]]$converged))
      {cat("Estimated model", i)
       if(!obj$estimated.models[[i]]$multi.model[[1]]$converged) cat(" NOT")
       cat(" converged.\n")
   }  }
 NextMethod("summary")
}

roots.forecast.cov.estimators.wrt.true <- function(obj, digits=4, mod=F)
 {cat("Estimated models:\n")
  if (!is.null(obj$trend.coef)) cat("trend coef: ", obj$trend.coef, "\n")
  if (!is.null(obj$estimated.models))
    {r <- vector("list",length(obj$estimated.models)) 
     for (i in 1:length(obj$estimated.models))
       {cat("estimation ", i, "\n")
        for (j in 1:length(obj$estimated.models[[i]]$multi.model))
         {r[[i]] <- vector("list",length(obj$estimated.models[[i]]$multi.model))
          cat("model ", j, "\n")
          r[[i]][[j]] <- roots(obj$estimated.models[[i]]$multi.model[[j]])
          if (mod) r[[i]][[j]] <- Mod(r[[i]][[j]])
          print(r[[i]][[j]], digits=digits)
          cat("\n")
         }
       }
    }
  invisible(r)
 }

extract.forecast.cov.estimators.from.model <- function(e,n)
  {# select indicated forecast.cov
   e$forecast.cov <- e$forecast.cov[[n]]
   e$estimation.methods <- e$estimation.methods[[n]]
   e$estimated.models   <- e$estimated.models[[n]]  
   e
}

combine.forecast.cov.estimators.wrt.true <- function(e1,e2)
  {if(! test.equal(e1$true.model, e2$true.model)) 
       warning("true.models are not the same.")
   if(! all(e1$seed == e2$seed)) 
       warning("seeds are not the same. Second one suppressed.")
   if (!all(c(e1$version[[1]] == e2$version[[1]],
              e1$version[[2]] == e2$version[[2]],
              e1$version[[3]] == e2$version[[3]],
              e1$version[[4]] == e2$version[[4]],
              e1$version[[5]] == e2$version[[5]],
              e1$version[[6]] == e2$version[[6]],
              e1$version[[7]] == e2$version[[7]],
              e1$version[[8]] == e2$version[[8]],
              e1$version[[9]] == e2$version[[9]]))) 
       warning("versions are not the same. Second one suppressed.")
   if(! all(e1$horizon == e2$horizon)) 
       stop("horizon's are not the same.")
   if(e1$discard.before != e2$discard.before) 
       warning("discard.before's are not the same. Second one suppressed.")
   if(e1$est.replications != e2$est.replications) 
       warning("est.replications's are not the same. Second one suppressed.")
   if(e1$pred.replications != e2$pred.replications) 
       warning("pred.replications's are not the same. Second one suppressed.")
   if(! (is.null(e1$simulation.args) & is.null(e2$simulation.args)) )
     if(! all(e1$simulation.args == e2$simulation.args)) 
       warning("simulation.args's are not the same. Second one suppressed.")
   e1$forecast.cov <- append(e1$forecast.cov, e2$forecast.cov)
   e1$estimation.methods <- append(e1$estimation.methods, e2$estimation.methods)
# fix   e1$multi.model <- append(e1$multi.model, e2$multi.model)  
   e1
}

combine.forecast.cov <- function(e1,e2)
  {if(! test.equal(e1$model, e2$model)) 
       warning("models are not the same. Second one suppressed.")
   if(! test.equal(e1$data, e2$data)) 
       warning("data is not the same. Second set suppressed.")
   if(! all(e1$sample.size == e2$sample.size))
       warning("sample.sizes are not the same. Second one suppressed.")
   if(! all(e1$horizons == e2$horizons)) 
       stop("horizon's are not the same.")
   if(e1$discard.before != e2$discard.before) 
       warning("discard.before's are not the same. Second one suppressed.")
   e1$forecast.cov <- append(e1$forecast.cov, e2$forecast.cov)
   e1
}

forecast.cov.reductions.wrt.true <-function(true.model, Spawn=.SPAWN, seed=NULL,
                       simulation.args=NULL,
                       est.replications=2, pred.replications=2,
                       discard.before=10, horizons=1:12,quiet=F,
                       estimation.methods=NULL,
                       criteria=NULL)
{# Calculate the forecasts cov of reduced models estimated from simulations of
 # true.model with an estimation method indicated by estimation.methods. 
 #  (estimation.methods is as in estimation.models BUT ONLY THE FIRST IS USED.)
 # discard.before is an integer indicating 1+the number of points in the
 # beginning of forecasts to discard for calculating forecast covariances.
 # criteria can  be a vector of criteria as in information.tests,
 #  (eg c("taic", "tbic") in which case the "best" model for each criteria
 #  is accounted separately. (ie. it is added to the beginning of the list of
 # estimated models)

 seed <- set.seed(seed)
 information.criteria <- NULL
 for (i in 1:est.replications)
        {data<-do.call("simulate",append(list(true.model), simulation.args))
         models <-estimate.models(data, trend=T,quiet=quiet,
                       estimation.methods=estimation.methods)
         models$multi.model <-reduced.models.Mittnik(models$multi.model[[1]]) # use only 1
         crit <- NULL
         for (m in models$multi.model) 
            crit<-rbind(crit,information.tests.calculations(l(m, data)))
         if(!is.null(criteria))
           {addmodels <- vector("list", length(criteria))
            for (i in 1:length(criteria))
              addmodels[[i]] <- models$multi.model[[order(crit[,criteria[i]])[1]]]
            models$multi.model <- append(addmodels, models$multi.model)
           }
         information.criteria <- append(information.criteria, list(crit))
         rn <- forecast.cov.wrt.true( models$multi.model, true.model, 
                    pred.replications=pred.replications, zero=T, quiet=quiet,
                    simulation.args=simulation.args, Spawn=Spawn,
                    horizons=horizons, discard.before=discard.before,
                    trend=cbind(1,1:periods(data)) %*% models$trend.coef)
         if (i==1)
           r<-rn[c("forecast.cov","forecast.cov.true","forecast.cov.zero","forecast.cov.trend")]
         else
          {for (j in 1:length(models$multi.model))
             r$forecast.cov[[j]] <- r$forecast.cov[[j]]*(i-1)/i + rn$forecast.cov[[j]]/i
           r$forecast.cov.true   <- r$forecast.cov.true*(i-1)/i + rn$forecast.cov.true/i
           r$forecast.cov.zero   <- r$forecast.cov.zero*(i-1)/i + rn$forecast.cov.zero/i
           r$forecast.cov.trend <- r$forecast.cov.trend*(i-1)/i + rn$forecast.cov.trend/i
          }
        }
  r<-append(r, list(true.model=true.model,
         estimation.methods=c(criteria,estimation.methods),
         seed=seed, version=version,
         horizons=horizons, 
         discard.before=discard.before, est.replications=est.replications,
         pred.replications=pred.replications, simulation.args=simulation.args, 
         information.criteria=information.criteria))
  class(r) <- c("forecast.cov.estimators.wrt.true", "forecast.cov")
  r
}

reduced.models.Mittnik <-function(largeModel)
{# Return a list of models with all smaller state dimesions.
  largeModel <- to.SS(largeModel)
  largeModel <- balance.Mittnik(largeModel, n=dim(largeModel$F)[1])
  r <- vector("list", dim(largeModel$F)[1])
  for (j in 1:length(r))
    {if(!is.null(largeModel$G)) z <-largeModel$G[1:j,,drop=F]
     else                       z <-NULL
     z <-list(F=largeModel$F[1:j,1:j,drop=F],G=z,
              H=largeModel$H[  , 1:j, drop=F],K= largeModel$K[1:j,,drop=F])
     r[[j]] <-set.parameters(TSmodel(z))
    }
  r
}


############################################################################
#
#       misc     <<<<<<<<<<
#
############################################################################


dimension <- function(m, ...)UseMethod("dimension")

dimension.SS <- function(model)
{if (is.null(model$G)) nt <- nrow(model$F)*2*nrow(model$H)
 else   nt <- nrow(model$F)*(ncol(model$G)+2*nrow(model$H))
 nt
}


############################################################################
#
#       experimental estimation techniques    <<<<<<<<<<
#
############################################################################
est.black.box2  <- function(data, estimation="est.VARX.ls", 
          lag.weight=.9, 
          reduction="reduction.Mittnik", 
          criterion="taic", 
          trend=F, 
          subtract.means=F,  re.add.means=T, 
          standardize=F, verbose=T, max.lag=12)
{if ((estimation!="est.VARX.ls") && (trend) )
     {cat("Trend estimation only support with est.VARX.ls.\n")
      cat("Proceeding using est.VARX.ls.\n")
      estimation<-"est.VARX.ls"
     }

 if(estimation=="est.VARX.ls")
     model <- est.VARX.ls(data,trend=trend, subtract.means=subtract.means, 
                         re.add.means=re.add.means, max.lag=max.lag, 
                         standardize=standardize, lag.weight=lag.weight)
 # else if(estimation=="est.VARX.ar")
 #     model <-est.VARX.ar(data, subtract.means=subtract.means, max.lag=max.lag)
 else
    stop("Only est.VARX.ls estimation technique is supported to date.\n")
 if (verbose) cat("First VAR model,              lags= ", 
          dim(model$model$A)[1]-1, ", -log likelihood = ", model$estimates$like[1], "\n")
 model <- l(to.SS(model),model$data) # data is standardized if standardize=T in estimation
 n <- dim(model$model$F)[1]
 if (verbose) cat("Equivalent    state space model, n= ", 
                  n, ", -log likelihood = ", model$estimates$like[1], "\n")
 if (1 < n)
   {model <- eval(call(reduction,model,criterion=criterion, verbose=verbose))
    if (verbose) cat("Final reduced state space model, n= ",
              dim(model$model$F)[1], ", -log likelihood = ", model$estimates$like[1], "\n")
   }
  if (verbose && exists.graphics.device()) check.residuals(model)
 model
}

best.TSestModel     <- function (models, sample.start=10, sample.end=NULL, criterion="aic", verbose=T)
{# return the best model from ... according to criterion
  #  models should be a list of models as returned by like.
  #  modeels[[i]]$estimates$pred is not recalculated but a sub-sample identified by 
  #  sample.start and  sample.end is used and the likelihood is recalculated. 
  #  If sample.end=NULL data is used to the end of the sample.
  #  taic might be a better default selection criteria but it is not available for ARMA models.
  values <- NULL
  for (lst in models ) 
    {z <- information.tests.calculations(lst, sample.start=sample.start, 
                  sample.end=sample.end)
     values <-rbind(values,z)
    }
  if (verbose)
     {cat("Criterion value for all models based on data starting in period: ",  
           sample.start, "\n")
      cat(values[,criterion], "\n")
     }
#Rbug rbind above looses dimnames
  dimnames(values) <- dimnames(z)
  opt <-order(values[,criterion])[1]  # minimum
  invisible(models[[opt]])
}

est.black.box3  <- function(data, estimation="est.VARX.ls", 
       lag.weight=1.0, 
       reduction="reduction.Mittnik", 
       criterion="aic", 
       trend=F, subtract.means=F,  re.add.means=T, 
       standardize=F, verbose=T, max.lag=12, sample.start=10)
  #  taic might be a better default selection criteria but it is not available for ARMA models.
{if ((estimation!="est.VARX.ls") && (trend) )
     {cat("Trend estimation only support with est.VARX.ls.\n")
      cat("Proceeding using est.VARX.ls.\n")
      estimation<-"est.VARX.ls"
     }
 models <- vector("list", max.lag)
 for (i in 1:max.lag)
   {if(estimation=="est.VARX.ls")
      models[[i]] <- est.VARX.ls(data,trend=trend, 
                          subtract.means=subtract.means,
                          re.add.means=re.add.means, max.lag=i, 
                          standardize=standardize, lag.weight=lag.weight)
    else
      stop("Only est.VARX.ls estimation technique is supported to date.\n")
   }
 model <- best.TSestModel(models, criterion=criterion, sample.start=sample.start, verbose=verbose)
 if (verbose) cat("Selected VAR model,              lags= ", 
          dim(model$model$A)[1]-1, ", -log likelihood = ", model$estimates$like[1], "\n")
 model <- l(to.SS(model),model$data) # data is standardized if standardize=T in estimation
 n <- dim(model$model$F)[1]
 if (verbose) cat("Equivalent    state space model,    n= ", 
                  n, ", -log likelihood = ", model$estimates$like[1], "\n")
 if (1 < n)
   {model <- eval(call(reduction,model,criterion=criterion, verbose=verbose))# , sample.start=sample.start))
    if (verbose) cat("Final reduced state space model, n= ",
              dim(model$model$F)[1], ", -log likelihood = ", model$estimates$like[1], "\n")
   }
  if (verbose && exists.graphics.device()) check.residuals(model)
 model
}

bft <- function(data, ...) est.black.box4(data, ...)

est.black.box4  <- function(data, estimation="est.VARX.ls", 
                lag.weight=1.0,  variable.weights=1, 
                reduction="reduction.Mittnik", 
                criterion="taic", 
                trend=F, subtract.means=F,  re.add.means=T, 
                standardize=F, verbose=T, max.lag=12, sample.start=10, warn=T)
{if ((estimation!="est.VARX.ls") && (trend) )
     {cat("Trend estimation only support with est.VARX.ls.\n")
      cat("Proceeding using est.VARX.ls.\n")
      estimation<-"est.VARX.ls"
     }
 models <- vector("list", max.lag)
 for (i in 1:max.lag)
   {if(estimation=="est.VARX.ls")
      {model<- est.VARX.ls(data,trend=trend, 
                          subtract.means=subtract.means,
                          re.add.means=re.add.means, max.lag=i, 
                          standardize=standardize, lag.weight=lag.weight,
                          warn=warn)
      }
    else if(estimation=="est.wt.variables")
      {model<- est.wt.variables(data, variable.weights,
                        estimation.args=list(trend=trend, 
                          subtract.means=subtract.means,
                          re.add.means=re.add.means, max.lag=i, 
                          standardize=standardize, lag.weight=lag.weight,
                          warn=warn) )
      }
    else
      stop("Estimation technique not yet is supported.\n")
    if (verbose) cat("Estimated  VAR   model       -log likelihood = ", 
        model$estimates$like[1],", lags= ",  dim(model$model$A)[1]-1,"\n")
    model <- l(to.SS(model),model$data,warn=warn) # data is standardized if standardize=T in estimation
    n <- dim(model$model$F)[1]
    if (verbose) cat("Equivalent state space model -log likelihood = ", 
               model$estimates$like[1], ",   n = ", n, "\n")
    if (1 < n)
      {model <- eval(call(reduction,model,criterion=criterion, verbose=verbose))# , sample.start=sample.start))
       if (verbose) cat("Final reduced state space model, n= ",
           dim(model$model$F)[1], ", -log likelihood = ", model$estimates$like[1], "\n")
      }
     models[[i]] <- model
   }
 model <- best.TSestModel(models, criterion=criterion, sample.start=sample.start, verbose=verbose)
 if (verbose && exists.graphics.device()) check.residuals(model)
 model
}

#  z<-est.black.box4(eg.data,  max.lag=3 ) 



############################################################################
#
#       procedure for testing functions   <<<<<<<<<<
#
############################################################################


dse3.function.tests <- function(verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-8, graphics=T)
{ max.error <- NA
  if (synopsis & !verbose) cat("All dse3 tests ...")
  if (verbose) cat("dse3 test 0 ... ")
  data <- example.BOC.93.4.data.all
  input.data(data) <- NULL
  mod1 <- TSmodel(est.VARX.ls(data))
  mod2 <- TSmodel(est.VARX.ar(data, re.add.means=F, warn=F))
  ok <- is.TSmodel(mod1)
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 1 ... ")
  z <- monte.carlo.simulations(mod1, replications=5, quiet=T)
  ok <- is.monte.carlo.simulation(z)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 2 ... ")
  ok <- test.equal(z, monte.carlo.simulations(mod1, replications=5,
                                     seed=get.seed(z), quiet=T))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 3 ... ")
  z <- eval.estimation(mod1, replications=3,  estimation="est.VARX.ls",
            estimation.args=NULL, criterion="TSmodel", quiet=T)
  ok <- is.estimation.evaluation(z)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 4 ... ")
  zz <-summary(parms(z), verbose=F)
  ok <- T   # could be improved
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 5 ... ")
  zz <- summary(roots(z), verbose=F)
  ok <- T   # could be improved
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 6 ... ")
  zz <- parms(z)
  ok <- is.estimation.evaluation(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 7 ... ")
  zz <- roots(z)
  ok <- is.estimation.evaluation(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 8a... ")
  z <- horizon.forecasts(mod1, data, horizons=c(6,12), discard.before=20)
  error <- max(abs(
    c(z$horizon.forecasts[,100,])
    -c( 0.0048425425521641824594, 0.0031489473295282835973, 0.0037730234730729999594,
    0.0024354234760485438289, 0.0040593859721713481878, 0.0031982930612152113414)))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("dse3 test 8b... ")
  z <- horizon.forecasts(l(to.SS(mod1), data), horizons=c(6,12), discard.before=20)
  error <- max(abs(
    c(z$horizon.forecasts[,100,])
    -c(  0.0048425425521641824594, 0.0031489473295282844646, 0.0037730234730729995257,
0.0024354234760485446963, 0.0040593859721713499225, 0.0031982930612152122088)))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }


  if (verbose) cat("dse3 test 9 ... ")
  zzz<-l(mod1,simulate(mod1))
  zz<-forecast.cov(zzz, discard.before=50, horizons=1:4)
  ok <- is.forecast.cov(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 10... ")
  ok <- test.equal(zz, forecast.cov(zzz$model, 
             data=zzz$data, discard.before=50, horizons=1:4))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 11... ")
  zz <-forecast.cov(mod1,mod2, data=data, discard.before=30, zero=T, trend=T)

  ok <- is.forecast.cov(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 12... ")
  zzz <-forecast.cov(to.SS(mod1),to.SS(mod2), data=data, 
                 discard.before=30, zero=T, trend=T)

  ok <- test.equal(zz,zzz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 13... ")
  zz <-out.of.sample.forecast.cov.estimators.wrt.data(data,
               estimation.methods = list(est.VARX.ar= list(max.lag=2, warn=F), 
                                         est.VARX.ls= list(max.lag=2)))
  ok <- is.forecast.cov(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 14... ")
  zz <- forecast.cov.wrt.true(list(mod1,mod2),mod1, 
               pred.replications=2, Spawn=F, quiet=T, trend=NULL, zero=T)
  ok <- is.forecast.cov(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 15... ")
  ok <- test.equal(zz, forecast.cov.wrt.true(list(mod1,mod2),mod1, 
          pred.replications=2, Spawn=.SPAWN, quiet=T, trend=NULL, zero=T,
          seed=get.seed(zz)))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 16... ")
  zz <- forecast.cov.estimators.wrt.true(mod1, Spawn=.SPAWN, quiet=T, 
         estimation.methods=list(est.VARX.ls=NULL, est.VARX.ar=list(warn=F)), 
         est.replications=2, pred.replications=2)
  ok <- is.forecast.cov(zz)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse3 test 17... ")
  ok <- test.equal(zz, forecast.cov.estimators.wrt.true(mod1, Spawn=F, 
           estimation.methods=list(est.VARX.ls=NULL,est.VARX.ar=list(warn=F)), 
           est.replications=2, pred.replications=2, seed=get.seed(zz)))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }
  if (graphics) {
  if (verbose) cat("dse3 test 18 (graphics) ... ")
  ok <- dse3.graphics.tests(verbose=verbose,  pause=F)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }
  }

  if (synopsis) 
    {if (verbose) cat("All dse3 tests completed")
     if (all.ok) cat(" ok\n")
     else    
       {cat(", some failed!")
        if(max.error > fuzz.small)
            cat(" max. error magnitude= ", max.error,")")
        cat("\n")
       }
    }

  invisible(all.ok)
}




dse3.graphics.tests <- function(verbose=T, synopsis=T,  pause=F)
{ if (synopsis & !verbose) cat("dse3 graphics tests ...")
  if (verbose) cat("  dse3 graphics test 1 ... ")
  # If no device is active then write to postscript file 
  if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",
                   width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
            {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
  if(pause) dev.ask(ask=T)

  data <- example.BOC.93.4.data.all
  input.data(data) <- NULL
  output.data(data) <- output.data(data, series=1)  # [,1,drop=F]
  mod1 <- TSmodel(est.VARX.ls(data,max.lag=3))
  mod2 <- TSmodel(est.VARX.ar(data,max.lag=3, aic=F, warn=F))

  z <- eval.estimation(mod1, replications=10,  estimation="est.VARX.ls",
            estimation.args=list(max.lag=3), criterion="TSmodel", quiet=T)
  distribution(parms(z)) 
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 2 ...")
  distribution(roots(z))
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 3 ...")
  z <- horizon.forecasts(mod1, data, horizons=c(6,12), discard.before=20)
  plot(z, Start=c(1985,1))
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 4 ...")

  zz <-forecast.cov(mod1,mod2, data=data,
                     discard.before=10, zero=T, trend=T)
  plot(zz)
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 5 ...")
  plot(zz, select.cov=c(1), select.trend=F)
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 6 ...")
  zz <-out.of.sample.forecast.cov.estimators.wrt.data(data,
               estimation.methods = list(est.VARX.ar=list(max.lag=2,warn=F),
                                         est.VARX.ls=list(max.lag=2))) 
  plot(zz, select.series=c(1))
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 7 ...")
  zz <- forecast.cov.wrt.true(list(mod1,mod2),mod1, 
               pred.replications=2, Spawn=.SPAWN, trend=NULL, zero=T, quiet=T)
  plot(zz, select.cov=c(1))
  if (verbose) cat("ok\n")

  if (verbose) cat("  dse3 graphics test 8 ...")
#Rbug ?? or in S too:
# est.VARX.ar can produce: Error ... all lags eliminated by AIC order selection.
#   depending on seed?
  zz <- forecast.cov.estimators.wrt.true(mod1, Spawn=.SPAWN, 
         estimation.methods=list(est.VARX.ls=NULL,est.VARX.ls=list(max.lag=2)), 
#        estimation.methods=list(est.VARX.ls=NULL,est.VARX.ar=NULL), 
         est.replications=2, pred.replications=2, quiet=T)
  plot(zz, select.cov=c(1))
  if (verbose) cat("ok\n")

  if (synopsis) 
    {if (verbose) cat("All dse3 graphics tests completed\n")
     else cat("completed\n")
    }
      
  invisible(T)
}


############################################################################
#
#       end
#
############################################################################
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



############################################################################

# Functions in this file are mainly for evaluating the information <<<<<<<<<<<<<
#    content of data series for predicting other series.           <<<<<<<<<<<<<


############################################################################
#
#  methods for "forecast.cov.estimators.wrt.data.subsets", "forecast.cov"
#
############################################################################


is.forecast.cov.estimators.wrt.data.subsets<-function(obj)
{  r <- "forecast.cov.estimators.wrt.data.subsets"==class(obj)[1]
   if(is.na(r)) r <- F
   r
}


print.forecast.cov.estimators.wrt.data.subsets <- function(obj, digits=4)
{cat("forecast.cov.estimators.wrt.data.subsets\n")
 cat("essential data:", obj$essential.data, "\n")
 cat("considering:", output.series.names(obj$all.data), 
                      input.series.names(obj$all.data), "\n")
 invisible(obj)
}

summary.forecast.cov.estimators.wrt.data.subsets <- function(obj, digits = 4)
{  cat("class: ", class(obj), "   ")
   cat(length(obj$horizons), " horizons\n")
   cat("essential data:", obj$essential.data, "\n")
   cat("considering:", output.series.names(obj$all.data), 
                        input.series.names(obj$all.data), "\n")
   NextMethod("summary")
}


############################################################################
#
#  methods for generating test data
#
############################################################################

gen.mine.data <- function(umodel, ymodel, uinput=NULL, sampleT=100, unoise=NULL, usd=1,ynoise=NULL, ysd=1, seed=NULL)
{# This function returns a TSdata object.
# umodel is used to generate data which is returned as $input and
# ymodel is used to generate data which is returned as $output.
# The result of umodel is used as input to ymodel so the input dimension of
# ymodel should be the output dimension of umodel. Typically the ymodel would
# be degenerate in some of the input variables so the effective inputs 
#  are a subset.
# If noise is NULL then an normal noise will be generated by simulate.
# This will be iid N(0,I).
# The seed will be set first if it is specified. 
# If unoise or ynoise are specified they should be as expected by simulate
#  for the specified umodel and ymodel.

#   eg:
#  umodel <- build.diagonal.model(build.input.models(all.data, max.lag=2))
#  umodel$A <- round(umodel$A, digits=2)
#  z  <- list(output=output.data(all.data), 
#             input = input.data(extract(all.data, n.input=1:2))
# class(z) <- "TSdata"
# ymodel <- est.VARX.ls(z, max.lag=3)$model   
# sim.data <- gen.mine.data(umodel, ymodel)

 if(!is.TSm.or.em(umodel)) TS.error.exit()
 if (is.TSestModel(umodel)) umodel <- umodel$model
 if(!is.TSm.or.em(ymodel)) TS.error.exit()
 if (is.TSestModel(ymodel)) ymodel <- ymodel$model
 if (input.dimension(ymodel) != output.dimension(umodel))
   stop("umodel output dimension must equal ymodel input dimension.")
 
 seed <- set.seed(seed)

 input <- simulate(umodel, input=uinput, sampleT=sampleT, 
                   noise=unoise, sd=usd, seed=seed)$output 

 output <- simulate(ymodel, input=input, sampleT=sampleT,
                   noise=ynoise, sd=ysd, seed=.Random.seed)$output
 r <- list(input=input, output=output,
           umodel=umodel, ymodel=ymodel, uinput=uinput, sampleT=sampleT, 
           unoise=unoise, usd=usd,ynoise=ynoise, ysd=ysd, seed=seed)
 class(r) <- "TSdata"
 r
}

build.input.models <- function(all.data, max.lag=NULL)
{# make a list of univariate models, one for each series in all.input.data(data)
 #   for use by build.diagonal.model. 
 n <- input.dimension(all.data)
 multi.models <- vector("list", n)
 for (i in seq(n))
   {data <-list(output= trim.na(extract(all.data, n=1, n.input=i))$input) 
    class(data) <- "TSdata"
    multi.models[[i]] <- est.VARX.ls(data, max.lag=max.lag)$model
   }
 multi.models
}

build.diagonal.model <- function(multi.models)
{# build one diagonal model from a list of models as returned  by 
 # build.input.models. Uses the AR part only. This can be used by gen.mine.data.
 n <- length(multi.models)
 lag <- 0
 for (i in seq(n)) lag <- max(lag, dim(multi.models[[i]]$A)[1])
 p <- 0
 for (i in seq(n))  p  <- p + dim(multi.models[[i]]$A)[3]
 model <- array(0, c(lag, p,p))
 p <- 0
 for (i in seq(n))
   {pi <- dim(multi.models[[i]]$A)[3]
    li <- dim(multi.models[[i]]$A)[1]
    model[ 1:li, (p+1):(p+pi), (p+1):(p+pi)] <-  multi.models[[i]]$A
    p <- p + pi
   }
 model <- list(A= model, B=array(diag(1, p), c(1,p,p)))
 class(model) <- c("ARMA",  "TSmodel")
 model
}



############################################################################
#
#  methods for stepwise mining
#
############################################################################


plot.mine.stepwise <- function(obj)
  {cases <- length(obj$stepwise$rss)
   o <- rev(order(obj$stepwise$rss))
   vo <- dim(obj$s.output.indicator)[2]
   plto <- t(matrix(1:vo, vo, cases)) * obj$s.output.indicator[o,]
   if (!is.null(obj$s.input.indicator))
     {vi <- dim(obj$s.input.indicator)[2]
      plti <- t(matrix(-1:(-vi), vi, cases)) * obj$s.input.indicator[o,]
      plt <- cbind(plti,plto)
     }
   plt[plt==0] <- NA
   matplot(0:(cases-1), plt, type="p", pch="+")
   y <- NULL
   io <-   obj$io.indicator   & (1==obj$lag.indicator)
   if (any(io)) y <- c(y,paste("output",  obj$v.indicator[io]))
   io <-  (!obj$io.indicator) & (0==obj$lag.indicator)
   if (any(io)) y <- c(y,paste(" input ",  obj$v.indicator[io]))
   cat("y axis above zero (outputs) and below zero (inputs) indicate", y, "\n")
   invisible()
  }

mine.stepwise <- function(data, essential.data=1,
      method="efroymson", f.crit=2, intercept=T,
      subtract.means=F,  standardize=F, 
      lags.in=6, lags.out=6, trend=F, plot.rss=T) 
{ # Data should be of class TSdata. essential.data must have length 1.
# standardize and subtract means may not make any sense ?????????
# The result is a list with the results of stepwise,..., and
# several vectors indicating information about the columns of the matrix
# passed to stepwise: 
#   io.indicator indicating an input (F) or output (T) variable
#   v.indicator  indicating which series
#   lag.indicator indicating the lag
# s.input.indicator and s.output.indicator are logic matrices
#    length(stepwise$rss) by m and p respectively indicating if
#   a series is included for each element of rss.
    data <- TSdata(data)
   m <- ncol(input.data(data))
   p <- ncol(output.data(data))
   if(is.null(m))  m <- 0
   N <- nrow(output.data(data))
   if (standardize)
     {svd.cov <- svd(var(output.data(data)))
      scalefac <- svd.cov$u %*% diag(1/svd.cov$d^.5, ncol=p)
      data <- scale(data, list(output=scalefac))
     }
   if (subtract.means)
    {if(m!=0)
       {input.means<-apply(input.data(data),2, mean)
        input.data(data)<-input.data(data)-t(matrix( input.means, m,N))
       }
     output.means <- apply(output.data(data),2, mean)
     output.data(data)  <- output.data(data) - t(matrix(output.means, p,N))
    }
 # The matrix Past is blocks of data:
 #  [ out-1 | out-2 | ... | out-max.lag | in | in-1 | ... | in-max.lag ]
 # so the result M has a corresponding structure.
 # If there is an input variable (m!=0) then it is shifted to give feedthrough 
 #    in one period. If lags.in=lags.out this has the result that a data point
 #    is lost at the beginning of the input series.
   if(m==0)
     {Past <- matrix(NA,N-lags.out, p*lags.out)
      io.indicator <- c(rep(T, p*lags.out))
      v.indicator  <- c(rep(1:p, lags.out)) 
      lag.indicator  <- c(t(matrix(1:lags.out, lags.out,p))) 
      for (i in 0:(lags.out-1)) 
         Past[,(1+p*i):(p*(1+i))] <-output.data(data)[(lags.out-i):(N-i-1),]
      Present <- output.data(data)[(lags.out+1):N, essential.data]
     }
   else 
     {shift <- max(lags.in+1, lags.out) # start pt. for Present
      Past <- matrix(NA,N-shift+1, p*lags.out+m*(1+lags.in))
      io.indicator <- c(rep(T, p*lags.out), rep(F, m*(1+lags.in)))
      v.indicator  <- c(rep(1:p, lags.out), rep(1:m, (1+lags.in))) 
      lag.indicator<- c(t(matrix(1:lags.out, lags.out,  p)), 
                        t(matrix(0:lags.in, (1+lags.in),m))) 
      for (i in 0:(lags.out-1)) 
        Past[,(1+p*i):(p*(1+i))] <-output.data(data)[(shift-1-i):(N-1-i),]
      for (i in 0:lags.in) 
        Past[,(p*lags.out+1+m*i):(p*lags.out+m*(1+i))] <-
                                   input.data(data) [(shift-i):(N-i),]
      Present <- output.data(data)[shift:N, essential.data]
     }
   dimnames(Past) <- list(NULL, c(
         paste(c(paste("out.v", matrix(1:p, p, lags.out), "L",sep="")),
               c(t(matrix(1:lags.out,  lags.out, p))), sep=""),
         paste(c(paste("in.v", matrix(1:m, m, lags.in), "L",sep="")),
               c(t(matrix(0:lags.in, 1+lags.in,  m))), sep="")) )
   plot.rss <- plot.rss & exists.graphics.device()
   if (plot.rss) par(mfcol=c(2,1))
   M <- stepwise(Past,Present, method=method,f.crit=f.crit, intercept=intercept,
                 plot=plot.rss)
   # Now construct an inidicator (s.indicator) of the series which are used in
   # each element of rss returned by tepwise.
   # The trick is to collapse obj$stepwise$which using obj$v.indicator so that
   #   any lags of a variable get lumped together.
   p <- v.indicator * io.indicator
   # part of the following (not the outer part) is an inner 
   # prod. with | in place of + and  & in place of *
   s.output.indicator <-  0 != (M$which %*% outer(p, 1:max(p),"==") )
   m <- v.indicator * !io.indicator
   if (max(m) !=0)
      s.input.indicator <- 0 != (M$which %*% outer(m, 1:max(m), "==") )

   M <- list(stepwise=M, io.indicator=io.indicator, v.indicator=v.indicator,
             lag.indicator=lag.indicator, Past=Past,
             lags.in=lags.in, lags.out=lags.out,
             s.input.indicator=s.input.indicator, 
             s.output.indicator=s.output.indicator)
   class(M) <- "mine.stepwise"
   if (plot.rss) plot(M)
   invisible(M)
}


############################################################################
#
#  methods for mining by splitting sample for estimation and forecast error
#
############################################################################


permute <- function(M)
  {# return matrix with rows indicating all possible selections of
   #    elements from seq(M). 0 in the result indicates omit. 
   # M is usually a positive integer. M=0 gives NULL.
   # Neg. M give -permute(abs(M)).
   if (is.null(M)) return(NULL)
   if (M==1) return(matrix(0:1, 2,1))
   if (M==-1) return(-matrix(0:1, 2,1))
   if (M==0) return(NULL)
   r <- permute(abs(M)-1) 
   sign(M)*rbind(cbind(r,abs(M)), cbind(r,0))
  }

mine.strip <-function(all.data, essential.data=1, 
                       estimation.sample=.5, 
                       discard.before=1, horizons=1:12,quiet=F,
                       estimation.methods=NULL,
                       step.size=NULL)
{# Calculate the predictions cov for essential.data of models estimated 
 # with estimation methods indicated by estimation.methods. 
 # estimation.methods is a list with syntax similar to programs
 #  for comparing estimation methods (eg. estimate.models), BUT ONLY 
 #  THE FIRST element (estimation method) is considered.
 # Essential.data indicates the subset of output variables to included in all
 #  models. It should be a vector of the indices. All possible combinations of
 #  input series and other output series data are considered. If omitted,
 #  essential.data is taken as the 
 #  first output series.
 # Only forecast covariances for essential data are returned.
 # discard.before is an integer indicating 1+the number of points in the
 # beginning of predictions to discard for calculating prediction covariances.
 # estimation.sample indicates the portion of the data to use for estimation.
  #If estimation.sample is an integer then it is used to indicate the number
  # of points in the sample to use for estimation. If it is a fracton it is
  # used to indicate the portion of points to use for estimation. The remainder
  # of the sample is used for evaluating predictions (ie. forcast covariance).

 # If step.size is NULL then all possible data permutations are attempted.
 #  Because S has a hard-coded limit in the number of synchronize calls this is
 #  not always possible (For loops call synchronize.) An error message:
 #    Error in synchronize(1): No room in database table
 #  If step.size is not NULL it should be a positive integer. In this case 
 #  variable permutions are divided up into
 #  steps of the given size. The result returned by the function can be used
 #  to continue from the last step:
 #      intermediate.result <- mine.strip(data, ...)
 #      intermediate.result <- mine.strip(intermediate.result)
 #      intermediate.result <- mine.strip(intermediate.result)
 #      result <- mine.strip(intermediate.result)
 #  This can be done either interactively or in a batch process, but cannot be
 #  done in a function because the database table is not cleared until the top
 #  level expression is complete.
 #  The class of an intermediate result is mine.strip.intermediate.result and
 #  the class of the final result is
 #         c("forecast.cov.estimators.wrt.data.subsets", "forecast.cov")
 #  If the final result is used in a call to mine.strip then it is just 
 #  returned, so extra calls do not cause errors and are very quick.
 #  This is useful when you are too lazy to calculate the exact number of steps.

  if (class(all.data)[1] == "forecast.cov.estimators.wrt.data.subsets")
       {cat("done.\n")
        return(all.data)
       }
  if (class(all.data)[1] == "mine.strip.intermediate.result")
    {r <- all.data$forecast.cov
     start <- 1+all.data$end
     estimation.sample <- all.data$estimation.sample
     discard.before <- all.data$discard.before
     quiet <- all.data$quiet
     step.size <- all.data$step.size
     variable.index <- all.data$variable.index
     m <- all.data$m
     p <- all.data$p
     multi.model <- all.data$multi.model
     essential.data <- all.data$essential.data
     estimation.methods <- all.data$estimation.methods
     horizons <- all.data$horizons
     all.data <- all.data$all.data
    }
  else
    {start <- 1
     if (estimation.sample < 1.0 )  estimation.sample <- 
           as.integer(round(estimation.sample*periods(all.data)))
     discard.before <- discard.before+estimation.sample

    #first  gen. combinations of non-essential data
     p <- output.dimension(all.data)
     m <-  input.dimension(all.data)
     M <- permute(m + p - length(essential.data) )
     # now combine essential data and permutations of non-essential data.
     if (is.null(M))
       {variable.index<-matrix(essential.data,length(essential.data),1)
        warning("essential.data seems to include all series, which does not make sense in call to mine.strip.")
       } 
     else
       variable.index<-cbind(
             t(matrix(essential.data,length(essential.data), nrow(M))), 
             array(c(0,seq(p)[-essential.data],seq(m))[1+M], dim(M)))
     r <- NULL
     if   (is.null(step.size)) step.size <- nrow(variable.index)
     else if (0 == step.size)  step.size <- nrow(variable.index)
     multi.model <- NULL
    }
 end <- min(nrow(variable.index), start+step.size-1)
 for (i in start:end )
   {if (m==0) data<-extract(all.data, variable.index[i,])
    else  
      {data<-extract(all.data, variable.index[i,1:p], 
             n.input=variable.index[i,(p+1):(p+m)])
      }
    if(0==length(input.data(data))) input.data(data) <- NULL
    if(0==length(output.data(data))) 
      stop("The variable selection has mistakenly eliminated all output variables.")
    models <-estimate.models(data, estimation.sample=estimation.sample,
                       trend=T,quiet=quiet,
                       estimation.methods=estimation.methods)
    multi.model <- append(multi.model, list(models$multi.model[[1]]))
    rn <- forecast.cov( models$multi.model[[1]], data=data, 
                    horizons=horizons, discard.before=discard.before)
    r<- append(r, list(
           rn["forecast.cov"][[1]][[1]][,essential.data,essential.data,drop=F]))
   }
  r<-list(forecast.cov=r,all.data=all.data, essential.data=essential.data,
         variable.index=variable.index,
         estimation.methods=estimation.methods,
         multi.model=multi.model,
         horizons=horizons, 
         discard.before=discard.before)
  if (end == nrow(variable.index))
     class(r) <- c("forecast.cov.estimators.wrt.data.subsets", "forecast.cov")
  else
    {r<-append(r, list(estimation.sample=estimation.sample,
             quiet=quiet, step.size=step.size, end=end, m=m,p=p))
     class(r) <- "mine.strip.intermediate.result"
    }
  r
}

# z <-mine.strip(example.BOC.93.4.data.all, essential.data=1, 
#      estimation.methods= list(est.VARX.ar=list(max.lag=3))) 

min.forecast.cov <- function(obj, select.series=1, verbose=T)
  {#obj is an object as returned by mine.strip
   #select the min cov for select.series only!!! at each horizon and print
   # the returned result is a vector indicating the element of forecast.cov which
   # was the min at each horizon. It is suitable as an argument to plot eg:
   #     plot(obj, select.cov=min.forecast.cov(obj))
   # The results of this are similar to the default results of 
   #   select.forecast.cov() cov info and information about the horizon
   #   where the model is optimal are given.

   N <- length(obj$forecast.cov)
   z <- matrix(0,length(obj$horizons),N)
   for (j in 1:N) z[,j]<-obj$forecast.cov[[j]][,select.series,select.series]
   m <- apply(z,1, min)
   r <- rep(NA,length(obj$horizons))
   for (j in 1:length(obj$horizons))
      r[j] <- (seq(N)[ z[j,]== m[j] ])[1] # only the first if more than 1 min
   if (verbose)
     {cat("              model     cov          using data\n")
      for (j in 1:length(obj$horizons))
         cat("horizon ", j,"   ",  r[j],"   ", m[j],  "   ", 
             obj$variable.index[r[j],],"\n")
     }
   invisible(r)
  }

select.forecast.cov <- function(obj, select.series=1, 
    select.cov.best=1,
    select.cov.bound=NULL,
    ranked.on.cov.bound=NULL,
    verbose=T)
  {#obj is an object as returned by mine.strip
   #select models with forecast cov for select.series meeting criteria.
   # The default select.cov.best=1 selects the best model at each horizon.
   #  select.cov.best=3 would select the best 3 models at each horizon.
   #     plot(select.forecast.cov(obj, select.cov.best=3))
   # If select.cov.bound is not NULL then  select.cov.best is ignored and
   #  any model which is better than the bound at all horizons is selected.
   #  select.cov.bound can be a vector of the same length as select.series,
   #  in which case corresponding elements are applied to the different series.
   #  any model which is better than the bound at all horizons is selected.
   # ranked.on.cov.bound is is used if it is not NULL and select.cov.bound is
   #  NULL. In this case select.cov.best is ignored.
   #  ranked.on.cov.bound should be a positive integer. The forecast
   #  covariances are ranked by there maximum over the horizon and the
   #  lowest number up to ranked.on.cov.bound are selected. This amounts
   #  to adjusting the covariance bound to allow for the given number of
   #  models to be selected. If select.series is a vector the results are 
   #  the best up to the given number on any series!
   # select.cov.bound can be a vector of the same length as select.series,
   #  in which case corresponding elements are applied to the different series.
   # If verbose=T then summary results are printed.
   # The returned result is a forecast.cov object like obj, but filtered
   #  to remove models which do not meet criteria.
   #     plot(select.forecast.cov(obj, select.cov.bound=20000))

   N <- length(obj$forecast.cov)
   r <- NULL
   if (!is.null(select.cov.bound))
     if (1 == length(select.cov.bound)) 
       select.cov.bound <- rep(select.cov.bound, length(select.series))
   for (i in 1:length(select.series)) 
     {z <- matrix(NA,length(obj$horizons),N)
      for (j in 1:N) 
         z[,j]<-obj$forecast.cov[[j]][,select.series[i],select.series[i]]
      if (!is.null(select.cov.bound))
         r <- c(r, seq(N)[apply((z <= select.cov.bound[i]),2, all)])
      else if (!is.null(ranked.on.cov.bound))
         r <- c(r, order(apply(z,2,max))[1: ranked.on.cov.bound])
      else
        {#r <- c(r, apply(z,1,sort.list)[ select.cov.best,])
         r <- c(r, apply(z,1,order)[ select.cov.best,])
        }
     }
   if (0==length(r)) stop("No forcasts meet the specified criterion.")
   r <- r[!apply(outer(r,r,"==") & 
          outer(seq(length(r)),seq(length(r)),"<"),  2,any)] #eliminate repeats
   r <- sort(r) 
   pred  <- vector("list",length(r))
   model <- vector("list",length(r))

   for (j in 1:length(r))
       {pred[[j]]           <- obj$forecast.cov[[r[j] ]]
#       model[[j]]          <- obj$multi.model[[r[j] ]]
       }
   obj$forecast.cov <- pred
#  obj$multi.model <- model
   obj$variable.index <- obj$variable.index[r,, drop=F]
   obj$selection.index <- r
   if (verbose)
     {cat("    model  using subset data series (output | input)\n")
      for (j in 1:length(obj$forecast.cov))
         cat( j,"   ", r[j],  "   ", 
             obj$variable.index[j,],"\n")
     }
   invisible(obj)
  }

exclude.forecast.cov <- function(obj, exclude.series=NULL)
  {# exlude results which depend on the indicated series from a 
   #  (forecast.cov.estimators.wrt.data.subsets forecast.cov) object.
   if (!is.null(exclude.series))
     {include<- !apply(0 != obj$variable.index[,exclude.series, drop=F], 1,any)
      obj$forecast.cov   <- obj$forecast.cov[include]
      obj$variable.index <- obj$variable.index[include,]
      obj$multi.model    <- obj$multi.model[include]
      # note all.data is not changed and variable.index still refers to it.
     }
   invisible(obj)
  }

############################################################################
#
#       procedure for testing functions     <<<<<<<<<<<<<
#
############################################################################


dse4.function.tests <- function(verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-7, graphics=T)
{max.error <- 0
 if (synopsis & !verbose) cat("All dse4 tests ...") 
 if (verbose) cat("dse4 test 1 ... ")
  z <- mine.strip(example.BOC.93.4.data.all, essential.data=c(1,2),
                   estimation.methods=list(est.VARX.ls=list(max.lag=3)))
  ok <- is.forecast.cov.estimators.wrt.data.subsets(z)
  all.ok <-  ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse4 test 2 ... ")
  z1 <- z$multi.model[[
       select.forecast.cov(z, select.cov.best=1, verbose=F)$selection.index[2]]]
  subdata <- extract(example.BOC.93.4.data.all,n=1:3,n.input=0)
  z2 <- estimate.models(subdata, estimation.sample =182, quiet = T, 
           estimation.methods = list(est.VARX.ls=list(max.lag=3)))
  output.data(subdata) <- output.data(subdata)[1:182,,drop=F]
#  input.data(subdata)  <- input.data(subdata) [1:182,,drop=F] not in subdata
  z3 <- est.VARX.ls(subdata, max.lag=3)
  ok <-      test.equal(z2$multi.model[[1]],z3$model)
  ok <- ok & test.equal(z2$multi.model[[1]],  z1)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("dse4 test 3 ... ")
#Rbug needs stepwise
if (is.R()) warning("skipping test 4 (requires stepwise).")
else
 {
   all.data <- list(input=example.BOC.93.4.data.all$output, 
                   output=example.BOC.93.4.data.all$input)
   class(all.data) <- "TSdata"
   umodel <- build.input.models(all.data, max.lag=2)
   umodel <- build.diagonal.model(umodel)
   z  <- list(output=output.data(all.data), 
              input = input.data(extract(all.data, 1, n.input=1:2)))
  class(z) <- "TSdata"
  ymodel <- est.VARX.ls(z, max.lag=3)$model 
  z <- ymodel$C
  ymodel$C <- array(0, c(dim(z)[1:2], output.dimension(umodel))) 
  ymodel$C[1:(dim(z)[1]), 1:(dim(z)[2]), 1:(dim(z)[3])] <- z 
  sim.data <- gen.mine.data(umodel, ymodel,
                 seed = c(21, 46, 16, 12, 51, 2, 31, 8, 42, 60, 7, 3))
  m.step <- mine.stepwise(sim.data, method="backward")
  error <- max(abs(m.step$stepwise$rss[c(1,27)] -
               c(47.537312899054931847, 4088283.2706551752053)))
  ok <- fuzz.large > error
  if (!ok) max.error <- max(error, max.error)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
 }

  if (graphics) {
  if (verbose) cat("dse4 test 4 (graphics) ... ")
  ok <- dse4.graphics.tests(verbose=verbose, pause=F)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }
  }

  if (synopsis) 
    {if (verbose) cat("All dse4 tests completed")
     if (all.ok) cat(" ok\n")
     else    
       {cat(", some failed!")
        if(max.error > fuzz.small)
            cat(" max. error magnitude= ", max.error,")")
        cat("\n")
       }
    }
  invisible(all.ok)
}

dse4.graphics.tests <- function(verbose=T, synopsis=T,  pause=F)
{ if (synopsis & !verbose) cat("dse4 graphics tests ...")
  if (verbose) cat("  dse4 graphics test 1 ...")

  # If no device is active then write to postscript file 
  if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
             {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
  if(pause) dev.ask(ask=T)

  z <- mine.strip(example.BOC.93.4.data.all, essential.data=c(1,2),
                   estimation.methods=list(est.VARX.ls=list(max.lag=3)))
  z <- plot(z)

  if (verbose) cat("ok\n")

  if (synopsis) 
    {if (verbose) cat("All dse4 graphics tests completed\n")
     else cat("completed\n")
    }
      
  invisible(T)
}


############################################################################
#
#       end
#
############################################################################
