addterm <- function(object, ...) UseMethod("addterm")

addterm.default <-
  function(object, scope, scale = 0, test=c("none", "Chisq"),
           k = 2, trace = FALSE, ...)
{
  if(missing(scope) || is.null(scope)) stop("no terms in scope")
  if(!is.character(scope))
    scope <- addScope(object, update.formula(object, scope))
  if(!length(scope))
    stop("no terms in scope for adding to object")
  ns <- length(scope)
  ans <- matrix(nrow = ns + 1, ncol = 2)
  dimnames(ans) <- list(c("<none>", scope), c("df", "AIC"))
  ans[1,  ] <- extractAIC(object, scale, k = k, ...)
  for(i in seq(ns)) {
    tt <- scope[i]
    if(trace > 1) cat("trying +", tt, "\n")
    nfit <- update(object, as.formula(paste("~ . +", tt)))
    ans[i+1,  ] <- extractAIC(nfit, scale, k = k, ...)
  }
  dfs <- ans[,1] - ans[1,1]
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, AIC = ans[,2])
  test <- match.arg(test)
  if(test == "Chisq") {
    dev <- ans[,2] - k*ans[, 1]
    dev <- dev[1] - dev; dev[1] <- NA
    nas <- !is.na(dev)
    P <- dev
    P[nas] <- 1 - pchisq(dev[nas], dfs[nas])
    aod[, c("LRT", "Pr(Chi)")] <- list(dev, P)
  }
  head <- c("Single term additions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

addterm.lm <-
  function(object, scope, scale = 0, test=c("none", "Chisq", "F"),
                    x = NULL, k = 2,...)
{
  Fstat <- function(table, RSS, rdf) {
    dev <- table$"Sum of Sq"
    df <- table$Df
    rms <- (RSS - dev)/(rdf - df)
    Fs <- (dev/df)/rms
    Fs[df < .Machine$double.eps] <- NA
    P <- Fs
    nnas <- !is.na(Fs)
    P[nnas] <- 1 - pf(Fs[nnas], df[nnas], rdf - df[nnas])
    list(Fs, P)    
  }

  if(missing(scope) || is.null(scope)) stop("no terms in scope")
  if(!is.character(scope))
    scope <- addScope(object, update.formula(object, scope))
  if(!length(scope))
    stop("no terms in scope for adding to object")
  oTerms <- attr(object$terms, "term.labels")
  int <- attr(object$terms, "intercept")
  ns <- length(scope)
  y <- object$residuals + predict(object)
  dfs <- numeric(ns+1)
  RSS <- numeric(ns+1)
  names(dfs) <- names(RSS) <- c("<none>", scope)
  dfs[1] <- object$rank
  RSS[1] <- deviance.lm(object)
  add.rhs <- paste(scope, collapse = "+")
  add.rhs <- eval(parse(text = paste("~ . +", add.rhs)))
  new.form <- update.formula(object, add.rhs)
  Terms <- terms(new.form)
  if(is.null(x)) {
    fc <- object$call
    fc$formula <- Terms
    fob <- list(call = fc)
    class(fob) <- class(object)
    m <- model.frame(fob, xlev = object$xlevels)
    x <- model.matrix(Terms, m, contrasts = object$contrasts)
  }
  n <- nrow(x)
  Terms <- attr(Terms, "term.labels")
  asgn <- attr(x, "assign")
  ousex <- match(asgn, match(oTerms, Terms), 0) > 0
  if(int) ousex[1] <- TRUE
  iswt <- !is.null(wt <- object$weights)
  for(tt in scope) {
    usex <- match(asgn, match(tt, Terms), 0) > 0
    X <- x[, usex|ousex, drop = FALSE]
    z <- if(iswt) lm.wfit(X, y, wt) else lm.fit(X, y)
    dfs[tt] <- z$rank
    RSS[tt] <- deviance.lm(z)
  }
  if(scale > 0) aic <- RSS/scale - n + k*dfs
  else aic <- n * log(RSS/n) + k*dfs
  dfs <- dfs - dfs[1]
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, "Sum of Sq" = c(NA, RSS[1] - RSS[-1]), 
                    RSS = RSS, AIC = aic, row.names = names(dfs),
                    check.names = FALSE)
  if(scale > 0) names(aod) <- c("Df", "Sum of Sq", "RSS", "Cp")
  test <- match.arg(test)
  if(test == "Chisq") {
    dev <- aod$"Sum of Sq"
    nas <- !is.na(dev)
    dev[nas] <- 1 - pchisq(dev[nas]/scale, aod$Df[nas])
    aod[, "Pr(Chi)"] <- dev
  } else if(test == "F") {
    rdf <- object$df.resid
    aod[, c("F Value", "Pr(F)")] <- Fstat(aod, aod$RSS[1], rdf)
  }
  head <- c("Single term additions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

add1.glm <- 
  function(object, scope, scale = 0, test=c("none", "Chisq"),
           x = NULL, k = 2,  ...)
{
  if(!is.character(scope))
    scope <- addScope(object, update.formula(object, scope))
  if(!length(scope))
    stop("no terms in scope for adding to object")
  oTerms <- attr(object$terms, "term.labels")
  int <- attr(object$terms, "intercept")
  ns <- length(scope)
#  rdf <- object$df.resid
  dfs <- numeric(ns+1)
  dev <- numeric(ns+1)
  names(dfs) <- names(dev) <- c("<none>", scope)
  dfs[1] <- object$rank
  dev[1] <- object$deviance
  add.rhs <- paste(scope, collapse = "+")
  add.rhs <- eval(parse(text = paste("~ . +", add.rhs)))
  new.form <- update.formula(object, add.rhs)
  Terms <- terms(new.form)
  if(is.null(x)) {
    fc <- object$call
    fc$formula <- Terms
    fob <- list(call = fc)
    class(fob) <- class(object)
    m <- model.frame(fob, xlev = object$xlevels)
    x <- model.matrix(Terms, m, contrasts = object$contrasts)
  }
  n <- nrow(x)
  y <- object$y
  if(is.null(y)) y <- model.response(model.frame(object), "numeric")
  wt <- model.weights(model.frame(object))
  if(is.null(wt)) wt <- rep(1, n)
  Terms <- attr(Terms, "term.labels")
  asgn <- attr(x, "assign")
  ousex <- match(asgn, match(oTerms, Terms), 0) > 0
  if(int) ousex[1] <- TRUE
  for(tt in scope) {
    usex <- match(asgn, match(tt, Terms), 0) > 0
    X <- x[, usex|ousex, drop = FALSE]
    z <-  glm.fit(X, y, wt, offset=object$offset,
                  family=object$family, control=object$control)
    dfs[tt] <- z$rank
    dev[tt] <- z$deviance
  }
  if (is.null(scale) || scale == 0) 
    dispersion <- summary(object, dispersion = NULL)$dispersion
  else dispersion <- scale
  if(object$family$family == "gaussian") {
    if(scale > 0) aic <- dev/scale - n + k*dfs
    else aic <- n * log(dev/n) + k*dfs
  } else aic <- dev/dispersion + k*dfs
  dfs <- dfs - dfs[1]
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, Deviance = dev, AIC = aic,
                    row.names = names(dfs), check.names = FALSE)
  test <- match.arg(test)
  if(test == "Chisq") {
    dev <- aod$Deviance
    nas <- !is.na(dev)
    dev[nas] <- 1 - pchisq(dev[nas]/dispersion, aod$Df[nas])
    aod[, "Pr(Chi)"] <- dev
  }
  head <- c("Single term additions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

add1.mlm <- function(...)
  stop("no add1 method implemented for mlm models")

dropterm <- function(object, ...) UseMethod("dropterm")

dropterm.default <- 
  function(object, scope, scale = 0, test=c("none", "Chisq"),
           k = 2, trace = FALSE, ...)
{
  tl <- attr(object$terms, "term.labels")
  if(missing(scope)) scope <- dropScope(object)
  else {
    if(!is.character(scope))
      scope <- attr(terms(update.formula(object, scope)), "term.labels")
    if(!all(match(scope, tl, FALSE)))
      stop("scope is not a subset of term labels")
  }
  ns <- length(scope)
  ans <- matrix(nrow = ns + 1, ncol = 2)
  dimnames(ans) <- list(c("<none>", scope), c("df", "AIC"))
  ans[1,  ] <- extractAIC(object, scale, k = k, ...)
  for(i in seq(ns)) {
    tt <- scope[i]
    if(trace > 1) cat("trying -", tt, "\n")
    nfit <- update(object, as.formula(paste("~ . -", tt)))
    ans[i+1,  ] <- extractAIC(nfit, scale, k = k, ...)
  }
  dfs <- ans[1,1] - ans[,1]
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, AIC = ans[,2])
  head <- c("Single term deletions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(test == "Chisq") {
    dev <- ans[, 2] - k*ans[, 1]
    dev <- dev - dev[1] ; dev[1] <- NA
    nas <- !is.na(dev)
    P <- dev
    P[nas] <- 1 - pchisq(dev[nas], dfs[nas])
    aod[, c("LRT", "Pr(Chi)")] <- list(dev, P)
  }
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

dropterm.lm <- function(object, scope, scale = 0, all.cols = TRUE,
                     test=c("none", "Chisq", "F"), k = 2, ...)
{
  setdiff <- function(x, y)
      if(length(x) == 0 || length(y) == 0) x else x[match(x, y, 0) == 0]

  x <- model.matrix(object)
  iswt <- !is.null(wt <- object$weights)
  n <- nrow(x)
  asgn <- attr(x, "assign")
  tl <- attr(object$terms, "term.labels")
  if(missing(scope)) scope <- dropScope(object)
  else {
    if(!is.character(scope))
      scope <- attr(terms(update.formula(object, scope)), "term.labels")
    if(!all(match(scope, tl, FALSE)))
      stop("scope is not a subset of term labels")
  }
  ndrop <- match(scope, tl)
  ns <- length(scope)
  rdf <- object$df.resid
  chisq <- deviance.lm(object)
  dfs <- numeric(ns)
  RSS <- numeric(ns)
  y <- object$residuals + predict(object)
  rank <- object$rank
  for(i in 1:ns) {
    ii <- seq(along=asgn)[asgn == ndrop[i]]                 
    if(all.cols) jj <- setdiff(seq(ncol(x)), ii)
    else jj <- setdiff(na.coef, ii)
    z <- if(iswt) lm.wfit(x[, jj, drop = FALSE], y, wt)
         else lm.fit(x[, jj, drop = FALSE], y)
    dfs[i] <- z$rank
    RSS[i] <- deviance.lm(z)
  }
  scope <- c("<none>", scope)
  dfs <- c(object$rank, dfs)
  RSS <- c(chisq, RSS)
  if(scale > 0) aic <- RSS/scale - n + k*dfs
  else aic <- n * log(RSS/n) + k*dfs
  dfs <- dfs[1] - dfs
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, "Sum of Sq" = c(NA, RSS[-1] - RSS[1]), 
                    RSS = RSS, AIC = aic, row.names = scope,
                    check.names = FALSE)
  if(scale > 0) names(aod) <- c("Df", "Sum of Sq", "RSS", "Cp")
  test <- match.arg(test)
  if(test == "Chisq") {
    dev <- aod$"Sum of Sq"
    nas <- !is.na(dev)
    dev[nas] <- 1 - pchisq(dev[nas]/scale, aod$Df[nas])
    aod[, "Pr(Chi)"] <- dev
  } else if(test == "F") {
    dev <- aod$"Sum of Sq"
    dfs <- aod$Df
    rdf <- object$df.resid
    rms <- aod$RSS[1]/rdf
    Fs <- (dev/dfs)/rms
    Fs[dfs < 1e-4] <- NA
    P <- Fs
    nas <- !is.na(Fs)
    P[nas] <- 1 - pf(Fs[nas], dfs[nas], rdf)
    aod[, c("F Value", "Pr(F)")] <- list(Fs, P)
  }
  head <- c("Single term deletions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

dropterm.mlm <- function(object, ...)
  stop("dropterm not implemented for mlm models")

dropterm.glm <-
  function(object, scope, scale = 0, test=c("none", "Chisq"), k = 2, ...)
{
  setdiff <- function(x, y)
    if(length(x) == 0 || length(y) == 0) x else x[match(x, y, 0) == 0]

  x <- model.matrix(object)
  iswt <- !is.null(wt <- object$weights)
  n <- nrow(x)
  asgn <- attr(x, "assign")
  tl <- attr(object$terms, "term.labels")
  if(missing(scope)) scope <- dropScope(object)
  else {
    if(!is.character(scope))
      scope <- attr(terms(update.formula(object, scope)), "term.labels")
    if(!all(match(scope, tl, FALSE)))
      stop("scope is not a subset of term labels")
  }
  ndrop <- match(scope, tl)
  ns <- length(scope)
  rdf <- object$df.resid
  chisq <- object$deviance
  dfs <- numeric(ns)
  dev <- numeric(ns)
  y <- object$y
  if(is.null(y)) y <- model.response(model.frame(object), "numeric")
  na.coef <- (1:length(object$coefficients))[!is.na(object$coefficients)]
  wt <- model.weights(model.frame(object))
  if(is.null(wt)) wt <- rep(1, n)
  rank <- object$rank
  for(i in 1:ns) {
    ii <- seq(along=asgn)[asgn == ndrop[i]]                 
    jj <- setdiff(seq(ncol(x)), ii)
    z <-  glm.fit(x[, jj, drop = FALSE], y, wt, offset=object$offset,
                  family=object$family, control=object$control)
    dfs[i] <- z$rank
    dev[i] <- z$deviance
  }
  scope <- c("<none>", scope)
  dfs <- c(object$rank, dfs)
  dev <- c(chisq, dev)
  if (is.null(scale) || scale == 0) 
    dispersion <- summary(object, dispersion = NULL)$dispersion
  else dispersion <- scale
  if(object$family$family == "gaussian") {
    if(scale > 0) aic <- dev/scale - n + k*dfs
    else aic <- n * log(dev/n) + k*dfs
  } else aic <- dev/dispersion + k*dfs
  dfs <- dfs[1] - dfs
  dfs[1] <- NA
  aod <- data.frame(Df = dfs, Deviance = dev, AIC = aic,
                    row.names = scope, check.names = FALSE)
  test <- match.arg(test)
  if(test == "Chisq") {
    dev <- aod$Deviance
    nas <- !is.na(dev)
    dev[nas] <- 1 - pchisq(dev[nas]/dispersion, aod$Df[nas])
    aod[, "Pr(Chi)"] <- dev
  }
  head <- c("Single term deletions", "\nModel:",
            deparse(as.vector(formula(object))))
  if(scale > 0)
    head <- c(head, paste("\nscale: ", format(scale), "\n"))
  class(aod) <- c("anova", "data.frame")
  attr(aod, "heading") <- head
  aod
}

addScope <- function(terms1, terms2)
{
  terms1 <- terms(as.formula(terms1))
  terms2 <- terms(as.formula(terms2))
  findScope(attr(terms1, "factor"), list(add = attr(terms2, "factor")))$add
}

dropScope <- function(terms1, terms2)
{
  terms1 <- terms(as.formula(terms1))
  f2 <- if(missing(terms2)) numeric(0)
  else attr(terms(as.formula(terms2)), "factor")
  findScope(attr(terms1, "factor"), list(drop = f2))$drop
}


