#
#  gnlm : A Library of Special Functions for Nonlinear Regression
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     gnlr3(y, dist="normal", pmu=NULL, pshape=NULL,
#	pfamily=NULL, mu=NULL, shape=NULL, family=NULL,
#	linear=NULL, exact=F, wt=1, delta=1, print.level=0,
#	typsiz=abs(p), ndigit=10, gradtol=0.00001, stepmax=10*sqrt(p%*%p),
#	steptol=0.00001, iterlim=100, fscale=1)
#
#  DESCRIPTION
#
#    A function to fit nonlinear regression models with a variety of
# three parameter distributions.

library.dynam("gnlm")
require(rmutil)

gnlr3 <- function(y, dist="normal", pmu=NULL, pshape=NULL,
	pfamily=NULL, mu=NULL, shape=NULL, family=NULL,
	linear=NULL, exact=F, wt=1, delta=1, print.level=0,
	typsiz=abs(p), ndigit=10, gradtol=0.00001, stepmax=10*sqrt(p%*%p),
	steptol=0.00001, iterlim=100, fscale=1){

pburr <- function(q, m, s, f) 1-(1+(q/m)^s)^-f
pglogis <- function(y, m, s, f) (1+exp(-sqrt(3)*(y-m)/(s*pi)))^-f
pgweibull <- function(y, s, m, f) (1-exp(-(y/m)^s))^f
phjorth <- function(y, m, s, f) 1-(1+s*y)^(-f/s)*exp(-(y/m)^2/2)
pginvgauss <- function(y, m, s, f)
	.C("pginvgauss",
		as.double(y),
		as.double(m),
		as.double(s),
		as.double(f),
		len=as.integer(n),
		eps=as.double(1.0e-6),
		pts=as.integer(5),
		max=as.integer(16),
		err=integer(1),
		res=double(n))$res
ppowexp <- function(y, m, s, f){
	z <- .C("ppowexp",
		as.double(y),
		as.double(m),
		as.double(s),
		as.double(f),
		len=as.integer(n),
		eps=as.double(1.0e-6),
		pts=as.integer(5),
		max=as.integer(16),
		err=integer(1),
		res=double(n))$res
	ifelse(y-m>0,0.5+z,0.5-z)}

call <- sys.call()
if(!missing(dist)&&!is.function(dist)){
	dist <- match.arg(dist,c("normal","inverse Gauss","logistic",
	"Hjorth","gamma","Burr","Weibull","extreme value","Student t",
	"power exponential"))}
if(!missing(pmu))npl <- length(pmu)
else npl <- 0
if(!missing(pshape))nps <- length(pshape)
else nps <- 0
if(!missing(pfamily))npf <- length(pfamily)
else npf <- 0
np <- npl+nps+npf
if(np<1)stop("At least one parameter must be estimated")
if(is.function(dist)){
	fcn <- dist
	dist <- "own"}
if(inherits(y,"response")){
	if(is.null(y$censor))y <- y$y
	else y <- cbind(y$y,y$censor)}
censor <- length(dim(y))==2&&ncol(y)==2
if(censor){
	n <- nrow(y)
	y[,2] <- as.integer(y[,2])
	if(any(y[,2]!=-1&y[,2]!=0&y[,2]!=1))
		stop("Censor indicator must be -1s, 0s, and 1s")
	cc <- ifelse(y[,2]==1,1,0)
	rc <- ifelse(y[,2]==0,1,ifelse(y[,2]==-1,-1,0))
	lc <- ifelse(y[,2]==-1,0,1)
	if(delta<=0&y[,2]==1)
		stop("All deltas for uncensored data must be positive")
	else {
		delta <- ifelse(delta<=0,0.000001,delta)
		delta <- ifelse(y[,1]-delta/2<=0,delta-0.00001,delta)}}
else {
	if(!is.vector(y,mode="double"))stop("y must be a vector")
	n <- length(y)
	if(min(delta)<=0)stop("All deltas for must be positive")}
if((dist!="logistic"&&dist!="Student t"&&dist!="power exponential")&&((censor&&any(y[,1]<=0))||
	(!censor&&any(y<=0))))stop("All response values must be > 0")
if(min(wt)<0)stop("All weights must be non-negative")
if(length(wt)==1)wt <- rep(wt,n)
if(length(delta)==1)delta <- rep(delta,n)
lin <- list(NULL,NULL,NULL)
if(!is.null(linear)){
	if(is.list(linear))lin <- linear
	else lin[[1]] <- linear}
if(is.language(mu))lin[[1]] <- mu
if(is.language(shape))lin[[2]] <- shape
if(is.language(family))lin[[3]] <- family
nlp <- npl
if(is.language(lin[[1]])){
	mt <- terms(lin[[1]])
	if(is.numeric(mt[[2]])){
		dm1 <- matrix(1)
		colnames(dm1) <- "(Intercept)"
		npt1 <- 1
		if(!is.function(mu)){
			mu1 <- function(p) p[1]*rep(1,n)
			nlp <- 1}
		else mu1 <- function (p) mu(p, p[1]*rep(1,n))}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm1 <- model.matrix(mt, mf)
		npt1 <- ncol(dm1)
		if(!is.function(mu)){
			mu1 <- function(p) dm1 %*% p[1:npt1]
			nlp <- npt1}
		else mu1 <- function (p) mu(p, dm1 %*% p[1:npt1])}
	if(npl<npt1)stop("Not enough initial estimates for mu")}
else if(!is.function(mu)){
	mu1 <- function(p) p[1]*rep(1,n)
	nlp <- 1}
else mu1 <- mu
if(nlp!=npl)
	stop("Number of initial estimates for mu does not correspond to model")
nlp <- nps
if(is.language(lin[[2]])){
	mt <- terms(lin[[2]])
	if(is.numeric(mt[[2]])){
		dm2 <- matrix(1)
		colnames(dm2) <- "(Intercept)"
		npt2 <- 1
		if(!is.function(shape)){
			sh1 <- function(p) p[1]*rep(1,n)
			nlp <- 1}
		else sh1 <- function(p) shape(p, p[1]*rep(1,n))}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm2 <- model.matrix(mt, mf)
		npt2 <- ncol(dm2)
		if(!is.function(shape)){
			sh1 <- function(p)dm2 %*% p[1:npt2]
			nlp <- npt2}
		else sh1 <- function(p)shape(p, dm2 %*% p[1:npt2])}
	if(nps<npt2)stop("Not enough initial estimates for shape")}
else if(!is.function(shape)){
	sh1 <- function(p) p[1]*rep(1,n)
	nlp <- 1}
else sh1 <- shape
if(nlp!=nps)
	stop("Number of initial estimates for shape does not correspond to model")
nlp <- npf
if(is.language(lin[[3]])){
	mt <- terms(lin[[3]])
	if(is.numeric(mt[[2]])){
		dm3 <- matrix(1)
		colnames(dm3) <- "(Intercept)"
		npt3 <- 1
		if(!is.function(family)){
			fa1 <- function(p) p[1]*rep(1,n)
			nlp <- 1}
		else fa1 <- function(p) family(p,p[1]*rep(1,n))}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm3 <- model.matrix(mt, mf)
		npt3 <- ncol(dm3)
		if(!is.function(family)){
			fa1 <- function(p) dm3 %*% p[1:npt3]
			nlp <- npt3}
		else fa1 <- function(p)family(p, dm3 %*% p[1:npt3])}
	if(npf<npt3)stop("Not enough initial estimates for family")}
else if(!is.function(family)){
	fa1 <- function(p) p[1]*rep(1,n)
	nlp <- 1}
else fa1 <- family
if(nlp!=npf)
	stop("Number of initial estimates for family does not correspond to model")
if(!is.numeric(mu1(pmu)))stop("The mean function must return numerical values")
if(!is.numeric(sh1(pshape)))
	stop("The shape function must return numerical values")
if(!is.numeric(fa1(pfamily)))
	stop("The family function must return numerical values")
p <- c(pmu,pshape,pfamily)
npl1 <- npl+1
np1 <- npl+nps
nps1 <- np1+1
if (!censor){
	ret <- switch(dist,
	normal={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				y <- y^f/f
				jy <- y^(2*f-1)*delta/(2*f)
				norm <- sign(f)*pnorm(0,m,s)
				ind <- f<0
				-wt*(log((pnorm(y+jy,m,s)-pnorm(y-jy,m,s)))
					-log(1-ind-norm))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				norm <- sign(f)*pnorm(0,m,s)
				ind <- f<0
				-wt*((f-1)*log(y)+log(dnorm(y^f/f,m,s))
					-log(1-ind-norm))}
			const <- -wt*log(delta)}},
	"power exponential"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*log(ppowexp(y+delta/2,m,s)
					-ppowexp(y-delta/2,m,s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				t <- 0.5*sh1(p[npl1:np1])
				f <- exp(fa1(p[nps1:np]))
				b <- 1+1/(2*f)
				wt*(t+(abs(y-mu1(p))/exp(t))^(2*f)/2+
					lgamma(b)+b*log(2))}
			const <- -wt*log(delta)}},
	"inverse Gauss"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*log(pginvgauss(y+delta/2,m,s,f)
					-pginvgauss(y-delta/2,m,s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*(-f*log(m)+(f-1)*log(y)-
					log(2*besselK(1/(s*m),abs(f)))-
					(1/y+y/m^2)/(2*s))}
			const <- -wt*log(delta)}},
	logistic={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*log(pglogis(y+delta/2,m,s,f)
					-pglogis(y-delta/2,m,s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				t <- sh1(p[npl1:np1])
				m <- (y-mu1(p))/exp(t)*sqrt(3)/pi
				wt*(-fa1(p[nps1:np])+m+t+(exp(fa1(p[nps1:np]))
					+1)*log(1+exp(-m)))}
			const <- -wt*(log(delta*sqrt(3)/pi))}},
	Hjorth={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*log(phjorth(y+delta/2,m,s,f)-
					phjorth(y-delta/2,m,s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*(-f*log(1+s*y)/s-(y/m)^2/2+
					log(y/m^2+f/(1+s*y)))}
			const <- -wt*log(delta)}},
        gamma={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				u <- (m/s)^f
				-wt*log(pgamma((y+delta/2)^f,s,u)
					-pgamma((y-delta/2)^f,s,u))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				u <- fa1(p[nps1:np])
				f <- exp(u)
				v <- s*f
				-wt*(v*(t-log(m))-(s*y/m)^f+u+(v-1)*log(y)
					-lgamma(s))}
			const <- -wt*log(delta)}},
	Burr={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*log(pburr(y+delta/2,m,s,f)-
					pburr(y-delta/2,m,s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				y1 <- y/m
				-wt*(log(f*s/m)+(s-1)*log(y1)
					-(f+1)*log(1+y1^s))}
			const <- -wt*log(delta)}},
        Weibull={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*log(pgweibull(y+delta/2,s,m,f)
					-pgweibull(y-delta/2,s,m,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				u <- fa1(p[nps1:np])
				f <- exp(u)
				y1 <- (y/m)^s
				-wt*(t+u+(s-1)*log(y)-s*log(m)+
					(f-1)*log(1-exp(-y1))-y1)}
			const <- -wt*log(delta)}},
        "Student t"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*log(pt((y+delta/2-m)/s,f)-
					pt((y-delta/2-m)/s,f))}
			const <- 0}
		else {
			fcn <- function(p) {
				s <- exp(0.5*sh1(p[npl1:np1]))
				-wt*log(dt((y-mu1(p))/s,exp(fa1(p[nps1:np])))/
					s)}
			const <- -wt*(log(delta))}},
        "extreme value"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				y1 <- y^f/f
				ey <- exp(y1)
				jey <- y^(f-1)*ey*delta/2
				norm <- sign(f)*exp(-m^-s)
				ind <- f>0
				-wt*(log((pweibull(ey+jey,s,m)
					-pweibull(ey-jey,s,m))/
					(1-ind+norm)))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				f <- fa1(p[nps1:np])
				y1 <- y^f/f
				norm <- sign(f)*exp(-m^-s)
				ind <- f>0
				-wt*(t+s*(y1-log(m))-(exp(y1)/m)^s
					+(f-1)*log(y)-
					log(1-ind+norm))}
			const <- -wt*log(delta)}},
	own={ const <- 0})}
else {
	ret <- switch(dist,
	normal={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				y <- y[,1]^f/f
				jy <- y[,1]^(2*f-1)*delta/(2*f)
				pn <- pnorm(y-jy,m,s)
				norm <- sign(f)*pnorm(0,m,s)
				ind <- f<0
				-wt*(cc*log((pnorm(y+jy,m,s)-pn))+
					log(lc-rc*(pn-(f>0)*norm)))/
					(1-ind-norm)}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(0.5*t)
				f <- fa1(p[nps1:np])
				norm <- sign(f)*pnorm(0,m,s)
				ind <- f<0
				-wt*(cc*(-(t+((y[,1]^f/f-m)/s)^2)/2+(f-1)*
					log(y[,1]))+log(lc-rc
					*(pnorm((y[,1]-delta/2)^f/f,m,s)
					-(f>0)*norm)))/(1-ind-norm)}
			const <- wt*cc*(log(2*pi)/2-log(delta))}},
	"power exponential"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				pp <- ppowexp(y[,1]-delta/2,m,s,f)
				-wt*(cc*log(ppowexp(y[,1]+delta/2,m,s,f)-pp)
					+log(lc-rc*pp))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- 0.5*sh1(p[npl1:np1])
				s <- exp(t)
				f <- exp(fa1(p[nps1:np]))
				b <- 1+1/(2*f)
				-wt*(cc*(-t-(abs(y[,1]-mu1(p))/s)^(2*f)/2-
					lgamma(b)-b*log(2))+log(lc-rc
					*ppowexp(y[,1]-delta/2,m,s,f)))}
			const <- -wt*cc*(log(delta))}},
	"inverse Gauss"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1])/2)
				f <- fa1(p[nps1:np])
				pg <- pginvgauss((y[,1]-delta/2)^f/f,m,s)
				-wt*(cc*log(pginvgauss(y[,1]+delta/2,m,s,f)-
					pg)+log(lc-rc*pg))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*(cc*(-f*log(m)+(f-1)*log(y[,1])-
					log(2*besselK(1/(s*m),abs(f)))-
					(1/y[,1]+y[,1]/m^2)/(2*s))+log(lc-rc
					*pginvgauss(y[,1]-delta/2,m,s,f)))}
			const <- -wt*cc*(log(delta))}},
	logistic={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))*sqrt(3)/pi
				f <- exp(fa1(p[nps1:np]))
				pl <- pglogis(y[,1]-delta/2,m,s,f)
				-wt*(cc*log(pglogis(y[,1]+delta/2,m,s,f)-pl)
					+log(lc-rc*pl))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))*sqrt(3)/pi
				y1 <- (y[,1]-m)/s
				u <- fa1(p[nps1:np])
				f <- exp(u)
				-wt*(cc*(u-y1-log(s)-(f+1)*log(1+exp(-y1)))
					+log(lc-rc*pglogis(y[,1]-delta/2,m,s,f)))}
			const <- -wt*cc*log(delta)}},
	Hjorth={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				ph <- phjorth(y[,1]-delta/2,m,s,f)
				-wt*(cc*log(phjorth(y[,1]+delta/2,m,s,f)-ph)
					+log(lc-rc*ph))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				-wt*(cc*(-f*log(1+s*y[,1])/s-(y[,1]/m)^2/2+
					log(y[,1]/m^2+f/(1+s*y[,1])))+
					log(lc-rc*phjorth(y[,1]
					-delta/2,m,s,f)))}
			const <- -wt*cc*log(delta)}},
        gamma={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				u <- (m/s)^f
				pg <- pgamma((y[,1]-delta/2)^f,s,u)
				-wt*(cc*log(pgamma((y[,1]+delta/2)^f,s,u)-pg)
					+log(lc-rc*pg))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				u <- fa1(p[nps1:np])
				f <- exp(u)
				v <- s*f
				-wt*(cc*(v*(t-log(m))-(s*y[,1]/m)^f+u+(v-1)*
					log(y[,1])-lgamma(s))+log(lc-rc
					*pgamma((y[,1]-delta/2)^f,s,(m/s)^f)))}
			const <- -wt*cc*log(delta)}},
	Burr={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				pb <- pburr(y[,1]-delta/2,m,s,f)
				-wt*(cc*log(pburr(y[,1]+delta/2,m,s,f)-pb)
					+log(lc-rc*pb))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				y1 <- y[,1]/m
				-wt*(cc*(log(f*s/m)+(s-1)*log(y1)
					-(f+1)*log(1+(y1)^s))+
					log(lc-rc*pburr(y[,1]
					-delta/2,m,s,f)))}
			const <- -wt*cc*log(delta)}},
        Weibull={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				pw <- pgweibull(y[,1]-delta/2,s,m,f)
				-wt*(cc*log(pgweibull(y[,1]+delta/2,s,m,f)-pw)
					+log(lc-rc*pw))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				u <- fa1(p[nps1:np])
				f <- exp(u)
				y1 <- (y[,1]/m)^s
				-wt*(cc*(t+u+(s-1)*log(y[,1])-s*log(m)+(f-1)
					*log(1-exp(-y1))-y1)+log(lc-rc*
					pgweibull(y[,1]-delta/2,s,m,f)))}
			const <- -wt*cc*log(delta)}},
        "Student t"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				ps <- pt((y[,1]-delta/2-m)/s,f)
				-wt*(cc*log(pt((y[,1]+delta/2-m)/s,f)-ps)
					+log(lc-rc*ps))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(0.5*sh1(p[npl1:np1]))
				f <- exp(fa1(p[nps1:np]))
				-wt*(cc*log(dt((y[,1]-m)/s,f)/s)
					+log(lc-rc*pt((y[,1]-delta/2-m)/s,f)))}
			const <- -wt*cc*(log(delta))}},
        "extreme value"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- exp(sh1(p[npl1:np1]))
				f <- fa1(p[nps1:np])
				y1 <- y[,1]^f/f
				ey <- exp(y1)
				jey <- y[,1]^(f-1)*ey
				norm <- sign(f)*exp(-m^-s)
				ind <- f>0
				pw <- pweibull(ey-jey*delta/2,s,m)
				-wt*(cc*log(pweibull(ey+jey*delta/2,s,m)-pw)
					+log(lc-rc*(pw-ind+(f>0)*norm))-
					log(1-ind+norm))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				t <- sh1(p[npl1:np1])
				s <- exp(t)
				f <- fa1(p[nps1:np])
				y1 <- y[,1]^f/f
				ey <- exp(y1)
				norm <- sign(f)*exp(-m^-s)
				ind <- f>0
				-wt*(cc*(t+s*(y1-log(m))-(ey/m)^s
					+(f-1)*log(y[,1]))+log(lc-rc*
					(pweibull(ey-y[,1]^(f-1)*ey*
					delta/2,s,m)-ind+(f>0)*norm))-
					log(1-ind+norm))}
			const <- -wt*cc*log(delta)}},
	own={const <- 0})}
fn <- function(p) sum(fcn(p))
if(fscale==1)fscale <- fn(p)
if(is.na(fscale))
	stop("Non-numerical function value: probably invalid initial values")
z0 <- nlm(fn, p=p, hessian=T, print.level=print.level, typsiz=typsiz,
	ndigit=ndigit, gradtol=gradtol, stepmax=stepmax, steptol=steptol,
	iterlim=iterlim, fscale=fscale)
z0$minimum <- z0$minimum+sum(const)
if(!is.language(lin[[1]]))cname <- paste("p",1:npl,sep="")
else {
     cname <- colnames(dm1)
     if(is.function(mu)&&length(cname)<npl)
	cname <- c(cname,paste("p",(length(cname)+1):npl,sep=""))}
if(!is.language(lin[[2]]))sname <- paste("p",1:nps,sep="")
else {
     sname <- colnames(dm2)
     if(is.function(shape)&&length(sname)<nps)
	sname <- c(sname,paste("p",(length(sname)+1):nps,sep=""))}
if(!is.language(lin[[3]]))fname <- paste("p",1:npf,sep="")
else {
     fname <- colnames(dm3)
     if(is.function(family)&&length(fname)<npf)
	fname <- c(fname,paste("p",(length(fname)+1):npf,sep=""))}
fitted.values <- as.vector(mu1(z0$estimate))
residuals <- y-fitted.values
if(np==1){
	cov <- 1/z0$hessian
	se <- sqrt(cov)}
else {
	a <- qr(z0$hessian)
	if(a$rank==np)cov <- solve(z0$hessian)
	else cov <- matrix(NA,ncol=np,nrow=np)
	se <- sqrt(diag(cov))}
like.comp <- as.vector(fcn(z0$estimate)+const)
if(is.function(mu))mu1 <- mu
if(is.function(shape))sh1 <- shape
if(is.function(family))fa1 <- family
z1 <- list(
	call=call,
	delta=delta,
	dist=dist,
	likefn=fcn,
	mu=mu1,
	shape=sh1,
	family=fa1,
	linear=lin,
	prior.weights=wt,
	censor=censor,
	maxlike=z0$minimum,
	fitted.values=fitted.values,
	residuals=residuals,
	like.comp=like.comp,
	aic=z0$minimum+np,
	df=sum(wt)-np,
	coefficients=z0$estimate,
	cname=cname,
	sname=sname,
	fname=fname,
	npl=npl,
	npm=0,
	nps=nps,
	npf=npf,
	se=se,
	cov=cov,
	corr=cov/(se%o%se),
	gradient=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z1) <- "gnlr"
return(z1)}
