#' @title Percent Yield Reduction (PYR)
#' @description
#' This function computes Percent Yield Reduction (PYR) using any traits (like yield) under stress and non-stress conditions. The lower values of PYR indicates greater tolerance. For more details see Farshadfar and Javadinia (2011).
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A data frame with \code{Gen}, \code{PYR}.
#'
#' \deqn{PYR = \frac{(YN - YS)}{YN} \times 100}
#'
#' @references Farshadfar, E. and Javadinia, J. (2011). Evaluation of chickpea (Cicer arietinum L.) genotypes for drought tolerance. Seed and Plant Improvement Journal, 27(4), 517–537.
#' @examples
#' out = PYR(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
PYR <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  out <- ((YN - YS) / YN) * 100
  out[YN == 0] <- NA_real_
  return(data.frame(Gen = as.character(Gen), PYR = out, row.names = NULL))
}
