# Functions for filtering

#' Global filter for peak finder
#'
#' \code{write_groups} implements the global filter for the peak finder.
#'
#' This filter tests a rectangle bounded by the peaks and troughs
#' in a group, and starting a new group if the rectangle
#' gets too large. The parameters minSlope and minYerror
#' determine how large a rectangle can get.
#'
#' @param X x-axis data as array
#' @param Y y-axis data as array
#' @param minSlope minimum slope between bottom left corner
#' of rectangle and top right corner
#' @param minYerror minimum y difference between bottom
#' left corner and top right corner
#' @param globalFilter whether the peak finder is actually
#' using this filter. If not, it just returns an
#' array of 0s.
#'
#' @return an integer array containing 1s where the group
#' should be split and 0s otherwise
write_groups <- function(X, Y, minSlope, minYerror, globalFilter = TRUE) {

  # If not using the globalFilter, return a vector of zeros
  if (!globalFilter) {return(rep(0, length(X)))}

  newgroup <- TRUE
  res <- vector('integer', length(X))
  for (i in 1:length(X)){
    # If the current point is the start of a new group of peaks and
    # troughs, move all rectangle boundaries to the position of that point.
    if (newgroup){
      minX <- X[i]
      maxX <- minX
      minY <- Y[i]
      maxY <- minY
      res[i] <- 0
      newgroup <- FALSE

    } else { # Update rectangle boundaries based on position of current point

      maxX <- X[i]
      if (Y[i] > maxY){
        maxY <- Y[i]
      } else if (Y[i] < minY){
        minY <- Y[i]
      }
      dY <- maxY - minY
      dX <- maxX - minX # Find dimensions of rectangle
      # Test if rectangle fits the grouping parameters
      # Using hyperbola test
      group <- (dY ^ 2) < (minSlope*dX)^2 + minYerror^2
      # If the rectangle is too large, start a new group
      res[i] <- ifelse(group, 0, 1)
      if (!group){
        newgroup <- TRUE
      }
    }
  }

  return(res)
}
