#' Find the surface level and split images
#'
#' \code{findSurface} - This function finds the best split of an image with a
#' horizontal or vertical line into one color (often red, e.g., for the
#' surface) and other colors (often black and white, e.g., for everything in
#' the ground). This works for any images. For the application on
#' minirhizotron root scans the function provides the additional functionality
#' of using the installation angle of the minirhizotron and the resolution of
#' the image to not only return the position of the line in pixels, but also
#' the corresponding depth in cm of the highest point of the top side of the
#' image, also called \code{depth_highpoint_cm} in other function of this
#' package.
#'
#' @param img_path (Optional, default = NULL) Character vector specifying
#' the image path of interest. This is only used if
#' \code{img} is set to NULL.
#' @param img Image (e.g., provided by the RootDetector) in the form of an
#' array with 3 dimensions for the RGB color channels (3 layers each containing
#' a 2-dim. numeric matrix with values between 0 and 1).
#' @param surface_col Color of the area to be split of the mask (default "red").
#' Can be of any of the three kinds of R color specifications, i.e., either a
#' color name (as listed by colors()), a hexadecimal string (see Details), or a
#' positive integer (indicates using palette()).
#' @param pos_highpoint_px Either one of "center" or "edge" (default),
#' indicating that scanning starts at the bottom or top facing side of the
#' minirhizotron, respectively, or it can also be an integer value (>=1 and
#' <=width of the \code{top_side} of the image) specifying where (left<->right)
#' in the top row of the image the highest point of the image lies, indicating
#' the column where the minimum points of the depth-level lines lie. \cr
#' "edge" is equivalent to using 1 or width of the \code{top_side} and
#' "center" to using half the width of the \code{top_side}.
#' @param radius_highpoint_px The radius specifying how large the are should be
#' to determine the split (default 10).
#' @param angle Numeric value >=0 and <=90 (default 45) specifying the
#' installation angle of the minirhizotron in degrees (angle between the ground
#' and the tube above the soil).
#' @param top_side One of "left","top" (default),"right","bottom". Indicates
#' where the upper side of the image is.
#' @param ppcm Numeric value specifying how many pixels there are per
#' centimeter (resolution of the image). Default NULL.
#' If \code{ppcm} is not NULL or NA, \code{ppi} is ignored.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#' @param ppi Numeric value specifying how many pixels there are per inch
#' (resolution of the image). Default NULL.
#' Leave/set \code{ppcm} to NULL or NA to use \code{ppi}.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#' @param return_all_results Specify if all checked overlaps with their
#' respective correlation score should be returned (default FALSE).
#'
#' @return \code{findSurface} A list with following features:
#' 'pos_surf_px' and 'depth_highpoint_cm'.
#' If \code{return_all_results} is set to true, a list also comprises a matrix
#' 'split_info' containing the information on all possible splits.
#' @export
#' @rdname findSurface
#'
#' @examples
#' # Example of finding the best row to horizontally split the image at the
#' # highpoint position.
#' mat_R <- matrix(c(1,1,1,1,1,
#'                   0,1,1,1,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_G <- matrix(c(0,0,0,0,1,
#'                   0,1,0,0,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_B <- matrix(c(0,0,0,0,1,
#'                   0,1,0,0,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' findSurface(img = array(c(mat_R,mat_G,mat_B), dim = c(dim(mat_R),3)),
#'             radius_highpoint_px = 1, top_side = "top", ppcm = 1,
#'             return_all_results = TRUE)
findSurface <- function(img_path = NULL, img = NULL,
                        surface_col = "red", pos_highpoint_px = "center",
                        radius_highpoint_px = 10,
                        angle = 45,
                        top_side = "left",
                        ppcm = NULL, ppi = NULL,
                        return_all_results = FALSE) {
  # Check input. ---------------------------------------------------------------
  if(angle<0 || angle>90){
    stop("The angle must be >=0 and <=90.")
  }
  if(!top_side %in% c("top", "left", "right", "bottom")){
    stop("Unknown 'top_side'.")
  }
  if(is.null(ppcm) || is.na(ppcm)){
    if(!is.null(ppi)){
      ppcm <- ppi2ppcm(ppi)
    } else {
      stop("No resolution specified ('ppcm' and 'ppi' are both NULL).")
    }
  }
  # Load the image. ------------------------------------------------------------
  # If a path is provided, load the image.
  if(is.null(img)){
    img <- png::readPNG(img_path)[,,1:3]
  } else { # If image is already loaded.
    # If the image is only a matrix, give it three layers.
    if(length(dim(img))!=3){
      stop("Array with 3 color channels needed.")
    }
  }
  # Rotate the image according to the top side, to always have the
  # top side on top. -----------------------------------------------------------
  if(top_side == "left"){
    img <- rotate_clockwise(img, degrees = 90)
  } else if(top_side == "right"){
    img <- rotate_clockwise(img, degrees = 270)
  } else if(top_side == "bottom"){
    img <- rotate_clockwise(img, degrees = 180)
  }
  # Where is the highpoint/upwards facing side of the tube in the images? ------
  if(pos_highpoint_px == "center"){
    pos_highpoint_px <- round(dim(img)[2]/2) # Still in the center even with gap.
  } else if(pos_highpoint_px == "edge"){
    pos_highpoint_px <- 1 # Still at the edge even with gap.
  } else {
    if(pos_highpoint_px<1 || pos_highpoint_px>dim(img)[2]){
      stop(paste0("'pos_highpoint_px' not within image."))
    }
  }
  if(radius_highpoint_px < 0){
    stop("Radius must be >=0.")
  }
  # Which part of the image should be used to determine the surface line? ------
  if(2*radius_highpoint_px+1 >= dim(img)[2]){
    # Use all columns if radius too large.
    cols_of_interest <- 1:dim(img)[2]
  } else {
    # Use subset (circular image wrap).
    cols_of_interest <- (pos_highpoint_px-radius_highpoint_px):
                        (pos_highpoint_px+radius_highpoint_px)
    numb_neg <- sum(cols_of_interest<=0)
    if(numb_neg>0){
      cols_of_interest <- c(cols_of_interest[-(1:numb_neg)],
                            (dim(img)[2]-numb_neg+1):(dim(img)[2]))
    }
    numb_above <- sum(cols_of_interest > dim(img)[2])
    if(numb_above>0){
      cols_of_interest <- c((1:numb_above),
                            cols_of_interest[-((dim(img)[2]-numb_above+1):
                                               (dim(img)[2]))])
    }
  }
  area_of_interest <- img[,cols_of_interest,, drop = FALSE]
  # Compute the best split for this area. --------------------------------------
  split_all_info <- splitImgHoriz(img = area_of_interest,
                                  surface_col = surface_col,
                                  return_all_results = return_all_results)
  #plot(split_all_info$split_info[,1], split_all_info$split_info[,2])
  split_px <- split_all_info[[1]]
  depth_px <- sin(angle/180 * pi)*(split_px-1)
  if(length(split_all_info)>1){
    surf_info <- list("pos_surf_px" = split_px,
                      "depth_highpoint_cm" = px2cm(depth_px, ppi = ppi,
                                                   ppcm = ppcm),
                      "split_info" = split_all_info[[2]])
  } else {
    surf_info <- list("pos_surf_px" = split_px,
                      "depth_highpoint_cm" = px2cm(depth_px, ppi = ppi,
                                                   ppcm = ppcm))
  }
  return(surf_info)
}
#' Find the surface level and split images
#'
#' \code{splitImgHoriz} - This function finds the best split of an image with a
#' horizontal line into one color on the top half (often red, e.g., for the
#' surface) and other colors on the bottom half (often black and white, e.g.,
#' for everything in the ground). This works for any images.\cr
#' Currently, the best split is determined as the one where there are the most
#' equal mismatches on both sides, i.e., the same number of pixels in the top
#' part of the image which have not the specified color as pixels in the bottom
#' part of the image which have this specific color.
#'
#' @return \code{splitImgHoriz} An integer value. The row where the upper
#' "surface" part of the image starts (the row included). Called 'pos_surf_px'
#' in the return value of the function \code{findSurface()}.
#' If \code{return_all_results} is set to true, a list containing this best
#' value 'best_split' as well as a matrix 'split_info' containing the
#' information on all possible splits is returned instead.
#'
#' @export
#' @rdname findSurface
#'
#' @examples
#' # Example of finding the best row to horizontally split the image.
#' # Note that the top row and one pixel in the second row is red. All others
#' # are either black or white.
#' mat_R <- matrix(c(1,1,1,1,
#'                   0,1,0,1,
#'                   0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE)
#' mat_G <- matrix(c(0,0,0,0,
#'                   0,1,0,0,
#'                   0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE)
#' mat_B <- matrix(c(0,0,0,0,
#'                   0,1,0,0,
#'                   0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE)
#' splitImgHoriz(img = array(c(mat_R,mat_G,mat_B), dim = c(dim(mat_R),3)),
#'               return_all_results = TRUE)
splitImgHoriz <- function(img, surface_col = "red",
                          return_all_results = FALSE) {
  pos_splits <- 1:(dim(img)[1]-1)
  surface_rgb <- as.vector(grDevices::col2rgb(surface_col, alpha = FALSE)/255)
  col_per_row <- sapply(1:dim(img)[1], function(row_i){
    all_cols_row <- matrix(img[row_i,,, drop = FALSE], ncol = 3, byrow = FALSE)
    col_matches <- sum(colSums(t(all_cols_row) == surface_rgb) == 3)
    return(col_matches)
  })
  if(sum(col_per_row) == 0){
    message("Specified color not found in image. Returning NA.")
    split_scores <- rep(NA, length(pos_splits))
    best_split <- NA
  } else if(sum(col_per_row) == dim(img)[1]*dim(img)[2]){
    message("Image has only the specified color. Returning NA.")
    split_scores <- rep(NA, length(pos_splits))
    best_split <- NA
  } else {
    split_scores <- sapply(pos_splits, function(split){
      # Not the color in the top part:
      dismatches_top <- split * dim(img)[2] - sum(col_per_row[1:split])
      # Color in the bottom part:
      dismatches_bot <- sum(col_per_row[(split+1):dim(img)[1]])
      return(abs(dismatches_top-dismatches_bot))
    })
    best_split <- pos_splits[which.min(split_scores)]
  }

  if(return_all_results){
    split_info <- list(best_split, cbind(pos_splits, split_scores))
    names(split_info) <- c("best_split","split_info")
    colnames(split_info$"split_info") <- c("possible_splits",
                                           "dismatch_difference")
    return(split_info)
  } else {
    return(best_split)
  }
}
