#' Generate data with skewed errors
#'
#' @param n Number of total observations
#' @param nr Number of observations with a different error distribution
#' @param p Number of predictors
#' @param dist_type Type of error distribution ("skew_normal", "skew_t", "skew_laplace")
#' @param ... Additional parameters for the error distribution
#'
#' @return A list with X (design matrix), Y (response), and e (error)
#' @export
#'
#' @examples
#' set.seed(123)
#' data <- serr(1000, 200, 5, "skew_t")
#' str(data)

serr<- function(n, nr, p, dist_type, ...) {
  beta <- sort(runif(p, 1, 5))
  X <- matrix(runif(n * p, 0, 1), ncol = p)

  if (dist_type == "skew_normal") {
    e1 <- sn::rsn(n - nr, xi = 0, omega = 1, alpha = 5)
    e2 <- sn::rsn(nr, xi = 0, omega = 1, alpha = 10)
  } else if (dist_type == "skew_t") {
    e1 <- sn::rst(n - nr, xi = 0, omega = 1, alpha = 5, nu = 5)
    e2 <- sn::rst(nr, xi = 0, omega = 1, alpha = 10, nu = 5)
  } else if (dist_type == "skew_laplace") {
    e1 <- LaplacesDemon::rslaplace(n - nr, mu = 0, beta = 1, alpha = 5)
    e2 <- LaplacesDemon::rslaplace(nr, mu = 0, beta = 2, alpha = 10)
  } else {
    stop("Unsupported distribution type")
  }

  e <- c(e1, e2)
  Y <- X %*% beta + e
  return(list(X = X, Y = Y, e = e))
}

