#' @importFrom volesti Hpolytope
#' @importFrom volesti sample_points

#' @title Core-center estimation by hit-and-run
#' @description Given a game with a full-dimensional core, this function computes a hit-and-run estimation of its core center.
#' @param v A characteristic function of a game with, as a vector.
#' @param k The number of points to be generated by the hit-and-run method, as an integer. By default, \code{k=1000}.
#' @param getpoints A logical value. By default, \code{getpoints=FALSE}. If set to\code{getpoints=TRUE}, the points generated by the hit-and-run method are also returned, as a matrix in which each row is a point.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return A hit-and-run estimation of the core-center, as a vector; and, if \code{getpoints=TRUE}, a matrix containing the points generated by the hit-and-run method.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}. A game is said to be balanced if its core is not empty.
#'
#' The core-center of a balanced game \eqn{v}, \eqn{CC(v)}, is defined as the expectation
#' of the uniform distribution over \eqn{C(v)}, and thus can be interpreted
#' as the centroid or center of gravity of \eqn{C(v)}.
#' Let \eqn{\mu} be the \eqn{(n-1)}-dimensional Lebesgue measure and let \eqn{V(C)=\mu(C(v))}
#' be the volume (measure) of the core. If \eqn{V(C)>0}, then, for each \eqn{i\in N},
#' \deqn{CC_i(v)=\frac{1}{V(C)}\int_{C(v)}x_i d\mu}.
#'
#' The hit-and-run method (Smith, 1984) is a Monte Carlo algorithm that generates
#' uniformly distributed random points within a bounded and convex region,
#' such as a polytope or a convex body in high-dimensional space.
#'
#' The hit-and-run method is based on the following process.
#' Step 0: choose a point inside the convex body.
#' Step 1: choose a uniformly distributed random direction from the unit sphere in the given dimension.
#' Step 2: determine the longest line segment that, starting from the chosen point and taking the chosen direction, remains entirely within the convex body.
#' Step 3: choose a uniformly distributed along the line segment random point.
#' Step 4: go to Step 1.
#' @examples
#' v1 <- claimsgame(E=8,d=c(3,5,6))
#' corecenterhitrun(v1,k=1e5)
#'
#' v2 <- c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' corecenterhitrun(v2,k=1e5)
#'
#' # Plotting the corecenter and its hit-and-run estimation:
#' plotcoreset(v2,solutions="corecenter",allocations=corecenterhitrun(v2))
#'
#' # Plotting the points generated by the hit-and-run method:
#' hrpoints <- corecenterhitrun(v2,k=100,getpoints=TRUE)$points
#' plotcoreset(v2,allocations=hrpoints)
#'
#' # What if the game is not full-dimensional because of a dummy player?
#' v3 <- c(440,0,0,0,440,440,440,15,14,7,455,454,447,60,500)
#' # For coredimension to detect that, tolerance has to be appropriate:
#' coredimension(v3,tol=100*.Machine$double.eps) # tolerance too small
#' coredimension(v3) # default tolerance, 1e-12, big enough
#'
#' # Now how to compute the hit-and-run estimation of the core-center?
#' # Knowing that player 1 is a dummy and that the core-center assigns
#' # dummies their individual worth...
#' v3.without1 <- subgame(v3,S=14) # subgame without player 1
#' ( cc.hr <- c(v3[1],corecenterhitrun(v3.without1,k=100)) )
#'
#' # Plotting the points when there is a dummy player:
#' points.without1 <- corecenterhitrun(v3.without1,k=100,getpoints=TRUE)$points
#' points.with1 <- cbind(v3[1],points.without1)
#' plotcoreset(v3,allocations=points.with1)
#'
#' # This function does not work if the core is not full-dimensional:
#' v4 <- c(0,0,0,0,2,5,0,5,0,0,10,2,5,5,10)
#' corecenterhitrun(v4,k=1e5)
#' @references Gonzalez-Díaz, J. & Sánchez-Rodríguez, E. (2007). A natural selection from the core of a TU game: the core-center. International Journal of Game Theory, 36(1), 27-46.
#' @references Espinoza-Burgos, N. H. (2020). \emph{Comparación de métodos exactos y aproximados para calcular el core-center del juego del aeropuerto}. TFM, Máster en Técnicas Estadísticas, \url{http://eio.usc.es/pub/mte/descargas/ProyectosFinMaster/Proyecto_1791.pdf}.
#' @references Smith, R. L. (1984). Efficient Monte Carlo Procedures for Generating Points Uniformly Distributed Over Bounded Regions. Operations Research, 32(6), 1296-1308.
#' @seealso \link{balancedcheck}, \link{corecentervalue}, \link{coredimension}, \link{corevertices}, \link{corevertices234}
#' @export

corecenterhitrun <- function(v, k = 1000, getpoints = FALSE, binary = FALSE) {

  ######################
  # trámites iniciales #
  ######################
  nC <- length(v)
  n <- log2(nC+1)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  # Vamos a guardar tanto v.lex como v.bin para ahorrar trabajo.
  if (binary == FALSE) {
    v.lex <- v
    v.bin <- lex2bin(v)
  } else {
    v.lex <- bin2lex(v)
    v.bin <- v
  }
  if (!balancedcheck(v.bin,binary=TRUE)) {
    message("The core of 'v' is empty.")
    return(NULL)
  }
  if (coredimension(v.bin,binary=TRUE)<n-1.5) {
    message("The core of 'v' is not full-dimensional.")
    return(NULL)
  }
  ##########################
  # fin trámites iniciales #
  ##########################

  ve <- corevertices(v.bin,binary=TRUE)
  start.point <- colMeans(ve[,-ncol(ve)])
  # tiempo<-proc.time()

  # Paso por binario y luego lo devuelvo a lex con la función Base.lex,
  # diría que volesti necesita así las entradas, no estoy segura

  Base=matrix(0,2^n-1,n)
  for (ii in 1:2^n-1){
    Base[ii,]=as.numeric(intToBits(ii)[1:n])
  }
  B <- cbind(Base,c(v.bin))

  Base.lex <- function(Base) {
    card <- rowSums(Base)
    binval <- apply(Base, 1, function(x) sum(2^((length(x)-1):0) * x))
    ord <- order(card, -binval)  # -binval para orden decreciente dentro de cada cardinalidad
    Base[ord, , drop = FALSE]
  }
  B <- cbind(Base.lex(Base),c(v.lex))

  # Subfunción que nos hace falta.
  nuevo_sistema<-function(A,b,k,ef){
    I=1:dim(A)[1]
    A_hat=NULL
    b_hat=NULL
    for(j in I){
      A_hat=rbind(A_hat,A[j,-k]-rep(1,dim(A)[2]-1)*A[j,k]) #Nueva matriz A_hat x<= b_hat sustituyendo la variable despejada
      b_hat=c(b_hat,b[j]-ef*A[j,k])
    }
    return(list(A_hat=A_hat,b_hat=b_hat))
  }
  # Fin subfunción que nos hace falta.

  B=nuevo_sistema(A=B[-nC,-(n+1)],b=B[-nC,(n+1)],k=n,ef=v[nC])
  B$A_hat=-B$A_hat;B$b_hat=-B$b_hat
  P = Hpolytope(A = B$A_hat, b = B$b_hat)
  points = sample_points(P,n=k,random_walk = list("startin_point" = start.point))
  # list("walk" = "BRDHR")
  corecenter=rowMeans(points)
  mu=c(corecenter,v[nC]-sum(corecenter))
  # tiempo<-proc.time()-tiempo
  pointsn <- rbind(points, v[nC] - colSums(points))
  pointsn <- t(pointsn) # trasponemos para que cada fila sea un punto

  # ÁdPS(29/05/2025): Comento la siguiente salida para que no salga.
  # return(list(corecenter=mu,tiempo=tiempo))

  # Solo devolvemos los puntos si nos los piden.
  if (getpoints == FALSE) {
    return(mu)
  }
  return(list(corecenter=mu,points=pointsn))

}
