#' @title Marginal contributions vector
#' @description Given a game and a permutation, this function returns the corresponding marginal contributions vector.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param permutation Position of the permutation in the Lehmer code order, as an integer.
#' @return The vector of marginal contributions.
#' @details Given a game \eqn{v\in G^N} and an order \eqn{\pi} of the players in \eqn{N},
#' the marginal contributions associated with order \eqn{\pi} is defined, for all \eqn{i \in N}, as
#' \eqn{m_i^{\pi}=v(Pre^{\pi}(i)\cap i)-v(Pre^{\pi}(i))}, being \eqn{Pre^{\pi}(i)=\{j:\pi(j)<\pi(i)\}}.
#' @examples
#' n <- 3
#' v <- c(1, 5, 10, 30, 60, 90, 200)
#' for (i in 1:factorial(n)) {
#'   cat("[", i, "]", paste(getpermutation(3,i)),"  ",
#'   paste(marginalvector(v,binary=FALSE,i)), "\n")
#' }
#' @seealso \link{getpermutation}, \link{getpermutationnumber}
#' @export
marginalvector <- function(v, binary = FALSE, permutation) {
  nC <- length(v)
  n <- log(nC + 1)/log(2)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (permutation < 0 || permutation%%1 != 0) {
    stop("'permutation' must be a positive integer.")
  }
  if (permutation > factorial(n)) {
    stop("'permutation' must not be greater than the factorial of its length.")
  }
  if (binary == FALSE) {
    v <- lex2bin(v)
  }
  marginal <- rep(0, n)
  permutation <- getpermutation(n, permutation)
  marginal[permutation[1]] <- v[2^(permutation[1] - 1)]
  for (ii in 2:n){ ## Cambiei neste bucle
    S <- permutation[1:(ii - 1)] ## Cambiei aquí, eliminando os wich
    SUi <- permutation[1:(ii)] ## Cambiei aquí, eliminando os wich
    marginal[permutation[ii]] <- v[sum(2^(SUi - 1))] - v[sum(2^(S - 1))] ## Cambiei aquí
  }
  return(marginal)
}
