#' Convert a Linear Model (lm) Object to Anytime-Valid lm (avlm)
#'
#' Converts an \code{lm} object into an anytime-valid version by storing a precision parameter
#' \code{g} as an attribute and updating the object's class. The resulting object supports anytime-valid inference.
#'
#' @param model An \code{lm} object from a linear model fit.
#' @param g An integer precision parameter for anytime-valid inference. Default is 1.
#' @param vcov_estimator Optional character string specifying the type of robust standard errors
#' to use. Must be one of \code{"HC0"}, \code{"HC1"}, \code{"HC2"}, or \code{"HC3"}.
#' If \code{NULL} (default), no robust variance estimation is applied.
#' @param ... Additional arguments passed to or from other methods.
#'
#' @return An enhanced \code{lm} object of class \code{avlm} with anytime-valid inference capabilities.
#'
#' @examples
#' # Fit a linear model on the built-in mtcars dataset
#' fit <- lm(mpg ~ wt + hp, data = mtcars)
#'
#' # Convert the standard lm object to an anytime-valid avlm object
#' av_fit <- av(fit, g = 1)
#'
#' # Print the summary of the anytime-valid model
#' summary(av_fit)
#'
#' @export
av.lm <- function(model, g = 1, vcov_estimator = NULL, ...) {
  attr(model, "g") <- g
  
  if (!is.null(vcov_estimator)) {
    # Validate choice
    valid_types <- c("HC0", "HC1", "HC2", "HC3")
    vcov_estimator <- match.arg(vcov_estimator, choices = valid_types)
  }
  attr(model, "vcov_estimator") <- vcov_estimator
  class(model) <- c("avlm", class(model))
  
  
  return(model)
}

#' Summary Method for Anytime-Valid lm (avlm) Objects
#'
#' Computes a summary for an \code{avlm} object, a linear model enhanced with anytime-valid inference.
#' In addition to the standard \code{lm} summary statistics, the p-values for the coefficient tests are
#' recalculated using an anytime-valid method.
#'
#' @param object An \code{avlm} object generated by \code{av.lm}.
#' @param ... Additional arguments passed to or from other methods.
#'
#' @return A summary object of class \code{summary.avlm} that includes updated p-values for the coefficients.
#' @export
summary.avlm <- function(object, ...) {
  # First get the regular lm summary
  summ <- NextMethod()
  
  # Change the class to include summary.avlm first 
  class(summ) <- c("summary.avlm", "summary.lm")
  
  # Add any additional elements specific to avlm
  g <- attr(object, "g")
  attr(summ, "g") <- g
  number_of_coefficients = dim(summ$cov.unscaled)[1]
  e = residuals(summ)
  n = length(e)
  p = number_of_coefficients - 1
  d = 1
  nu = n - p - d
  
  
  vcov_estimator = attr(object, "vcov_estimator")
  if(is.null(vcov_estimator)){
    t = summ$coefficients[, 't value']
    t2 = t ^ 2
    log_G_t_values = log_G_t(t2, nu, n, g)
    summ$coefficients[,4] = p_G_t(log_G_t_values)
  }else{
    h <- hatvalues(object)
    weights <- switch(vcov_estimator,
                      "HC0" = e^2,
                      "HC1" = e^2 * (n/(n-number_of_coefficients)),
                      "HC2" = e^2 / (1-h),
                      "HC3" = e^2 / (1-h)^2,
                      stop("Unsupported vcov estimator type")
    )
    X <- model.matrix(object)
    XV_hatX <- crossprod(X,  X * weights)
    U <- chol(solve(XV_hatX))
    XX <- crossprod(X)
    X_star <- U %*% XX
    
    beta_hat <- object$coefficients
    asymp_precision = crossprod(X_star)
    intercept_index <- which(names(beta_hat) == "(Intercept)")
    beta_hat_no_intercept <- beta_hat[-intercept_index]
    asymp_precision_no_intercept <- asymp_precision[-intercept_index, -intercept_index]
    fstatistic = t(beta_hat_no_intercept) %*% asymp_precision_no_intercept %*% beta_hat_no_intercept
    summ$fstatistic[1] = fstatistic
    

    asymp_covariance = solve(asymp_precision)
    std_errs <- sqrt(diag(asymp_covariance))
    t<- as.vector(beta_hat / std_errs)
    t2 = t ^ 2
    log_G_t_values = log_G_t(t2, nu, n, g)
    summ$coefficients[,2] = std_errs
    summ$coefficients[,3] = t
    summ$coefficients[,4] = p_G_t(log_G_t_values)
    
  }
  
  attr(summ, "vcov_estimator") = vcov_estimator
  return(summ)
}


#' Print Method for summary.avlm Objects
#'
#' Custom print method for \code{summary.avlm} objects that mimics the standard \code{summary.lm} output,
#' but replaces p-value column headers and includes anytime-valid annotations.
#' @param x An object of class \code{summary.avlm} containing the summary of an anytime-valid lm object.
#' @param digits The number of significant digits to use when printing. Defaults based on system options.
#' @param signif.stars Logical indicating whether significance stars should be printed.
#' @param ... Additional arguments passed to or from other methods.
#'
#' @return Invisibly returns the summary object.
#' @export
print.summary.avlm <- function(x, digits = max(3L, getOption("digits") - 3L), 
                               signif.stars = getOption("show.signif.stars"), ...) {
  # Don't call NextMethod() - we'll implement our own printing
  # that closely follows the standard summary.lm printing
  
  cat("\nCall:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"), "\n\n", sep = "")
  
  # Print residuals
  resid <- x$residuals
  cat("Residuals:\n")
  if (length(resid) > 5) {
    rq <- quantile(resid)
    names(rq) <- c("Min", "1Q", "Median", "3Q", "Max")
    print(rq, digits = digits)
  } else {
    print(resid, digits = digits)
  }
  
  # Print coefficients
  cat("\nCoefficients:\n")
  regression_table <- capture.output({
    printCoefmat(x$coefficients, digits = digits, signif.stars = signif.stars, ...)
  })
  # Replace the p-value column header text
  output <- sub("Pr\\(>\\|t\\|\\)", "p value", regression_table)
  
  # Print the modified output
  cat(paste(output, collapse = "\n"))
  
  vcov_estimator = attr(x, "vcov_estimator")
  # Print residual standard error
  if(is.null(vcov_estimator)){
    cat("\nResidual standard error:", format(x$sigma, digits = digits), 
        "on", x$df[2], "degrees of freedom")
  }

  # Print NA handling message if needed
  if (nzchar(mess <- naprint(x$na.action))) 
    cat("\n (", mess, ")\n", sep = "")
  else cat("\n")
  
  # Print R-squared and F-statistic with custom p-value
  if (!is.null(x$fstatistic)) {
    cat("Multiple R-squared:  ", formatC(x$r.squared, digits = digits),
        ",\tAdjusted R-squared:  ", formatC(x$adj.r.squared, digits = digits), 
        "\n", sep = "")

    # Calculate custom p-value for F-statistic
    g <- attr(x, "g")
    f <- x$fstatistic[1]
    n = length(x$residuals)
    d = x$fstatistic[2]
    nu <- x$fstatistic[3] #n - p - d
    log_G_f_value = log_G_f(f, d, nu, n, g)
    f_spvalue = p_G_f(log_G_f_value)
    
    # Print F-statistic with custom p-value
    cat("F-statistic:", formatC(f, digits = digits), 
        "on", d, "and", nu, "DF,  p-value:", 
        format.pval(f_spvalue, digits = digits), "\n")
  }
  
  cat("Anytime-Valid: TRUE\n")
  if(!is.null(vcov_estimator)){
    cat("Robust Standard Error Type:", vcov_estimator, "\n")
  }

  invisible(x)
}


#' Confidence Intervals for Anytime-Valid lm (avlm) Objects
#'
#' Computes confidence intervals for the coefficients of an \code{avlm} object by adjusting
#' the standard errors using the precision parameter \code{g} and an anytime-valid approach.
#'
#' @param object An \code{avlm} object.
#' @param parm A specification of which parameters are to be given confidence intervals; can be a vector of numbers or names.
#' @param level The confidence level required for the intervals. Defaults to 0.95.
#' @param ... Additional arguments passed to or from other methods.
#'
#' @return A matrix with the lower and upper confidence limits for the specified parameters.
#' 
#' @examples
#' # Fit a simple linear model using the mtcars dataset
#' fit <- lm(mpg ~ wt + hp, data = mtcars)
#'
#' # Convert the standard lm object to an anytime-valid avlm object with precision parameter g = 1
#' av_fit <- av(fit, g = 1)
#'
#' # Calculate and print confidence intervals for the coefficients
#' conf_intervals <- confint(av_fit)
#' print(conf_intervals)
#' 
#' @export
confint.avlm <- function(object, parm, level = 0.95, ...) {
  # Get the standard confidence intervals from lm
  # We'll use the standard method as a starting point
  orig_confint <- NextMethod()
  
  # Extract needed components for custom CI calculation
  coefs <- coef(object)
  se <- summary(object)$coefficients[, 2]
  g <- attr(object, "g")  # The avlm-specific parameter
  n <- length(object$residuals)
  number_of_coefficients = length(se)
  nu = n - number_of_coefficients
  
  # Parameter selection (same as in the standard method)
  if (missing(parm)) {
    parm <- names(coefs)
  } else if (is.numeric(parm)) {
    parm <- names(coefs)[parm]
  }
  
  # Calculate custom confidence intervals that incorporate g
  alpha <- 1 - level
  t_rad <- t_radius(g, n, number_of_coefficients, alpha)

  
  # Create the CI matrix
  custom_ci <- matrix(NA, length(parm), 2)
  rownames(custom_ci) <- parm
  colnames(custom_ci) <- c(paste(format(100 * alpha/2, digits = 3), "%"), 
                           paste(format(100 * (1 - alpha/2), digits = 3), "%"))
  
  # Fill in the confidence intervals
  for (i in seq_along(parm)) {
    param_name <- parm[i]
    if (param_name %in% names(coefs)) {
      idx <- which(names(coefs) == param_name)
      custom_ci[i, 1] <- coefs[idx] - t_rad * se[idx]
      custom_ci[i, 2] <- coefs[idx] + t_rad * se[idx]
    }
  }
  
  return(custom_ci)
}