% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design.R
\name{design}
\alias{design}
\title{Create a complete experimental design with graph of design layout and skeletal ANOVA table}
\usage{
design(
  type,
  treatments,
  reps,
  nrows,
  ncols,
  brows = NA,
  bcols = NA,
  byrow = TRUE,
  sub_treatments = NULL,
  fac.names = NULL,
  fac.sep = c("", " "),
  buffer = NULL,
  plot = TRUE,
  rotation = 0,
  size = 4,
  margin = FALSE,
  save = FALSE,
  savename = paste0(type, "_design"),
  plottype = "pdf",
  seed = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{type}{The design type. One of \code{crd}, \code{rcbd}, \code{lsd}, \code{split}, \code{strip}, or a crossed factorial specified as \verb{crossed:<base>} where \verb{<base>} is one of \code{crd}, \code{rcbd}, or \code{lsd}.}

\item{treatments}{A vector containing the treatment names or labels. For split-plot designs, these treatments are applied to whole-plots. For strip-plot designs, these treatments are applied to row-strips (entire rows within each block receive the same treatment).}

\item{reps}{The number of replicates. Ignored for Latin Square Designs.}

\item{nrows}{The number of rows in the design.}

\item{ncols}{The number of columns in the design.}

\item{brows}{For RCBD, split-plot and strip-plot designs. The number of rows in a block.}

\item{bcols}{For RCBD, split-plot and strip-plot designs. The number of columns in a block.}

\item{byrow}{For split-plot and strip-plot designs. Logical (default \code{TRUE}). Controls the within-block arrangement when there are multiple valid layouts.}

\item{sub_treatments}{A vector of treatments for the sub-plot factor (required for \code{split} and \code{strip}). For strip-plot designs, these treatments are applied to column-strips (entire columns within each block receive the same treatment). To apply treatments to columns instead of rows, swap the \code{treatments} and \code{sub_treatments} arguments.}

\item{fac.names}{Allows renaming of the \code{A} level of factorial designs by passing (optionally named) vectors of new labels to be applied to the factors within a list. See examples and details for more information.}

\item{fac.sep}{The separator used by \code{fac.names}. Used to combine factorial design levels. If a vector of 2 levels is supplied, the first separates factor levels and label, and the second separates the different factors.}

\item{buffer}{A string specifying the buffer plots to include for plotting. Default is \code{NULL} (no buffers plotted). Other options are "edge" (outer edge of trial area), "rows" (between rows), "columns" (between columns), "double row" (a buffer row each side of a treatment row) or "double column" (a buffer row each side of a treatment column). "blocks" (a buffer around each treatment block) will be implemented in a future release.}

\item{plot}{Logical (default \code{TRUE}). If \code{TRUE}, display a plot of the generated design. A plot can always be produced later using \code{\link[=autoplot]{autoplot()}}.}

\item{rotation}{Rotate the text output as Treatments within the plot. Allows for easier reading of long treatment labels. Takes positive and negative values being number of degrees of rotation from horizontal.}

\item{size}{Increase or decrease the text size within the plot for treatment labels. Numeric with default value of 4.}

\item{margin}{Logical (default \code{FALSE}). Expand the plot to the edges of the plotting area i.e. remove white space between plot and axes.}

\item{save}{One of \code{FALSE} (default)/\code{"none"}, \code{TRUE}/\code{"both"}, \code{"plot"} or \code{"workbook"}. Specifies which output to save.}

\item{savename}{A file name for the design to be saved to. Default is the type of the design combined with "_design".}

\item{plottype}{The type of file to save the plot as. Usually one of \code{"pdf"}, \code{"png"}, or \code{"jpg"}. See \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}} for all possible options.}

\item{seed}{Logical (default \code{TRUE}). If \code{TRUE}, return the seed used to generate the design. If a numeric value, use that value as the seed for the design.}

\item{quiet}{Logical (default \code{FALSE}). Hide the output.}

\item{...}{Additional parameters passed to \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}} for saving the plot.}
}
\value{
A list containing a data frame with the complete design (\verb{$design}), a ggplot object with plot layout (\verb{$plot.des}), the seed (\verb{$seed}, if \code{return.seed = TRUE}), and the \code{satab} object (\verb{$satab}), allowing repeat output of the \code{satab} table via \code{cat(output$satab)}.
}
\description{
Create a complete experimental design with graph of design layout and skeletal ANOVA table
}
\details{
Supported designs are Completely Randomised (\code{crd}), Randomised Complete Block (\code{rcbd}), Latin Square (\code{lsd}), split-plot (\code{split}), strip-plot (\code{strip}), and crossed factorial designs via \verb{crossed:<base>} where \verb{<base>} is \code{crd}, \code{rcbd}, or \code{lsd} (e.g. \code{crossed:crd}).

If \code{save = TRUE} (or \code{"both"}), both the plot and the workbook will be saved to the current working directory, with filename given by \code{savename}. If one of either \code{"plot"} or \code{"workbook"} is specified, only that output is saved. If \code{save = FALSE} (the default, or equivalently \code{"none"}), nothing will be output.

\code{fac.names} can be supplied to provide more intuitive names for factors and their levels in factorial and split plot designs. They can be specified in a list format, for example \code{fac.names = list(A_names = c("a", "b", "c"), B_names = c("x", "y", "z"))}. This will result a design output with a column named \code{A_names} with levels \verb{a, b, c} and another named \code{B_names} with levels \verb{x, y, z}. Labels can also be supplied as a character vector (e.g. \code{c("A", "B")}) which will result in only the treatment column names being renamed. Only the first two elements of the list will be used, except in the case of a 3-way factorial design.

\code{...} allows extra arguments to be passed to \code{ggsave()} for output of the plot. The details of possible arguments can be found in  \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}}.
}
\examples{
# Completely Randomised Design
des.out <- design(type = "crd", treatments = c(1, 5, 10, 20),
                  reps = 5, nrows = 4, ncols = 5, seed = 42)

# Randomised Complete Block Design
des.out <- design("rcbd", treatments = LETTERS[1:11], reps = 4,
                  nrows = 11, ncols = 4, brows = 11, bcols = 1, seed = 42)

# Latin Square Design
# Doesn't require reps argument
des.out <- design(type = "lsd", c("S1", "S2", "S3", "S4"),
                  nrows = 4, ncols = 4, seed = 42)

# Factorial Design (Crossed, Completely Randomised)
des.out <- design(type = "crossed:crd", treatments = c(3, 2),
                  reps = 3, nrows = 6, ncols = 3, seed = 42)

# Factorial Design (Crossed, Completely Randomised), renaming factors
des.out <- design(type = "crossed:crd", treatments = c(3, 2),
                  reps = 3, nrows = 6, ncols = 3, seed = 42,
                  fac.names = list(N = c(50, 100, 150),
                                   Water = c("Irrigated", "Rain-fed")))

# Factorial Design (Crossed, Randomised Complete Block Design),
# changing separation between factors
des.out <- design(type = "crossed:rcbd", treatments = c(3, 2),
                  reps = 3, nrows = 6, ncols = 3,
                  brows = 6, bcols = 1,
                  seed = 42, fac.sep = c(":", "_"))

# Factorial Design (Nested, Latin Square)
trt <- c("A1", "A2", "A3", "A4", "B1", "B2", "B3")
des.out <- design(type = "lsd", treatments = trt,
                  nrows = 7, ncols = 7, seed = 42)

# Split plot design
des.out <- design(type = "split", treatments = c("A", "B"), sub_treatments = 1:4,
                  reps = 4, nrows = 8, ncols = 4, brows = 4, bcols = 2, seed = 42)

# Alternative arrangement of the same design as above
des.out <- design(type = "split", treatments = c("A", "B"), sub_treatments = 1:4,
                  reps = 4, nrows = 8, ncols = 4, brows = 4, bcols = 2,
                  byrow = FALSE, seed = 42)

# Strip plot design
des.out <- design(type = "strip", treatments = c("A", "B", "C"), sub_treatments = 1:4,
                  reps = 4, nrows = 12, ncols = 4, brows = 3, bcols = 4, seed = 42)

}
