# ------------------------------------------------------------------------
# 
# "thetaRB<-function(X,ato)" 
# 
# Compute the Extremal Index for Atomic Markov Chains Using Block Division
#
# ------------------------------------------------------------------------
##

#' @aliases thetaRB
#' @title Compute the Extremal Index for Atomic Markov Chains Using Regenerative
#' Blocks
#' @description This function divides the input dataset into blocks for an 
#' atomic Markov chain and calculates the extremal index using quantile-based thresholds.
#' @param X A numeric vector representing the Markov chain.
#' @param ato A numeric value; the atom of the Markov chain. 
#' @details The function uses \code{GetBlocks} to divide \code{X} into blocks 
#' according to the atom \code{ato}. 
#' High quantiles from \code{X} are generated and used as thresholds to compute 
#' statistics on each block, including sub-maximums 
#' and block indices. These statistics are then used to calculate the extremal index,
#'  a measure of extreme value clustering, across the blocks.
#' @return A numeric vector representing the extremal index at various quantile thresholds.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P. and Clémençon, S. (2006).
#' Regeneration-based statistics for Harris recurrent
#'   Markov chains.  
#' \emph{Lecture notes in Statistics}, vol. \bold{187}, pp. 1-54, Springer.
#' 
#' Bertail, P., Clémençon, S. and Tressou, J. (2009). Extreme values statistics for Markov chains 
#' via the (pseudo-) regenerative method. \emph{Extremes}, \bold{12}, 327–360.
#' 
#' 
#' @seealso \code{\link{GetBlocks}},
#' \code{\link{regenboot}}.
#' @keywords "Regenerative Block" "Markov chains" "Atom" "Extremal index"
#' @export
#' @examples 
#' X = genMM1(1000, 1, 2)
#' thetaB <- thetaRB(X, ato=0)
#' plot(thetaB)


thetaRB<-function(X,ato)
{
  RES=GetBlocks(X, ato, func=max)
  dataset=RES[[1]]
  Block=RES[[2]]
  l_n=RES[[3]]
  
  # Take high level quantiles ( the extremal index is computed as a function of u
  perc=seq(from=0.9,by=0.001,length.out=100)
  u=quantile(X,perc)
  # compute statistics on blocks
  Bnumber=Block[,1]
  lB=Block[,2]
  Submax=Block[,3]
  #First=Block[,4]
  #Submax_1=Block[,5]
  n=length(X)
  n_u=length(u)
  n_b=length(Submax)
  # compute extremal index based on blocks
  thetaB=(colSums((matrix(rep(Submax,n_u),n_b,n_u)>matrix(rep(u,n_b),n_b,n_u,byrow=TRUE)))
          /apply (rbind(matrix(1,1,n_u), colSums((matrix(rep(X,n_u),n,n_u)>matrix(rep(u,n),n,n_u,byrow=TRUE)))),2,max))
  return(thetaB)
}
