#' Names, sets, and class CLI assertions
#'
#' These functions wrap \pkg{checkmate} checks related to names, sets,
#' permutations, choices, and class membership, returning the check result
#' (`check_*_cli()`) or raising a `cli::cli_abort()` error via
#' [make_assertion()] (`assert_*_cli()`).
#' They are useful for validating input names, restricting values to a fixed
#' set, checking set relationships, or ensuring that objects inherit from one
#' or more expected classes, while emitting consistent CLI-styled error
#' messages.
#'
#' @param x Object to be checked.
#'   This is the value whose type, length, or other properties are validated.
#'
#' @param y Second object used in set-relationship checks.
#'   For functions such as `check_set_equal_cli()` and `check_disjunct_cli()`,
#'   this is the object against which `x` is compared (e.g., another vector
#'   of labels or IDs).
#'
#' @param ... Additional arguments passed on to the corresponding
#'   \pkg{checkmate} function (e.g. `lower`, `upper`, `any.missing`,
#'   `all.missing`, `min.len`, `null.ok`).
#'
#' @param .var.name Character scalar used in error messages to refer to
#'   the checked object. Defaults to [checkmate::vname()], which tries to
#'   infer the variable name from the calling context.
#'
#' @param add Optional [checkmate::AssertCollection] to which
#'   assertion failures are added instead of triggering an immediate
#'   error. Defaults to `NULL`, which causes a `cli::cli_abort()` on
#'   failure.
#'
#' - Names:
#'   - [check_names_cli()], [assert_names_cli()] check that object names satisfy
#'     constraints such as being non-missing, unique, or matching a pattern.
#' - Set membership and subsets:
#'   - [check_subset_cli()], [assert_subset_cli()] ensure that all elements of
#'     `x` are contained in `choices`.
#'   - [check_choice_cli()], [assert_choice_cli()] ensure that scalar `x` is
#'     one of a set of allowed values (`choices`).
#'   - [check_permutation_cli()], [assert_permutation_cli()] ensure that `x`
#'     is a permutation of integers `1:n`.
#' - Set relationships:
#'   - [check_set_equal_cli()], [assert_set_equal_cli()] check that two sets
#'     contain the same elements (order ignored).
#'   - [check_disjunct_cli()], [assert_disjunct_cli()] check that two sets are
#'     disjoint.
#' - Class membership:
#'   - [check_class_cli()], [assert_class_cli()] ensure that `x` inherits from
#'     at least one of the given classes.
#'   - [check_multi_class_cli()], [assert_multi_class_cli()] are similar but
#'     designed for objects that may have multiple acceptable classes.
#'
#' In all `assert_*_cli()` functions, `x` is the object being checked and
#' `.var.name` is used only for error message construction; the return value is
#' always `x` (invisibly) on success.
#'
#' @returns
#'
#' - `check_*_cli()` functions return `TRUE` on success or a character vector
#'   describing the failure, exactly like the corresponding
#'   \pkg{checkmate} checks.
#' - `assert_*_cli()` functions return `x` invisibly on success and either:
#'   - raise a `cli::cli_abort()` error with bullet-style messages, or
#'   - push messages into an [checkmate::AssertCollection] if `add` is supplied.
#'
#' @examples
#'
#' ## Names:
#'
#' x <- c(a = "a", b = "b")
#'
#' # Check that all names are non-missing and unique
#' check_names_cli(x, type = "named")
#'
#' y <- c("a", "b")
#' try(check_names_cli(y, type = "named"))        # failure: no names
#'
#' assert_names_cli(x, type = "named")
#' try(assert_names_cli(y, type = "named"))  # cli-styled error
#'
#' ## Subsets and choices:
#'
#' allowed <- c("rf", "xgboost", "nn")
#'
#' # Subset: every element of x must be in allowed
#' check_subset_cli(c("rf", "nn"), choices = allowed)
#' try(check_subset_cli(c("rf", "svm"), choices = allowed))   # failure
#'
#' assert_subset_cli(c("rf", "nn"), choices = allowed)
#' try(assert_subset_cli(c("rf", "svm"), choices = allowed))
#'
#' # Choice: scalar x must be one of allowed
#' check_choice_cli("rf", choices = allowed)
#' try(check_choice_cli("svm", choices = allowed))            # failure
#'
#' assert_choice_cli("xgboost", choices = allowed)
#' try(assert_choice_cli("svm", choices = allowed))
#'
#' ## Permutations:
#'
#' # Valid permutation of 1:5
#' check_permutation_cli(5:1, 1:5)
#'
#' # Not a permutation: duplicates or missing values
#' try(check_permutation_cli(c(1, 2, 2, 4, 5), 1:5))              # failure
#'
#' assert_permutation_cli(c(1, 2, 3, 4), 1:4)
#' try(assert_permutation_cli(c(1, 3, 4, 4), 1:4))
#'
#' ## Set relationships:
#'
#' a <- c("sp1", "sp2", "sp3")
#' b <- c("sp3", "sp2", "sp1")
#' c <- c("sp1", "sp4")
#'
#' # Equality of sets (order ignored)
#' check_set_equal_cli(a, b)
#' try(check_set_equal_cli(a, c))                 # failure
#'
#' assert_set_equal_cli(a, b)
#' try(assert_set_equal_cli(a, c))
#'
#' # Disjointness
#' d <- c("x", "y")
#' e <- c("z", "w")
#'
#' check_disjunct_cli(d, e)
#' try(check_disjunct_cli(d, c("y", "z")))        # failure: "y" is shared
#'
#' assert_disjunct_cli(d, e)
#' try(assert_disjunct_cli(d, c("y", "z")))
#'
#' ## Class and multi-class:
#'
#' df <- data.frame(x = 1:3)
#'
#' # Single/inheritance class checks
#' check_class_cli(df, "data.frame")
#' try(check_class_cli(df, "matrix"))             # failure
#'
#' assert_class_cli(df, "data.frame")
#' try(assert_class_cli(df, "matrix"))
#'
#' # Multi-class: object must inherit from at least one allowed class
#' allowed_classes <- c("data.frame", "tbl_df")
#' check_multi_class_cli(df, allowed_classes)
#'
#' # A plain numeric vector will fail this multi-class check
#' v <- 1:3
#' try(check_multi_class_cli(v, allowed_classes)) # failure
#'
#' assert_multi_class_cli(df, allowed_classes)
#' try(assert_multi_class_cli(v, allowed_classes))
#'
#' @seealso
#'   [checkmate::check_names()], [checkmate::check_subset()],
#'   [checkmate::check_choice()], [checkmate::check_permutation()],
#'   [checkmate::check_set_equal()], [checkmate::check_disjunct()],
#'   [checkmate::check_class()], [checkmate::check_multi_class()],
#'   [make_assertion()], [assert_cli()]
#'
#' @name checkcli-names-sets
#' @family checkCLI
#' @import checkmate
#' @import cli
#' @importFrom glue glue
#' @importFrom purrr map2

#' @rdname checkcli-names-sets
#' @export
check_names_cli <- function(...) {
  checkmate::check_names(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_names_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_names(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_subset_cli <- function(...) {
  checkmate::check_subset(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_subset_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_subset(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_permutation_cli <- function(...) {
  checkmate::check_permutation(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_permutation_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_permutation(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_choice_cli <- function(...) {
  checkmate::check_choice(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_choice_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_choice(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_set_equal_cli <- function(...) {
  checkmate::check_set_equal(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_set_equal_cli <- function(x, y, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_set_equal(x, y, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_disjunct_cli <- function(...) {
  checkmate::check_disjunct(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_disjunct_cli <- function(x, y, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_disjunct(x, y, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-names-sets
#' @export
check_class_cli <- function(...) {
  checkmate::check_class(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_class_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  res <- make_assertion(
    x,
    checkmate::check_class(x, ...),
    .var.name,
    add
  )
  invisible(res)
}

#' @rdname checkcli-names-sets
#' @export
check_multi_class_cli <- function(...) {
  checkmate::check_multi_class(...)
}

#' @rdname checkcli-names-sets
#' @export
assert_multi_class_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_multi_class(x, ...),
      .var.name,
      add
    )
  )
}
