% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cmdensity.R
\name{cmdensity}
\alias{cmdensity}
\title{Multi-class Circular Density Curve}
\usage{
cmdensity(
  funlist,
  funprop = 1,
  radius = 1/sqrt(base::pi),
  area.prop = TRUE,
  total.area = 1,
  n = 500,
  nlabels = 4,
  cols = NULL,
  borders = NULL,
  xlim = NULL,
  ylim = NULL,
  main = NULL,
  type = c("null", "compass", "clock"),
  add = FALSE,
  x.legend = "bottomright",
  y.legend = NULL,
  fill = TRUE,
  lty = 1,
  lwd = 1
)
}
\arguments{
\item{funlist}{a list of functions which can be used to calculate the
density values for each class, evaluated at given points defined by
the first argument of the functions. The set of points is a sequence
from \eqn{0} to \eqn{2\pi}, with length \code{n}.}

\item{funprop}{proportions for functions. It is 1 by default. A user can
choose different proportions for the functions so as to represent
different numbers of observations. If they do not add up to the number
of functions (k), it will be normalised so that \code{sum(classprop) =
k}.}

\item{radius}{the radius of the reference circle.}

\item{area.prop}{logical; if \code{TRUE}, an area-proportional
transformation is applied; if \code{FALSE}, a height-proportional
transformationis applied.}

\item{total.area}{a positive number specifying the total area under all the
density curves. If \code{total.area = NULL}, no scaling is applied, the
plot is in the original scale. If \code{area.prop = TRUE}, the total area 
is automatically unity without scaling.}

\item{n}{the number of points used to plot each density curve.  The
larger the number is, the more accurate the curve is.}

\item{nlabels}{integer, for the number of levels to be plotted; if
\code{0}, no label is plotted.}

\item{cols}{the colors to fill the area under each density curve, with
the same order as the class.}

\item{borders}{the colors of the borders.}

\item{xlim}{numeric vectors of length 2, giving the x coordinates
ranges.}

\item{ylim}{numeric vectors of length 2, giving the y coordinates
ranges.}

\item{main}{the main title (on top)}

\item{type}{the type of circular data, one of the values \code{"null"},
\code{"compass"} or \code{"clock"}.  If \code{"null"}, no special
lables plotted for directions. If \code{"compass"}, the four cardinal
directions are printed inside the reference circle. If \code{"clock"},
labels for 24 hours are printed inside the reference circle.}

\item{add}{logical; if \code{TRUE}, density curves are superimposed to
the current plot, for example, the circular histograms, rose diagrams
and stacked dot plots.}

\item{x.legend}{x coordinate to plot the legend.}

\item{y.legend}{y coordinate to plot the legend.}

\item{fill}{logical. If \code{TRUEt}, fills the regions with colors
under/between the density curves. If \code{FALSE}, only the density
curves are plotted.}

\item{lty}{line width}

\item{lwd}{line width}
}
\value{
No return value
}
\description{
Function \code{cmdensity} can be used to plot 2-dimensional 
  density curves for circular data with multiple classes. The density curves 
  are stacked to avoid any overlap.
}
\examples{
# Load and pre-process the dataset
library(circular)
data("pigeons", package = "circular")
x = pigeons[,2] / 180 * pi                  # bearing
y = pigeons[,1]                             # treatment
vs = split(x, factor(y, unique(y)))    # list of classified value
prop = sapply(vs, length) / length(x)  # proportion of each class

# Define the kde function for each class using von Mises kernels
dvm = function(x, mu=0, kappa=1)  # von Mises density
  exp(kappa * cos(x - mu)) * (2 * pi * besselI(kappa, 0))^(-1)
kdevm = function(x, x0, bw=0.3) 
  rowMeans(outer(x, x0, dvm, 0.5 / (1 - exp(-bw^2 / 2))))
fs = list(function(x) kdevm(x, x0=vs[[1]]),
          function(x) kdevm(x, x0=vs[[2]]),
          function(x) kdevm(x, x0=vs[[3]]))

# stacked density curves for 3 classes
cmdensity(fs)                         # 1:1:1
cmdensity(fs, prop)                   # using proportions for functions

}
\references{
Xu, D. and Wang, Y. (2020). Area-proportional Visualization for
Circular Data. \emph{Journal of Computational and Graphical
Statistics}, \bold{29}, 351-357.
}
\seealso{
\code{\link{cdensity}}, \code{\link{cmhist}}
}
\author{
Danli Xu <dxu452@aucklanduni.ac.nz>, Yong Wang <yongwang@auckland.ac.nz>
}
\concept{stacked circular density curve}
