% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groupFit.R
\name{groupFit}
\alias{groupFit}
\title{groupFit}
\usage{
groupFit(breaks, counts, data, discrete, distr, N, params, initials,
  pfixed, bootstrap = FALSE, numLoops = 5000, known = FALSE,
  pave = FALSE, imhof = TRUE)
}
\arguments{
\item{breaks}{Vector defining the breaks in each group.}

\item{counts}{Vector containing the frequency of counts in each group.}

\item{data}{Vector containing values of discrete random variables.}

\item{discrete}{Logical. Is the distribution discrete?}

\item{distr}{Character; the name of the distribution users want to fit the data to.

   Included continuous distributions are: "exp", "gamma", "lnorm", "norm", "unif" and "weibull".

   Included discrete distributions are: "binom" and "pois".

   User defined distributions are supported as "user".

   Short-hand or full spelling of these distributions will be recognized, case insensitive.}

\item{N}{Number of trials, used only for the binomial distribution.}

\item{params}{Vector of distribution parameters. This is only required when known == TRUE. groupFit will estimate the parameters if known == FALSE.}

\item{initials}{Vector of distribution parameters to use as starting points for calculating MLEs.}

\item{pfixed}{Vector of known probabilities for corresponding counts vector. pfixed must be provided when distr = "user".}

\item{bootstrap}{Logical. Should p-values be calculated via bootstrapping?}

\item{numLoops}{Number of Bootstrap iterations. Set to be 5000 by default.}

\item{known}{Logical. Set to be TRUE if the parameters are known and do not need to be estimated.}

\item{pave}{Logical. Set to be FALSE if the probabilities in groups are used; set to TRUE if the average of probabilities of groups j and j + 1 are used. See page 2 of Spinelli (2001) for more details.}

\item{imhof}{Logical. Set to be TRUE if Imhof's method from the package "CompQuadForm" is used to approximate the null distributions.}
}
\value{
List containing the components:

\item{estimates}{The estimated parameters of the distribution}

\item{stats}{Data frame containing the goodness of fit statistics}

\item{pvals}{Data frame containing p-values for the goodness of fit statistics}
}
\description{
Fits grouped continuous data or discrete data to a distribution and computes Cramer-Von Mises Goodness of Fit statistics and p-values.
}
\details{
For grouped continuous data: call groupFit with arguments breaks, counts, and distr to fit the data.

   For discrete data call groupFit with data and distr to fit the data. If distr = "binom", then be sure to call groupFit with N as well.

   Provide initials to suggest starting points for parameter estimation.

   Provide params and set known = TRUE to test goodness of fit when parameters are known.

   Set bootstrap = TRUE to use bootstrapping to estimate p-values rather than using asymptotic distributions.

   Set imhof = FALSE when \eqn{a+bX^2_p} is used to approximate the null distributions. See page 5 of Spinelli (2001) for details.

   groupFit can test the fit of user defined discrete distributions. To do so, set distr to "user", and provide the vector pfixed, where each cell contains the probability corresponding to that same cell in counts.
}
\examples{

#Poisson Example (Spinelli 1994) p36

counts <- c(9, 22, 6, 2, 1, 0, 0, 0, 0)
vals <- 0:8
data <- rep(vals, counts)

groupFit(data = data, distr = "pois")


# When the parameters are unknown
#(Spinelli 1994) p56

counts <- c(57, 203, 383, 525, 532, 408, 273, 139, 45, 27, 10, 4, 0, 1, 1)
vals <- 0:14
data <- rep(vals, counts)

(pois_fit <- groupFit(data = data, distr = "pois"))

#Binomial example when the parameter is unknown
#Spinelli (1994) P92.
N=12
counts= c(185, 1149, 3265, 5475, 6114, 5194,
          3067, 1331, 403, 105, 14, 4, 0)
vals <- 0:12
data <- rep(vals, counts)

(binom_fit <- groupFit(data = data, N = N, distr = "binom"))

#When the parameter is assumed known and is equal to 1/3

(binom_fit <- groupFit(data = data, N = N, distr = "binom", params = 1/3, known = TRUE))

#uniform example (Choulakian, Lockhart and Stephens(1994) Example 2, p8)

counts <- c(10, 19, 18, 15, 11, 13, 7, 10, 13, 23, 15, 22)

(uni_fit <- groupFit(0:12, counts, distr = "unif"))


#uniform example (Choulakian, Lockhart and Stephens(1994) Example 3, p8)
counts <- c(1, 4, 11, 4, 0)
probability <- c(0.05, 0.3, 0.3, 0.3, 0.05)
breaks <- c(0, cumsum(probability))

#with bootstrapping
(uni_fit1 <- groupFit(breaks, counts, distr = "unif", bootstrap = TRUE, numLoops = 500))

#without bootstrapping
(uni_fit2 <- groupFit(breaks, counts, distr = "unif", bootstrap = FALSE))

#exponential example (Spinelli 2001)

breaks <- c(0, 2, 6, 10, 14, 18, 22, 26)
counts <- c(21, 9, 5, 2, 1, 1, 0)

(exp_fit <- groupFit(breaks, counts, distr = "exp", pave = TRUE))


#Example Sun, Stephens & Spinelli (2012) set 2.
breaks <- c(0, 2, 6, 10, 14, 18, 22, 26)
counts <- c(21, 9, 5, 2, 1, 1, 0)
breaks[1] <- 1e-6
breaks[8] <- Inf

(weibull_fit <- groupFit(breaks, counts, distr = "Weibull"))


#Example Sun, Stephens & Spinelli (2012) set 3.

breaks <- c(0, seq(0.5, 6.5, 1) )
counts <- c(32, 12, 3, 6, 0, 0, 1)
breaks[1] <- 1e-6
breaks[8] <- Inf

(weibull_fit <- groupFit(breaks, counts, distr = "Weibull"))


#Example Sun, Stephens & Spinelli (2012) set 3.
breaks <- c(0, 2, 6, 10, 14, 18, 22, 26)
counts <- c(21, 9, 5, 2, 1, 1, 0)
breaks[1] <- 1e-6
breaks[8] <- Inf

(gamma_fit <- groupFit(breaks, counts, distr = "exp"))


#Example Sun, Stephens & Spinelli (2012) set 3.
breaks <- c(0, seq(0.5, 6.5, 1) )
counts <- c(32, 12, 3, 6, 0, 0, 1)
breaks[1] <- 1e-6
breaks[8] <- Inf

(gamma_fit <- groupFit(breaks, counts, distr = "gamma"))


#More examples

breaks <- c(0, seq(0.5, 6.5, 1) )
counts <- table(cut(rgamma(100, 3, 1/3), breaks))
breaks[8] <- Inf

#setting pave to true
(exp_fit <- groupFit(breaks, counts, distr = "exp", initials = 0.2, pave = TRUE))

#Setting known to true, with params
(gamma_fit <- groupFit(breaks, counts, distr = "gamma",
                      params = c(3, 1/3), known = TRUE))

#with bootstrapping, specifying the number of loops.
(lnorm_fit <- groupFit(breaks, counts, distr = "lnorm",
                      bootstrap = TRUE, numLoops = 1000))

#fitting with both pave and imhof set to false
#by setting imhof to false, we use a+bX^2_p to approximate
#the distribution of the goodness-of-fit Statistics
(weibull_fit <- groupFit(breaks, counts, distr = "weibull",
                        pave = TRUE, imhof = FALSE))

#Using the user defined distribution to test for Benford's law

#genomic data, Lesperance et al (2016)

genomic <- c(48, 14, 12, 6, 18, 5, 7, 8, 9)
phat <- log10(1+1/1:9)

(fit <- groupFit(counts = genomic, distr = "user", pfixed = phat, imhof = FALSE, pave = TRUE))

}
\references{
V. Choulakian, R. A. Lockhart & M. A. Stephens (1994). Cramer-vonMises statistics for discrete distributions.
The Canadian Journal of Statistics,2 2,125-137.

J.J.Spinelli (2001). Testing fit for the grouped exponential distribution.
The Canadian Journal of Statistics,29,451-458.

J.J. Spinelli (1994). Cramer-vonMises statistics for discrete distributions.
Phd thesis. Simon Fraser University, Burnaby, Canada.

S. Z. Sun, J.J. Spinelli & M. A. Stephens (2012).  Testing fit for the grouped gamma and weibull distributions.
Technical report. Simon Fraser University, Burnaby, Canada.
}
\seealso{
\code{\link{distrFit}}: Parameter estimation function
}
\author{
Shaun Zheng Sun and Dillon Duncan
}
