% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dd_calc.R
\name{dd_calc}
\alias{dd_calc}
\alias{dd_simp_avg}
\alias{dd_sng_tri}
\alias{dd_sng_sine}
\alias{dd_dbl_tri}
\alias{dd_dbl_sine}
\title{Estimate degree days from daily data}
\usage{
dd_calc(
  daily_min,
  daily_max,
  nextday_min = daily_min,
  thresh_low = NULL,
  thresh_up = NULL,
  method = c("sng_tri", "dbl_tri", "sng_sine", "dbl_sine", "simp_avg")[0],
  cutoff = c("horizontal", "vertical", "intermediate")[1],
  digits = 2,
  cumulative = FALSE,
  no_neg = TRUE,
  simp_avg_zero_method = 1,
  interpolate_na = FALSE,
  quiet = FALSE,
  debug = FALSE
)

dd_simp_avg(
  daily_min,
  daily_max,
  thresh_low,
  thresh_up = NULL,
  simp_avg_zero_method = 1,
  digits = 2,
  cumulative = FALSE,
  quiet = FALSE
)

dd_sng_tri(
  daily_min,
  daily_max,
  thresh_low = NULL,
  thresh_up = NULL,
  cutoff = c("horizontal", "vertical", "intermediate")[1],
  digits = 2,
  cumulative = FALSE,
  quiet = FALSE
)

dd_sng_sine(
  daily_min,
  daily_max,
  thresh_low = NULL,
  thresh_up = NULL,
  cutoff = c("horizontal", "vertical", "intermediate")[1],
  digits = 2,
  cumulative = FALSE,
  quiet = FALSE
)

dd_dbl_tri(
  daily_min,
  daily_max,
  nextday_min = daily_min,
  thresh_low = NULL,
  thresh_up = NULL,
  cutoff = c("horizontal", "vertical", "intermediate")[1],
  digits = 2,
  cumulative = FALSE,
  quiet = FALSE
)

dd_dbl_sine(
  daily_min,
  daily_max,
  nextday_min = daily_min,
  thresh_low = NULL,
  thresh_up = NULL,
  cutoff = c("horizontal", "vertical", "intermediate")[1],
  digits = 2,
  cumulative = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{daily_min}{Daily minimum temperature}

\item{daily_max}{Daily maximum temperature}

\item{nextday_min}{Minimum temp the day after}

\item{thresh_low}{Lower development threshold temperature}

\item{thresh_up}{Upper development threshold temperature}

\item{method}{Estimation method}

\item{cutoff}{Estimation cutoff method}

\item{digits}{Number of decimal places to round results to}

\item{cumulative}{Return cumulative values}

\item{no_neg}{Set negative values to zero}

\item{simp_avg_zero_method}{How to handle temperatures in the simple average method that fall outside the upper and lower thresholds (see details)}

\item{interpolate_na}{Interpolate missing values, logical}

\item{quiet}{Suppress messages, logical}

\item{debug}{Show additional messages}
}
\value{
A vector of estimated degree day values (either daily or cumulative, depending on the value of \code{cumulative})
}
\description{
Estimate degree days from daily min and max temperature
}
\details{
Units for \code{daily_min}, \code{daily_max}, \code{thresh_low}, and \code{thresh_up} should all be the same
(i.e., all Fahrenheit or all Celsius). The function does not check for unit consistency.

\code{nextday_min} is required for the double-triangle and the double-sine methods. These methods use the minimum temperature
of the following day to model temperatures in the 2nd half of the day. If omitted or NA, the daily minimum temperature will be
substituted.

\code{no_neg = TRUE} sets negative values to zero. This is generally preferred when using degree days to predict the timing of
 development milestones, if one assumes that growth can not go backwards.

The simple average method is taken from McMaster and Wilhelm (\href{https://digitalcommons.unl.edu/cgi/viewcontent.cgi?article=1086&context=usdaarsfacpub}{1997}). This method requires passing a lower threshold (also called the base temp). There are two ways of handling temperatures that fall below the base temperature. Most studies and applications use the default method (\code{simp_avg_zero_method = 1}) which simply 'zeroes out' average daily temperatures that fall below the base temp. Some studies (notably corn) use method 2, which truncates the daily minimum and maximum temperature before computing the simple average. Method 2 also allows you to pass an upper threshold. For details, see McMaster and Wilhelm.

Missing values (NAs) in the temperatures will result in NA degree days. If \code{interpolate_na = TRUE}, missing degree days will
be interpolated. NAs in the middle of the series will be linearly interpolated, and NAs at the ends will be filled with the
adjacent values.
}
\section{Functions}{
\itemize{
\item \code{dd_simp_avg()}: Estimate degree days using the simple avg method

\item \code{dd_sng_tri()}: Estimate degree days using the single-triangle method

\item \code{dd_sng_sine()}: Estimate degree days using the single-sine method

\item \code{dd_dbl_tri()}: Estimate degree days using the double-triangle method

\item \code{dd_dbl_sine()}: Estimate degree days using the double-sine method

}}
\examples{
daily_temps <- system.file("extdata/espartoa-weather-2020.csv", package = "degday") \%>\%
  read.csv() \%>\%
    dplyr::mutate(date = as.Date(date)) \%>\%
    dplyr::slice(1:10)
daily_temps
## Simple average method
dd_simp_avg(daily_min = daily_temps$tmin,
            daily_max = daily_temps$tmax,
            thresh_low = 55)
## Single sine method
dd_sng_sine(daily_min = daily_temps$tmin, daily_max = daily_temps$tmax,
            thresh_low = 55, thresh_up = 93.9)
## Single triangle method
dd_sng_tri(daily_min = daily_temps$tmin, daily_max = daily_temps$tmax,
           thresh_low = 55, thresh_up = 93.9)
## Add next day min temp as an additional column
daily_temps_plus_tmin_next <- daily_temps \%>\% dplyr::mutate(tmin_next = dplyr::lead(tmin, n = 1))
daily_temps_plus_tmin_next
## Double-triangle method
dd_dbl_tri(daily_min = daily_temps_plus_tmin_next$tmin,
           daily_max = daily_temps_plus_tmin_next$tmax,
           nextday_min = daily_temps_plus_tmin_next$tmin_next,
           thresh_low = 55, thresh_up = 93.9)

}
