% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/histo.R
\name{histo}
\alias{histo}
\title{Histogram with Added Options}
\usage{
histo(x, dis = "none", dis_shift = NULL, integer_breaks = NULL,
  colors = rep("black", length(dis)), lty = 1:length(dis),
  legend_form = ifelse(length(dis) == 1, 0, 1), aic_decimals = 1,
  points_list = NULL, axis_list = NULL, legend_list = NULL, ...)
}
\arguments{
\item{x}{Numeric vector of values.}

\item{dis}{Character vector indicating which distributions should be used to
add fitted PDF/PMF to the histogram. If not \code{"none"}, choices for each
element are:

\code{"beta"}

\code{"binom"} (must specify \code{size})

\code{"cauchy"}

\code{"chisq"}

\code{"exp"}

\code{"f"}

\code{"gamma"}

\code{"geom"}

\code{"hyper"} (must specify total number of balls in urn, \code{N}, and
number of balls drawn each time, \code{k})

\code{"lnorm"}

\code{"nbinom"} (must specify \code{size})

\code{"norm"}

\code{"pois"},

\code{"t"}

\code{"unif"}

\code{"weibull"}}

\item{dis_shift}{Numeric value for shifting the fitted PDF/PMF along the
x-axis of the histogram.}

\item{integer_breaks}{If \code{TRUE}, integers covering the range of \code{x}
are used for breaks, so there is one bin for each integer. Useful for
discrete distributions that don't take on too many unique values.}

\item{colors}{Character vector of colors for each PDF/PMF.}

\item{lty}{Integer vector specifying line types for each curve.}

\item{legend_form}{Integer value controlling what type of legend to include.
Choices are 0 for no legend, 1 for legend naming each distribution, and 2 for
legend naming each distribution and the corresponding AIC.}

\item{aic_decimals}{Integer value for number of decimals for AIC.}

\item{points_list}{Optional list of inputs to pass to
\code{\link[graphics]{points}} function, which is used to add the fitted
PDF/PMF.}

\item{axis_list}{Optional list of inputs to pass to
\code{\link[graphics]{axis}}.}

\item{legend_list}{Optional list of inputs to pass to 
\code{\link[graphics]{legend}}.}

\item{...}{May include arguments to pass to \code{\link[graphics]{hist}}
and/or parameter values needed for certain distributions (\code{size} if
\code{dis = "binom"} or \code{dis = "nbinom"}, \code{N} and \code{k} if
\code{dis = "hyper"}).}
}
\value{
Histogram with fitted PDFs/PMFs if requested.
}
\description{
Similar to base R function \code{\link[graphics]{hist}}, but with two added
features: (1) Can overlay one or more fitted probability density/mass
functions (PDFs/PMFs) for any univariate distribution supported in R (see
\code{\link[stats]{Distributions}}); and (2) Can generate more of a barplot
type histogram, where each possible value gets its own bin centered over its
value (useful for discrete variables with not too many possible values).
}
\details{
When \code{x} takes on whole numbers, you typically want to set
\code{dis_shift = -0.5} if \code{right = TRUE}
(\code{\link[graphics]{hist}}'s default) and \code{dis_shift = 0.5} if
\code{right = FALSE}. The function will do this internally by default.

To illustrate, suppose a particular bin represents \code{(7, 10]}. Its
midpoint will be at \code{x = 8.5} on the graph. But if input values are
whole numbers, this bin really only includes values of 8, 9, and 10, which
have a mean of 9. So you really want \code{f(9)} to appear at \code{x = 8.5}.
This requires shifting the curve to the left 0.5 units, i.e. setting
\code{dis_shift = -0.5}.

When \code{x} takes on whole numbers with not too many unique values, you may
want the histogram to show one bin for each integer. You can do this by
setting \code{integer_breaks = TRUE}. By default, the function sets
\code{integer_breaks = TRUE} if \code{x} contains whole numbers with 10 or
fewer unique values.
}
\examples{
# Sample 10,000 Poisson(2) values and commpare default hist vs. histo
set.seed(123)
x <- rpois(n = 10000, lambda = 2)
par(mfrow = c(1, 2))
hist(x, main = "hist function")
histo(x, main = "histo function")

# Sample 10,000 lognormal(0, 0.35) values. Create histogram with curves
# showing fitted lognormal, normal, and Gamma PDFs.
set.seed(123)
x <- rlnorm(n = 10000, meanlog = 0, sdlog = 0.35)
par(mfrow = c(1, 1))
histo(x, c("lnorm", "norm", "gamma"), main = "X ~ Lognormal(0, 0.35)")

# Generate 10,000 Binomial(8, 0.25) values. Create histogram, specifying
# size = 5, with blue line/points showing fitted PMF.
set.seed(123)
x <- rbinom(n = 10000, size = 5, prob = 0.25)
par(mfrow = c(1, 1))
histo(x, dis = "binom", size = 5, colors = "blue", 
      points_list = list(type = "b"))

}
