#' Calculate Vulnerability Scores Using Expert-Based or Model-Derived Overall Exposure
#' and Sensitivity (Including Adaptive Capacity) Scores
#'
#' This function calculates the state indicator ~ pressures ~ type specific vulnerability,
#' from exposure scores and sensitivity scores. The function can either be used
#' with the output from \code{\link{calc_exposure}} or \code{\link{model_exposure}}
#' and \code{\link{calc_sensitivity}} or \code{\link{model_sensitivity}}.
#'
#' @param exposure_results a data frame containing the output from \code{\link{calc_exposure}}
#'        or \code{\link{model_exposure}}.
#' @param sensitivity_results  a data frame containing the output from \code{\link{calc_sensitivity}}
#'        or \code{\link{model_sensitivity}}.
#' @param method_vulnerability a character string specifying the method for aggregating
#'        the trait based vulnerabilities, available are mean (default), median,
#'        maximum, and minimum.
#' @param method_uncertainty a character string specifying the method for the
#'        aggregation of the uncertainty scores from exposure and sensitivity.
#'        Available are mean (default), median, maximum, and minimum.
#'
#' @details For expert scores the following equation is applied
#' \itemize{\item(sensitivity + adaptive capacity) + exposure,}
#' or in case of negative sensitivity values:
#' \itemize{\item(sensitivity + adaptive capacity) - exposure.}
#' Trait based sensitivity and adaptive capacity scores will be assessed individually
#' and then aggregated to one vulnerability score per state indicator and pressure
#' combination. The aggregation method can be chosen with the \code{method_vulnerability}
#' argument.
#' For modelling scores sensitivity and exposure scores are summed up. If the exposure
#' trend and the sensitivity score have the same direction, e.g. a decreasing trend
#' in exposure and a negative sensitivity score, then the vulnerability effect
#' is assigned as positive. If they have opposing directions, e.g. an increasing
#' exposure, while sensitivity is negative, then the vulnerability is negative.
#' Vulnerability scores can range only from -10 to 10, aligning with the *ecorisk*
#' framework.
#'
#'
#' @return a data frame containing state indicator, pressure, type and the vulnerability
#' and associated uncertainty score.
#'
#' @seealso \code{\link{calc_exposure}}, \code{\link{calc_sensitivity}},
#'          \code{\link{model_exposure}}, \code{\link{model_sensitivity}},
#'          \code{\link{status}}, \code{\link{risk}}
#'
#' @export
#'
#' @examples
#' # Using demo output data from the calc_exposure() and calc_sensitivity()
#' # functions:
#' vulnerability(
#'   exposure_results = ex_output_calc_exposure,
#'   sensitivity_results = ex_output_calc_sensitivity
#' )
#'
#' \donttest{
#'   ### Demo Expert-Based Pathway
#'   # - using the example scoring datasets 'ex_expert_exposure',
#'   #   and 'ex_expert_sensitivity'
#'
#'   # Calculate (mean) exposure score:
#'   exp_expert <- calc_exposure(
#'     pressures = ex_expert_exposure$pressure,
#'     components = ex_expert_exposure[ ,2:5],
#'     uncertainty = ex_expert_exposure[ ,6:9],
#'     method = "mean" # default
#'   )
#'   # Calculate (mean) sensitivity (and adaptive capacity) score:
#'   sens_ac_expert <- calc_sensitivity(
#'     indicators = ex_expert_sensitivity$indicator,
#'     pressures = ex_expert_sensitivity$pressure,
#'     type = ex_expert_sensitivity$type,
#'     sensitivity_traits = ex_expert_sensitivity[ ,4:8],
#'     adaptive_capacities = ex_expert_sensitivity[ ,9:13],
#'     uncertainty_sens = ex_expert_sensitivity[ ,14:18],
#'     uncertainty_ac = ex_expert_sensitivity[ ,19:23],
#'     method = "mean"
#'   )
#'   # Calculate vulnerability using the mean (default):
#'   vulnerability(
#'     exposure_results = exp_expert,
#'     sensitivity_results = sens_ac_expert
#'   )
#'   # Calculate vulnerability using the median and maximum:
#'   vulnerability(
#'     exposure_results = exp_expert,
#'     sensitivity_results = sens_ac_expert,
#'     method_vulnerability = "median",
#'     method_uncertainty = "maximum"
#'   )
#'
#'
#'   ### Demo Model-Based Pathway
#'   # - using the demo time series 'pressure_ts_baltic' and 'indicator_ts_baltic'
#'
#'   # Model exposure score:
#'   exp_model <- model_exposure(
#'     pressure_time_series = pressure_ts_baltic,
#'     base_years = c(start = 1984, end = 1994),
#'     current_years = c(start = 2010, end = 2016)
#'   )
#'
#'   # Model sensitivity score:
#'   sens_ac_model <- model_sensitivity(
#'     indicator_time_series = indicator_ts_baltic,
#'     pressure_time_series = pressure_ts_baltic,
#'     current_years = c(start = 2010, end = 2016)
#'   )
#'   # Add manually adaptive capacity scores (otherwise zero):
#'   sens_ac_model$adaptive_capacity <- c(rep(1, 8), rep(-1, 8))
#'
#'   # Calculate vulnerability using the mean (default):
#'   vulnerability(
#'     exposure_results = exp_model,
#'     sensitivity_results = sens_ac_model
#'   )
#'
#' }

vulnerability <- function(exposure_results, sensitivity_results,
  method_vulnerability = "mean", method_uncertainty = "mean" ) {

  # Shorten arguments
  exposure <- exposure_results
  sens_ac <- sensitivity_results
  method_v <- method_vulnerability
  method_unc <- method_uncertainty

  # Test if all sensitivity pressures are in exposure
  if (all(unique(sens_ac$pressure) %in% unique(exposure$pressure)) == FALSE) {
    exp_names <- unique(exposure$pressure)
    sen_names <- unique(sens_ac$pressure)
    no_match <- which(!(sen_names %in% exp_names))
    warning(paste(c("The following pressure names in the sensitivity data set
                    are not included in the pressure data set:",
      sen_names[c(no_match)],
      "\nPlease ensure that all pressure names of the sensitivity data set are
      in the pressure data set; also check for spelling errors. The affected
      lines are not analysed further."),
      collapse = " "))
    skip_names <- sen_names[no_match]
    sens_ac <- sens_ac[-which(sens_ac$pressure %in% skip_names), ]
  }

  # Test if methods are correct
  if (!(method_v %in% c("mean", "median", "maximum", "minimum"))) {
    warning("Method for aggregation of the vulnerability traits not recognised.
            Will use method = arithmetic mean instead.")
  }
  if (!(method_unc %in% c("mean", "median", "maximum", "minimum"))) {
    warning("Method for uncertainty aggregation not recognised.
            Will use method = arithmetic mean instead.")
  }

  # ---
  # Since traits can have different effect directions for the same
  # PRESS~IND~type combination, the vulnerability has to be
  # calculated for each trait individually and after that the
  # arithmetic mean of all trait vulnerabilities can be calculated

  press_out <- sens_ac$pressure
  ind_out <- sens_ac$indicator
  type_out <- sens_ac$type
  v_score <- rep(0, nrow(sens_ac))
  unc <- rep(0, nrow(sens_ac))
  pathway_out <- sens_ac$pathway

  # Get trait-based sensitivity and ac scores from expert pathway
  sens_traits_col <- which(grepl("^sens_original", names(sens_ac)))
  ac_traits_col <- which(grepl("^ac_original", names(sens_ac)))
  # Use general scores in model pathway and trait-specific scores in
  # expert pathway
  if (length(sens_traits_col) == 0) {
    sens_traits <- sens_ac$sensitivity
  } else {
    sens_traits <- sens_ac[ ,sens_traits_col]
  }
  if (length(ac_traits_col) == 0) {
    ac_traits <- sens_ac$adaptive_capacity
  } else {
    ac_traits <- sens_ac[ ,ac_traits_col]
  }

  # Use input from sens_ac function to calculate the trait based vulnerability
  for (i in 1:nrow(sens_ac)) {
    # extract the pressure in the current row
    helper_press <- sens_ac$pressure[i]

    # First calculate adjsen in case of scores as vectors (stemming
    # from modeling function or general scores)
    if (is.null(ncol(sens_traits)) && is.null(ncol(ac_traits))) {
      if (is.na(as.numeric(sens_traits[i])) == TRUE) {
        helper <- NA
      } else if (as.numeric(sens_traits[i]) == 0) {
        # if sensitivity is 0, vulnerability will be 0
        helper <- 999  # code for sensitivity is 0, so vulnerability has to be in the end 0
      } else if (is.na(as.numeric(ac_traits[i])) == TRUE) {
        # in case AC is NA, vulnerability is determined by
        # sensitivity and exposure only
        helper <- as.numeric(sens_traits[i])
      } else {
        # calculate adjsen
        helper <- as.numeric(sens_traits[i]) + as.numeric(ac_traits[i])
      }
    } else {
      # sensitivity and adaptive capacity specific for each trait
      helper <- rep(0, ncol(sens_traits))
      for (l in 1:length(sens_traits[i, ])) {
        if (is.na(sens_traits[i, l]) == TRUE) {
          helper[l] <- NA
        } else if (sens_traits[i, l] == 0) {
          # test if sensitivity is 0 for any trait
          helper[l] <- 999  # code for sensitivity is 0, so vulnerability has to be in the end 0
        } else if (is.na(as.numeric(ac_traits[i, l])) == TRUE) {
          # in case AC is NA, vulnerability is determined by
          # sensitivity and exposure only
          helper[l] <- as.numeric(sens_traits[i, l])
        } else {
          # otherwise calculate adjsen, here 0 can be created due to
          # the combination of sen and ac
          helper[l] <- as.numeric(sens_traits[i, l]) + as.numeric(ac_traits[i,
            l])
        }
      }
    }  # end of adjsen calculation


    # Test that score is not greater than 5 or smaller than -5
    for (k in 1:length(helper)) {
      if (is.na(helper[k]) == TRUE) {
        helper[k] <- NA
      } else if (helper[k] == 999) {
        helper[k] <- 999
      } else if (helper[k] == 0) {
        helper[k] <- 0
      } else {
        if (helper[k] > 0) {
          if (helper[k] > 5) {
            helper[k] <- 5
          }
          if (helper[k] < 1) {
            helper[k] <- 1
          } else {
            helper[k] <- helper[k]
          }
        } else {
          if (helper[k] < -5) {
            helper[k] <- -5
          }
          if (helper[k] > -1) {
            helper[k] <- -1
          } else {
            helper[k] <- helper[k]
          }
        }
      }

    }  # end of test

    helper_v <- rep(0, length(helper))


    # Now add exp to adjsen
    for (j in 1:length(helper)) {
      if (is.na(helper[j]) == TRUE) {
        helper_v[j] <- NA
      } else if (helper[j] == 999) {
        # if sensitivity was 0, v must be 0 in the end
        helper_v[j] <- 0
      } else if (exposure[exposure$pressure == helper_press, ]$exposure == 0) {
        # if the exposure to a pressure is 0, then v is also 0
        helper_v[j] <- 0
      } else {
        if ("comp_direction" %in% names(exposure)) {
          # exposure from modelling pathway
          if (exposure[exposure$pressure == helper_press, ]$comp_direction == "increase" &
              helper[j] > 0) {
            helper_v[j] <- helper[j] + exposure[exposure$pressure == helper_press, ]$exposure
          } else if (exposure[exposure$pressure == helper_press, ]$comp_direction == "increase" &
              helper[j] <= 0) {
            helper_v[j] <- helper[j] - exposure[exposure$pressure == helper_press, ]$exposure
          } else if (exposure[exposure$pressure == helper_press, ]$comp_direction == "decrease" &
              helper[j] > 0) {
            helper_v[j] <- -1 * helper[j] - exposure[exposure$pressure == helper_press, ]$exposure
          } else {
            # exposure = decrease and sensitivity = negative
            helper_v[j] <- -1 * helper[j] + exposure[exposure$pressure == helper_press, ]$exposure
          }
        } else {
          # exposure from scoring pathway if adjsen is zero, then the
          # ind is actually impacted but ac compensates it, therefore
          # it is impacted by the pressure
          if (helper[j] > 0) {
            helper_v[j] <- helper[j] + exposure[exposure$pressure == helper_press, ]$exposure
          } else {
            helper_v[j] <- helper[j] - exposure[exposure$pressure == helper_press, ]$exposure
          }
        }
      }

      # Test that score is not exceeding ±10
      if (is.na(helper_v[j]) == TRUE) {
        helper_v[j] <- NA
      } else {
        if (helper_v[j] > 10) {
          helper_v[j] <- 10
        }
        if (helper_v[j] < -10) {
          helper_v[j] <- -10
        } else {
          helper_v[j] <- helper_v[j]
        }
      }

    }  # end of vulnerability calculation


    # then aggregate
    if (method_v == "median") {
      v_score[i] <- stats::median(helper_v, na.rm = TRUE)
    }
    if (method_v == "maximum") {
      v_score[i] <- max(helper_v, na.rm = TRUE)
    }
    if (method_v == "minimum") {
      v_score[i] <- min(helper_v, na.rm = TRUE)
    }
    if (!(method_v %in% c("median", "maximum", "minimum"))) {
      v_score[i] <- mean(helper_v, na.rm = TRUE)
    }


    # Now aggregate uncertainty
    if (method_unc == "median") {
      unc[i] <- stats::median(c(sens_ac$uncertainty_sen[i], sens_ac$uncertainty_ac[i],
        exposure[exposure$pressure == helper_press, ]$uncertainty),
        na.rm = TRUE)
    }
    if (method_unc == "maximum") {
      unc[i] <- max(c(sens_ac$uncertainty_sen[i], sens_ac$uncertainty_ac[i],
        exposure[exposure$pressure == helper_press, ]$uncertainty),
        na.rm = TRUE)
    }
    if (method_unc == "minimum") {
      unc[i] <- min(c(sens_ac$uncertainty_sen[i], sens_ac$uncertainty_ac[i],
        exposure[exposure$pressure == helper_press, ]$uncertainty),
        na.rm = TRUE)
    }
    if (!(method_unc %in% c("median", "maximum", "minimum"))) {
      unc[i] <- mean(c(sens_ac$uncertainty_sen[i], sens_ac$uncertainty_ac[i],
        exposure[exposure$pressure == helper_press, ]$uncertainty),
        na.rm = TRUE)
    }


  }  # end of sens_ac-loop

  output_vulnerability <- data.frame(
    indicator = ind_out,
    pressure = press_out,
    type = type_out,
    pathway = pathway_out,
    vulnerability = v_score,
    uncertainty = unc
  )

  return(output_vulnerability)
}

