% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.r
\name{predict,GRaster-method}
\alias{predict,GRaster-method}
\alias{predict}
\title{Make predictions from a linear or generalized linear model to a GRaster}
\usage{
\S4method{predict}{GRaster}(object, model, type = "response")
}
\arguments{
\item{object}{A \code{GRaster} with one or more layers.}

\item{model}{An \code{lm} or \code{glm} model object.}

\item{type}{Character: Type of prediction to make. This can be either \code{link} (default; predictions are made on the scale of the link function) or \code{response} (predictions are made on the scale of the response variable). This function can only make predictions on the scale of the response for the identity, logit, log, or cloglog (complementary log-log) link functions.}
}
\value{
A \code{GRaster}.
}
\description{
This version of the \code{predict()} function make predictions to a set of \code{GRaster}s from a model object.

The model must be either a linear model, which is of class \code{lm} and typically created using the \code{\link[stats:lm]{stats::lm()}} function or a generalized linear model (GLM), which is class \code{glm} and typically created using \code{\link[stats:glm]{stats::glm()}}. Other packages can also create \code{lm} or \code{glm} objects, but they may not work in this function. For example, generalized additive models, which can be created using the \code{gam()} function in the \strong{mgcv} package, inherit the \code{glm} class, but cannot be used in this function. However, \code{glm} objects created with the \strong{speedglm} package should work with this function.

This \code{predict()} function can handle:
\itemize{
\item Linear predictors and intercepts like \code{1 + x};
\item Quadratic terms like \code{x^2} (or, in \strong{R} formula notation, \code{I(x^2)});
\item Two-way interaction terms between scalars like \code{x1:x2} and \code{x1 * x2};
\item Categorical predictors (i.e., categorical \code{GRaster}s; see \code{vignette("GRasters", package = "fasterRaster")});
\item Two-way interactions between a categorical predictor and a scalar predictor; and
\item Two-way interactions between categorical predictors.
}
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

### This example creates a simple model of Dypsis distribution using
# elevation, distance to forest, land cover class, and nearness to rivers.

# Elevation raster, forest cover in year 2000, land cover class, and
# points where Dypsis plants have been collected
madElev <- fastData("madElev")
madForest2000 <- fastData("madForest2000")
madCover <- fastData("madCover")
madRivers <- fastData("madRivers")
madDypsis <- fastData("madDypsis")

# Convert SpatRasters to GRasters and sf vectors to GVectors:
elev <- fast(madElev)
forest <- fast(madForest2000)
cover <- fast(madCover)
rivers <- fast(madRivers)
dypsis <- fast(madDypsis)

# Distance to forest
distToForest <- distance(forest, unit = "m")
distToForest <- log1p(distToForest) # log(x + 1) of distance
names(distToForest) <- "distToForest"

# "Stack" elevation and forest cover
continuous <- c(elev, distToForest)

# Scale continuous predictors to mean of 0 and sd of 1
continuousScaled <- scale(continuous)
names(continuousScaled) <- c("elevation", "distToForest")

# Project land cover raster
coverProj <- project(cover, continuousScaled)

# Near a river?
riverBuffer <- buffer(rivers, 5000)
nearRiver <- rasterize(riverBuffer, elev, background = 0)
names(nearRiver) <- "nearRiver"
levels(nearRiver) <- data.frame(value = 0:1, label = c("far", "near"))

# Combine continuous/categorical data
covariateRasters <- c(continuousScaled, coverProj, nearRiver)
plot(covariateRasters)

# Extract environmental values at Dypsis locations:
presEnv <- extract(covariateRasters, dypsis, cats = TRUE)
presEnv$presBg <- 1
head(presEnv)

# Extract elevation and forest cover at 2000 background sites:
bgEnv <- spatSample(covariateRasters, size = 3000, values = TRUE, cats = TRUE)
bgEnv <- bgEnv[stats::complete.cases(bgEnv), ]
bgEnv <- bgEnv[1:2000, ]
bgEnv$presBg <- 0
head(bgEnv)

# Combine presence and background data:
env <- rbind(presEnv, bgEnv)

# Calibrate model:
form <- presBg ~ elevation + distToForest +
  I(distToForest^2) + elevation * distToForest +
  madCover + nearRiver

model <- stats::glm(form, data = env, family = stats::binomial)
summary(model)

# Make predictions and map:
prediction <- predict(covariateRasters, model, type = "response")
prediction

# Not a great model!
plot(prediction, main = "Predicted")
plot(dypsis, pch = 1, add = TRUE)

}
}
\seealso{
\code{\link[terra:predict]{terra::predict()}}; \code{\link[stats:predict]{stats::predict()}}
}
