% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/project.r
\name{project,GRaster-method}
\alias{project,GRaster-method}
\alias{project}
\alias{project,GVector-method}
\title{Change the coordinate reference system of a GRaster or GVector}
\usage{
\S4method{project}{GRaster}(
  x,
  y,
  align = FALSE,
  method = NULL,
  fallback = TRUE,
  res = "fallback",
  wrap = FALSE,
  verbose = FALSE
)

\S4method{project}{GVector}(x, y, wrap = FALSE)
}
\arguments{
\item{x}{A \code{GRaster} or \code{GVector} to be projected.}

\item{y}{A character or \code{GLocation} object (i.e., typically a \code{GRaster} or \code{GVector}): Used to set the focal \code{GRaster} or \code{GVector}'s new CRS (and resolution and possibly extent, for \code{GRaster}s).}

\item{align}{Logical: If \code{FALSE} (default), and \code{x} and \code{y} are \code{GRaster}s, then the extent of \code{x} will be cropped to the extent of \code{y}. If \code{TRUE}, no cropping is performed.}

\item{method}{Character or \code{NULL} (for \code{GRaster}s only): Method to use to conduct the transformation (rasters only). Partial matching is used.
\itemize{
\item \code{NULL} (default): Automatically choose based on raster properties (\code{near} for categorical data, \code{bilinear} for continuous data).
\item \code{"near"}: Nearest neighbor. Best for categorical data, and often a poor choice for continuous data.  If \code{\link[=datatype]{datatype()}} is \code{integer}, this method will be used by default.
\item \code{"bilinear"}: Bilinear interpolation (default for non-categorical data; uses weighted values from 4 cells).
\item \code{"bicubic"}: Bicubic interpolation (uses weighted values from 16 cells).
\item \code{"lanczos"}: Lanczos interpolation (uses weighted values from 25 cells).
}

\emph{Note #1}: If \code{x} and \code{y} are \code{GRaster}s, and \code{res = "terra"}, then the same \code{method} is used to resample \code{x} to the resolution of \code{y} before projecting \code{x}.

\emph{Note #2}: Methods that use multiple cells will cause the focal cell to become \code{NA} if there is at least one cell with an \code{NA} in the cells it draws from. These \code{NA} cells can often be filled using the \code{fallback} argument.}

\item{fallback}{Logical (for projecting \code{GRaster}s only): If \code{TRUE} (default), then use "lower" resampling methods to fill in \code{NA} cells when a "higher" resampling method is used. For example, if \code{method = "bicubic"}, \code{NA} cells will be filled in using the \code{bilinear} method, except when that results in \code{NA}s, in which case the \code{near} method will be used. Fallback causes fewer cells to revert to \code{NA} values, so may be better at capturing complex "edges" (e.g., coastlines). Fallback does increase processing time because each "lower" method must be applied, then results merged. Fallback is not used if \code{method = "near"}.}

\item{res}{Character (for projecting \code{GRaster}s only): Method used to set the resolution of a \code{GRaster} in the new CRS. This can be one of three options.  Partial matching is used and case ignored:
\itemize{
\item \code{"terra"}: This method creates an output raster that is as close as possible in values and resolution to the one that \code{\link[terra:project]{terra::project()}} would create. However, for large rasters (i.e., many cells), this can fail because \code{terra::project()} encounters memory limits (it is used internally to create a template). This method resamples the focal raster in its starting CRS, then projects it to the destination CRS.
\item \code{"template"}: This method can only be used if \code{y} is a \code{GRaster}. The output will have the same resolution as \code{y} and possibly the same extent (depending on the value of \code{align}). However, unlike the \code{"terra"} method, cell values will not necessarily be as close as possible to what \code{\link[terra:project]{terra::project()}} would generate (unless \code{method = "near"}). Unlike the \code{"terra"} method, this method does not resample the focal raster in its starting CRS before projecting. For large rasters it will be faster than the \code{"terra"} method (especially if \verb{"method = "near"}), and it should be less likely to fail because of memory limits.
\item Two numeric values: Values for the new resolution (x- and y-dimensions).
\item \code{"center"}: This method locates the centroid of the raster to be projected (in the same CRS as the original raster). It then creates four points north, south, east, and west of the centroid, each spaced one cell's width from the centroid. This set of points is then projected to the new CRS. The new cell size in the x-dimension will be the average of the distance between the east and west points from the centroid, and in the y-dimension the average from the centroid to the north and south points.
\item \code{"fallback"} (default): This applies the \code{terra} method first, but if that fails, then tries \code{template}, then \code{center}. This process can take a long time for large rasters.
}}

\item{wrap}{Logical:
\itemize{
\item \code{GRaster}s: When projecting rasters that "wrap around" (i.e., whole-world rasters or rasters that have edges that actually circle around to meet on the globe), \code{wrap} should be \code{TRUE} to avoid removing rows and columns from the "edge" of the map. The default is \code{FALSE}.
\item \code{GVector}s: When projecting vectors that span the international date line at 180E/W, \code{wrap} should be \code{TRUE} to avoid an issue where the coordinates are incorrectly mapped to the range -180 to 180.
}}

\item{verbose}{Logical (for projecting \code{GRaster}s only): If \code{TRUE}, display progress. Default is \code{FALSE}.}
}
\value{
A \code{GRaster} or \code{GVector}.
}
\description{
\code{project()} changes the coordinate reference system (CRS) of a \code{GRaster} or \code{GVector}. It has three use cases:
\itemize{
\item \code{x} is a \code{GRaster} and \code{y} is a \code{GRaster}: \code{x} will be projected to the CRS of \code{y} and resampled to have the same resolution as \code{y}. If argument \code{align} is \code{FALSE}, then it will also be cropped to the extent of \code{y}.
\item \code{x} is a \code{GRaster} and \code{y} is a \code{GVector} or a CRS string (typically in Well-Known Text format): \code{x} will be projected to the CRS specified by \code{y} and resampled but not cropped.
\item \code{x} is a \code{GVector} and \code{y} is a \code{GRaster}, \code{GVector}, or CRS string: The vector will be projected to the CRS of \code{y}.
}
}
\details{
When projecting a raster, the "fallback" methods in \strong{GRASS} tool \code{r.import} are actually used, even though the \code{method} argument takes the strings specifying non-fallback methods. See the manual page for the \code{r.import} \strong{GRASS} tool.
}
\examples{
if (grassStarted()) {

### Setup for all examples

library(sf)
library(terra)

# Climate raster, elevation raster, rivers vector
madElev <- fastData("madElev")
madRivers <- fastData("madRivers")
madChelsa <- fastData("madChelsa")

# Convert objects into fasterRaster formats
chelsa <- fast(madChelsa)
elev <- fast(madElev)
rivers <- fast(madRivers)

### Project raster without resampling
elevWGS84 <- project(elev, crs(chelsa))
elevWGS84

### Project raster and resample to resolution of another raster
elevWGS84Resamp <- project(elev, chelsa)
elevWGS84Resamp

res(elevWGS84)
res(elevWGS84Resamp)
res(chelsa)

### Project vector
riversWGS84 <- project(rivers, chelsa)
riversWGS84
cat(crs(rivers)) # using "cat()" to make it look nice
cat(crs(riversWGS84))

}
}
\seealso{
\code{\link[terra:project]{terra::project()}}, \code{\link[sf:st_transform]{sf::st_transform()}}, \strong{GRASS} manual pages for modules \code{r.proj} and \code{v.proj} (see \code{grassHelp("r.proj")} and \code{grassHelp("v.proj")})
}
