% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_gdp.R
\name{est_gdp}
\alias{est_gdp}
\title{Gaussian differential privacy parameters lower bounding empirical trade-off points}
\usage{
est_gdp(x, dp = 2L)
}
\arguments{
\item{x}{One or more f-DP trade-off specifications to be lower bounded. Accepts the same flexible input types as \code{\link[=fdp]{fdp()}}:
\itemize{
\item A function (user-defined or built-in, e.g., \code{\link[=lap]{lap()}}) that when called with a numeric vector \code{alpha} returns a data frame with columns \code{alpha} and \code{beta};
\item A data frame with columns \code{alpha} and \code{beta} containing empirical trade-off points;
\item A numeric vector of length 101 (interpreted as \code{beta} values on the canonical grid \code{alpha = seq(0, 1, by = 0.01)}).
}
The function extracts all Type-I/Type-II error coordinates and finds the minimal \eqn{(\varepsilon, \delta)}-DP parameters lower bounding them.}

\item{dp}{Integer scalar specifying the number of decimal places of precision for the result (with careful rounding employed to ensure the bound holds).
Must be a non-negative integer.
Default is \code{2L}.}
}
\value{
A GDP trade-off function object (see \code{\link[=gdp]{gdp()}}) with class \code{c("fdp_gdp_tradeoff", "function")}.
This represents the tightest \eqn{\mu}-GDP trade-off function that lower bounds the input \code{x}.
}
\description{
Estimates the minimal Gaussian differential privacy (GDP) parameter \eqn{\mu} that provides a valid lower bound for a collection of empirical or analytically-derived trade-off points.
\strong{Note:} due to the numerical optimisation involved, this is only an approximation.
}
\details{
Given a set of trade-off points \eqn{\{(\alpha_i, \beta_i)\}_{i=1}^n} representing Type-I and Type-II error rates, this function numerically solves for the smallest \eqn{\mu \ge 0} such that the \eqn{\mu}-GDP trade-off function
\deqn{G_\mu(\alpha) = \Phi\left(\Phi^{-1}(1-\alpha) - \mu\right)}
satisfies \eqn{G_\mu(\alpha_i) \le \beta_i} for all \eqn{i = 1, \ldots, n}, where \eqn{\Phi} denotes the standard normal cumulative distribution function.

\strong{Warning:} since this is a numerical optimisation on a finite set of trade-off points, there is no mathematical guarantee of correctness.
As such, the \eqn{\mu} found ought best to be viewed as an approximate lower bound on the true values, since there could be intermediate trade-off points that are not supplied which cause the true values to be larger.

This function may be useful for post-hoc privacy auditing, privacy budget allocation, or mechanism comparison.
}
\examples{
# Estimate GDP from manually specified empirical trade-off points
# These could come from empirical measurements or privacy audits
empirical_points <- data.frame(
  alpha = c(0.00, 0.05, 0.10, 0.25, 0.50, 1.00),
  beta  = c(1.00, 0.93, 0.87, 0.72, 0.43, 0.00)
)
result <- est_gdp(empirical_points)
result

# Visualise how well the GDP bound fits the empirical points
fdp(empirical_points, result)

# Find the GDP lower bound for a Laplace mechanism.
lap_mechanism <- lap(1.5)
gdp_bound <- est_gdp(lap_mechanism)
gdp_bound

# Compare the Laplace mechanism with its GDP lower bound
fdp(lap_mechanism, gdp_bound)

# Control precision with the dp parameter
result_1dp <- est_gdp(empirical_points, dp = 1L)
result_3dp <- est_gdp(empirical_points, dp = 3L)
# Higher precision gives tighter bounds
fdp(empirical_points, result_1dp, result_3dp)
}
\references{
Dong, J., Roth, A. and Su, W.J. (2022). “Gaussian Differential Privacy”. \emph{Journal of the Royal Statistical Society Series B}, \strong{84}(1), 3–37. \doi{10.1111/rssb.12454}.
}
\seealso{
\code{\link[=gdp]{gdp()}} for constructing GDP trade-off functions with known \eqn{\mu},
\code{\link[=fdp]{fdp()}} for visualising and comparing trade-off functions,
\code{\link[=gdp_to_epsdelta]{gdp_to_epsdelta()}} for converting from GDP to classical \eqn{(\varepsilon, \delta)}-DP,
\code{\link[=est_epsdelta]{est_epsdelta()}} for estimating classical DP parameters from trade-off points.
}
