% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GloPointPr.R
\name{GloPointPrReg}
\alias{GloPointPrReg}
\title{Global Cox point process regression.}
\usage{
GloPointPrReg(xin = NULL, tin = NULL, T0 = NULL, xout = NULL, optns = list())
}
\arguments{
\item{xin}{An n by p matrix with input measurements of the predictors.}

\item{tin}{A list holding the sample of event times of each replicated point process, where the ith element of the list \code{tin} holds the event times of the point process corresponding to the ith row of \code{xin}.}

\item{T0}{A positive scalar that defines the time window [0,\code{T0}] where the replicated Cox point processes are observed.}

\item{xout}{A k by p matrix with output measurements of the predictors. Default is \code{xin}.}

\item{optns}{A list of control parameters specified by \code{list(name=value)}.}
}
\value{
A list containing the following components:
\item{xout}{Input \code{xout}.}
\item{dSup}{Support of each estimated (up to a constant) conditional intensity regression function in the columns of \code{intensityReg}.}
\item{intensityReg}{A matrix of dimension \code{length(dSup)} by \code{nrow(xout)} holding the estimated intensity regression functions up to a constant over the support grid \code{dSup}, where each column corresponds to a predictor level in the corresponding row of \code{xout}.}
\item{xin}{Input \code{xin}.}
\item{optns}{A list of control options used.}
}
\description{
Global Fréchet regression for replicated Cox point processes with respect to \eqn{L^2}-Wasserstein distance on shape space and Euclidean 2-norm on intensity factor space.
}
\details{
Available control options are \code{bwDen} (see \code{\link{LocDenReg}} for this option description) and
\describe{
\item{L}{Upper Lipschitz constant for quantile space; numeric -default: 1e10.}
\item{M}{Lower Lipschitz constant for quantile space; numeric -default: 1e-10.}
\item{dSup}{User defined output grid for the support of kernel density estimation used in \code{CreateDensity()} for mapping from quantile space to shape space. This grid must be in [0,\code{T0}]. Default is an equidistant grid with \code{nqSup}+2 points.}
\item{nqSup}{A scalar with the number of equidistant points in (0,1) used to obtain the empirical quantile function from each point process. Default: 500.}
}
}
\examples{
\donttest{
n=100
alpha_n=sqrt(n)
alpha1=2.0
beta1=1.0
gridQ=seq(0,1,length.out=500+2)[2:(500+1)]
X=runif(n,0,1)#p=1
tau=matrix(0,nrow=n,ncol=1)
for(i in 1:n){
  tau[i]=alpha1+beta1*X[i]+truncnorm::rtruncnorm(1, a=-0.3, b=0.3, mean = 0, sd = 1.0)
}
Ni_n=matrix(0,nrow=n,ncol=1)
u0=0.4
u1=0.5
u2=0.05
u3=-0.01
tin=list()
for(i in 1:n){
  Ni_n[i]=rpois(1,alpha_n*tau[i])
  mu_x=u0+u1*X[i]+truncnorm::rtruncnorm(1,a=-0.1,b=0.1,mean=0,sd=1)
  sd_x=u2+u3*X[i]+truncnorm::rtruncnorm(1,a=-0.02,b=0.02,mean=0,sd=0.5)
  if(Ni_n[i]==0){
    tin[[i]]=c()
  }else{
    tin[[i]]=truncnorm::rtruncnorm(Ni_n[i],a=0,b=1,mean=mu_x,sd=sd_x) #Sample from truncated normal
  }
}
res=GloPointPrReg(
  xin=matrix(X,ncol=1),tin=tin,
  T0=1,xout=matrix(seq(0,1,length.out=10),ncol=1),
  optns=list(bwDen=0.1)
)
}
}
\references{
\cite{Petersen, A., & Müller, H.-G. (2019). "Fréchet regression for random objects with Euclidean predictors." The Annals of Statistics, 47(2), 691--719.}

\cite{Gajardo, Á. and Müller, H.-G. (2022). "Cox Point Process Regression." IEEE Transactions on Information Theory, 68(2), 1133-1156.}
}
