% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LocCorReg.R
\name{LocCorReg}
\alias{LocCorReg}
\title{Local Fréchet regression for correlation matrices}
\usage{
LocCorReg(x, M, xOut = NULL, optns = list())
}
\arguments{
\item{x}{an n by p matrix or data frame of predictors.}

\item{M}{a q by q by n array (resp. a list of q by q matrices) where
\code{M[, , i]} (resp. \code{M[[i]]}) contains the i-th correlation matrix
of dimension q by q.}

\item{xOut}{an m by p matrix or data frame of output predictor levels.
It can be a vector of length p if m = 1.}

\item{optns}{A list of options control parameters specified by
\code{list(name=value)}. See `Details'.}
}
\value{
A \code{corReg} object --- a list containing the following fields:
\item{fit}{a list of estimated correlation matrices at \code{x}.}
\item{predict}{a list of estimated correlation matrices at \code{xOut}.
  Included if \code{xOut} is not \code{NULL}.}
\item{residuals}{Frobenius distance between the true and
  fitted correlation matrices.}
\item{xOut}{the output predictor level used.}
\item{optns}{the control options used.}
}
\description{
Local Fréchet regression for correlation matrices
  with Euclidean predictors.
}
\details{
Available control options are
\describe{
\item{metric}{choice of metric. \code{'frobenius'} and \code{'power'} are supported,
  which corresponds to Frobenius metric and Euclidean power metric,
  respectively. Default is Frobenius metric.}
\item{alpha}{the power for Euclidean power metric. Default is 1 which
  corresponds to Frobenius metric.}
\item{kernel}{Name of the kernel function to be chosen from \code{'gauss'},
  \code{'rect'}, \code{'epan'}, \code{'gausvar'} and \code{'quar'}. Default is \code{'gauss'}.}
\item{bw}{bandwidth for local Fréchet regression, if not entered
  it would be chosen from cross validation.}
\item{digits}{the integer indicating the number of decimal places (round)
  to be kept in the output. Default is NULL, which means no round operation.}
}
}
\examples{
# Generate simulation data
\donttest{
n <- 100
q <- 10
d <- q * (q - 1) / 2
xOut <- seq(0.1, 0.9, length.out = 9)
x <- runif(n, min = 0, max = 1)
y <- list()
for (i in 1:n) {
  yVec <- rbeta(d, shape1 = sin(pi * x[i]), shape2 = 1 - sin(pi * x[i]))
  y[[i]] <- matrix(0, nrow = q, ncol = q)
  y[[i]][lower.tri(y[[i]])] <- yVec
  y[[i]] <- y[[i]] + t(y[[i]])
  diag(y[[i]]) <- 1
}
# Frobenius metric
fit1 <- LocCorReg(x, y, xOut,
  optns = list(metric = "frobenius", digits = 2)
)
# Euclidean power metric
fit2 <- LocCorReg(x, y, xOut,
  optns = list(
    metric = "power", alpha = .5,
    kernel = "epan", bw = 0.08
  )
)
}
}
\references{
\itemize{
\item \cite{Petersen, A. and Müller, H.-G. (2019). Fréchet regression for random objects with Euclidean predictors. The Annals of Statistics, 47(2), 691--719.}
}
}
