% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_hdr_1d.R
\name{get_hdr_1d}
\alias{get_hdr_1d}
\title{Computing the highest density regions of a 1D density}
\usage{
get_hdr_1d(
  x = NULL,
  method = "kde",
  probs = c(0.99, 0.95, 0.8, 0.5),
  n = 512,
  range = NULL,
  hdr_membership = TRUE,
  fun,
  args = list()
)
}
\arguments{
\item{x}{A vector of data}

\item{method}{Either a character (\code{"kde"}, \code{"norm"}, \code{"histogram"}, \code{"freqpoly"}, or \code{"fun"}) or \verb{method_*_1d()} function.
See the "The \code{method} argument" section below for details.}

\item{probs}{Probabilities to compute HDRs for.}

\item{n}{Resolution of grid representing estimated density and HDRs.}

\item{range}{Range of grid representing estimated density and HDRs.}

\item{hdr_membership}{Should HDR membership of data points (\code{x}) be computed?}

\item{fun}{Optional, a probability density function, must be vectorized in its first argument.
See the "The \code{fun} argument" section below for details.}

\item{args}{Optional, a list of arguments to be provided to \code{fun}.}
}
\value{
\code{get_hdr_1d} returns a list with elements \code{df_est} (\code{data.frame}), \code{breaks} (named \code{numeric}), and \code{data} (\code{data.frame}).
\itemize{
\item \code{df_est}: the estimated HDRs and density evaluated on the grid defined by \code{range} and \code{n}.
The column of estimated HDRs (\code{df_est$hdr}) is a numeric vector with values from \code{probs}.
The columns \code{df_est$fhat} and \code{df_est$fhat_discretized} correspond to the estimated density
on the original scale and rescaled to sum to 1, respectively.
\item \code{breaks}: the heights of the estimated density (\code{df_est$fhat}) corresponding to the HDRs specified by \code{probs}.
Will always have additional element \code{Inf} representing the cutoff for the 100\% HDR.
\item \code{data}: the original data provided in the \code{data} argument.
If \code{hdr_membership} is set to \code{TRUE}, this includes a column (\code{data$hdr_membership})
with the HDR corresponding to each data point.
}
}
\description{
\code{get_hdr_1d} is used to estimate a 1-dimensional density and compute corresponding HDRs.
The estimated density and HDRs are represented in a discrete form as a grid, defined by arguments \code{range} and \code{n}.
\code{get_hdr_1d} is used internally by layer functions \code{stat_hdr_rug()} and \code{stat_hdr_rug_fun()}.
}
\section{The \code{method} argument}{

The density estimator used to estimate the HDRs is specified with the \code{method} argument.
The simplest way to specify an estimator is to provide a character value to \code{method},
for example \code{method = "kde"} specifies a kernel density estimator.
However, this specification is limited to the default behavior of the estimator.

Instead, it is possible to provide a function call, for example: \code{method = method_kde_1d()}.
This is slightly different from the function calls provided in \code{get_hdr()}, note the \verb{_1d} suffix.
In many cases, these functions accept parameters governing the density estimation procedure.
Here, \code{method_kde_1d()} accepts several parameters related to the choice of kernel.
For details, see \code{?method_kde_1d}.
Every method of univariate density estimation implemented has such corresponding \verb{method_*_1d()} function,
each with an associated help page.

Note: \code{geom_hdr_rug()} and other layer functions also have \code{method} arguments which behave in the same way.
For more details on the use and implementation of the \verb{method_*_1d()} functions,
see \code{vignette("method", "ggdensity")}.
}

\section{The \code{fun} argument}{

If \code{method} is set to \code{"fun"}, \code{get_hdr_1d()} expects a univariate probability
density function to be specified with the \code{fun} argument.
It is required that \code{fun} be a function of at least one argument (\code{x}).
Beyond this first argument, \code{fun} can have arbitrarily many arguments;
these can be set in \code{get_hdr_1d()} as a named list via the \code{args} parameter.

Note: \code{get_hdr_1d()} requires that \code{fun} be vectorized in \code{x}.
For an example of an appropriate choice of \code{fun}, see the final example below.
}

\examples{
x <- rnorm(1e3)

# Two ways to specify `method`
get_hdr_1d(x, method = "kde")
get_hdr_1d(x, method = method_kde_1d())

\dontrun{

# If parenthesis are omitted, `get_hdr_1d()` errors
get_hdr_1d(df, method = method_kde_1d)

# If the `_1d` suffix is omitted, `get_hdr_1d()` errors
get_hdr_1d(x, method = method_kde())
}

# Adjust estimator parameters with arguments to `method_kde_1d()`
get_hdr_1d(x, method = method_kde_1d(kernel = "triangular"))

# Estimate different HDRs with `probs`
get_hdr_1d(x, method = method_kde_1d(), probs = c(.975, .6, .2))

# Compute "population" HDRs of specified univariate pdf with `method = "fun"`
f <- function(x, sd = 1) dnorm(x, sd = sd)
get_hdr_1d(method = "fun", fun = f, range = c(-5, 5))
get_hdr_1d(method = "fun", fun = f, range = c(-5, 5), args = list(sd = .5))


}
