\name{HHGramImage}
\alias{HHGramImage}
\title{Display Hilbert spectrogram}
\description{This function displays the Hilbert spectrogram of EMD and EEMD results.}
\usage{
HHGramImage(hgram, time.span = NULL, freq.span = NULL, amp.span = NULL, 
    clustergram = FALSE, cluster.span = NULL, imf.list = NULL, 
    fit.line = FALSE, scaling = "none", grid = TRUE, colorbar = TRUE, 
    backcol = c(0, 0, 0), colormap = NULL, pretty = FALSE, ...)
}
\arguments{
\item{hgram}{Data structure generated by \code{\link{HHRender}}.}
\item{time.span}{Time span to render spectrogram over.  \code{NULL} will draw the spectrogram over the entire signal.}
\item{freq.span}{Frequency span to render spectrogram over.  \code{NULL} plots everything up to the max frequency set when \code{\link{HHRender}} was run.}
\item{amp.span}{This is the amplitude span to plot, everything below is set to \code{backcol}, everything above is set to max color, \code{NULL} scales to the range in the signal.}
\item{clustergram}{If \code{TRUE}, plot the number of times data occupies a given pixel instead of plotting the signal amplitude.
This is akin to the \code{weight} component returned by the \code{as.image} function in the \code{fields} package.
Only applies when using EEMD.
Default is \code{FALSE}.}
\item{cluster.span}{Plots only parts of the signal that have a certain number of data points per pixel (see notes below).  
This only applies when using EEMD.  
The pixel range is defined as \code{c(AT LEAST, AT MOST)}.}
\item{imf.list}{A vector of IMFs to plot on the spectrogram, the others will not be shown.
You must set \code{combine.imfs = FALSE} in \code{\link{HHRender}} for this to work correctly.}
\item{fit.line}{If \code{TRUE}, plot a red line on the trace that shows the part of the signal represented by the spectrogram}.
\item{scaling}{determines whether to apply a logarithmic (\code{"log"}), or square root (\code{"sqrt"}) scaling to the amplitude data, default is \code{"none"}}
\item{grid}{Boolean - whether to display grid lines or not}
\item{colorbar}{Boolean - whether to display amplitude colorbar or not}
\item{backcol}{What background color to use behind the spectrogram, in a 3 element vector: \code{c(red, green, blue)}}
\item{colormap}{What palette object to use for the spectrogram, defaults to \code{rainbow}}
\item{pretty}{Boolean - to choose nice axes values or to use exactly the ranges given}
\item{\dots}{This function supports some optional parameters as well:
\itemize{
    \item{trace.format}{ - the format of the trace minima and maxima in sprintf format}
    \item{img.x.format}{ - the format of the X axis labels of the image in sprintf format}
    \item{img.y.format}{ - the format of the Y axis labels of the image in sprintf format}
    \item{colorbar.format}{ - the format of the colorbar labels in sprintf format}
    \item{cex.lab}{ - the font size of the image axis labels}
    \item{cex.colorbar}{ - the font size of the colorbar}
    \item{cex.trace}{ - the font size of the trace axis labels}
    \item{img.x.lab}{ - the X - axis label of the image, it defaults to "time"}
    \item{img.y.lab}{ - the Y - axis label of the image, it defaults to "frequency"}
    \item{main}{ - adds a title to the figure}}}
}

\details{
This function plots the image generated by \code{\link{HHRender}} along with the original signal trace.
The plotter can use data from both EMD and EEMD runs.
When it plots EEMD data, it shows the time frequency plot of every single trial at once.
The \code{cluster.span} option is useful in this case because it can distinguish ``signal'' (pixels where multiple trials intersect) from ``noise'' (whether from EEMD or from nature) where only one trial contributes data.
}

\value{
\item{img}{The spectrogram image, suitable for plotting with the generic \code{image} function}
}

\note{
Using the option \code{combine.imfs = FALSE} in \code{\link{HHRender}} will cause \code{HHGramImage} to run much, much slower.
Unless you have a compelling reason to plot certain IMFs (as opposed to all of them combined), I recommend against using this.

It may take some trial and error to get a nice image.
For example, if the data points are too small (and thus the spectrogram looks like a mist of fine points rather than continuous frequency bands), try rerunning \code{\link{HHRender}}, but with lower frequency resolution.
If the spectrogram is extremely noisy, try defining \code{cluster.span} - this usually gets rid of most of the random noise.
For example, a \code{cluster.span} of \code{c(3, 10)} only keeps pixels that have data from at least 3 trials, up to 10.  
Most noise pixels will only have one trial contributing data. 
The upper limit (10) is a formality - it does not make much sense at this point to put an upper limit on trial intersections unless you are interested in the \bold{noise} component isolated from the signal.
}

\author{Daniel Bowman \email{danny.c.bowman@gmail.com}}

\seealso{
\code{\link{FTGramImage}}, \code{\link{HHRender}}, \code{\link{HHSpecPlot}}
}
\examples{
data(PortFosterEvent)

trials <- 10
nimf <- 10
noise.amp <- 6.4e-07
trials.dir <- "test"

set.seed(628)
#Run EEMD (this may take some time)
\dontrun{EEMD(sig, tt, noise.amp, trials, nimf, trials.dir = trials.dir)}

#Compile the results
\dontrun{EEMD.result <- EEMDCompile(trials.dir, trials, nimf)}

#Calculate spectrogram
dt  <-  0.1
dfreq  <-  0.1
\dontrun{hgram <- HHRender(EEMD.result, dt, dfreq)}


#Plot spectrogram 
time.span <- c(4, 10)
freq.span <- c(0, 25)
\dontrun{HHGramImage(hgram, time.span, freq.span,  
pretty = TRUE, img.x.format = "\%.1f", img.y.format = "\%.0f", 
main = "Port Foster event - ensemble Hilbert spectrogram")}

#Plot intersections

\dontrun{HHGramImage(hgram, time.span, freq.span, amp.span = c(1, 5),  
clustergram = TRUE, pretty = TRUE, img.x.format = "\%.1f", colorbar.format = "\%.0f",
img.y.format = "\%.0f", main = "Port Foster event - signal stability")}

#Decluster
#show only areas with stable signal 
#i.e. each pixel has data from at least 3 EEMD trials
\dontrun{HHGramImage(hgram, time.span = time.span, freq.span = freq.span,
cluster.span = c(, 10), pretty = TRUE, img.x.format = "\%.1f", 
img.y.format = "\%.0f",
main = "Port Foster event - ensemble Hilbert spectrogram")}

#Log amplitude plot

\dontrun{HHGramImage(hgram, time.span = time.span, freq.span = freq.span,
scaling = "log", pretty = TRUE, img.x.format = "\%.1f", img.y.format = "\%.0f",
main = "Port Foster event - ensemble Hilbert spectrogram with log amplitude")}

#Log frequency plot
dfreq <- 0.001
\dontrun{hgram=HHRender(EEMD.result, dt, dfreq, scaling = "log")}
\dontrun{HHGramImage(hgram, time.span, freq.span = c(0, 2),          
pretty = TRUE, img.x.format = "\%.1f", img.y.format = "\%.2f",
img.y.lab = "log frequency",
main = "Port Foster event - ensemble Hilbert spectrogram with log frequency")}

#Only show IMF 1
dfreq <- 0.1
\dontrun{hgram=HHRender(EEMD.result, dt, dfreq, combine.imfs = FALSE)}
\dontrun{HHGramImage(hgram, time.span, freq.span, imf.list = 1,
pretty = TRUE, img.x.format = "\%.1f", img.y.format = "\%.0f",
main = "Port Foster event - ensemble Hilbert spectrogram of IMF 1")}
}
\keyword{ts}
\keyword{hplot}
