#' Mod. Sainte Lague formula
#'
#' @rdname ElecFuns
#'
#' @section Details: The modsaintelague function works with the same procedure as 
#' the dhondt function, but in this case the sequence of numbers used for the
#' division is only comprised by odd numbers except for the first one, which is 
#' 1.4 instead of 1. It ends up being: 1.4, 3, 5, 7 and so on. It uses an amount 
#' of numbers equal to m. 
#' @export
#'
#' @examples
#' ## Modified Sainte-Lague without threshold: 
#' 
#' modsaintelague(v=example, m=3)
#'  
#' ## Modified Sainte-Lague with 20% threshold: 
#' 
#' modsaintelague(v=example, m=3, threshold=0.2) 
#' 
modsaintelague  <- function(v,
                           m,
                           threshold = 0.0,
                           ...){
  if(nrow(v) > 1){
    stop("Modified Sainte-Lague undefined for ranked votes.")
  }
  if(threshold > max(v)/sum(v)){
    stop("Threshold is higher than maximum proportion of votes")
  }
  # apply threshold
  if(sum(prop.table(v) < threshold) != ncol(v)){
    v[1, prop.table(v) < threshold]  <- 0         
  }   
  divisors = 2 * 1:m - 1
  divisors[1] <- 1.4
  div_table <- outer(c(v), divisors, "/")
  hi_quot <- order(c(div_table), decreasing = TRUE)[1:m]
  seats <- table(factor(arrayInd(hi_quot, dim(div_table))[,1], levels = 1:ncol(v)))
  names(seats) <- colnames(v)
  return (seats)
}
