% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knn_forecast_boot_intervals.R
\name{knn.forecast.boot.intervals}
\alias{knn.forecast.boot.intervals}
\title{KNN Forecast Bootstrap Prediction Intervals}
\usage{
knn.forecast.boot.intervals(
  Sim.Mat.in,
  f.index.in,
  k.in,
  y.in,
  burn.in = NULL,
  B = 200,
  return.simulations = FALSE,
  level = 0.95
)
}
\arguments{
\item{Sim.Mat.in}{numeric and symmetric matrix of similarities (recommend use of \code{S_w}, see \code{SwMatrixCalc()}).}

\item{f.index.in}{numeric vector indicating the indices of \code{Sim.Mat.in} and \code{y.in} which correspond to the time order of the points to be forecast.}

\item{k.in}{integer value indicating the the number of nearest neighbors to be considered in forecasting, must be \code{>= 1}.}

\item{y.in}{numeric vector of the response series to be forecast.}

\item{burn.in}{integer value which indicates how many points at the start of the series to set aside as eligible neighbors before calculating forecast errors to be re-sampled.}

\item{B}{integer value representing the number of bootstrap replications, this will be the number of forecasts simulated and used to calculate outputs, must be \code{>= 1}.}

\item{return.simulations}{logical value indicating whether to return all simulated forecasts.}

\item{level}{numeric value over the range (0,1) indicating the confidence level for the prediction intervals.}
}
\value{
list of the following components:
\describe{
\item{lb}{numeric vector of the same length as \code{f.index.in}, with the estimated lower bound of the prediction interval.}
\item{ub}{numeric vector of the same length as \code{f.index.in}, with the estimated upper bound of the prediction interval.}
\item{mean}{numeric vector of the same length as \code{f.index.in}, with the mean of the \code{B} simulated paths for each forecasted point.}
\item{median}{numeric vector of the same length as \code{f.index.in}, with the median of the \code{B} simulated paths for each forecasted point.}
\item{simulated.paths}{numeric matrix where each of the \code{B} rows contains a simulated path for the points in \code{f.index.in}, only returned if \code{return.simulations = TRUE}.}
}
}
\description{
A function for forecasting using KNN regression with prediction intervals. The approach is based on the description of
"Prediction intervals from bootstrapped residuals" from chapter 5.5 of Hyndman R, Athanasopoulos G (2021) \url{https://otexts.com/fpp3/prediction-intervals.html#prediction-intervals-from-bootstrapped-residuals},
modified as needed for use with KNN regression. The algorithm starts by calculating a pool of forecast errors to later
sample from. If there are \code{n} points prior to the first observation indicated in \code{f.index.in} then there will be \code{n - k.in} errors generated by one-step ahead forecasts
starting with the point of the response series at the index \code{k.in + 1}. The first \code{k.in} points cannot be estimated because
a minimum of \code{k.in} eligible neighbors would be needed. The optional \code{burn.in} argument can be used to increase the number
of points from the start of the series that need to be available as neighbors before calculating errors for the pool. Next, \code{B}
possible paths the series could take are simulated using the pool of errors. Each path is simulated by calling \code{knn.forecast()}, estimating the first point in \code{f.index.in}, adding a sampled forecast error, then adding
this value to the end of the series. This process is then repeated for the next point in \code{f.index.in} until all have been estimated. The final output
interval estimates are calculated for each point in \code{f.index.in} by taking the appropriate percentiles of the corresponding simulations of that point.
The mean and medians are also calculated from these simulations. One important implication of this behavior is that the mean forecast output from this function can
differ from the point forecast produced by \code{knn.forecast()} alone.
}
\examples{
data("simulation_master_list")
series.index <- 15
ex.series <- simulation_master_list[[series.index]]$series.lin.coef.chng.x

# Weights pre tuned by random search. In alpha, beta, gamma order
pre.tuned.wts <- c(0.2148058, 0.2899638, 0.4952303)
pre.tuned.k <- 5

df <- data.frame(ex.series)
# Generate vector of time orders
df$t <- c(1:nrow(df))

# Generate vector of periods
nperiods <- simulation_master_list[[series.index]]$seasonal.periods
df$p <- rep(1:nperiods, length.out = nrow(df))

# Pull corresponding exogenous predictor(s)
X <- as.matrix(simulation_master_list[[series.index]]$x.chng)


# Calculate the weighted similarity matrix using Sw
Sw.ex <- SwMatrixCalc(
  t.in = df$t,
  p.in = df$p, nPeriods.in = nperiods,
  X.in = X,
  weights = pre.tuned.wts
)

n <- length(ex.series)
# Index we want to forecast
f.index <- c((n - 5 + 1):length(ex.series))

interval.forecast <- knn.forecast.boot.intervals(
  Sim.Mat.in = Sw.ex,
  f.index.in = f.index,
  y.in = ex.series,
  k.in = pre.tuned.k
)
}
\seealso{
\itemize{
\item \code{\link[=knn.forecast]{knn.forecast()}} for the function called to perform knn regression.
\item \code{\link[=SwMatrixCalc]{SwMatrixCalc()}} for the function to calculate a matrix with the recommended similarity measure.
\item Hyndman R, Athanasopoulos G (2021),"Forecasting: Principles and Practice, 3rd ed", Chapter 5.5, \url{https://otexts.com/fpp3/prediction-intervals.html#prediction-intervals-from-bootstrapped-residuals}.
For background on the algorithm this function is based on.
}
}
