% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lavaan.mi.R
\name{lavaan.mi}
\alias{lavaan.mi}
\alias{cfa.mi}
\alias{sem.mi}
\alias{growth.mi}
\title{Fit a lavaan Model to Multiple Imputed Data Sets}
\usage{
lavaan.mi(model, data, ...)

cfa.mi(model, data, ...)

sem.mi(model, data, ...)

growth.mi(model, data, ...)
}
\arguments{
\item{model}{The analysis model can be specified using lavaan
\code{\link[lavaan:model.syntax]{lavaan::model.syntax()}} or a parameter table (as generated by
\code{\link[lavaan:model.syntax]{lavaan::lavaanify()}} or returned by \code{\link[lavaan:parTable]{lavaan::parTable()}}).}

\item{data}{A a \code{list} of imputed data sets, or an object class from
which imputed data can be extracted. Recognized classes are
\code{lavaan.mi} (stored in the \verb{@DataList} slot),
\code{amelia} (created by the Amelia package), or
\code{mids} (created by the mice package).}

\item{\dots}{additional arguments to pass to \code{\link[lavaan:lavaan]{lavaan::lavaan()}} or
\code{\link[lavaan:lavaanList]{lavaan::lavaanList()}}. See also \code{\link[lavaan:lavOptions]{lavaan::lavOptions()}}.
Note that \code{lavaanList} provides parallel computing options, as well as
a \verb{FUN=} argument so the user can extract custom output after the model
is fitted to each imputed data set (see \strong{Examples}).  TIP: If a
custom \code{FUN} is used \emph{and} \code{parallel = "snow"} is requested,
the user-supplied function should explicitly call \code{library} or use
\code{\link[base]{::}} for any functions not part of the base distribution.}
}
\value{
A \linkS4class{lavaan.mi} object
}
\description{
This function fits a lavaan model to a list of imputed data sets.
}
\note{
This functionality was originally provided via \code{runMI()} in the
\code{semTools} package, but there are differences.  See the README file
on the GitHub page for this package (find link in DESCRIPTION).
}
\examples{
data(HS20imps) # import a list of 20 imputed data sets

## specify CFA model from lavaan's ?cfa help page
HS.model <- '
  visual  =~ x1 + x2 + x3
  textual =~ x4 + x5 + x6
  speed   =~ x7 + x8 + x9
'

## fit model to imputed data sets
fit <- cfa.mi(HS.model, data = HS20imps)

\donttest{
summary(fit, fit.measures = TRUE, fmi = TRUE)
summary(fit, standardized = "std.all", rsquare = TRUE)
}

## You can pass other lavaanList() arguments, such as FUN=, which allows
## you to save any custom output from each imputation's fitted model.

## An example with ordered-categorical data:
data(binHS5imps) # import a list of 5 imputed data sets

## Define a function to save a list with 2 custom outputs per imputation:
##  - zero-cell tables
##  - the obsolete "WRMR" fit index
myCustomFunc <- function(object) {
  list(wrmr      = lavaan::fitMeasures(object, "wrmr"),
       zeroCells = lavaan::lavInspect(object, "zero.cell.tables"))
}
## fit the model
catout <- cfa.mi(HS.model, data = binHS5imps, ordered = TRUE,
                 FUN = myCustomFunc)
## pooled results
\donttest{
summary(catout)
}

## extract custom output (per imputation)
sapply(catout@funList, function(x) x$wrmr) # WRMR for each imputation
catout@funList[[1]]$zeroCells # zero-cell tables for first imputation
catout@funList[[2]]$zeroCells # zero-cell tables for second imputation ...


}
\references{
Enders, C. K. (2010). \emph{Applied missing data analysis}.
New York, NY: Guilford.

Rubin, D. B. (1987). \emph{Multiple imputation for nonresponse in surveys}.
New York, NY: Wiley. \doi{10.1002/9780470316696}
}
\seealso{
\code{\link[=poolSat]{poolSat()}} for a more efficient method to obtain SEM results
for multiple imputations
}
\author{
Terrence D. Jorgensen (University of Amsterdam; \email{TJorgensen314@gmail.com})
}
