% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/match.R
\name{Match}
\alias{Match}
\title{Match Value Against Multiple Values}
\usage{
Match(x, ...)
}
\arguments{
\item{x}{object to match}

\item{...}{conditions used for matching, separated from the returned
value by \code{->}. Multiple conditions can be associated with the
same return value using \code{|}. Each matching statement must be
separated by a comma. See "Details" below. Use \code{.} to represent
the default *(else ...)* condition.}
}
\value{
an object based on the matched clause. An Error is produced if no
match is found.
}
\description{
Functional programming style matching using \code{->} to separate
conditions from associated return values.
}
\details{
Unlike \code{\link{switch}}, \code{Match} accepts a variety of
different condition statements. These can character, numeric,
or logical values, functions, symbols, language objects, enums,
etc. For example, \code{"hello" -> 1} tests whether the object is
equal to \code{"hello"}. If so, the function returns \code{1},
otherwise the next condition is tested. \code{<-} can also be used.
If so, the condition & return expression are reversed: \code{1 <- "hello"}
also tests \code{"hello"} and returns \code{1}.

Each condition is tested sequentially by calling the appropriate method
of \code{\link{match_cond}}. If the condition is a character value, then
\code{match_cond.character} is called, and so on. If a match is
confirmed, the right-hand side is evaluated and returned.

For atomic vectors - numeric, logical, or character - \code{Match} will
check for equality. All resulting values must be \code{TRUE} to match.
Lists and environments are checked using \code{identical}. If a
function is placed within the condition, then the function will be evaluated
on object \code{x}. If the result is logical and \code{TRUE}, then it is
considered a match. A non-logical result will be checked again using
\code{\link{match_cond}}. Failed function calls with an error are treated
as a non-match rather than stopping \code{Match}. Expressions are evaluated
similar to functions.

The period \code{.} is a special condition in \code{Match}. When alone,
it is treated as the "default" condition that always matches. When used
as a call, though, it matches values within object \code{x} and/or attaches
the individual items within \code{x} for use in the return expression.
For example, \code{x = c(1, 2)} will be matched with the condition
\code{.(1, second)}. This is because the first values are identical
\code{(1 == 1)}. Furthermore, \code{second = 2} for use in the return
expression. Preface a symbol with \code{..} to evaluate it and check
for equality. \code{...} can be used to denote any number of unspecified
objects.

The period call \code{.()} can also be used to test named member of \code{x},
though all objects in \code{.()} must be named to do so. For example, the
condition \code{.(a = 5, b=)} tests whether \code{x} contains \code{"a"}
with a value of \code{5} and \code{"b"} with any value.

If \code{function(...)} is used on the left hand side, then it may
need to be surrounded by parentheses for the parser to properly recognize
it. The \code{\%fn\%} infix function has be provided as syntactic sugar
for developing functions for matching.

Similar to many functional languages, \code{(first:rest)} can be used
as a condition to extract the first element and the rest from any
vector as long as the vector is sufficiently long. Variables used on the
left hand side can be called on the right hand side expression.

Matching an \code{\link{Enum}} causes symbols to represent possible
variants. For example, \code{None -> "none"} would try to match the
variant of \code{x} with \code{None}. If it succeeds, then \code{Match}
will return \code{"none"}. A function call on the left-hand side for an
Enum is treated as a variant and its inside arguments, which are made
available in the result expression. So, \code{Some(var) -> sqrt(var)} would
attempt to match on the variant \code{Some}. If it matches, then the
inside is exposed as the variable \code{var} for the right-hand side to
use. The number of objects in the variant on the left-hand side must
match the number of objects inside of \code{x} or else an error will
populate.

Regex conditions can be used when matching strings by surrounding the
expression in braces. For example, the condition "{[ab]*}" is equivalent
to using \code{grepl("\[ab\]*", ...)}. The braces must be the first and
last characters to trigger a regex match.

Call \code{\link{fallthrough}} within a return expression to stop evaluating
the expression and return to matching. This can be convenient for complex
matching conditions or to execute code for side-effects, such as printing.
}
\examples{
## Matching to functions, characters, regex, and default
Match(
 "abc",
 is.numeric       -> "Not a character!",
 is.character     -> {
   print("Found a character!")
   fallthrough()
 },
 "a" | "b" | "c"  -> "It's a letter!",
 "{bc}"           -> "Contains 'bc'!",
 .                -> "Can be anything!"
)


## Unwrapping a Result enum
val <- Result$Ok("hello world!")

Match(
  val,
  Ok(w)  -> w,
  Err(s) -> s
)

## Using functions
# If 'function' is used on the lhs, surround in '()'
# Alternatively, use \%fn\% notation
Match(
  1:10,
  (function(i) mean(i) < 5)  -> TRUE,
  i \%fn\% (mean(i) >= 5) -> FALSE
)

## Extracting parts
x <- list(a = 5, b = 6, c = 7)
Match(
  x,
  .(a=, d=2)  -> "won't match, no 'd'",
  .(a=5, b=)  -> "will match, a == '5'",
  (x:xs)      -> {
    print(x)  # 5
    print(xs) # list(b=6, c=7)
    "will match, since not empty"
  },
  .           -> "this matches anything!"
)

z     <- c(1,2,3,4)
first <- 1
Match(
  z,
  .(0, ...) -> "no match, first is 1 not 0",
  .(1, 2)   -> "no match, z has 4 elements",
  .(x, 2, ...) -> paste("match, x = ", x),
  .(..first, ...) -> "match, since 'first' == 1"
)


}
