% lwc-13-11-2006: First draft
% lwc-12-10-2007: major changes
% lwc-17-02-2010: Add fs.cl and fs.cl.1
%
\name{frankvali}
\alias{frankvali}
\alias{fs.cl}
\alias{fs.cl.1}
\alias{frankvali.formula}
\alias{frankvali.default}
\alias{print.frankvali}
\alias{summary.frankvali}
\alias{print.summary.frankvali}

\title{
  Estimates Feature Ranking Error Rate with Resampling
}

% ----------------------------------------------------------------------------
\description{
  Estimates error rate of feature ranking with resampling methods.
}

% ----------------------------------------------------------------------------
\usage{
frankvali(dat, \dots)
\method{frankvali}{default}(dat,cl,cl.method = "svm", fs.method="fs.auc",
          fs.order=NULL, fs.len="power2", pars = valipars(),
          tr.idx=NULL,\dots)

\method{frankvali}{formula}(formula, data = NULL, \dots, subset, na.action = na.omit)

fs.cl(dat,cl,fs.order=colnames(dat), fs.len=1:ncol(dat), 
      cl.method = "svm", pars = valipars(), all.fs=FALSE, \dots)
        
fs.cl.1(dat,cl,fs.order=colnames(dat), cl.method = "svm", 
        pars = valipars(), agg_f=FALSE,\dots)

}

% ----------------------------------------------------------------------------
\arguments{
  \item{formula}{
    A formula of the form \code{groups ~ x1 + x2 + \dots}  That is, the
    response is the grouping factor and the right hand side specifies
    the (non-factor) discriminators.
  }
  \item{data}{
    Data frame from which variables specified in \code{formula} are
    preferentially to be taken.
  }
  \item{dat}{
    A matrix or data frame containing the explanatory variables if no formula is
    given as the principal argument.
  }
  \item{cl}{
    A factor specifying the class for each observation if no formula principal 
    argument is given.
  }
  \item{cl.method}{
    Classification method to be used. Any classification methods can be employed 
    if they have method \code{predict} (except \code{knn}) with output of predicted class
    label or one component with name of \code{class} in the returned list, such as 
    \code{randomForest}, \code{svm}, \code{knn} and \code{lda}. 
  }
  \item{fs.method}{
    Feature ranking method to be used.  If \code{fs.order} is not \code{NULL}, it will be
    overridden. 
  }             
  \item{fs.order}{
   A vector of ordered feature order. In \code{frankvali} its default is \code{NULL} and then the 
   feature selection will be performed on the training data. 
  }
  \item{fs.len}{
    Feature length used for validation. For details, see \code{\link{get.fs.len}}. 
  }
  \item{pars}{
    A list of resampling scheme method such as \emph{Cross-validation},
	  \emph{Stratified cross-validation}, \emph{Leave-one-out cross-validation}, 
  	\emph{Randomised validation (holdout)}, \emph{Bootstrap}, \emph{.632 bootstrap}
  	and \emph{.632 plus bootstrap}, and control parameters for the calculation of accuracy. 
  	See \code{\link{valipars}} for details.
  }
  \item{tr.idx}{
  User defined index of training samples. Can be generated by \code{trainind}.
  }
  \item{all.fs}{ 
   A logical value indicating whether all features should be used for evaluation.
  }
  \item{agg_f}{ 
   A logical value indicating whether aggregated features should be used for 
   evaluation.
  }
  \item{\dots}{Additional parameters to \code{fs.method} or \code{cl.method}.}
  \item{subset}{
    Optional vector, specifying a subset of observations to be used.
  }
  \item{na.action}{
    Function which indicates what should happen when the data
    contains \code{NA}'s, defaults to \code{\link{na.omit}}.
  } 
}

% ----------------------------------------------------------------------------
\details{
  These functions validate the selected feature subsets by classification and resampling methods.  
  
  It can take any classification model if its argument format 
  is \code{model(formula, data, subset, ...)} and their corresponding 
  method \code{predict.model(object, newdata, ...)} can either return the only 
  predicted class label or in a list with name as \code{class}, such as 
  \code{lda} and \code{pcalda}.  
  
  The resampling method can be one of \code{cv}, \code{scv}, \code{loocv}, 
  \code{boot}, \code{632b} and \code{632pb}.
  
  The feature ranking method can take one of \code{fs.rf}, \code{fs.auc}, \code{fs.welch}, 
  \code{fs.anova}, \code{fs.bw}, \code{fs.snr}, \code{fs.kruskal}, \code{fs.relief} and \code{fs.rfe}. 

}

% ----------------------------------------------------------------------------
\value{
  \code{frankvali} returns an object of class including the components:
  \item{fs.method}{Feature ranking method used.}
  \item{cl.method}{Classification method used.}
  \item{fs.len}{Feature lengths used.}
  \item{fs.rank}{Final feature ranking. It is obtained based on \code{fs.list}
        by Borda vote method.}
  \item{err.all}{Error rate for all computation.}
  \item{err.iter}{Error rate for each iteration.}
  \item{err.avg}{Average error rate for all iterations.}
  \item{sampling}{Sampling scheme used.} 
  \item{niter}{Number of iterations.}
  \item{nboot}{
    Number of bootstrap replications if the sampling method is 
    one of \code{boot}, \code{632b} and \code{632pb}.
  }
  \item{nfold}{Fold of cross-validations if the sampling is \code{cv} or \code{scv}.}
  \item{nrand}{Number of replications if the sampling is \code{random}.}
  \item{fs.list}{Feature list of all computation if \code{fs.order} is \code{NULL}.}


	\code{fs.cl} and \code{fs.cl.1} return a matrix with columns of \code{acc} (accuracy), 
	\code{auc}(area under ROC curve) and \code{mar}(class margin).

}
% ----------------------------------------------------------------------------
\author{
  Wanchang Lin 
}

\seealso{
  \code{\link{feat.rank.re}}, \code{\link{frank.err}}, \code{\link{valipars}}, 
  \code{\link{boxplot.frankvali}}, \code{\link{get.fs.len}}
  
}

\note{
  \code{fs.cl} is the simplified version of \code{frankvali}. Both \code{frankvali}
  and \code{fs.cl} are used for validation of aggregated features from top to
  bottom only, but \code{fs.cl.1} can be used for validation of either individual 
  or aggregated features.
}

% ----------------------------------------------------------------------------
\examples{
data(abr1)
dat <- abr1$pos
x   <- preproc(dat[,110:500], method="log10")  
y   <- factor(abr1$fact$class)        

dat <- dat.sel(x, y, choices=c("1","2"))
x.1 <- dat[[1]]$dat
y.1 <- dat[[1]]$cls

len  <- c(1:20,seq(25,50,5),seq(60,90,10),seq(100,300,50))
pars <- valipars(sampling="boot",niter=2, nreps=4)
res  <- frankvali(x.1,y.1,cl.method = "knn", fs.method="fs.auc",
                  fs.len=len, pars = pars)
res
summary(res)
boxplot(res)  

\dontrun{
## or apply feature selection with re-sampling procedure at first
fs  <- feat.rank.re(x.1,y.1,method="fs.auc",pars = pars)

## then estimate error of feature selection.
res.1  <- frankvali(x.1,y.1,cl.method = "knn", fs.order=fs$fs.order,
                    fs.len=len, pars = pars)
res.1

## use formula
data.bin <- data.frame(y.1,x.1)

pars <- valipars(sampling="cv",niter=2,nreps=4)
res.2  <- frankvali(y.1~., data=data.bin,fs.method="fs.rfe",fs.len=len, 
                    cl.method = "knn",pars = pars)
res.2

## examples of fs.cl and fs.cl.1
fs <- fs.rf(x.1, y.1)
res.3 <- fs.cl(x.1,y.1,fs.order=fs$fs.order, fs.len=len,
               cl.method = "svm", pars = pars, all.fs=TRUE)

ord <- fs$fs.order[1:50]
## aggregated features
res.4 <- fs.cl.1(x.1,y.1,fs.order=ord, cl.method = "svm", pars = pars,
                 agg_f=TRUE)
               
## individual feature
res.5 <- fs.cl.1(x.1,y.1,fs.order=ord, cl.method = "svm", pars = pars,
                 agg_f=FALSE)
                 
}
}
\keyword{classif}
