#' Double logistic function for the time-series data.
#'
#' @description Apply a time-series model, double logistic function
#' , to the time-series trait data.
#' This function was originally developed for the time-series data of rice CIg data.
#' Fitting the time-series model is done by the "two-step procedure".
#' For more information, see Taniguchi et al. (under review).
#'
#' @param dat data.frame including date and trait (e.g. canopy height).
#' @param x Column name (character) for the date after sowing or planting.
#' @param y Column name (character) for the trait.
#' @param returnModels Logical value whether to return the time-series model object. Default is F.
#' @param start Start values to estimate 'r2', 'r3', 'd2' and 'd3'. Default is 'c(r2 = 0.05, r3 = 0.05, d2 = 40, d3 = 100)'.
#' @param upper Upper bounds to estimate 'r2', 'r3', 'd2' and 'd3'. Default is 'c(r2 = 1, r3 = 1, d2 = 200, d3 = 200)'.
#' @param lower Lower bounds to estimate 'r2', 'r3', 'd2' and 'd3'. Default is 'c(r2 = 0, r3 = 0, d2 = 0, d3 = 0)'.
#'
#' @return
#' doubleLogis function returns the vector of estimated parameter values.
#' If returnModels = TRUE, this function also returns the nls regression object.
#'
#' @examples
#' library(phenolocrop)
#' riceCIg_eg |>
#'    doubleLogis("x", "CIg")
#'
#' @references
#' S. Taniguchi et al. (2025) Phenology analysis for trait prediction using UAVs
#' in a MAGIC rice population with different transplanting protocols.
#'  Frontiers in Artificial Intelligence, 7, 1477637.
#'
#' @importFrom stats nls
#'
#' @export

doubleLogis <- function(dat, x, y,
                        returnModels = FALSE,
                        start = c(r2 = 0.05,
                                  r3 = 0.05,
                                  d2 = 40,
                                  d3 = 100),
                        upper = c(r2 = 1,
                                  r3 = 1,
                                  d2 = 200,
                                  d3 = 200),
                        lower = c(r2 = 0,
                                  r3 = 0,
                                  d2 = 0,
                                  d3 = 0)){

  x <- dat |> purrr::pluck(x)
  y <- dat |> purrr::pluck(y)
  dat_doublelogis <- data.frame(x = x, y = y)

  xmax <- x |> max()
  if(upper[3] < xmax){
    upper[3] <- xmax
  }

  ymax <- y |> max()
  ymin <- 0

  nls_res <- tryCatch(nls(y ~  ymin + ymax *
                             (1/(1 + exp(r2 * (d2 - x))) - (1/(1 + exp(r3 * (d3 - x))))),
                           data = dat_doublelogis,
                           algorithm = "port",
                           start = start,
                           upper = upper,
                           lower = lower),
                       error = function(e){
                         stop(message =
                                "Failed to apply nls. Modifying the start, upper, and lower values may solve the problem.")
                       })

  nls_res_summ <- summary(nls_res)
  param <- nls_res_summ$coefficients[, "Estimate"]
  param <- c(param, ymax = ymax)
  if(returnModels == FALSE){
    return(param)
  }else{
    return(list(param, nls_res))
  }
}
