% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/penalized.R
\name{penalized_est}
\alias{penalized_est}
\title{Penalized Parameter Estimation for Longitudinal CFA Models}
\usage{
penalized_est(
  x,
  w,
  pen_par_id = NULL,
  pen_diff_id = NULL,
  pen_fn = "l0a",
  pen_gr = NULL,
  se = "none",
  opt_control = list()
)
}
\arguments{
\item{x}{A fitted lavaan model object from which estimation components will
be extracted.}

\item{w}{Numeric scalar. Penalty weight (multiplier) applied to the penalty
terms.}

\item{pen_par_id}{Integer vector of parameter IDs to apply the penalty function
directly to, in the same order as returned by \code{lavaan::coef()} and by
\code{\link[lavaan:parTable]{lavaan::partable()}}, with only the free elements.}

\item{pen_diff_id}{List of matrices containing parameter IDs. For each matrix,
the penalty is applied to the pairwise differences of parameters in the same
column indicated by the IDs. For matrices with names starting with "loading",
the log transformation is applied before computing differences.}

\item{pen_fn}{A character string (\code{"l0a"} or \code{"alf"}) or a function that computes
the penalty. Default is \code{"l0a"}.}

\item{pen_gr}{A function that computes the gradient of the penalty function.
If \code{pen_fn} is \code{"l0a"} or \code{"alf"}, this is automatically set.}

\item{se}{Character string specifying the type of standard errors to compute.
Options are \code{"none"} (default; no standard errors) or \code{"robust.huber.white"}
(robust sandwich estimator using numerical Hessian and first-order information,
which is the same as used in the \code{"mlr"} estimator).}

\item{opt_control}{A list of control parameters passed to \code{\link[stats:nlminb]{stats::nlminb()}}.
Default includes \code{eval.max = 2e4}, \code{iter.max = 1e4}, and \code{abs.tol = 1e-20}.}
}
\value{
A lavaan model object updated with the penalized parameter estimates.
The returned object includes an attribute \code{opt_info} containing the
optimization information returned by \code{nlminb()}.
}
\description{
Performs penalized estimation on a lavaan model object by optimizing a
penalized objective function. The function extracts the objective function
from a lavaan model, applies a penalty function to specified parameters
or pairwise differences of parameters, and returns an updated model with
the optimized parameter estimates.
}
\details{
The function uses \code{nlminb()} to minimize a penalized objective function that
combines the standard lavaan objective function with a penalty term. Only the
parameter estimates and the log-likelihood should be interpreted. The
returned object was not "fitted" (\code{do.fit = FALSE}) to avoid users
interpreting the standard errors, which are generally not valid with
penalized estimation. The degrees of freedom may also be inaccurate. If the
optimization does not converge (convergence code != 0), a warning is issued.
}
\section{Warning}{

The returned object is not fitted using standard ML. Standard errors reported
by \code{summary()} or \code{parameterEstimates()} will be missing unless
\code{se = "robust.huber.white"} was specified. Even then, they are based on an
experimental sandwich approximation and should be interpreted with caution.
}

\examples{
library(lavaan)

# Define a longitudinal factor model with PoliticalDemocracy data
model <- "
  dem60 =~ y1 + y2 + y3 + y4
  dem65 =~ y5 + y6 + y7 + y8
  dem60 ~~ dem65
  dem60 ~~ 1 * dem60
  dem65 ~~ NA * dem65
  dem60 ~ 0
  dem65 ~ NA * 1
  y1 ~~ y5
  y2 ~~ y6
  y3 ~~ y7
  y4 ~~ y8
"

# Fit the model without constraints first to get parameter table
fit_un <- cfa(model, data = PoliticalDemocracy, std.lv = TRUE,
              meanstructure = TRUE, do.fit = FALSE)

# Get parameter IDs
pt <- parTable(fit_un)
# Loadings
load_60 <- pt$free[pt$op == "=~" & pt$lhs == "dem60"]
load_65 <- pt$free[pt$op == "=~" & pt$lhs == "dem65"]
# Intercepts
int_60 <- pt$free[pt$op == "~1" & pt$lhs \%in\% c("y1", "y2", "y3", "y4")]
int_65 <- pt$free[pt$op == "~1" & pt$lhs \%in\% c("y5", "y6", "y7", "y8")]

# Apply penalized estimation to penalize differences in loadings and intercepts
pen_fit <- penalized_est(
    x = fit_un,
    w = 0.03,
    pen_diff_id = list(
        loadings = rbind(load_60, load_65),
        intercepts = rbind(int_60, int_65)
    ),
    pen_fn = "l0a"
)

# Compare parameter estimates
summary(pen_fit)

}
\seealso{
\code{\link[lavaan]{lavaan}}, \code{\link[stats]{nlminb}}
}
