% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hsp_compat.R
\name{hsp_compat}
\alias{hsp_compat}
\alias{hsp_read_manual_input}
\alias{hsp_read_opt_sky_coef}
\alias{hsp_write_sky_points}
\alias{hsp_write_sun_coord}
\title{HSP compatibility functions}
\usage{
hsp_read_manual_input(path_to_HSP_project, img_name)

hsp_read_opt_sky_coef(path_to_HSP_project, img_name)

hsp_write_sky_points(sky_points, path_to_HSP_project, img_name)

hsp_write_sun_coord(sun_row_col, path_to_HSP_project, img_name)
}
\arguments{
\item{path_to_HSP_project}{character vector of length one. Path to the HSP
project folder (e.g., \code{"C:/Users/johndoe/Documents/HSP/projects/my_prj/"}).}

\item{img_name}{character vector of length one (e.g., \code{"DSCN6342.pgm"} or
\code{"DSCN6342"}). See \emph{About HSP software}.}

\item{sky_points}{\code{data.frame} with columns \code{row} and \code{col}.}

\item{sun_row_col}{numeric vector of length two. Raster coordinates (row,
column) of the solar disk.}
}
\value{
See \emph{Functions}
}
\description{
Read and write legacy files from HSP (HemiSPherical Project Manager) projects
to interoperate with existing workflows. Intended for legacy support; not
required when working fully within \code{rcaiman}.
}
\section{About HSP software}{


HSP (introduced in \insertCite{Lang2013}{rcaiman}, based on the method in
\insertCite{Lang2010}{rcaiman}) runs exclusively on Windows. HSP stores
pre-processed images as PGM files in the \code{manipulate} subfolder of each
project (itself inside the \code{projects} folder).
}

\section{Functions}{


\describe{
\item{\code{hsp_read_manual_input()}}{read sky marks and sun position
defined manually within an HSP project; returns a named list with
components \code{weight}, \code{max_points}, \code{angle},
\code{point_radius}, \code{sun_row_col}, \code{sky_points}, and
\code{zenith_dn}.}

\item{\code{hsp_read_opt_sky_coef()}}{read optimized CIE sky coefficients
from an HSP project; returns a numeric vector of length five.}

\item{\code{hsp_write_sky_points()}}{write a file with sky point
coordinates compatible with HSP; creates a file on disk.}

\item{\code{hsp_write_sun_coord()}}{write a file with solar disk
coordinates compatible with HSP; creates a file on disk.}
}
}

\examples{
\dontrun{
# NOTE: assumes the working directory is the HSP project folder (e.g., an RStudio project).

# From HSP to R in order to compare ---------------------------------------
r <- read_caim("manipulate/IMG_1013.pgm")
z <- zenith_image(ncol(r), lens())
a <- azimuth_image(z)
manual_input <- read_manual_input(".", "IMG_1013")
sun_row_col <- manual_input$sun_row_col
sun_angles <- zenith_azimuth_from_row_col(
  z, a,
  sun_row_col[1],
  sun_row_col[2]
)
sun_angles <- as.vector(sun_angles)

sky_points <- manual_input$sky_points
rr <- extract_rr(r, z, a, sky_points)
model <- fit_cie_model(rr, sun_angles)
sky <- cie_image(
  z, a,
  model$sun_angles,
  model$coef
) * model$rr$zenith_dn
plot(r / sky)

r <- read_caim("manipulate/IMG_1013.pgm")
sky_coef <- read_opt_sky_coef(".", "IMG_1013")
sky_m <- cie_image(z, a, sun_angles, sky_coef)
sky_m <- cie_sky_manual * manual_input$zenith_dn
plot(r / sky_m)

# From R to HSP  ----------------------------------------------------------
r <- read_caim("manipulate/IMG_1014.pgm")
z <- zenith_image(ncol(r), lens())
a <- azimuth_image(z)
m <- !is.na(z)
g <- sky_grid_segmentation(z, a, 10)
bin <- binarize_with_thr(caim$Blue, thr_isodata(caim$Blue[m]))
bin <- select_sky_region(z, 0, 85) & bin

sun_angles <- estimate_sun_angles(r, z, a, bin, g)
sun_row_col <- row_col_from_zenith_azimuth(
  z, a,
  sun_angles["z"],
  sun_angles["a"]
) \%>\% as.numeric()
write_sun_coord(sun_row_col, ".", "IMG_1014")

sky_points <- extract_sky_points(r, bin, g)
write_sky_points(sky_points, ".", "IMG_1014")
}
}
\references{
\insertAllCited{}
}
