% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_caim.R
\name{read_caim}
\alias{read_caim}
\title{Read a canopy image from a file}
\usage{
read_caim(path = NULL, upper_left = NULL, width = NULL, height = NULL)
}
\arguments{
\item{path}{character vector of length one.  Path to an image file, including
extension. If \code{NULL}, an example image is returned.}

\item{upper_left}{numeric vector of length two. Pixel coordinates of the
upper-left corner of the ROI, in the format \code{c(column, row)}.}

\item{width, height}{numeric vector of length one. Size (in pixels) of the
rectangular ROI to read.}
}
\value{
Numeric \link[terra:SpatRaster-class]{terra::SpatRaster}, typically with layers named \code{"Red"},
\code{"Green"}, and \code{"Blue"}. If the file format or metadata prevents automatic
layer naming, names will be inferred and a warning may be issued.
}
\description{
Reads a born-digital image (typically RGB-JPEG or RGB-TIFF) using
\code{\link[terra:rast]{terra::rast()}} and returns a \link[terra:SpatRaster-class]{terra::SpatRaster} object. Optionally, it can
extract a rectangular region of interest (ROI) specified by the user.
}
\details{
This function is intended for importing color hemispherical photographs, such
as those obtained with digital cameras equipped with fisheye lenses. For raw
image files (e.g., NEF, CR2), see \code{\link[=read_caim_raw]{read_caim_raw()}}.

Internally, this is a wrapper around \code{\link[terra:rast]{terra::rast()}}, so support for image
formats depends on the capabilities of the \code{terra} package.

If no arguments are provided, a sample image will be returned.
}
\note{
The example image was created from a raw photograph taken with a Nikon Coolpix
5700 and a FC-E9 auxiliary lens, processed with the following code:

\if{html}{\out{<div class="sourceCode">}}\preformatted{zenith_colrow <- c(1290, 988)/2
diameter <- 756
z <- zenith_image(diameter, lens("Nikon_FCE9"))
a <- azimuth_image(z)
m <- !is.na(z)
caim <- read_caim_raw("DSCN4606.NEF")
caim <- crop_caim(caim, zenith_colrow - diameter/2, diameter, diameter)
caim <- correct_vignetting(caim, z, c(0.0638, -0.101))
caim <- c(mean(caim$Y, caim$M), caim$G, caim$C)
caim <- fisheye_to_equidistant(caim, z, a, m, radius = 300, k = 1)
write_caim(caim, "example.tif", 16)
}\if{html}{\out{</div>}}
}
\section{Selecting a Region of Interest}{

To load a specific subregion from the image, use the arguments \code{upper_left},
\code{width}, and \code{height}. These are expressed in raster coordinates, similar to
a spreadsheet layout: \strong{columns first, then rows}. In other words, specify
coordinates as \code{c(column, row)}, \strong{not} \code{c(row, column)}, which is typical
in \code{data.frame} objects.

While any image editor can be used to obtain these values, this function was
tested with \href{https://imagej.net/ij/}{ImageJ}, particularly the Fiji
distribution. A recommended workflow:
\enumerate{
\item Open the image in Fiji.
\item Draw a rectangular selection.
\item Go to \emph{Edit > Selection > Specify...} to read \code{upper_left}, \code{width}, and \code{height}.
}
}

\examples{
path <- system.file("external/DSCN4500.JPG", package = "rcaiman")
zenith_colrow <- c(1276, 980)
diameter <- 756*2
caim <- read_caim(path, zenith_colrow - diameter/2, diameter, diameter)
plot(caim$Blue)
}
\seealso{
\code{\link[=write_caim]{write_caim()}}
}
