% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/remulateActor.R
\name{remulateActor}
\alias{remulateActor}
\title{Simulate Relational Event Data - Actor-Oriented model}
\usage{
remulateActor(
  rateEffects,
  choiceEffects,
  actors,
  endTime,
  events = NULL,
  startTime = 0,
  initial = 0,
  riskset = NULL,
  memory = c("full", "window", "window_m", "decay"),
  memoryParam = NULL
)
}
\arguments{
\item{rateEffects}{A \code{formula} object specifying the statistics used to 
simulate the network under the actor rate sub-model.}

\item{choiceEffects}{A \code{formula} object specifying the statistics used to 
simulate the network under the receiver choice sub-model.}

\item{actors}{A numeric or character vector representing the actor names.}

\item{endTime}{A numeric value specifying the end time up to which the 
network should be simulated.}

\item{events}{[Optional] An integer specifying the maximum number of events 
to simulate.}

\item{startTime}{[Optional] A numeric value (default = 0) indicating the time 
at which the simulation should start.}

\item{initial}{[Optional] A numeric or \code{data.frame} object (default = 0) 
specifying how to initialize the network.  
- If an integer is provided, it represents the number of random events to 
sample before beginning data generation.  
- If a \code{data.frame} is provided with columns (time, sender, receiver), 
it serves as an edgelist of initial events, after which subsequent events 
are predicted.}

\item{riskset}{[Optional] A \code{matrix} with columns (sender, receiver) 
defining a custom risk set.}

\item{memory}{[Optional] A string (default = "full") specifying the memory 
type used for computing statistics.  
- `"full"`: Uses the entire event history.  
- `"window"`: Considers only events occurring within a specified time window.  
- `"window_m"`: Considers only a specified number of most recent events.  
- `"decay"`: Applies an exponential decay, where older events contribute 
less based on elapsed time.}

\item{memoryParam}{[Optional] A numeric value (> 0) defining the memory 
parameter based on the selected memory type:  
- `"window"`: Length of the time window.  
- `"window_m"`: Number of past events to consider.  
- `"decay"`: Half-life (i.e., time until an event's weight is reduced to half).}
}
\value{
An object of class \code{"remulateActor"}. A data.frame containing the simulated event sequence with columns (time, sender, receiver).
The \code{"remulateActor"} object has the following attributes::
\describe{
  \item{evls}{A \code{matrix} containing the event list with columns (dyad, time), 
  where \code{dyad} represents the index of the (sender, receiver) pair in the risk set.}
  \item{rateStatistics}{An array of dimensions \code{M x N x P}, where:  
  - \code{M} is the number of events,  
  - \code{N} is the number of actors,  
  - \code{P} is the number of sender rate statistics.}
  \item{choiceStatistics}{An array of dimensions \code{M x D x Q}, where:  
  - \code{M} is the number of events,  
  - \code{D} is the number of dyads in the risk set,  
  - \code{Q} is the number of receiver choice statistics.}
  \item{rateParams}{A named list of rate model parameters corresponding to the 
  specified rate statistics.}
  \item{choiceParams}{A named list of choice model parameters corresponding to 
  the specified choice statistics.}
  \item{riskset}{A \code{matrix} with columns (sender, receiver) representing 
  the risk set used in the simulation.}
  \item{actors}{A \code{data.frame} mapping the actor names provided by the user 
  to the integer IDs used in internal computations.}
#'   \item{initial}{A A numeric or \code{data.frame} object representing the network initialization, 
  which can be a number of random initialization events or
  a \code{data.frame} specifying pre-existing ties.}
  \item{rateEffects}{A \code{formula} object specifying the effects included in the rate sub-model.}
  \item{choiceEffects}{A \code{formula} object specifying the effects included in the choice sub-model.}
  \item{density}{A numeric value indicating the density of the generated network, 
  defined as the number of observed ties divided by \code{N*(N-1)}, where 
  \code{N} is the number of actors.}
}
}
\description{
A function to simulate relational event data by sampling from an
actor-oriented relational event model.
}
\details{
#' If time is irrelevant and only a specific number of events are desired, set time to Inf. 
If both time and events are supplied then the function 
stops simulating whenever the first stop condition is met

A list of available statistics for actor rate model. 
See \link{remulateActorEffects} for details on effects: 
\itemize{
 \item \code{baseline()}
 \item \code{indegreeSender()}
 \item \code{outdegreeSender()}
 \item \code{totaldegreeSender()}
 \item \code{ospSender()}
 \item \code{otpSender()}
 \item \code{send()}
 \item \code{interact()}
}

A list of available statistics for receiver choice model. 
See \link{remulateActorEffects} for details on effects: :
\itemize{
 \item \code{inertia()}
 \item \code{reciprocity()}
 \item \code{indegreeReceiver()}
 \item \code{outdegreeReceiver()}
 \item \code{totaldegreeReceiver()}
 \item \code{otp()}
 \item \code{itp()}
 \item \code{osp()}
 \item \code{isp()}
 \item \code{psABBA()}
 \item \code{psABBY()}
 \item \code{psABXA()}
 \item \code{psABXB()}
 \item \code{psABXY()}
 \item \code{psABAY()}
 \item \code{recencyContinue()}
 \item \code{recencySendReceiver()}
 \item \code{recencyReceiveReceiver()} 
 \item \code{rrankReceive()} 
 \item \code{receive()}
 \item \code{dyad()}
 \item \code{same()}
 \item \code{average()}
 \item \code{difference()}
 \item \code{minimum()}
 \item \code{maximum()}
 \item \code{interact()}
}
}
\examples{
 # To generate events up to time '50' in a network of 25 actors with 
 # 200 random initial events
 
 # Exogenous attributes data.frame
 cov <- data.frame(
   id   = 1:25, 
   time = rep(0, 25), 
   sex  = sample(c(0,1), 25, replace = TRUE, prob = c(0.4, 0.6)), 
   age  = sample(20:30, 25, replace = TRUE) 
 )
 
 # Effects specification
 rateform <- ~ remulate::baseline(-6) + 
              remulate::indegreeSender(0.01) + 
              remulate::send(0.02, variable = "age", attr_actors = cov) + 
              remulate::interact(0.01, indices = c(2, 3))
 
 choiceform <- ~ remulate::inertia(0.01) + 
               remulate::reciprocity(-0.03) + 
               remulate::interact(0.01, indices = c(2, 1))
 
 # Calling remulateActor
 remulate::remulateActor(
   rateform, 
   choiceform, 
   actors  = 1:25, 
   endTime = 100, 
   initial = 200, 
   events  = 500, 
 )
  
 # To predict events, given an edgelist of initial events
 initialREH <- data.frame(
   time = seq(0.5, 100, 0.5), 
   sender = sample(1:25, 200, TRUE), 
   receiver = sample(1:25, 200, TRUE)
 )
 
 remulate::remulateActor(
   rateform, 
   choiceform, 
   actors  = 1:25, 
   endTime = 200, 
   initial = initialREH, 
   events  = 500
 )
}
\references{
Lakdawala, R., Mulder, J., & Leenders, R. (2025).
*Simulating Relational Event Histories: Why and How*.
arXiv:2403.19329.
}
