% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comp_accu.R
\name{comp_accu_freq}
\alias{comp_accu_freq}
\title{Compute accuracy metrics of current classification results.}
\usage{
comp_accu_freq(hi = freq$hi, mi = freq$mi, fa = freq$fa, cr = freq$cr, w = 0.5)
}
\arguments{
\item{hi}{The number of hits \code{\link{hi}} (or true positives).}

\item{mi}{The number of misses \code{\link{mi}} (or false negatives).}

\item{fa}{The number of false alarms \code{\link{fa}} (or false positives).}

\item{cr}{The number of correct rejections \code{\link{cr}} (or true negatives).}

\item{w}{The weighting parameter \code{w} (from 0 to 1)
for computing weighted accuracy \code{wacc}.
Default: \code{w = .50} (i.e., yielding balanced accuracy \code{bacc}).}
}
\value{
A list \code{\link{accu}} containing current accuracy metrics.
}
\description{
\code{comp_accu_freq} computes a list of current accuracy metrics
from the 4 essential frequencies (\code{\link{hi}},
\code{\link{mi}}, \code{\link{fa}}, \code{\link{cr}})
that constitute the current confusion matrix and
are contained in \code{\link{freq}}.
}
\details{
Currently computed accuracy metrics include:

\enumerate{

   \item \code{acc}: Overall accuracy as the proportion (or probability)
   of correctly classifying cases or of \code{\link{dec_cor}} cases:

   \code{acc = dec_cor/N = (hi + cr)/(hi + mi + fa + cr)}

   Values range from 0 (no correct prediction) to 1 (perfect prediction).

   \item \code{wacc}: Weighted accuracy, as a weighted average of the
   sensitivity \code{\link{sens}} (aka. hit rate \code{\link{HR}}, \code{\link{TPR}},
   \code{\link{power}} or \code{\link{recall}})
   and the the specificity \code{\link{spec}} (aka. \code{\link{TNR}})
   in which \code{\link{sens}} is multiplied by a weighting parameter \code{w}
   (ranging from 0 to 1) and \code{\link{spec}} is multiplied by
   \code{w}'s complement \code{(1 - w)}:

   \code{wacc = (w * sens) + ((1 - w) * spec)}

   If \code{w = .50}, \code{wacc} becomes \emph{balanced} accuracy \code{bacc}.

   \item \code{mcc}: The Matthews correlation coefficient (with values ranging from -1 to +1):

   \code{mcc = ((hi * cr) - (fa * mi)) / sqrt((hi + fa) * (hi + mi) * (cr + fa) * (cr + mi))}

   A value of \code{mcc = 0} implies random performance; \code{mcc = 1} implies perfect performance.

   See \href{https://en.wikipedia.org/wiki/Matthews_correlation_coefficient}{Wikipedia: Matthews correlation coefficient}
   for additional information.

   \item \code{f1s}: The harmonic mean of the positive predictive value \code{\link{PPV}}
   (aka. \code{\link{precision}})
   and the sensitivity \code{\link{sens}} (aka. hit rate \code{\link{HR}},
   \code{\link{TPR}}, \code{\link{power}} or \code{\link{recall}}):

   \code{f1s =  2 * (PPV * sens) / (PPV + sens)}

   See \href{https://en.wikipedia.org/wiki/F1_score}{Wikipedia: F1 score} for additional information.

}

Notes:

\itemize{

   \item Accuracy metrics describe the \emph{correspondence} of decisions (or predictions) to actual conditions (or truth).

   There are several possible interpretations of accuracy:

   \enumerate{

     \item as \emph{probabilities} (i.e., \code{acc} being the proportion of correct classifications,
     or the ratio \code{\link{dec_cor}}/\code{\link{N}}),

     \item as \emph{frequencies} (e.g., as classifying a population of \code{\link{N}}
     individuals into cases of \code{\link{dec_cor}} vs. \code{\link{dec_err}}),

     \item as \emph{correlations} (e.g., see \code{mcc} in \code{\link{accu}}).

   }

   \item Computing exact accuracy values based on probabilities (by \code{\link{comp_accu_prob}}) may differ from
   accuracy values computed from (possibly rounded) frequencies (by \code{\link{comp_accu_freq}}).

   When frequencies are rounded to integers (see the default of \code{round = TRUE}
   in \code{\link{comp_freq}} and \code{\link{comp_freq_prob}}) the accuracy metrics computed by
   \code{comp_accu_freq} correspond to these rounded values.
   Use \code{\link{comp_accu_prob}} to obtain exact accuracy metrics from probabilities.

   }
}
\examples{
comp_accu_freq()  # => accuracy metrics for freq of current scenario
comp_accu_freq(hi = 1, mi = 2, fa = 3, cr = 4)  # medium accuracy, but cr > hi

# Extreme cases:
comp_accu_freq(hi = 1, mi = 1, fa = 1, cr = 1)  # random performance
comp_accu_freq(hi = 0, mi = 0, fa = 1, cr = 1)  # random performance: wacc and f1s are NaN
comp_accu_freq(hi = 1, mi = 0, fa = 0, cr = 1)  # perfect accuracy/optimal performance
comp_accu_freq(hi = 0, mi = 1, fa = 1, cr = 0)  # zero accuracy/worst performance, but see f1s
comp_accu_freq(hi = 1, mi = 0, fa = 0, cr = 0)  # perfect accuracy, but see wacc and mcc

# Effects of w:
comp_accu_freq(hi = 3, mi = 2, fa = 1, cr = 4, w = 1/2)  # equal weights to sens and spec
comp_accu_freq(hi = 3, mi = 2, fa = 1, cr = 4, w = 2/3)  # more weight to sens
comp_accu_freq(hi = 3, mi = 2, fa = 1, cr = 4, w = 1/3)  # more weight to spec

## Contrasting comp_accu_freq and comp_accu_prob:
# (a) comp_accu_freq (based on rounded frequencies):
freq1 <- comp_freq(N = 10, prev = 1/3, sens = 2/3, spec = 3/4)   # => hi = 2, mi = 1, fa = 2, cr = 5
accu1 <- comp_accu_freq(freq1$hi, freq1$mi, freq1$fa, freq1$cr)  # => accu1 (based on rounded freq).
# accu1
#
# (b) comp_accu_prob (based on probabilities):
accu2 <- comp_accu_prob(prev = 1/3, sens = 2/3, spec = 3/4)      # => exact accu (based on prob).
# accu2
all.equal(accu1, accu2)  # => 4 differences!
#
# (c) comp_accu_freq (exact values, i.e., without rounding):
freq3 <- comp_freq(N = 10, prev = 1/3, sens = 2/3, spec = 3/4, round = FALSE)
accu3 <- comp_accu_freq(freq3$hi, freq3$mi, freq3$fa, freq3$cr)  # => accu3 (based on EXACT freq).
# accu3
all.equal(accu2, accu3)  # => TRUE (qed).


}
\references{
Consult \href{https://en.wikipedia.org/wiki/Confusion_matrix}{Wikipedia: Confusion matrix}
for additional information.
}
\seealso{
\code{\link{accu}} for all accuracy metrics;
\code{\link{comp_accu_prob}} computes exact accuracy metrics from probabilities;
\code{\link{num}} for basic numeric parameters;
\code{\link{freq}} for current frequency information;
\code{\link{txt}} for current text settings;
\code{\link{pal}} for current color settings;
\code{\link{popu}} for a table of the current population.

Other metrics: 
\code{\link{acc}},
\code{\link{accu}},
\code{\link{comp_acc}()},
\code{\link{comp_accu_prob}()},
\code{\link{comp_err}()},
\code{\link{err}}

Other functions computing probabilities: 
\code{\link{comp_FDR}()},
\code{\link{comp_FOR}()},
\code{\link{comp_NPV}()},
\code{\link{comp_PPV}()},
\code{\link{comp_acc}()},
\code{\link{comp_accu_prob}()},
\code{\link{comp_comp_pair}()},
\code{\link{comp_complement}()},
\code{\link{comp_complete_prob_set}()},
\code{\link{comp_err}()},
\code{\link{comp_fart}()},
\code{\link{comp_mirt}()},
\code{\link{comp_ppod}()},
\code{\link{comp_prob}()},
\code{\link{comp_prob_freq}()},
\code{\link{comp_sens}()},
\code{\link{comp_spec}()}
}
\concept{functions computing probabilities}
\concept{metrics}
