% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.R
\name{rrda.cv}
\alias{rrda.cv}
\title{Cross-validation for Ridge Redundancy Analysis}
\usage{
rrda.cv(
  Y,
  X,
  maxrank = NULL,
  lambda = NULL,
  num.lambda = 50,
  nfold = 5,
  folds = NULL,
  sample.X = 1000,
  sample.Y = 1000,
  scale.X = FALSE,
  scale.Y = FALSE,
  center.X = TRUE,
  center.Y = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{Y}{A numeric matrix of response variables.}

\item{X}{A numeric matrix of explanatory variables.}

\item{maxrank}{A numeric vector specifying the maximum rank of the coefficient Bhat. Default is \code{NULL}, which sets it to \code{(min(15, min(dim(X), dim(Y))))}.}

\item{lambda}{A numeric vector of ridge penalty values. Default is \code{NULL}, where the lambda values are automatically chosen.}

\item{num.lambda}{A number of lambda generated (only when the lambda is not given by user). Default is 50.}

\item{nfold}{The number of folds for cross-validation. Default is 5.}

\item{folds}{A vector specifying the folds. Default is \code{NULL}, which randomly assigns folds.}

\item{sample.X}{A number of variables sampled from X for the lamdba range estimate. Default is 1000.}

\item{sample.Y}{A number of variables sampled from Y for the lamdba range estimate. Default is 1000.}

\item{scale.X}{Logical indicating if \code{X} should be scaled. If \code{TRUE}, scales \code{X}. Default is \code{FALSE}.}

\item{scale.Y}{Logical indicating if \code{Y} should be scaled. If \code{TRUE}, scales \code{Y}. Default is \code{FALSE}.}

\item{center.X}{Logical indicating if \code{X} should be centered. If \code{TRUE}, scales \code{X}. Default is \code{TRUE}.}

\item{center.Y}{Logical indicating if \code{Y} should be centered. If \code{TRUE}, scales \code{Y}. Default is \code{TRUE}.}

\item{verbose}{Logical indicating. If \code{TRUE}, the function displays information about the function call. Default is \code{TRUE}.}
}
\value{
A list containing the cross-validated MSE matrix, lambda values, rank values, and the optimal lambda and rank.
}
\description{
This function performs cross-validation to evaluate the performance of Ridge Redundancy Analysis (RDA) models. It calculates the mean squared error (MSE) for different ranks and ridge penalty values through cross-validation folds. The function also supports centering and scaling of the input matrices.

The range of lambda for the cross-validation is automatically calculated following the method of "glmnet" (Friedman et al., 2010). When we have a matrix of response variables (Y; n times q matrix) and a matrix of explanatory variables (X; n times p matrix), the largest lambda for the validation is obtained as follows

\deqn{ \lambda_{\text{max}} = \frac{\max_{j \in \{1, 2, \dots, p\}} \sqrt{\sum_{k=1}^{q} \left( \sum_{i=1}^{n}  (x_{ij}\cdot y_{ik})  \right)^2}}{N \times 10^{-3}}}

Then, we define \eqn{\lambda_{min}=10^{-4}\lambda_{max}}, and the sequence \eqn{\lambda} is generated based on the range.

Also, to reduce the computation, the variable sampling is performed for the large matrix of X and Y (by default, when the number of the variables is over 1000). Alternatively, the range of lambda can be specified manually.
}
\examples{
\dontrun{
set.seed(10)
simdata<-rdasim1(n = 100,p = 200,q = 200,k = 3)
X <- simdata$X
Y <- simdata$Y
cv_result<- rrda.cv(Y = Y, X = X, maxrank = 5, nfold = 5)
rrda.summary(cv_result = cv_result)

##Complete Example##



# library(future) # <- if you want to compute in parallel

# plan(multisession) # <- if you want to compute in parallel
# cv_result<- rrda.cv(Y = Y, X = X, maxrank = 5, nfold = 5) # cv
# plan(multisession) # <- To come back to sequential computing

# rrda.summary(cv_result = cv_result) # cv result

p <- rrda.plot(cv_result) # cv result plot
print(p)
h <- rrda.heatmap(cv_result) # cv result heatmao
print(h)

estimated_lambda<-cv_result$opt_min$lambda  # selected parameter
estimated_rank<-cv_result$opt_min$rank # selected parameter

Bhat <- rrda.fit(Y = Y, X = X, nrank = estimated_rank,lambda = estimated_lambda) # fitting
Bhat_mat<-rrda.coef(Bhat)
Yhat_mat <- rrda.predict(Bhat = Bhat, X = X) # prediction
Yhat<-Yhat_mat[[1]][[1]][[1]] # predicted values

cor_Y_Yhat<-diag(cor(Y,Yhat)) # correlation
summary(cor_Y_Yhat)
}
}
