% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf_mediation.R
\name{semPower.powerMediation}
\alias{semPower.powerMediation}
\title{semPower.powerMediation}
\usage{
semPower.powerMediation(
  type,
  comparison = "restricted",
  bYX = NULL,
  bMX = NULL,
  bYM = NULL,
  Beta = NULL,
  indirect = NULL,
  estimateDirectEffects = TRUE,
  nullEffect = "ind = 0",
  nullWhichGroups = NULL,
  standardized = TRUE,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{bYX}{the slope (direct effect) for X -> Y. A list for multiple group models. Can be \code{NULL} if \code{Beta} is set.}

\item{bMX}{the slope for X -> M. A list for multiple group models. Can be \code{NULL} if \code{Beta} is set.}

\item{bYM}{the slope for M -> Y. A list for multiple group models. Can be \code{NULL} if \code{Beta} is set.}

\item{Beta}{can be used instead of \code{bYX}, \code{bMX}, and \code{bYM}: matrix of regression weights connecting the latent factors (all-Y notation). Exogenous variables must be in the first row(s), so the upper triangular of Beta must be zero. A list for multiple group models.}

\item{indirect}{\code{NULL} unless \code{Beta} is set. Otherwise a list of vectors of size 2 indicating the elements of \code{Beta} that define the indirect effect of interest, e.g. \code{list(c(2, 1), c(3, 2))}. See details.}

\item{estimateDirectEffects}{Whether to estimate all direct effects (\code{TRUE}, the default). If \code{FALSE}, only direct effects unequal zero in the population are estimated.}

\item{nullEffect}{defines the hypothesis of interest, must be one of \code{'ind = 0'} (the default) to test whether the indirect effect is zero or \code{'indA = indB'} to test for the equality of indirect effects across groups. See details.}

\item{nullWhichGroups}{for \code{nullEffect = 'indA = indB'}, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{standardized}{whether all parameters should be standardized (\code{TRUE}, the default). If \code{FALSE}, all regression relations are unstandardized.}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. In case of a simple mediation, the order of factors is X, M, Y. See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analysis concerning indirect effect(s) in a mediation model.
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising
in the context of mediation:
\itemize{
\item \code{nullEffect = 'ind = 0'}: Tests the hypothesis that an indirect effect is zero.
\item \code{nullEffect = 'indA = indB'}: Tests the hypothesis that an indirect effect is equal in two or more groups. This is currently only possible for models without latent variables.
}

The indirect effect of interest can be specified in two ways:
\itemize{
\item If a simple mediation involving three variables of the form \code{X -> M -> Y} is assumed, the arguments
\code{bYX}, \code{bMX}, and \code{bYM} are used to define the respective slopes, e. g.  \code{bYX = .4}, \code{bMX = .5}, and \code{bYM = .3} translates to
\verb{X -- .5 --> M -- .3 --> Y} and  \verb{X -- .4 --> Y}.
\item More complex mediation structures can be defined by providing the \code{Beta} matrix along with \code{indirect} specifying which paths define the indirect effect. See examples below. To specify residual correlations, use \code{\link[=semPower.powerPath]{semPower.powerPath()}} in conjunction with \code{\link[=semPower.powerLav]{semPower.powerLav()}}.
}

Notes on implementation:
\itemize{
\item For models without latent variables, \code{nullEffect = 'ind = 0'} and \code{nullEffect = 'indA = indB'} constrain the indirect effect to zero and to equality, respectively, yielding the test described in Tofighi & Kelley (2020).
\item For models with latent variables and \code{nullEffect = 'ind = 0'}, power is (sometimes roughly) approximated by constraining the smallest slope contained in the indirect effect to zero.
\item For models with latent variables multiple groups (i. e., \code{nullEffect = 'indA = indB'}), there is currently no way to determine power.
}

Tofighi, D., & Kelley, K. (2020). Improved inference in mediation analysis: Introducing the model-based constrained optimization procedure. \emph{Psychological Methods, 25(4)}, 496–515. https://doi.org/10.1037/met0000259

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of factors).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} need to be defined.
If the model contains observed variables only, use \code{Lambda = diag(x)} where \code{x} is the number of variables.

Note that in case of a simple mediation model involving three variables, the order of the factors is X, M, Y, i. e., the first factor is treated as X, the second as M, and the third as Y. In case of a more complex mediation defined via the \code{Beta} matrix, the order of factors matches the order of \code{Beta}.

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.
}
\examples{
\dontrun{
# simple case of X -> M -> Y mediation in the form of
# X -- .30 --> M -- .40 --> Y
# X --------- .25 --------> Y
# determine the required N to detect the indirect effect of >= .12 (= .3 * .4) 
# with a power of 95\% on alpha = 5\%, where   
# X is measured by 3 indicators loading by .5 each, 
# M is measured by 5 indicators loading by .6 each, 
# Y is measured by 4 indicators loading by .7 each.
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05,
                                    bYX = .25, bMX = .3, bYM = .4,
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7)
                                    )
# show summary
summary(powerMed)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerMed$modelH1, sample.cov = powerMed$Sigma,
sample.nobs = powerMed$requiredN, sample.cov.rescale = FALSE)
lavaan::sem(powerMed$modelH0, sample.cov = powerMed$Sigma,
sample.nobs = powerMed$requiredN, sample.cov.rescale = FALSE)

# same as above, but determine power with N = 500 on alpha = .05
powerMed <- semPower.powerMediation(type = 'post-hoc',
                                    alpha = .05, N = 500,
                                    bYX = .25, bMX = .3, bYM = .4,
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7))


# same as above, but determine the critical chi-square with N = 500 so that alpha = beta
powerMed <- semPower.powerMediation(type = 'compromise',
                                    abratio = 1, N = 500
                                    bYX = .25, bMX = .3, bYM = .4,
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7)
                                    )


# same as above, but compare to the saturated model
# (rather than to the less restricted model)
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05,
                                    comparison = 'saturated',
                                    bYX = .25, bMX = .3, bYM = .4,
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7)
                                    )
                                    

# same as above, but assuming observed variables only (Lambda = diag(3))
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05,
                                    bYX = .25, bMX = .3, bYM = .4,
                                    Lambda = diag(3)
                                    )
                                    


# same mediation model as above, but specifying Beta and indirect
Beta <- matrix(c(
  #   X    M    Y
  c(.00, .00, .00),    # X
  c(.30, .00, .00),    # M
  c(.25, .40, .00)     # Y
), byrow = TRUE, ncol = 3)
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05
                                    Beta = Beta, 
                                    indirect = list(c(2, 1), c(3, 2)),
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7)
                                    )

# Beta for a more complex mediation hypothesis
# of the form X -- .2 --> M1 -- .3 --> M2 -- .40 -> Y 
# (and all other effects being zero)
# using a reduced loading matrix to define that
# X is measured by 3 indicators loading by .4, .5, .8 
# M1 is measured by 4 indicators loading by .7, .6, .5, .8
# M2 is measured by 5 indicators loading by .5, .6, .3, .4, .6 
# Y is measured by 4 indicators loading by .6, .7, .8
Beta <- matrix(c(
  c(.00, .00, .00, .00),       # X
  c(.20, .00, .00, .00),       # M1
  c(.00, .30, .00, .00),       # M2
  c(.00, .00, .40, .00)        # Y
), byrow = TRUE, ncol = 4)
loadings <- list(
  c(0.4, 0.5, 0.8),           # X
  c(0.7, 0.6, 0.5, 0.8),      # M1
  c(0.5, 0.6, 0.3, 0.4, 0.6), # M2
  c(0.6, 0.7, 0.8)            # Y
)
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05,
                                    Beta = Beta, 
                                    indirect = list(c(2, 1), 
                                                    c(3, 2), 
                                                    c(4, 3)),
                                    loadings = loadings)
                                    

# Determine required N to detect that the indirect effect 
# in group 1 (of .2 * .3 = .09) differs from the indirect effect 
# in group 2 (of .3 * .5 = .15).
# The direct effect of X on Y is .25 in both groups.  
# The model is based on observed variables only (Lambda = diag(3))
# Both groups are sized equally (N = list(1, 1)).
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05, N = list(1, 1),
                                    nullEffect = 'indA = indB',
                                    bYX = list(.25, .25), 
                                    bMX = list(.2, .3), 
                                    bYM = list(.3, .5),
                                    Lambda = diag(3)
                                    )

# same as above, but specifying Beta 
Beta1 <- matrix(c(
  c(.00, .00, .00),    # X
  c(.20, .00, .00),    # M
  c(.25, .30, .00)     # Y
), byrow = TRUE, ncol = 3)
Beta2 <- matrix(c(
  c(.00, .00, .00),    # X
  c(.30, .00, .00),    # M
  c(.25, .50, .00)     # Y
), byrow = TRUE, ncol = 3)
powerMed <- semPower.powerMediation(type = 'a-priori',
                                    alpha = .05, beta = .05, N = list(1, 1),
                                    nullEffect = 'indA = indB',
                                    Beta = list(Beta1, Beta2), 
                                    indirect = list(c(2, 1), c(3, 2)),
                                    Lambda = diag(3)
                                    )

# request a simulated post-hoc power analysis with 500 replications.
set.seed(300121)
powerMed <- semPower.powerMediation(type = 'post-hoc',
                                    alpha = .05, N = 500,
                                    bYX = .25, bMX = .3, bYM = .4,
                                    nIndicator = c(3, 5, 4),
                                    loadM = c(.5, .6, .7),
                                    simulatedPower = TRUE, 
                                    simOptions = list(nReplications = 500))
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
